/*
 * Copyright (C) 2004  Stefan Kleine Stegemann
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <Foundation/NSException.h>
#include <Foundation/NSString.h>
#include <Foundation/NSDictionary.h>
#include <Foundation/NSNotification.h>
#include <AppKit/NSAttributedString.h>
#include <AppKit/NSBezierPath.h>
#include <AppKit/NSColor.h>
#include <AppKit/NSClipView.h>
#include "OutlinePanel.h"

/* Initialize constants.  */
NSString* N_OutlineItemSelected  = @"N_OutlineItemSelected";
NSString* UserInfoKeyOutlineItem = @"UserInfoKeyOutlineItem";


/* ----------------------------------------------------- */
/*  Interface NoOutlineView                              */
/* ----------------------------------------------------- */

@interface NoOutlineView : NSView
{
   NSAttributedString* text;
}

+ (NoOutlineView*) sharedInstance;

@end


/* ----------------------------------------------------- */
/*  Interface OutlineState                               */
/* ----------------------------------------------------- */

@interface OutlineState : NSObject
{
   @public
   NSPoint scrollState;
   int     selectedRow;
}
@end


/* ----------------------------------------------------- */
/*  Implementation OutlinePanel                          */
/* ----------------------------------------------------- */

/**
 * Non-Public methods.
 */
@interface OutlinePanel (Private)
- (void) _setNoOutlineView;
- (void) _setRootItems: (NSArray*)items;
- (void) _saveOutlineState;
- (void) _restoreOutlineState;
- (void) _outlineItemSelected: (id)sender;
@end


@implementation OutlinePanel

- (id) initWithContentRect: (NSRect)contentRect
                 styleMask: (unsigned int)styleMask
                   backing: (NSBackingStoreType)backingType
                     defer: (BOOL)flag
{
   if ((self = [super initWithContentRect: contentRect
                                styleMask: styleMask
                                  backing: backingType
                                    defer: flag]))
   {
      scrollView         = nil;
      outlineView        = nil;
      currentOutlineName = nil;
      rootItems          = nil;
      outlineStates      = [[NSMutableDictionary alloc] init];
      displaysOutline   = NO;
   }
   
   return self;
}


- (void) dealloc
{
   RELEASE(outlineView);
   RELEASE(currentOutlineName);
   RELEASE(outlineStates);

   [self _setRootItems: nil];

   [super dealloc];
}


- (void) showOutlineForDocument: (PDFDocument*)aDocument
                      usingName: (NSString*)aName
{
   NSAssert(aDocument, @"no document");
   NSAssert(aName, @"must specify a name for the document's outline");

   if ([aName isEqualToString: currentOutlineName])
   {
      NSLog(@"outline did not change");
      return;
   }

   [self _saveOutlineState];
   
   if ([outlineView selectedRow] >= 0)
   {
      [outlineView deselectRow: [outlineView selectedRow]];
   }

   RELEASE(currentOutlineName);
   currentOutlineName = [aName copy];

   if ([aDocument hasOutline])
   {
      if (!displaysOutline)
      {
         [scrollView setDocumentView: outlineView];
      }

      [self _setRootItems: [[aDocument outline] items]];
      displaysOutline = YES;
   }
   else
   {
      [self _setNoOutlineView];
   }

   [outlineView reloadData];
   [[self contentView] setNeedsDisplay: YES];

   [self _restoreOutlineState];
}


- (void) outlineWillClose: (NSString*)anOutlineName
{
   // if the outline that will be closed is the
   // outline that is displayed, remove the outline
   // from panel
   if ([anOutlineName isEqualToString: currentOutlineName])
   {
      [self _saveOutlineState];

      RELEASE(currentOutlineName);
      currentOutlineName = nil;
      [self _setNoOutlineView];
   }

   [outlineStates removeObjectForKey: anOutlineName];
}


- (NSOutlineView*) outlineView
{
   return outlineView;
}


- (void) awakeFromNib
{
   RETAIN(outlineView);

   [outlineView setDoubleAction: @selector(_outlineItemSelected:)];

   [self _setNoOutlineView];
}

@end


/* ----------------------------------------------------- */
/*  Category NSOutlineViewDataSource                     */
/* ----------------------------------------------------- */

@implementation OutlinePanel (NSOutlineViewDataSource)

- (int) outlineView: (NSOutlineView *)outlineView numberOfChildrenOfItem: (id)item
{
   if (!item)
   {
      return [rootItems count];
   }
   
   return [(PDFOutlineItem*)item countKids];
}


- (id) outlineView: (NSOutlineView *)outlineView child: (int)index ofItem: (id)item
{
   NSArray* items;

   if (!item)
   {
      items = rootItems;
   }
   else
   {
      items = [(PDFOutlineItem*)item kids];
   }

   return [items objectAtIndex: index];
}


- (BOOL) outlineView: (NSOutlineView *)outlineView isItemExpandable: (id)item
{
   return ([(PDFOutlineItem*)item countKids] > 0);
}


- (id) outlineView: (NSOutlineView *)outlineView 
objectValueForTableColumn: (NSTableColumn *)tableColumn
            byItem:(id)item
{
   return [(PDFOutlineItem*)item title];
}

@end


/* ----------------------------------------------------- */
/*  Category Private                                     */
/* ----------------------------------------------------- */

@implementation OutlinePanel (Private)

- (void) _setNoOutlineView
{
   NSSize contentSize;
   
   contentSize = [scrollView contentSize];
   [[NoOutlineView sharedInstance] setFrame: 
         NSMakeRect(0, 0, contentSize.width, contentSize.height)];

   [scrollView setDocumentView: [NoOutlineView sharedInstance]];

   displaysOutline = NO;
   [self _setRootItems: nil];
}


- (void) _setRootItems: (NSArray*)items
{
   RELEASE(rootItems);
   rootItems = RETAIN(items);
}


/** Save the state of the current outline.  */
- (void) _saveOutlineState
{
   OutlineState* theState;
   NSRect        vRect;

   if ((!currentOutlineName) || (!displaysOutline))
   {
      return;
   }

   theState = [outlineStates objectForKey: currentOutlineName];
   if (!theState)
   {
      theState = [[OutlineState alloc] init];
      [outlineStates setObject: theState forKey: currentOutlineName];
      RELEASE(theState);
   }

   vRect = [[scrollView contentView] visibleRect];
   theState->scrollState = NSMakePoint(NSMinX(vRect), NSMinY(vRect));

   theState->selectedRow = [outlineView selectedRow];
}


/** Restore the state for the current outline.  */
- (void) _restoreOutlineState
{
   OutlineState* theState;

   if ((!currentOutlineName) || (!displaysOutline))
   {
      return;
   }

   theState = [outlineStates objectForKey: currentOutlineName];
   if (theState)
   {
      [[scrollView contentView] scrollToPoint: theState->scrollState];
   
      if (theState->selectedRow >= 0)
      {
         [outlineView selectRow: theState->selectedRow byExtendingSelection: NO];
      }
   }
   else
   {
      [[scrollView contentView] scrollToPoint: NSMakePoint(0, 0)];
   }
}

- (void) _outlineItemSelected: (id)sender
{
   id             outlineItem;
   NSDictionary*  userInfoDict;

   if ([sender selectedRow] == -1)
   {
      return;
   }

   outlineItem = [sender itemAtRow: [sender selectedRow]];
   NSAssert(outlineItem, @"selected item is nil");

   userInfoDict = [NSDictionary dictionaryWithObject: outlineItem
                                              forKey: UserInfoKeyOutlineItem];

   [[NSNotificationCenter defaultCenter]
      postNotificationName: N_OutlineItemSelected
                    object: self
                  userInfo: userInfoDict];
}

@end


/* ----------------------------------------------------- */
/*  Implementation NoOutlineView                         */
/* ----------------------------------------------------- */

@implementation NoOutlineView

- (id) initWithFrame: (NSRect)aFrame
{
   if ((self = [super initWithFrame: aFrame]))
   {
      NSMutableDictionary* textAttrs;

      [self setAutoresizingMask: (NSViewWidthSizable | NSViewHeightSizable)];

      textAttrs = [[NSMutableDictionary alloc] init];
      [textAttrs setObject: [NSFont labelFontOfSize: 18.0] forKey: NSFontAttributeName];
      [textAttrs setObject: [NSColor blackColor] forKey: NSForegroundColorAttributeName];

      text = [[NSAttributedString alloc] initWithString: @"no outline available"
                                             attributes: textAttrs];

      RELEASE(textAttrs);
   }

   return self;
}


- (void) dealloc
{
   RELEASE(text);
   [super dealloc];
}


+ (NoOutlineView*) sharedInstance
{
   static NoOutlineView* sharedNoOutlineView = nil;

   if (!sharedNoOutlineView)
   {
      sharedNoOutlineView = [[NoOutlineView alloc] initWithFrame: NSMakeRect(0, 0, 10, 10)];
   }

   return sharedNoOutlineView;
}

- (BOOL) isOpaque
{
   return YES;
}


- (void) drawRect: (NSRect)aRect;
{
   NSPoint textOrigin;

   // fill background
   [[NSColor windowBackgroundColor] set];
   [NSBezierPath fillRect: NSMakeRect(0, 0, NSWidth([self frame]), NSHeight([self frame]))];

   // draw text in the view's center
   textOrigin = NSMakePoint((NSWidth([self frame]) / 2) - ([text size].width / 2),
                            (NSHeight([self frame]) / 2) - ([text size].height / 2));
   [text drawAtPoint: textOrigin];
}

@end


/* ----------------------------------------------------- */
/*  Interface OutlineState                               */
/* ----------------------------------------------------- */

@implementation OutlineState

- (id) init
{
   if ((self = [super init]))
   {
      scrollState = NSZeroPoint;
   }

   return self;
}


- (void) dealloc
{
   NSLog(@"dealloc OutlineState");
   [super dealloc];
}

@end
