/* -*- C++ -*-

  This file is part of ViPEC
  Copyright (C) 1991-2000 Johan Rossouw (jrossouw@alcatel.altech.co.za)

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU Library General Public License as
  published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU Library General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

#ifndef GRAPHVIEW_H
#define GRAPHVIEW_H

#include <Types.h>
#include <DataVector.h>

#include <qdialog.h>
#include <qpainter.h>
#include <qprinter.h>
#include <qstring.h>
#include <qlist.h>
#include <qpoint.h>
#include <math.h>
#include <qfont.h>
#include <qmap.h>

class QMenuBar;
class DrawingFrame;
class TableFrame;

typedef QMap<QString,DataVector> SeriesMap;

class GraphView : public QDialog
{
  Q_OBJECT
public:
  enum GraphViewType { unknown, gridView, smithView, tableView };

  GraphView( GraphViewType viewType, const char * name=0, WFlags f=0 );
  virtual ~GraphView();

  void setName(const char* value);
  const QString& getName() const;
  
  void setTitle(const char* value);
  const QString& getTitle() const;
  GraphViewType getViewType() const;

  void addSeries(const QString& title, uint size);
  void addData(const QString& title, TReal x, TReal y, uint size);
  void emptySeries();
  bool isEmpty();

  virtual void draw(QPainter* p) = 0;
  virtual void drawCursors(QPainter* p) = 0;

  virtual void mouseMoveEvent( QMouseEvent* );
  virtual void mousePressEvent( QMouseEvent* );
  
public slots:
  void reDraw();
  void print();
  virtual void save();
  
signals:
  void fontChanged();
  
protected:
  
  QFont getTitleFont(QPaintDevice* p);
  QFont getLabelFont(QPaintDevice* p);
  QMenuBar& menuBar();

  virtual void print( QPainter * ) = 0;
  virtual void drawCursors(QPainter* p, QRect textRect);
  
  virtual QPoint  toClient( TReal xValue, TReal yValue ) = 0;
  virtual TComplex fromClient( int x, int y ) = 0;
  virtual TComplex transformCoordinates( TComplex coord ) = 0;
  
  virtual void setDefaults();
  virtual void computeScaling(QPainter *) = 0;
  virtual bool isInsideView(const QPoint& p) = 0;
  
  void setCursorLabel(QString label);
  bool showMarkers();
  
private slots:
  void toggleMarkers();
  void setTitleFont();
  void setLabelFont();
  
private:
  void createMenus();
  void setFont(QFont& font);
  double getFontScale(QPaintDevice* p);

protected:
  TReal scale_;
  DrawingFrame* frame_;
  TableFrame* table_;
  GraphViewType viewType_;
  SeriesMap seriesMap_;
  
private:
  static QPrinter* printer_;
  
  bool showMarkers_;
  
  QString name_;
  QString viewTitle_;
  QString cursorLabel_;
  QMenuBar* menuBar_;
  
  static QFont titleFont_;
  static QFont labelFont_;
  
  //Store for cursor position
  bool     showCursor_ [2];
  TComplex clickedPos_ [2];
  QPoint   currentPos_;
    
};

#endif

