/* Copyright (C) 1999 Rafal Wierzbicki <rafal@mcss.mcmaster.ca>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


static char *cvsident="$Id: wallp.c,v 1.4 1999/03/25 23:04:59 rafal Exp $";

#include "wallp.h"

#include <sys/stat.h>

static char save_file[128];
static int QUIT = 0;

enum image_style STYLE = NONE;
Display *dpy;
GtkWidget *test_area = NULL;
int PREVIEW_ONLY = 0;
int AUTO_START = 0;
char init_entry[7];
GSList *simple_list=NULL;
int RANDOM=0;
int WHARF=0;


static void parse_opts (int argc, char *argv[]);
static int parse_control(char *control_file);
static int read_image_list(char *list_file);
static void f_save_file(void);
static void check_prefs_dir(void);

void simple_list_add (char *data)
{
    char *tmp;

    tmp = (char *)malloc (255);
    strcpy (tmp, data);
    simple_list = g_slist_append (simple_list, tmp);
}

void simple_list_remove (char *data)
{
    simple_list = g_slist_remove (simple_list, data);
}

void simple_list_clear ()
{
    g_slist_free (simple_list);
}

char *style_string(void) {

    switch(STYLE) {
    case TILE:
        return "TILE";
    case ASPECT:
        return "ASPECT";
    case CENTER:
        return "CENTER";
    case SCALE:
        return "SCALE";
    default:
        return "unknown";
    }
}

int parse_style(char *string) {

    if( strcmp(string, "TILE") == 0 )
        return TILE;
    else if( strcmp(string, "CENTER") == 0 )
        return CENTER;
    else if( strcmp(string, "SCALE") == 0 )
        return SCALE;
    else if( strcmp(string, "ASPECT") == 0  || strcmp(string, "STRETCH") == 0)
        return ASPECT;
    else
        return NONE;
}

static void
check_prefs_dir(void)
{
    struct stat st;
    char dir[128];

    snprintf(dir, 127, "%s%s", getenv("HOME"), "/.wallp");
    if(stat(dir, &st) < 0)
        mkdir(dir, S_IRUSR | S_IWUSR | S_IXUSR);
}

static void
parse_opts (int argc, char *argv[])
{
    int i;

    for (i = 0; i < argc; i++) {
        if (strcmp (argv[i], "-style") == 0) {
            if (argv[i + 1]) {
                if (strcmp (argv[i+1], "center") == 0) {
                    STYLE = CENTER;
                    i = i + 1;
                }
                else if (strcmp (argv[i+1], "tile") == 0) {
                    STYLE = TILE;
                    i = i + 1;
                }
                else if (strcmp (argv[i+1], "scale") == 0) {
                    STYLE = SCALE;
                    i = i + 1;
                }
                else if (strcmp (argv[i+1], "aspect") == 0) {
                    STYLE = ASPECT;
                    i = i + 1;
                }
                else {
                    fprintf (stderr, "Style unknown, you can choose tile, scale, aspect, or center\n");
                    exit (-1);
                }
            }
            else {
                fprintf (stderr, "style requires an option, see -help\n");
                exit (-1);
            }
        }
        else if (strcmp (argv[i], "-start") == 0)
            AUTO_START = 1;
        else if (strcmp (argv[i], "-quit") == 0) {
            if (argv[i+1]) {
                QUIT = i + 1;
                i = i + 1;
            }
            else {
                fprintf (stderr, "need filename for -quit\n");
                exit (-1);
            }
        }
        else if (strcmp (argv[i], "-preview") == 0)
        {
            PREVIEW_ONLY = 1;
        }
        else if (strcmp (argv[i], "-random") == 0)
        {
            RANDOM = 1;
        }
        else if (strcmp (argv[i], "-version") == 0)
        {
            printf ("WallP %s\n%s\n", VERSION, cvsident);
            exit (0);
        }
        else if (strcmp (argv[i], "--version") == 0)
        {
            printf ("WallP %s\n%s\n", VERSION, cvsident);
            exit (0);
        }
        else if (strcmp (argv[i], "-wharf") == 0)
        {
            WHARF = 1;
        }
        else if (strcmp (argv[i], "-help")==0) {
            fprintf (stderr, "Options: -preview, -start, -quit <image>, -style [<center> <tile> <scale> <aspect>]\n");
            exit (-1);
        }
        else {
            if (i > 0) {
                fprintf (stderr, "unknown option %s, see -help\n", argv[i]);
                exit (-1);
            }
        }
    }
}

static int
parse_control(char *control_file) {

    FILE *control;
    char tmp[128];

    if( (control = fopen(control_file, "r")) == NULL ) {
        fprintf(stderr, "%s not found, will create one later\n", control_file);
        return 0;
    }

    memset(tmp, '\0', 128);
    memset (init_entry, '\0', 7);

    if(fscanf(control, "%s", tmp)) { /* interval */
        snprintf (init_entry, 6, "%s", tmp);
#ifdef VERBOSE
        printf("%s\n", tmp);
#endif
    }
    else {
        fprintf(stderr, "Error reading control file\n");
        return 0;
    }

    if(fscanf(control, "%s", tmp)) { /* style */
        STYLE = parse_style(tmp);
#ifdef VERBOSE
        printf("%s\n", style_string());
#endif
    }
    else {
        fprintf(stderr, "Error reading control file\n");
        return 0;
    }

    fclose(control);
    return 1;
}

static int
read_image_list(char *list_file) {

    FILE *list;
    char tmp[128];
    int count = 0; /* how many lines read */

    if( (list = fopen(list_file, "r")) == NULL ) {
        fprintf(stderr, "Could not read list from %s\n", list_file);
        return 0;
    }

    while (1) {
        fgets (tmp, 127, list);
        tmp[strlen(tmp)-1] = '\0';
        if (!feof(list)) {
            if (RANDOM || WHARF)
                simple_list_add (tmp);
            else
                list_add (tmp);
            ++count;
        }
        else
            break;
    }

    fclose(list);

    return count;
}

static void
f_save_file (void)
{
    if (!sprintf (save_file, "%s%s", getenv("HOME"), "/.wallp/list"))
        fprintf (stderr, "Could not save the image list! ~/.wallp/list\n");
}

void write_control_file (void)
{
    FILE *control;
    char tmp[128];

    snprintf(tmp, 127, "%s%s", getenv("HOME"), "/.wallp/control");

    if( (control = fopen(tmp, "w")) == NULL ) {
        check_prefs_dir(); /* maybe the dir does not exist */
        if( (control = fopen(tmp, "w")) == NULL ) {
            fprintf(stderr, "Could not write to %s\n", tmp);
        }
    }

#ifdef VERBOSE
    printf("%s\n", gtk_entry_get_text(entry));
    printf("%s\n", style_string());
#endif
    fprintf(control, "%s\n", gtk_entry_get_text(entry));
    fprintf(control, "%s\n", style_string());
    fclose(control);
}

void dump_list_to_file (void)
{
    char *item_data[1];
    int i;
    FILE *fp;

    if (!(fp = fopen (save_file, "w"))){
        check_prefs_dir(); /* maybe the dir did not exist */
        if (!(fp = fopen (save_file, "w"))){
            fprintf (stderr, "Could not save the image list in %s\n",
                     save_file);
            return;
        }
    }

    for (i = 0; i < no_of_list_items; i++) {
        gtk_clist_get_text (GTK_CLIST(list), i, 0, item_data);
        fprintf (fp, "%s\n", item_data[0]);
    }
    fclose (fp);
}

int
main (int argc, char *argv[])
{
    char tmp[128];
    int result;

    srandom (time (NULL));

    parse_opts (argc, argv);
    gtk_init (&argc, &argv);
    gtk_rc_init();

    snprintf(tmp, 127, "%s%s", getenv("HOME"), "/.wallp/control");

    if (STYLE == 0)
        STYLE = TILE;

    result = parse_control (tmp);

#ifdef VERBOSE
    printf ("%d => %s\n", result, tmp);
#endif

    if (QUIT)
    {
        init_wallp ();
        if (put_image_on_root (argv[QUIT], STYLE)) {
            return 0;
        }
        else
            return -1;
    }

    if (RANDOM)
    {
        init_wallp ();
        snprintf(tmp, 127, "%s%s", getenv("HOME"), "/.wallp/list");
        if (result)
        {
            read_image_list(tmp);
            no_of_list_items = g_slist_length (simple_list);
            if (no_of_list_items == 0)
            {
                fprintf (stderr,
                         "Run WallP without -wharf to create the list of images first\n");
                return 1;
            }
            put_image_on_root (g_slist_nth_data (simple_list, random_image()), STYLE);
            return 0;
        }
        return 1;
    }

    if (WHARF)
    {
        wharf_create_gui ();
        init_wallp();
        snprintf(tmp, 127, "%s%s", getenv("HOME"), "/.wallp/list");
        read_image_list (tmp);
        no_of_list_items = g_slist_length (simple_list);
        if (no_of_list_items == 0)
        {
            fprintf (stderr,
                     "Run WallP without -wharf to create the list of images first\n");
            return 1;
        }
        wharf_logo();
        gtk_main();
        return 0;
    }

    create_list ();

    f_save_file();

    if( result ) { /* if control exists, try to read list */
        snprintf(tmp, 127, "%s%s", getenv("HOME"), "/.wallp/list");
        result = read_image_list(tmp);

#ifdef VERBOSE
        printf("%d => %s\n", result, tmp);
#endif
    }

    if( ! result ) { /* list does not exist, read in old style config */
        snprintf(tmp, 127, "%s%s", getenv("HOME"), "/.wallp_save");
        read_list_from_file (tmp);
    }

    init_wallp ();
    wallp_logo();
    gtk_main ();

    return 0;
}

