// Copyright (c) The University of Cincinnati.
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
// THE SOFTWARE, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.

// Authors: Radharamanan Radhakrishnan  ramanan@ececs.uc.edu
//          Philip A. Wilsey            phil.wilsey@uc.edu

//---------------------------------------------------------------------------
//
// $Id: Transceiver.cpp
//
//---------------------------------------------------------------------------

#include "Transceiver.h"
#include "MsgAggregatingCommunicationManager.h"
using std::cerr;
using std::endl;

Transceiver::Transceiver() : numberOfMsgs( 0 ),
			     lpId( 0 ),
			     currentPosition( buffer ),
			     messageManagerHandle( 0 ),
			     commPhyLib( 0 ){}

Transceiver::~Transceiver() {}

void
Transceiver::setLpId(unsigned int id) {
  lpId = id;
}

void
Transceiver::setMessageManagerHandle(MessageManager* msgMgrPtr) {
  messageManagerHandle = msgMgrPtr;
}

unsigned int
Transceiver::readMsgSize() {
  int* ptr;
  ptr = (int*)currentPosition;
  // The following operation is REALLY COMPILER, ARCHITECTURE dependent
  int val = *ptr;
  ptr++;
  currentPosition = (char*) ptr;
  return val;
}

const SerializedInstance *
Transceiver::readMessage() {
  unsigned int msgSize = readMsgSize();
  char *buf = cppStrDup( currentPosition, msgSize );
  currentPosition = currentPosition + msgSize;
  return new SerializedInstance( buf, msgSize );
}


void
Transceiver::writeMsgSize( int msgSize ){
  int* ptr;
  ptr = (int*)currentPosition;
  // The following operation is REALLY COMPILER, ARCHITECTURE dependent
  *ptr = msgSize;
  ptr++;
  currentPosition = (char*) ptr;
}

void
Transceiver::writeMessage(int msgSize, char* message) {
  memcpy(currentPosition, message, msgSize);
  currentPosition = currentPosition + msgSize;
}


void
Transceiver::sendMessage() {
  unsigned int size = currentPosition - (char *)&numberOfMsgs;
  char *buf = cppStrDup( (char *)&numberOfMsgs, size );
  commPhyLib->physicalSend( new SerializedInstance( buf, 
						    size ), 
			    lpId);
  numberOfMsgs = 0;
  currentPosition = buffer;
}

void
Transceiver::receiveMessage() {
  numberOfMsgs = 0;
  currentPosition = buffer;
  const SerializedInstance *newMsg = commPhyLib->physicalProbeRecv();
  if( newMsg != 0 ){
    if( newMsg->getSize() > INTERNAL_BUFFER_SIZE + sizeof(int) ){
      cerr << "Transceiver::receiveMessage message size greater "
	   << "than INTERNAL_BUFFER_SIZE+sizeof(int):"
	   << INTERNAL_BUFFER_SIZE+sizeof(int) << endl;
      abort();
    }
    else{
      memcpy( buffer, &(*newMsg->getData().begin()), newMsg->getSize() );
      delete newMsg;
    }
  }
}

bool
Transceiver::canWriteMessage(int size) {
  if((currentPosition + size) < ((char*)buffer + INTERNAL_BUFFER_SIZE)) {
    return true;
  }
  else {
    return false;
  }
}

bool
Transceiver::canReceiveMessage(int) {
  if(numberOfMsgs == 0) {
    return true;
  }
  else {
    return false;
  }
}

void
Transceiver::probeToSend() {
  if(numberOfMsgs > 0) {
    sendMessage();
  }
}

void
Transceiver::probeToReceive() {
  if(numberOfMsgs == 0) {
    receiveMessage();
  }
}

void
Transceiver::setCommPhyInterface(PhysicalCommunicationLayer *phyLib){
  commPhyLib = phyLib;
}
