# ldap-useradmin-lib.pl
# Module sponsored by
# Sanitaetsbetrieb Brixen  - Azienda Sanitaria di Bressanone
# www.sb-brixen.it         - www.as-bressanone.it

do '../web-lib.pl';
&init_config();
require '../ui-lib.pl';
&foreign_require("useradmin", "user-lib.pl");

%utext = &load_language("useradmin");
foreach $t (keys %utext) {
	$text{$t} ||= $utext{$t};
	}
%mconfig = %useradmin::config;
foreach my $k (keys %config) {
	if ($config{$k} ne "") {
		$mconfig{$k} = $config{$k};
		}
	}

eval "use Net::LDAP";
if ($@) { $net_ldap_error = $@; }
else { $got_net_ldap++; }
eval "use Net::IMAP";
if ($@) { $net_imap_error = $@; }
else { $got_net_imap++; }

$secret_file = "/etc/ldap.secret";
$samba_class = $config{'samba_class'} || "sambaAccount";
$cyrus_class = $config{'imap_class'} || "SuSEeMailObject";

if ($config{'charset'}) {
	$force_charset = $config{'charset'};
	}

# get_nss_config()
# Parses the NSS config file into a hash reference
sub get_nss_config
{
return undef if (!$config{'auth_ldap'});
if (!%nss_config_cache) {
	open(CONF, $config{'auth_ldap'});
	while(<CONF>) {
		s/\r|\n//g;
		s/#.*$//;
		if (/^\s*(\S+)\s*(.*)/) {
			$nss_config_cache{lc($1)} = $2;
			}
		}
	close(CONF);
	}
return \%nss_config_cache;
}

# ldap_connect(return-error)
# Connect to the LDAP server and return a handle to the Net::LDAP object
sub ldap_connect
{
local $conf = &get_nss_config();
local ($host, $port);
if ($conf) {
	$host = $conf->{'host'};
	$port = $conf->{'port'};
	}
else {
	$host = $config{'ldap_host'};
	$port = $config{'ldap_port'};
	}
$port ||= 389;
local $ldap = Net::LDAP->new($host, port => $port);
if (!$ldap) {
	local $err = &text('conn_econn',
			   "<tt>$host</tt>","<tt>$port</tt>");
	if ($_[0]) { return $err; }
	else { &error($err); }
	}

# Start TLS if configured
if ($config{'ldap_tls'}) {
	$ldap->start_tls;
	}

local ($dn, $password);
if ($config{'login'}) {
	# Use the configured login
	$dn = $config{'login'};
	$password = $config{'pass'};
	}
elsif ($conf->{'rootbinddn'}) {
	# Use the root login if we have one
	$dn = $conf->{'rootbinddn'};
	open(SECRET, $secret_file);
	chop($password = <SECRET>);
	close(SECRET);
	}
else {
	# Use the normal login
	$dn = $conf->{'binddn'};
	$password = $conf->{'bindpw'};
	}
local $mesg;
if ($dn) {
	$mesg = $ldap->bind(dn => $dn, password => $password);
	}
else {
	$mesg = $ldap->bind(anonymous => 1);
	}
if (!$mesg || $mesg->code) {
	local $err = &text('conn_elogin', "<tt>$conf{'host'}</tt>",
		     $dn, $mesg ? $mesg->error : "Unknown error");
	if ($_[0]) { return $err; }
	else { &error($err); }
	}
return $ldap;
}

# get_user_base()
sub get_user_base
{
local $conf = &get_nss_config();
local $base = $config{'user_base'} || $conf->{'nss_base_passwd'} || $conf->{'base'};
$base =~ s/\?.*$//;
return $base;
}

# get_group_base()
sub get_group_base
{
local $conf = &get_nss_config();
local $base = $config{'group_base'} || $conf->{'nss_base_group'} || $conf->{'base'};
$base =~ s/\?.*$//;
return $base;
}

# imap_connect(return-error)
# Connect and login to the IMAP server
sub imap_connect
{
local $imap = new Net::IMAP($config{'imap_host'});
if (!$imap) {
	local $err = &text('imap_econn', "<tt>$config{'imap_host'}</tt>");
	if ($_[0]) { return $err; }
	else { &error($err); }
	}
local $rv = $imap->login($config{'imap_login'}, $config{'imap_pass'});
if ($rv->{'Status'} ne 'ok') {
	local $err = &text('imap_elogin', "<tt>$config{'imap_host'}</tt>",
			   "<tt>$config{'imap_login'}</tt>", $rv->{'Text'});
	if ($_[0]) { return $err; }
	else { &error($err); }
	}
return $imap;
}

# samba_password(password)
# Converts a plain text string into two Samba passwords (nt and lm) with the
# ntpasswd program.
sub samba_password
{
local ($nt, $lm);
&foreign_check("samba") || &error($text{'usave_esamba'});
&foreign_require("samba", "smbhash.pl");
$nt = &samba::nthash($_[0]);
$lm = &samba::lmhash($_[0]);
return ($nt, $lm);
}

# encrypt_password(string)
sub encrypt_password
{
&seed_random();
if ($config{'md5'} == 3) {
	# LDAP MD5 encryption
	local $qp = quotemeta($_[0]);
	local $out = `$config{'slappasswd'} -h '{md5}' -s $qp 2>/dev/null`;
	$out =~ s/\s+$//;
	$out =~ s/^\{md5\}//i;
	return $out;
	}
if ($config{'md5'} == 1) {
	# Unix MD5 encryption
	&foreign_require("useradmin", "user-lib.pl");
	return &useradmin::encrypt_md5($_[0]);
	}
elsif ($config{'md5'} == 0) {
	# Standard Unix crypt
	local $salt = chr(int(rand(26))+65).chr(int(rand(26))+65);
	return crypt($_[0], $salt);
	}
else {
	# No encryption!
	return $_[0];
	}
}

# list_users()
# Returns a list of users, in the same format as the useradmin module
sub list_users
{
if (!defined(@list_users_cache)) {
	local $ldap = &ldap_connect();
	local $base = &get_user_base();
	local $rv = $ldap->search(base => $base,
			    filter => '(objectClass=posixAccount)');
	local $u;
	foreach $u ($rv->all_entries) {
		local %uinfo = &dn_to_hash($u);
		push(@list_users_cache, \%uinfo);
		}
	$ldap->unbind();
	}
return @list_users_cache;
}

# create_user(&user)
# Given a user details hash in the same format as the useradmin module, add
# it to the LDAP database
sub create_user
{
local $ldap = &ldap_connect();
local $base = &get_user_base();
local $_[0]->{'dn'} = "uid=$_[0]->{'user'},$base";
local @classes = ( "posixAccount", "shadowAccount",
		   split(/\s+/, $config{'other_class'}),
		   @{$_[0]->{'ldap_class'}} );
local @attrs = &user_to_dn($_[0]);
push(@attrs, &split_props($config{'props'}, $_[0]));
push(@attrs, @{$_[0]->{'ldap_attrs'}});
push(@attrs, "objectClass" => \@classes);
local $rv = $ldap->add($_[0]->{'dn'}, attr => \@attrs);
if ($rv->code) {
	&error(&text('usave_eadd', $rv->error));
	}
push(@list_users_cache, $_[0]) if (defined(@list_users_cache));
$ldap->unbind();
}

# delete_user(&user)
# Given a user details hash in the same format as the useradmin module, removes
# it from the LDAP database
sub delete_user
{
local $ldap = &ldap_connect();
local $rv = $ldap->delete($_[0]->{'dn'});
if ($rv->code) {
	&error(&text('usave_edelete', $rv->error));
	}
$ldap->unbind();
@list_users_cache = grep { $_ ne $_[0] } @list_users_cache
        if (defined(@list_users_cache));
}

# modify_user(&olduser, &newuser)
sub modify_user
{
local $ldap = &ldap_connect();
local $base = &get_user_base();
local @attrs = &user_to_dn($_[1]);
push(@attrs, &split_props($config{'mod_props'}, $_[1]));
push(@attrs, @{$_[0]->{'ldap_attrs'}});
local $rv = $ldap->modify($_[0]->{'dn'}, replace => { @attrs });
if ($rv->code) {
	&error(&text('usave_emod', $rv->error));
	}
local $newdn = "uid=$_[1]->{'user'},$base";
if (!&same_dn($newdn, $_[0]->{'dn'})) {
	# Re-named too!
	$rv = $ldap->moddn($_[0]->{'dn'}, newrdn => "uid=$_[1]->{'user'}");
	if ($rv->code) {
		&error(&text('usave_emoddn', $rv->error));
		}
	$_[1]->{'dn'} = $newdn;
	}
if ($_[0] ne $_[1] && &indexof($_[0], @list_users_cache) != -1) {
	# Update old object in cache
	%{$_[0]} = %{$_[1]};
	}
$ldap->unbind();
}

# list_groups()
# Returns a list of groups, in the same format as the useradmin module
sub list_groups
{
if (!defined(@list_groups_cache)) {
	local $ldap = &ldap_connect();
	local $base = &get_group_base();
	local $rv = $ldap->search(base => $base,
			    filter => '(objectClass=posixGroup)');
	local $g;
	foreach $g ($rv->all_entries) {
		local %ginfo = &dn_to_hash($g);
		push(@list_groups_cache, \%ginfo);
		}
	$ldap->unbind();
	}
return @list_groups_cache;
}

# create_group(&group)
# Given a group details hash in the same format as the useradmin module, add
# it to the LDAP database
sub create_group
{
local $ldap = &ldap_connect();
local $base = &get_group_base();
local $_[0]->{'dn'} = "cn=$_[0]->{'group'},$base";
local @classes = ( "posixGroup" );
local @attrs = &group_to_dn($_[0]);
push(@attrs, @{$_[0]->{'ldap_attrs'}});
push(@attrs, "objectClass" => \@classes);
local $rv = $ldap->add($_[0]->{'dn'}, attr => \@attrs);
if ($rv->code) {
	&error(&text('gsave_eadd', $rv->error));
	}
push(@list_groups_cache, $_[0]) if (defined(@list_groups_cache));
$ldap->unbind();
}

# delete_group(&group)
# Given a group details hash in the same format as the useradmin module, removes
# it from the LDAP database
sub delete_group
{
local $ldap = &ldap_connect();
local $rv = $ldap->delete($_[0]->{'dn'});
if ($rv->code) {
	&error(&text('gsave_edelete', $rv->error));
	}
$ldap->unbind();
@list_groups_cache = grep { $_ ne $_[0] } @list_groups_cache
        if (defined(@list_groups_cache));
}

# modify_group(&oldgroup, &newgroup)
sub modify_group
{
local $ldap = &ldap_connect();
local $base = &get_group_base();
local @attrs = &group_to_dn($_[1]);
push(@attrs, @{$_[0]->{'ldap_attrs'}});
local $rv = $ldap->modify($_[0]->{'dn'}, replace => { @attrs });
if ($rv->code) {
	&error(&text('gsave_emod', $rv->error));
	}
local $newdn = "cn=$_[1]->{'group'},$base";
if (!&same_dn($newdn, $_[0]->{'dn'})) {
	# Re-named too!
	$rv = $ldap->moddn($_[0]->{'dn'}, newrdn => "cn=$_[1]->{'group'}");
	if ($rv->code) {
		&error(&text('gsave_emoddn', $rv->error));
		}
	$_[1]->{'dn'} = $newdn;
	}
if ($_[0] ne $_[1] && &indexof($_[0], @list_groups_cache) != -1) {
	# Update old object in cache
	%{$_[0]} = %{$_[1]};
	}
$ldap->unbind();
}

# dn_to_hash(&ldap-object)
# Given a LDAP object containing user or group details, convert it to a hash
# in the same format uses by the useradmin module
sub dn_to_hash
{
if ($_[0]->get_value("uid")) {
	local %user = ( 'dn' => $_[0]->dn(),
			'user' => $_[0]->get_value("uid"),
			'uid' => $_[0]->get_value("uidNumber"),
			'gid' => $_[0]->get_value("gidNumber"),
			'real' => $_[0]->get_value("cn"),
			'home' => $_[0]->get_value("homeDirectory"),
			'shell' => $_[0]->get_value("loginShell"),
			'pass' => $_[0]->get_value("userPassword"),
			'change' => $_[0]->get_value("shadowLastChange") || "",
			'expire' => $_[0]->get_value("shadowExpire") || "",
			'min' => $_[0]->get_value("shadowMin") || "",
			'max' => $_[0]->get_value("shadowMax") || "",
			'warn' => $_[0]->get_value("shadowWarning") || "",
			'inactive' => $_[0]->get_value("shadowInactive") || "",
		      );
	$user{'pass'} =~ s/^{[a-z0-9]+}//i;
	return %user;
	}
else {
	local @members = $_[0]->get_value('memberUid');
	local %group = ( 'dn' => $_[0]->dn(),
			 'group' => $_[0]->get_value("cn"),
			 'gid' => $_[0]->get_value("gidNumber"),
			 'pass' => $_[0]->get_value("userPassword"),
			 'members' => join(",", @members),
			);
	return %group;
	}
}

# user_to_dn(&user)
# Given a useradmin-style user hash, returns a list of properties
sub user_to_dn
{
local $pfx = $_[0]->{'pass'} =~ /^\{[a-z0-9]+\}/i ? undef :
	     $config{'md5'} == 1 || $config{'md5'} == 3 ? "{md5}" :
	     $config{'md5'} == 0 ? "{crypt}" : "";
return ( "cn" => $_[0]->{'real'},
	 "uid" => $_[0]->{'user'},
	 "uidNumber" => $_[0]->{'uid'},
	 "loginShell" => $_[0]->{'shell'},
	 "homeDirectory" => $_[0]->{'home'},
	 "gidNumber" => $_[0]->{'gid'},
	 "userPassword" => $pfx.$_[0]->{'pass'},
	 $_[0]->{'change'} eq '' ? ( ) :
		( "shadowLastChange" => $_[0]->{'change'} ),
	 $_[0]->{'expire'} eq '' ? ( ) :
		( "shadowExpire" => $_[0]->{'expire'} ),
	 $_[0]->{'min'} eq '' ? ( ) :
		( "shadowMin" => $_[0]->{'min'} ),
	 $_[0]->{'max'} eq '' ? ( ) :
		( "shadowMax" => $_[0]->{'max'} ),
	 $_[0]->{'warn'} eq '' ? ( ) :
		( "shadowWarn" => $_[0]->{'warn'} ),
	 $_[0]->{'inactive'} eq '' ? ( ) :
		( "shadowInactive" => $_[0]->{'inactive'} )
	);
}

sub group_to_dn
{
local @members = split(/,/, $_[0]->{'members'});
return ( "cn" => $_[0]->{'group'},
	 "gidNumber" => $_[0]->{'gid'},
	 "userPassword" => $_[0]->{'pass'},
	 @members ? ( "memberUid" => \@members ) : ( ) );
}

# making_changes()
# Called before the LDAP database has been updated, to run the pre-changes
# command.
sub making_changes
{
if ($config{'pre_command'} =~ /\S/) {
	local $out = &backquote_logged("($config{'pre_command'}) 2>&1 </dev/null");
	return $? ? $out : undef;
	}
return undef;
}

# made_changes()
# Called after the LDAP database has been updated, to run the post-changes
# command.
sub made_changes
{
if ($config{'post_command'} =~ /\S/) {
	local $out = &backquote_logged("($config{'post_command'}) 2>&1 </dev/null");
	return $? ? $out : undef;
	}
return undef;
}

# Just call the useradmin function of the same name
sub set_user_envs
{
return &useradmin::set_user_envs(@_);
}

# Just call the useradmin function of the same name
sub set_group_envs
{
return &useradmin::set_group_envs(@_);
}

# Does nothing, because no LDAP locking is needed
sub lock_user_files { }

# Does nothing, because no LDAP locking is needed
sub unlock_user_files { }

# split_props(text, &user)
sub split_props
{
local %pmap;
foreach $p (split(/\t+/, &substitute_template($_[0], $_[1]))) {
	if ($p =~ /^(\S+):\s*(.*)/) {
		push(@{$pmap{$1}}, $2);
		}
	}
local @rv;
local $k;
foreach $k (keys %pmap) {
	local $v = $pmap{$k};
	if (@$v == 1) {
		push(@rv, $k, $v->[0]);
		}
	else {
		push(@rv, $k, $v);
		}
	}
return @rv;
}

# substitute_template(text, &hash)
# Given some text and a hash reference, for each ocurrance of $FOO or ${FOO} in
# the text replaces it with the value of the hash key foo
sub substitute_template
{
# Add some extra fixed parameters to the hash
local %hash = %{$_[1]};
$hash{'hostname'} = &get_system_hostname();

# Actually do the substition
local $rv = $_[0];
local $s;
foreach $s (keys %hash) {
	local $us = uc($s);
	local $sv = $hash{$s};
	$rv =~ s/\$\{\Q$us\E\}/$sv/g;
	$rv =~ s/\$\Q$us\E/$sv/g;
	if ($sv) {
		$rv =~ s/\$\{IF-\Q$us\E\}(\n?)([\000-\377]*?)\$\{ELSE-\Q$us\E\}(\n?)([\000-\377]*?)\$\{ENDIF-\Q$us\E\}(\n?)/\2/g;
		$rv =~ s/\$\{IF-\Q$us\E\}(\n?)([\000-\377]*?)\$\{ENDIF-\Q$us\E\}(\n?)/\2/g;

		$rv =~ s/\$IF-\Q$us\E(\n?)([\000-\377]*?)\$ELSE-\Q$us\E(\n?)([\000-\377]*?)\$ENDIF-\Q$us\E(\n?)/\2/g;
		$rv =~ s/\$IF-\Q$us\E(\n?)([\000-\377]*?)\$ENDIF-\Q$us\E(\n?)/\2/g;
		}
	else {
		$rv =~ s/\$\{IF-\Q$us\E\}(\n?)([\000-\377]*?)\$\{ELSE-\Q$us\E\}(\n?)([\000-\377]*?)\$\{ENDIF-\Q$us\E\}(\n?)/\4/g;
		$rv =~ s/\$\{IF-\Q$us\E\}(\n?)([\000-\377]*?)\$\{ENDIF-\Q$us\E\}(\n?)//g;

		$rv =~ s/\$IF-\Q$us\E(\n?)([\000-\377]*?)\$ELSE-\Q$us\E(\n?)([\000-\377]*?)\$ENDIF-\Q$us\E(\n?)/\4/g;
		$rv =~ s/\$IF-\Q$us\E(\n?)([\000-\377]*?)\$ENDIF-\Q$us\E(\n?)//g;
		}
	}

# Remove un-replaced IF blocks
$rv =~ s/\$\{IF-([A-Z0-9_]+)\}(\n?)([\000-\377]*?)\$\{ELSE-$1\}(\n?)([\000-\377]*?)\$\{ENDIF-$1\}(\n?)/\5/g;
$rv =~ s/\$IF-([A-Z0-9_]+)(\n?)([\000-\377]*?)\$ELSE-$1(\n?)([\000-\377]*?)\$ENDIF-$1(\n?)/\5/g;
$rv =~ s/\$\{IF-([A-Z0-9_]+)\}(\n?)([\000-\377]*?)\$\{ENDIF-$1\}(\n?)//g;
$rv =~ s/\$IF-([A-Z0-9_]+)(\n?)([\000-\377]*?)\$ENDIF-$1(\n?)//g;
return $rv;
}

# build_user_used([&uid-hash], [&shell-list], [&username-hash])
# Fills in a hash with used UIDs and shells
sub build_user_used
{
setpwent();
local @u;
while(@u = getpwent()) {
	$_[0]->{$u[2]}++ if ($_[0]);
	push(@{$_[1]}, $u[8]) if ($_[1] && $u[8]);
	$_[2]->{$u[0]}++ if ($_[2]);
	}
endpwent();
local $u;
foreach $u (&list_users()) {
	$_[0]->{$u->{'uid'}}++ if ($_[0]);
	push(@{$_[1]}, $u->{'shell'}) if ($_[1] && $u->{'shell'});
	$_[2]->{$u->{'user'}}++ if ($_[2]);
	}
}

# build_group_used([&uid-hash], [&groupname-hash])
sub build_group_used
{
setgrent();
local @g;
while(@g = getgrent()) {
	$_[0]->{$g[2]}++ if ($_[0]);
	$_[1]->{$g[0]}++ if ($_[1]);
	}
endgrent();
local $g;
foreach $g (&list_groups()) {
	$_[0]->{$g->{'gid'}}++ if ($_[0]);
	$_[1]->{$g->{'group'}}++ if ($_[1]);
	}
}

# allocate_uid(&uids-used)
sub allocate_uid
{
local $rv = $mconfig{'base_uid'};
while($_[0]->{$rv}) {
	$rv++;
	}
return $rv;
}

# allocate_gid(&gids-used)
sub allocate_gid
{
local $rv = $mconfig{'base_gid'};
while($_[0]->{$rv}) {
	$rv++;
	}
return $rv;
}

# same_dn(dn1, dn2)
# Returns 1 if two DNs are the same
sub same_dn
{
local $dn0 = join(",", split(/,\s*/, $_[0]));
local $dn1 = join(",", split(/,\s*/, $_[1]));
return lc($dn0) eq lc($dn1);
}

# all_getpwnam(username)
# Look up a user by name, and return his details in array format. Searches
# both LDAP and the local users DB.
sub all_getpwnam
{
local @uinfo = getpwnam($_[0]);
if (scalar(@uinfo)) {
	return wantarray ? @uinfo : $uinfo[2];
	}
local $u;
foreach $u (&list_users()) {
	return &pw_user_rv($u, wantarray, 'uid')
		if ($u->{'user'} eq $_[0]);
	}
return wantarray ? () : undef;
}

# all_getpwuid(uid)
# Look up a user by UID, and return his details in array format. Searches
# both LDAP and the local users DB.
sub all_getpwuid
{
local @uinfo = getpwuid($_[0]);
if (scalar(@uinfo)) {
	return wantarray ? @uinfo : $uinfo[0];
	}
local $u;
foreach $u (&list_users()) {
	return &pw_user_rv($u, wantarray, 'user')
		if ($u->{'uid'} == $_[0]);
	}
return wantarray ? () : undef;
}

# all_getgrgid(gid)
# Look up a group by GID, and return its details in array format. Searches
# both LDAP and the local groups DB.
sub all_getgrgid
{
local @ginfo = getgrgid($_[0]);
if (scalar(@ginfo)) {
	return wantarray ? @ginfo : $ginfo[0];
	}
local $g;
foreach $g (&list_groups()) {
	return &gr_group_rv($g, wantarray, 'group')
		if ($g->{'gid'} == $_[0]);
	}
return wantarray ? () : undef;
}

# all_getgrnam(groupname)
# Look up a group by name, and return its details in array format. Searches
# both LDAP and the local groups DB.
sub all_getgrnam
{
local @ginfo = getgrnam($_[0]);
if (scalar(@ginfo)) {
	return wantarray ? @ginfo : $ginfo[2];
	}
local $g;
foreach $g (&list_groups()) {
	return &gr_group_rv($g, wantarray, 'gid')
		if ($g->{'group'} eq $_[0]);
	}
return wantarray ? () : undef;
}

sub gr_group_rv
{
return $_[1] ? ( $_[0]->{'group'}, $_[0]->{'pass'}, $_[0]->{'gid'},
		 $_[0]->{'members'} ) : $_[0]->{$_[2]};
}

sub pw_user_rv
{
return $_[0] ? ( $_[0]->{'user'}, $_[0]->{'pass'}, $_[0]->{'uid'},
		 $_[0]->{'gid'}, undef, undef, $_[0]->{'real'},
		 $_[0]->{'home'}, $_[0]->{'shell'}, undef ) : $_[0]->{$_[2]};
}

# auto_home_dir(base, username, groupname)
# Returns an automatically generated home directory, and creates needed
# parent dirs
sub auto_home_dir
{
local $pfx = $_[0] eq "/" ? "/" : $_[0]."/";
if ($mconfig{'home_style'} == 0) {
	return $pfx.$_[1];
	}
elsif ($mconfig{'home_style'} == 1) {
	&mkdir_if_needed($pfx.substr($_[1], 0, 1));
	return $pfx.substr($_[1], 0, 1)."/".$_[1];
	}
elsif ($mconfig{'home_style'} == 2) {
	&mkdir_if_needed($pfx.substr($_[1], 0, 1));
	&mkdir_if_needed($pfx.substr($_[1], 0, 1)."/".
			 substr($_[1], 0, 2));
	return $pfx.substr($_[1], 0, 1)."/".
	       substr($_[1], 0, 2)."/".$_[1];
	}
elsif ($mconfig{'home_style'} == 3) {
	&mkdir_if_needed($pfx.substr($_[1], 0, 1));
	&mkdir_if_needed($pfx.substr($_[1], 0, 1)."/".
			 substr($_[1], 1, 1));
	return $pfx.substr($_[1], 0, 1)."/".
	       substr($_[1], 1, 1)."/".$_[1];
	}
elsif ($mconfig{'home_style'} == 4) {
	return $_[0];
	}
elsif ($mconfig{'home_style'} == 5) {
	return $pfx.$_[2]."/".$_[1];
	}
}



1;

