/* -*-c-*- */
/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

%option prefix="cisco_ios"
%option outfile="cisco_ios.cc"
%option noyywrap

%{
using namespace std;

#define YY_NO_UNPUT

#define IPLEN 16

#include <unistd.h>
#include <string.h>
#include <ctype.h>

#include "cisco_ios.h"
#include "common.h"
#include "defs.h"

enum {
  CI_OPT_NONE,
  CI_OPT_HOST,
  CI_OPT_MSEC,
  CI_OPT_PORT,
  CI_OPT_MISSING,
  CI_OPT_TYPE
};

#define SHORTLEN 32
#define SHORTLEN_S "32"

static void cisco_ios_parse_date(char* input, unsigned char mode);
static void cisco_ios_parse_src(char* input, unsigned char mode);
static void cisco_ios_parse_dst(char* input, unsigned char mode);
static void cisco_ios_parse_if(char* input);

static wf_logentry* logentry;
%}

MONTH	"Jan"|"Feb"|"Mar"|"Apr"|"May"|"Jun"|"Jul"|"Aug"|"Sep"|"Oct"|"Nov"|"Dec"
STRING	[a-zA-Z-][a-zA-Z0-9._-]*
LOGHOST	[0-9.a-zA-Z_-]*
DIGIT	[0-9]
NUMBER	{DIGIT}+
BYTE	{DIGIT}{1,3}
IPADDR	{BYTE}\.{BYTE}\.{BYTE}\.{BYTE}
PORT	{DIGIT}{1,5}
HEXDIGIT	[0-9a-fA-F]
HEXNUM	"0x"{HEXDIGIT}+
LIST	[a-zA-Z0-9._>-]*
IOS	"%SEC-6-IPACCESSLOG"("P"|"DP"|"RP"|"NP"|"S")":"
TARGET	"denied"|"permitted"
PROTO	"tcp"|"udp"|"icmp"|"igmp"|"gre"|"ospf"|"ipinip"|"pim"|{NUMBER}

%%

{MONTH}[ ]{1,2}{DIGIT}{1,2}[ ]{DIGIT}{2}:{DIGIT}{2}:{DIGIT}{2}[ ]{LOGHOST}	cisco_ios_parse_date(cisco_iostext, CI_OPT_HOST);
{NUMBER}":"	/* ignore */
{DIGIT}"y"{DIGIT}{1,2}"w:"	/* ignore */
{DIGIT}{1,2}"w"{DIGIT}"d:"	/* ignore */
{DIGIT}"d"{DIGIT}{2}"h:"	/* ignore */
{MONTH}[ ]{1,2}{DIGIT}{1,2}[ ]{DIGIT}{2}:{DIGIT}{2}:{DIGIT}{2}"."{DIGIT}{3}":"	cisco_ios_parse_date(cisco_iostext, CI_OPT_MSEC);
{MONTH}[ ]{1,2}{DIGIT}{1,2}[ ]{DIGIT}{2}:{DIGIT}{2}:{DIGIT}{2}":"	cisco_ios_parse_date(cisco_iostext, CI_OPT_NONE);
{IOS}		/* ignore */
"list "{LIST}[ ]{TARGET}[ ]{PROTO}[ ]{IPADDR}"("{PORT}")"	cisco_ios_parse_src(cisco_iostext, CI_OPT_PORT);
"list "{LIST}[ ]{TARGET}[ ]{PROTO}[ ]{IPADDR}	cisco_ios_parse_src(cisco_iostext, CI_OPT_NONE);
"list "{LIST}[ ]{TARGET}[ ]{IPADDR}		cisco_ios_parse_src(cisco_iostext, CI_OPT_MISSING);
"-> "{IPADDR}"("{PORT}"),"			cisco_ios_parse_dst(cisco_iostext, CI_OPT_PORT);
"-> "{IPADDR}" ("{NUMBER}"/"{NUMBER}"),"	cisco_ios_parse_dst(cisco_iostext, CI_OPT_TYPE);
"-> "{IPADDR}","				cisco_ios_parse_dst(cisco_iostext, CI_OPT_NONE);
{NUMBER}" packet"("s")?	{
  logentry->count = atoi(cisco_iostext);
  // =| CISCO_IOS_COUNT;
}
"("[A-Za-z0-9 /\._\*-]*")"	cisco_ios_parse_if(cisco_iostext);
[ ]+		/* ignore whitespace */
[\n]		/* ignore */
{STRING}	if (verbose) fprintf(stderr, _("Unrecognized token: %s\n"), cisco_iostext);
.		if (verbose) fprintf(stderr, _("Unrecognized character: %s\n"), cisco_iostext);

%%

static void
cisco_ios_parse_date(char* input, unsigned char mode) {
  int day, hour, minute, second;
  char smonth[4], hostname[33];
#ifdef IRIX
  char tmp[SHOSTLEN];
#endif
#ifdef LOGDOTS
  char *remove_dot;
#endif

  if (mode == CI_OPT_HOST) {
    if (sscanf(input, "%3s %2d %2d:%2d:%2d %32s",
	       smonth, &day, &hour, &minute, &second,
#ifndef IRIX
		    hostname) != 6)
     return;
   logentry->hostname = hostname;
#else
		    tmp) != 6)
     return;
   if (tmp[2] == ':')
     logentry->hostname = tmp + 3;
#endif

#ifdef LOGDOTS
    remove_dot = strstr(opt.line->hostname, ".");
    if(remove_dot != NULL)
      *remove_dot = '\0';
#endif
  }
  else if (mode == CI_OPT_MSEC) {
    int msec;
    if (sscanf(input, "%3s %2d %2d:%2d:%2d.%3d:",
	       smonth, &day, &hour, &minute, &second, &msec) != 6)
     return;
  }
  else if (mode == CI_OPT_NONE) {
    if (sscanf(input, "%3s %2d %2d:%2d:%2d:",
               smonth, &day, &hour, &minute, &second) != 5)
      return;
  }

  logentry->start_time = build_time(smonth, day, hour, minute, second);

  // |CISCO_IOS_DATE;
}

static void
cisco_ios_parse_src(char* input, unsigned char mode) {
  char proto[9], ip[IPLEN], chainlabel[SHORTLEN + 1], branchname[SHORTLEN + 1];
  int shost1, shost2, shost3, shost4;

  if (mode == CI_OPT_PORT) {
    int sport;
    if (sscanf(input, "list %" SHORTLEN_S "s %" SHORTLEN_S "s %8s %3d.%3d.%3d.%3d(%5d)",
	       chainlabel, branchname, proto,
	       &shost1, &shost2, &shost3, &shost4, &sport) != 8)
      return;
    logentry->chainlabel = chainlabel;
    logentry->branchname = branchname;
    logentry->sport = sport;
  }
  else if (mode == CI_OPT_NONE) {
    if (sscanf(input, "list %" SHORTLEN_S "s %" SHORTLEN_S "s %8s %3d.%3d.%3d.%3d",
	       chainlabel, branchname, proto,
	       &shost1, &shost2, &shost3, &shost4) != 7)
      return;
    logentry->chainlabel = chainlabel;
    logentry->branchname = branchname;
  }
  else if (mode == CI_OPT_MISSING)
    return;

  snprintf(ip, IPLEN, "%d.%d.%d.%d", shost1, shost2, shost3, shost4);
  if (logentry->sipaddr.set(ip) == false)
    return;

  // |CISCO_IOS_SRC;

  if (!strncmp(proto, "tcp", 3)) logentry->protocol = IPPROTO_TCP;
  else if (!strncmp(proto, "udp", 3)) logentry->protocol = IPPROTO_UDP;
  else if (!strncmp(proto, "icmp", 4)) logentry->protocol = IPPROTO_ICMP;
  else if (!strncmp(proto, "igmp", 4)) logentry->protocol = IPPROTO_IGMP;
  else if (!strncmp(proto, "gre", 3)) logentry->protocol = IPPROTO_GRE; /* RFC1701/1702 */
  else if (!strncmp(proto, "ospf", 4)) logentry->protocol = 89; // ALL@@2;
  else if (!strncmp(proto, "ipinip", 6)) logentry->protocol = IPPROTO_IPIP;
  else if (!strncmp(proto, "pim", 3)) logentry->protocol = IPPROTO_PIM;
  else if (isdigit((int)proto[0])) logentry->protocol = atoi(proto);

  // |CISCO_IOS_PROTO;
}

static void
cisco_ios_parse_dst(char* input, unsigned char mode) {
  char ip[IPLEN];
  int dhost1, dhost2, dhost3, dhost4;

  if (mode == CI_OPT_PORT) {
    int dport;
    if (sscanf(input, "-> %3d.%3d.%3d.%3d(%5d),",
	       &dhost1, &dhost2, &dhost3, &dhost4, &dport) != 5)
      return;
    logentry->dport = dport;
  }
  else if (mode == CI_OPT_TYPE) {
    int sport, dport;
    if (sscanf(input, "-> %3d.%3d.%3d.%3d (%2d/%2d),",
	       &dhost1, &dhost2, &dhost3, &dhost4, &sport, &dport) != 6)
      return;
    logentry->sport = sport;
    logentry->dport = dport;
  }
  else if (mode == CI_OPT_NONE) {
    if (sscanf(input, "-> %3d.%3d.%3d.%3d,",
		    &dhost1, &dhost2, &dhost3, &dhost4) != 4)
      return;
  }
  else
    return;

  snprintf(ip, IPLEN, "%d.%d.%d.%d", dhost1, dhost2, dhost3, dhost4);
  if (logentry->dipaddr.set(ip) == false)
    return;

  // |CISCO_IOS_DST;
}

static void
cisco_ios_parse_if(char* input) {
  logentry->input_iface = input;
}


wf_inmodule_cisco_ios::wf_inmodule_cisco_ios() {
}

bool
wf_inmodule_cisco_ios::match(const string& line) const {
  return (line.find(": %SEC-6-IPACCESSLOG") != string::npos); // optimal
}

enum wf_logentry_parsing_result
wf_inmodule_cisco_ios::parse(wf_logentry** retlogentry,
  const string& line, int linenum) {
  logentry = new wf_logentry();
  if (logentry == NULL)
     return WF_LOGENTRY_PARSING_ERROR;

  cisco_ios_scan_string(line.c_str());
  cisco_ioslex();

  logentry->format = "cisco_ios";

  /*
  if (parser == (CISCO_IOS_DATE|CISCO_IOS_SRC|CISCO_IOS_DST))
  */
    *retlogentry = logentry;
    return WF_LOGENTRY_PARSING_OK;

  if (verbose)
    fprintf(stderr, _("cisco_ios parse error in line %d, ignoring.\n"),
	    linenum);

  delete logentry;
  return WF_LOGENTRY_PARSING_ERROR;
}

extern "C" wf_inmodule*
wf_inmodule_cisco_ios_init() {
  return new wf_inmodule_cisco_ios();
}
