/*
  libwftk - Worldforge Toolkit - a widget library
  Copyright (C) 2002 Malcolm Walker <malcolm@worldforge.org>
  Based on code copyright  (C) 1999-2002  Karsten Laux 

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.
  
  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.
  
  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the
  Free Software Foundation, Inc., 59 Temple Place - Suite 330,
  Boston, MA  02111-1307, SA.
*/

#ifndef EVENT_QUEUE_H
#define EVENT_QUEUE_H

#include <cassert>
#include <queue>

#include <sigc++/object.h>
#if SIGC_MAJOR_VERSION == 1 && SIGC_MINOR_VERSION == 0
#include <sigc++/signal_system.h>
#else
#include <sigc++/signal.h>
#endif
#include <sigc++/object_slot.h>

#include "timer.h"

namespace wftk {

template<class C = SigC::Slot0<bool> >
class EventQueue : public SigC::Object
{
 public:
  ///
  EventQueue(unsigned iterations = 1);
  ///
  virtual ~EventQueue();

  ///
  void push(const C&);

 protected:
  /// regularly called by the wftk mainloop with the top element in the queue
  /**
   * Calls operator() on the object by default, so you can pass in, e.g.,
   * a SigC::Slot0<bool>.
   **/
  virtual bool handle(C& c) {return c();}
  /// called if queue is destroyed before events are handled
  virtual void cleanup(C&) {}

 private:
  void act();

  wftk::Timer event_pump_; // zero duration timer, always called if running
  std::queue<C> queue_;
  unsigned iterations_;
};

} // namespace

template<class C>
wftk::EventQueue<C>::EventQueue(unsigned iterations) :
	event_pump_(0, false), iterations_(iterations)
{
  assert(iterations);
  event_pump_.alarm.connect(SigC::slot(*this, &EventQueue::act));
}

template<class C>
wftk::EventQueue<C>::~EventQueue()
{
  while(!queue_.empty()) {
    cleanup(queue_.front());
    queue_.pop();
  }
}

template<class C>
void
wftk::EventQueue<C>::push(const C& c)
{
  queue_.push(c);
  event_pump_.run();
}

template<class C>
void
wftk::EventQueue<C>::act()
{
  unsigned i = 0;

  while(!queue_.empty() && i++ < iterations_)
    if(handle(queue_.front()))
      queue_.pop();

  if(queue_.empty())
    event_pump_.halt();
}

#endif // EVENT_QUEUE_H
