
#ifndef _SIMPLE_CONFIG_H_
#define _SIMPLE_CONFIG_H_

/**@name SimpleConfig - Class to read and write conf files easily in c++
 *
 * <b>Copyright</b><br>
 * This library is free software; you can redistribute it and/or      
 * modify it under the terms of the GNU Library General Public        
 * License as published by the Free Software Foundation; either       
 * version 2 of the License, or (at your option) any later version.   
 * <p>                                                                   
 * This library is distributed in the hope that it will be useful,    
 * but WITHOUT ANY WARRANTY; without even the implied warranty of     
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU  
 * Library General Public License for more details.                   
 * <p>                                                                   
 * You should have received a copy of the GNU Library General Public  
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * @author Ricardo Villalba &lt;rvm@linuxfan.com&gt;
 */
//@{ 

#include <appconf.h>

class FileConfig;


// ----------------------------------------------------------------------------
/**
  SimpleConfig derives from FileConfig and adds methods for reading and writing
  some extra types.
  <p>
  <b>Descripcin en espaol</b>
  <p>
  SimpleConfig est basado en appconf (deriva de FileConfig), pero
  al contrario que appconf, SimpleConfig no usa ficheros de configuracin
  por defecto (/etc/appname.conf, o ~/.appname.conf). El motivo es que
  el nombre por defecto no me serva para algunos de mis programas.

  <p>
  Aparte de eso, la clase SimpleConfig aade mtodos para leer y escribir
  tipos <i>int</i> y <i>bool</i>.

  <p>
  Ejemplo:

  <pre>
  SimpleConfig conf("testsimple.conf");

  conf.setGroup("General");
  conf.writeNumEntry("Int", 10 );
  conf.writeBoolEntry("Bool", true );
  conf.writeEntry("String", "Hola" );

  conf.setGroup("Another_group");
  conf.writeBoolEntry("Check_list", false );

  conf.sync();
  </pre>

  <p>
  Este ejemplo escribir en el fichero "testsimple.conf" en el directorio
  actual, con este contenido:

  <pre>
  [General]
  Int = 10
  Bool = true
  String = Hola
  [Another_group]
  Check_list = false
  </pre>

  <p>
  Luego el contenido de este fichero puede leerse as de fcil:

  <pre>
  conf.setGroup("General");
  int num= conf.readNumEntry("Int", 0 );
  bool activate= conf.readBoolEntry("Bool", false );
  cout << "String:" << conf.readEntry("String", "default" ) << endl;

  </pre>

  <b>Description in English</b>

  <p>
  SimpleConfig is based on appconf (derives from FileConfig), but
  this class doesn't use the default file configuration names that
  appconf does (/etc/appname.conf or ~/.appname.conf). The reason
  is that those names weren't good for some of my programmes.

  <p>
  Apart of that, class SimpleConfig add methods for reading and writing
  <i>int</i> and <i>bool</i> types.

  <p>
  Example:

  <pre>
  SimpleConfig conf("testsimple.conf");

  conf.setGroup("General");
  conf.writeNumEntry("Int", 10 );
  conf.writeBoolEntry("Bool", true );
  conf.writeEntry("String", "Hola" );

  conf.setGroup("Another_group");
  conf.writeBoolEntry("Check_list", false );

  conf.sync();
  </pre>

  <p>
  This example will write in the file "testsimple.conf" in the current 
  directory, these lines:

  <pre>
  [General]
  Int = 10
  Bool = true
  String = Hola
  [Another_group]
  Check_list = false
  </pre>

  <p>
  Then the content of that file could be read this way:

  <pre>
  conf.setGroup("General");
  int num= conf.readNumEntry("Int", 0 );
  bool activate= conf.readBoolEntry("Bool", false );
  cout << "String:" << conf.readEntry("String", "default" ) << endl;

  </pre>
*/
// ----------------------------------------------------------------------------
 
class SimpleConfig : public FileConfig
{

public:
	/** @name Constructores y destructor */
	//@{
    	/**
	     ////Constructor simple, para usar con setFileName()
         Simple constructor, to use with setFileName()
		*/
	SimpleConfig();

		/**
		 ////@memo Constructor con nombre de fichero 
		 ////@param fileName Nombre de fichero que se utilizar para el
         ////       fichero de configuracin

         @memo Constructor with file name
         @param fileName File name that will be used for the configuration name
		*/
	SimpleConfig(const char *fileName);

		/**
		 Destructor
		*/
	~SimpleConfig();
	//@}

	/**
	 ////Establece y lee el fichero de configuracin especificado
	 ////@param fileName Nombre del fichero 

     Sets and read the specified configuration name
     @param fileName File name
	*/
	void setFileName(const char *fileName);
	
	//Bool writeEntry(const char *key, const char *sValor);
	//Bool writeEntry(const char *key, long int Value);
	//Bool writeEntry(const char *key, double Value);

	/**
	 ////Guarda una entrada de tipo bool
	 ////@param key La clave 
	 ////@param bValue El nuevo valor 
	 ////@return TRUE si no hubo ningn problema, FALSE en caso contrario

     Stores an entry of type bool
     @param key The key
     @param bValue The new value
     @return TRUE on no error, FALSE in other case
	*/		
	bool writeBoolEntry(const char *key, bool bValue);

	/**
	 ////Guarda una entrada de tipo entero
     ////@param key La clave 
	 ////@param iValue El nuevo valor 
	 ////@return TRUE si no hubo ningn problema, FALSE en caso contrario

     Stores an entry of type int
     @param key The key
     @param iValue The new value
     @return TRUE on no error, FALSE in other case
	*/
	bool writeNumEntry(const char *key, int iValue);

	/**
	 ////Recupera el valor de una entrada de tipo bool previamente almacenado
     ////@param key La clave
     ////@param bDefault Valor por defecto
 	 ////@return El valor de la clave o bDefault si no se encontr

	 Retrieves the value of an entry of type bool previously stored
	 @param key The key
	 @param bDefault Default value
	 @return Value of the key or bDefault if not found
	*/
	bool readBoolEntry(const char *key, bool bDefault) const;

	/**
	 ////Recupera el valor de una entrada de tipo int previamente almacenado
     ////@param key La clave
     ////@param iDefault Valor por defecto
 	 ////@return El valor de la clave o iDefault si no se encontr

	 Retrieves the value of an entry of type int previously stored
	 @param key The key
     @param iDefault Default value
     @return Value of the key or bDefault if not found
	*/
	int readNumEntry(const char *key, int iDefault) const;

	/**
	 ////Guarda todas las entradas en el fichero especificado
     Save all entries in the specified file
	*/
	bool save(const char *fileName);

	/**
	 ////Guarda todas las entradas en el fichero de configuracin
     ////(el especificado en el constructor o en el mtodo setFileName

     Save all entries in the configuration file (the one specified in
     constructor or method setFileName)
	*/
	bool sync();

	/**
     ////Equivale a setCurrentPath(const char *path)
     This method is the same as setCurrentPath(const char *path)
	*/
	void setGroup(const char *path) { setCurrentPath(path); };

	/**
     ////Equivale a const char *getCurrentPath()
     This method is the same as const char *getCurrentPath()
	*/
	const char* getGroup() { return getCurrentPath(); };

protected:
	/*
	 ////Nombre del fichero de configuracin
     Name of configuration file
	*/
	char filename[255];

};

//@}

#endif
