/* scriptop.cc
 * This file belongs to Worker, a filemanager for UNIX/X11.
 * Copyright (C) 2002-2004 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
/* $Id: scriptop.cc,v 1.16 2004/10/27 21:08:38 ralf Exp $ */

#include "scriptop.h"
#include "listermode.h"
#include "normalmode.h"
#include "worker.h"
#include "wpucontext.h"
#include "execlass.h"

const char *ScriptOp::name="ScriptOp";

ScriptOp::ScriptOp() : FunctionProto()
{
  type = SCRIPT_NOP;
  label = NULL;
  stack_nr = 0;
  push_string = NULL;
  push_useoutput = false;
  if_test = NULL;
  if_label = NULL;
  description_buf_size = 1024;
  description_buf = (char*)_allocsafe( description_buf_size );
  dodebug = false;
  wpu_recursive = false;
  wpu_take_dirs = false;
  wintype = SCRIPT_WINDOW_LEAVE;
  change_progress = false;
  change_text = false;
  progress = NULL;
  progress_useoutput = false;
  wintext = NULL;
  wintext_useoutput = false;
  hasConfigure = true;
}

ScriptOp::~ScriptOp()
{
  if ( label != NULL ) _freesafe( label );
  if ( push_string != NULL ) _freesafe( push_string );
  if ( if_test != NULL ) _freesafe( if_test );
  if ( if_label != NULL ) _freesafe( if_label );
  if ( progress != NULL ) _freesafe( progress );
  if ( wintext != NULL ) _freesafe( wintext );
  _freesafe( description_buf );
}

ScriptOp*
ScriptOp::duplicate() const
{
  ScriptOp *ta=new ScriptOp();
  ta->setType( type );
  ta->setLabel( label );
  ta->setStackNr( stack_nr );
  ta->setPushString( push_string );
  ta->setPushUseOutput( push_useoutput );
  ta->setIfTest( if_test );
  ta->setIfLabel( if_label );
  ta->setDoDebug( dodebug );
  ta->setWPURecursive( wpu_recursive );
  ta->setWPUTakeDirs( wpu_take_dirs );
  ta->setWinType( wintype );
  ta->setChangeProgress( change_progress );
  ta->setChangeText( change_text );
  ta->setProgress( progress );
  ta->setProgressUseOutput( progress_useoutput );
  ta->setWinText( wintext );
  ta->setWinTextUseOutput( wintext_useoutput );
  return ta;
}

bool
ScriptOp::isName(const char *str)
{
  if(strcmp(str,name)==0) return true; else return false;
}

const char *
ScriptOp::getName()
{
  return name;
}

int
ScriptOp::run( WPUContext *wpu, ActionMessage *msg )
{
  char *tstr, *tstr2, *tstr3;
  ExeClass *ec;
  int erg, l, exeerror;
  
  switch ( type ) {
    case SCRIPT_PUSH:
      /* 1.resolv push_string
         2.if push_useoutput, execute string and get output
         3.push string to stack nr
       */
      if ( push_string != NULL ) {
        tstr = wpu->parse( push_string, a_max( EXE_STRING_LEN - 1024, 256 ), ( ( push_useoutput == true ) ? true : false ) );
        if ( tstr != NULL ) {
          if ( push_useoutput == true ) {
            ec = new ExeClass( ExeClass::EXECUTE_NORMAL, msg->getWorker() );
            ec->cdDir( wpu->getBaseDir() );
            ec->addCommand( "%s", tstr );
            tstr2 = ec->getOutput( &exeerror );
            delete ec;
            
            if ( exeerror == 0 ) {
              if ( tstr2 != NULL ) {
                if ( dodebug == true ) {
                  Requester *req;
                  req = new Requester( Worker::getAGUIX() );
                  tstr3 = (char*)_allocsafe( strlen( catalog.getLocale( 485 ) ) +
                                             strlen( tstr2 ) +
                                             A_BYTESFORNUMBER( stack_nr ) + 1 );
                  sprintf( tstr3, catalog.getLocale( 485 ), tstr2, stack_nr );
                  req->request( catalog.getLocale( 124 ), tstr3, catalog.getLocale( 11 ) );
                  _freesafe( tstr3 );
                  delete req;
                }
                wpu->push( stack_nr, tstr2 );
                _freesafe( tstr2 );
              }
            } else {
              if ( tstr2 != NULL ) _freesafe( tstr2 );
              if ( Worker::getRequester() != NULL ) {
                char *reqstr;
                reqstr = (char*)_allocsafe( strlen( catalog.getLocale( 480 ) ) + strlen( tstr ) + 1 );
                sprintf( reqstr, catalog.getLocale( 480 ), tstr );
                Worker::getRequester()->request( catalog.getLocale( 347 ), reqstr, catalog.getLocale( 11 ) );
                _freesafe( reqstr );
              }
              wpu->continueAtLabel( NULL );
            }
          } else {
            if ( dodebug == true ) {
              Requester *req;
              req = new Requester( Worker::getAGUIX() );
              tstr3 = (char*)_allocsafe( strlen( catalog.getLocale( 485 ) ) +
                                         strlen( tstr ) +
                                         A_BYTESFORNUMBER( stack_nr ) + 1 );
              sprintf( tstr3, catalog.getLocale( 485 ), tstr, stack_nr );
              req->request( catalog.getLocale( 124 ), tstr3, catalog.getLocale( 11 ) );
              _freesafe( tstr3 );
              delete req;
            }
            wpu->push( stack_nr, tstr );
          }
          _freesafe( tstr );
        }
      }
      break;
    case SCRIPT_LABEL:
      /* nothing to do, label is just symbolic */
      break;
    case SCRIPT_IF:
      l = wpu->getCurrentLine();
      if ( if_test != NULL ) {
        erg = wpu->parse_if ( if_test );
        if ( dodebug == true ) {
          Requester *req;
          req = new Requester( Worker::getAGUIX() );
          tstr = (char*)_allocsafe( strlen( catalog.getLocale( 484 ) ) +
                                    strlen( if_test ) +
                                    A_BYTESFORNUMBER( erg ) + 1 );
          sprintf( tstr, catalog.getLocale( 484 ), if_test, erg );
          req->request( catalog.getLocale( 124 ), tstr, catalog.getLocale( 11 ) );
          _freesafe( tstr );
          delete req;
        }
        if ( erg == 1 ) {
          if ( wpu->continueAtLabel( if_label ) != 0 ) {
            Requester *req;
            req = new Requester( Worker::getAGUIX() );
            tstr = (char*)_allocsafe( strlen( catalog.getLocale( 459 ) ) +
                                      strlen( if_label ) +
                                      A_BYTESFORNUMBER( l ) + 1 );
            sprintf( tstr, catalog.getLocale( 459 ), if_label, l + 1 );
            req->request( catalog.getLocale( 347 ), tstr, catalog.getLocale( 11 ) );
            _freesafe( tstr );
            delete req;
          }
        } else if ( erg == 0 ) {
        } else {
          Requester *req;
          req = new Requester( Worker::getAGUIX() );
          tstr = (char*)_allocsafe( strlen( catalog.getLocale( 460 ) ) + A_BYTESFORNUMBER( l ) + 1 );
          sprintf( tstr, catalog.getLocale( 460 ), l + 1 );
          req->request( catalog.getLocale( 347 ), tstr, catalog.getLocale( 11 ) );
          _freesafe( tstr );
          delete req;
          wpu->continueAtLabel( NULL );
        }
      } else {
        Requester *req;
        req = new Requester( Worker::getAGUIX() );
        tstr = (char*)_allocsafe( strlen( catalog.getLocale( 460 ) ) + A_BYTESFORNUMBER( l ) + 1 );
        sprintf( tstr, catalog.getLocale( 460 ), l + 1 );
        req->request( catalog.getLocale( 347 ), tstr, catalog.getLocale( 11 ) );
        _freesafe( tstr );
        delete req;
        wpu->continueAtLabel( NULL );
      }
      break;
    case SCRIPT_END:
      // I use a "feature" of wpu: NULL label will jump to the end
      wpu->continueAtLabel( NULL );
      break;
    case SCRIPT_POP:
      wpu->pop( stack_nr );
      break;
    case SCRIPT_SETTINGS:
      wpu->setRecursive( wpu_recursive );
      wpu->setTakeDirs( wpu_take_dirs );
      break;
    case SCRIPT_WINDOW:
      switch ( wintype ) {
        case SCRIPT_WINDOW_OPEN:
          wpu->openWin();
          break;
        case SCRIPT_WINDOW_CLOSE:
          wpu->closeWin();
          break;
        default:
          break;
      }
      if ( change_progress == true ) {
        if ( progress != NULL ) {
          tstr = wpu->parse( progress, a_max( EXE_STRING_LEN - 1024, 256 ), ( ( progress_useoutput == true ) ? true : false ) );
          if ( tstr != NULL ) {
            if ( progress_useoutput == true ) {
              ec = new ExeClass( ExeClass::EXECUTE_NORMAL );
              ec->cdDir( wpu->getBaseDir() );
              ec->addCommand( "%s", tstr );
              tstr2 = ec->getOutput( &exeerror );
              delete ec;
              
              if ( exeerror == 0 ) {
                _freesafe( tstr );
                if ( tstr2 != NULL ) {
                  tstr = tstr2;
                } else {
                  tstr = NULL;
                }
              } else {
                if ( tstr2 != NULL ) _freesafe( tstr2 );
                if ( Worker::getRequester() != NULL ) {
                  char *reqstr;
                  reqstr = (char*)_allocsafe( strlen( catalog.getLocale( 504 ) ) + strlen( tstr ) + 1 );
                  sprintf( reqstr, catalog.getLocale( 504 ), tstr );
                  Worker::getRequester()->request( catalog.getLocale( 347 ), reqstr, catalog.getLocale( 11 ) );
                  _freesafe( reqstr );
                }
                _freesafe( tstr );
                tstr = NULL;
              }
            }
            if ( tstr != NULL ) {
              l = atoi( tstr );
              _freesafe( tstr );
              if ( ( l < 0 ) || ( l > 100 ) ) l = 0;
              wpu->setWinProgress( l );
            }
          }
        }
      }
      if ( change_text == true ) {
        if ( wintext != NULL ) {
          tstr = wpu->parse( wintext, a_max( EXE_STRING_LEN - 1024, 256 ), ( ( wintext_useoutput == true ) ? true : false ) );
          if ( tstr != NULL ) {
            if ( wintext_useoutput == true ) {
              ec = new ExeClass( ExeClass::EXECUTE_NORMAL );
              ec->cdDir( wpu->getBaseDir() );
              ec->addCommand( "%s", tstr );
              tstr2 = ec->getOutput( &exeerror );
              delete ec;
              if ( exeerror == 0 ) {
                _freesafe( tstr );
                if ( tstr2 != NULL ) {
                  tstr = tstr2;
                } else {
                  tstr = NULL;
                }
              } else {
                if ( tstr2 != NULL ) _freesafe( tstr2 );
                if ( Worker::getRequester() != NULL ) {
                  char *reqstr;
                  reqstr = (char*)_allocsafe( strlen( catalog.getLocale( 504 ) ) + strlen( tstr ) + 1 );
                  sprintf( reqstr, catalog.getLocale( 504 ), tstr );
                  msg->getWorker()->getRequester()->request( catalog.getLocale( 347 ), reqstr, catalog.getLocale( 11 ) );
                  _freesafe( reqstr );
                }
                _freesafe( tstr );
                tstr = NULL;
              }
            }
            if ( tstr != NULL ) {
              wpu->setWinText( tstr );
              _freesafe( tstr );
            }
          }
        }
      }
      break;
    case SCRIPT_GOTO:
      if ( wpu->continueAtLabel( label ) != 0 ) {
        if ( label != NULL ) {
          l = wpu->getCurrentLine();
          Requester *req;
          req = new Requester( Worker::getAGUIX() );
          tstr = (char*)_allocsafe( strlen( catalog.getLocale( 514 ) ) +
                                    strlen( label ) +
                                    A_BYTESFORNUMBER( l ) + 1 );
          sprintf( tstr, catalog.getLocale( 514 ), label, l + 1 );
          req->request( catalog.getLocale( 347 ), tstr, catalog.getLocale( 11 ) );
          _freesafe( tstr );
          delete req;
        }
      }
      break;
    default:
      break;
  }
  return 0;
}

const char *
ScriptOp::getDescription()
{
  int s = description_buf_size - 1;
  std::string str1;
  char buf[A_BYTESFORNUMBER(stack_nr)];
  
  sprintf( buf, "%d", stack_nr );
#if 1
  str1 = catalog.getLocaleCom( 44 );
  str1 += ": ";
#else
  str1 = "";
#endif
  switch( type ) {
    case SCRIPT_PUSH:
      str1 += "push ";
      if ( push_useoutput == true )
        str1 += "the output of ";
      str1 += ( push_string != NULL ) ? push_string : "";
      str1 += " at stack ";
      str1 += buf;
      break;
    case SCRIPT_POP:
      str1 += "pop ";
      str1 += buf;
      break;
    case SCRIPT_LABEL:
      str1 += ( label != NULL ) ? label : "";
      str1 += ":";
      break;
    case SCRIPT_IF:
      str1 += "if ( ";
      str1 += ( if_test != NULL ) ? if_test : "";
      str1 += " ) then goto ";
      str1 += ( if_label != NULL ) ? if_label : "";
      break;
    case SCRIPT_END:
      str1 += "end";
      break;
    case SCRIPT_SETTINGS:
      str1 += "set recursive=";
      if ( wpu_recursive == true )
        str1 += "true, ";
      else
        str1 += "false, ";
      str1 += "take dirs=";
      if ( wpu_take_dirs == true )
        str1 += "true";
      else
        str1 += "false";
      break;
    case SCRIPT_WINDOW:
      str1 += "change user window";
      break;
    case SCRIPT_GOTO:
      str1 += "goto ";
      str1 += ( label != NULL ) ? label : "";
      break;
    default:
      str1 += "nop";
      break;
  }
  strncpy( description_buf, str1.c_str(), s );
  description_buf[s] = '\0';
  return description_buf;
}

bool ScriptOp::isLabel( const char *str )
{
  if ( str != NULL ) {
    if ( type == SCRIPT_LABEL ) {
      if ( label != NULL ) {
        if ( strcmp( str, label ) == 0 ) return true;
      }
    }
  }
  return false;
}

void ScriptOp::setType( scriptop_type_t ntype )
{
  type = ntype;
}

void ScriptOp::setLabel( const char *nlabel )
{
  if ( label != NULL ) _freesafe( label );
  if ( nlabel != NULL ) {
    label = dupstring( nlabel );
  } else {
    label = NULL;
  }
}

void ScriptOp::setStackNr( int nv )
{
  if ( nv >= 0 ) {
    stack_nr = nv;
  }
}

void ScriptOp::setPushString( const char *nstr )
{
  if ( push_string != NULL ) _freesafe( push_string );
  if ( nstr != NULL ) {
    push_string = dupstring( nstr );
  } else {
    push_string = NULL;
  }
}

void ScriptOp::setPushUseOutput( bool nv )
{
  push_useoutput = nv;
}

void ScriptOp::setIfTest( const char *nstr )
{
  if ( if_test != NULL ) _freesafe( if_test );
  if ( nstr != NULL ) {
    if_test = dupstring( nstr );
  } else {
    if_test = NULL;
  }
}

void ScriptOp::setIfLabel( const char *nstr )
{
  if ( if_label != NULL ) _freesafe( if_label );
  if ( nstr != NULL ) {
    if_label = dupstring( nstr );
  } else {
    if_label = NULL;
  }
}

ScriptOp::scriptop_type_t ScriptOp::getType()
{
  return type;
}

const char *ScriptOp::getLabel()
{
  return label;
}

int ScriptOp::getStackNr()
{
  return stack_nr;
}

const char *ScriptOp::getPushString()
{
  return push_string;
}

bool ScriptOp::getPushUseOutput()
{
  return push_useoutput;
}

const char *ScriptOp::getIfTest()
{
  return if_test;
}

const char *ScriptOp::getIfLabel()
{
  return if_label;
}

int ScriptOp::configure()
{
  AGUIX *aguix = Worker::getAGUIX();
  Button *okb,*cb;
  AWindow *win;
  FieldListView *lv;
  ChooseButton *chb;
  Text *ttext;
  int tw,ttw,tth,ttx,tty;
  AGMessage *msg;
  int endmode=-1;
  char *tstr;
  char buf[ A_BYTESFORNUMBER( int ) ];
  AWindow *push_win, *label_win, *if_win, *nop_win, *pop_win, *end_win, *set_win, *win_win, *goto_win;
  int lx, ly;
  GUIElement *bs[2];
  StringGadget *popsg;
  AWindow *winlist[9];
  int cur_win;
  int trow;
  
  ttw=tth=10;
  ttx=tty=5;
  win=new AWindow(aguix);

  tstr=(char*)_allocsafe(strlen(catalog.getLocale(293))+strlen(getName())+1);
  sprintf(tstr,catalog.getLocale(293),getName());
  win->create(NULL,10,10,ttw,tth,0,tstr);
  _freesafe(tstr);

  ttext = (Text*)win->add( new Text( aguix, ttx, tty, catalog.getLocale( 461 ), 1 ) );
  tty += ttext->getHeight() + 5;

  lv = (FieldListView*)win->add( new FieldListView( aguix, ttx, tty, 100, 50, 0 ) );
  lv->setVBarState( 0 );
  lv->setHBarState( 0 );
  lv->setDisplayFocus( true );
  lv->setAcceptFocus( true );
  trow = lv->addRow();
  lv->setText( trow, 0, catalog.getLocale ( 462 ) );
  lv->setPreColors( trow, FieldListView::PRECOLOR_ONLYACTIVE );
  trow = lv->addRow();
  lv->setText( trow, 0, catalog.getLocale ( 463 ) );
  lv->setPreColors( trow, FieldListView::PRECOLOR_ONLYACTIVE );
  trow = lv->addRow();
  lv->setText( trow, 0, catalog.getLocale ( 464 ) );
  lv->setPreColors( trow, FieldListView::PRECOLOR_ONLYACTIVE );
  trow = lv->addRow();
  lv->setText( trow, 0, catalog.getLocale ( 465 ) );
  lv->setPreColors( trow, FieldListView::PRECOLOR_ONLYACTIVE );
  trow = lv->addRow();
  lv->setText( trow, 0, catalog.getLocale ( 466 ) );
  lv->setPreColors( trow, FieldListView::PRECOLOR_ONLYACTIVE );
  trow = lv->addRow();
  lv->setText( trow, 0, catalog.getLocale ( 467 ) );
  lv->setPreColors( trow, FieldListView::PRECOLOR_ONLYACTIVE );
  trow = lv->addRow();
  lv->setText( trow, 0, catalog.getLocale ( 486 ) );
  lv->setPreColors( trow, FieldListView::PRECOLOR_ONLYACTIVE );
  trow = lv->addRow();
  lv->setText( trow, 0, catalog.getLocale ( 488 ) );
  lv->setPreColors( trow, FieldListView::PRECOLOR_ONLYACTIVE );
  trow = lv->addRow();
  lv->setText( trow, 0, catalog.getLocale ( 515 ) );
  lv->setPreColors( trow, FieldListView::PRECOLOR_ONLYACTIVE );
  lv->maximizeX();
  lv->maximizeY();
  if ( type == SCRIPT_PUSH ) lv->setActiveRow( 1 );
  else if ( type == SCRIPT_POP ) lv->setActiveRow( 2 );
  else if ( type == SCRIPT_LABEL ) lv->setActiveRow( 3 );
  else if ( type == SCRIPT_IF ) lv->setActiveRow( 4 );
  else if ( type == SCRIPT_END ) lv->setActiveRow( 5 );
  else if ( type == SCRIPT_SETTINGS ) lv->setActiveRow( 6 );
  else if ( type == SCRIPT_WINDOW ) lv->setActiveRow( 7 );
  else if ( type == SCRIPT_GOTO ) lv->setActiveRow( 8 );
  else lv->setActiveRow( 0 );

  tty += lv->getHeight() + 5;  

  tw = ttx + lv->getWidth() + 5;
  if(tw>ttw) ttw=tw;
  ttx=5;

  push_win = new AWindow( aguix );
  push_win->create( win, ttx, tty, 10, 10, 0, "" );
  push_win->setBorderWidth( 0 );
  lx = 0;
  ly = 0;
  ttext = (Text*)push_win->add( new Text( aguix, lx, ly, catalog.getLocale( 468 ), 1 ) );

  ly += ttext->getHeight() + 5;

  chb = (ChooseButton*)push_win->add( new ChooseButton( aguix, lx, ly,
                                                        ( push_useoutput == true ) ? 1 : 0,
                                                        catalog.getLocale( 469 ), LABEL_RIGHT, 1, 0 ) );
  ly += chb->getHeight() + 5;

  sprintf( buf, "%d", stack_nr );
  ttext = (Text*)push_win->add( new Text( aguix, lx, ly, catalog.getLocale( 470 ), 1 ) );
  lx += ttext->getWidth() + 5;
  StringGadget *psnsg = (StringGadget*)push_win->add( new StringGadget( aguix,
                                                                        lx,
                                                                        ly,
                                                                        20 * aguix->getCharWidth(),
                                                                        buf,
                                                                        0 ) );

  ly += psnsg->getHeight() + 5;
  lx = 0;

  AGUIX::centerElementsY( ttext, psnsg );

  ttext = (Text*)push_win->add( new Text( aguix, lx, ly, catalog.getLocale( 471 ), 1 ) );
  lx += ttext->getWidth() + 5;
  StringGadget *pssg = (StringGadget*)push_win->add( new StringGadget( aguix,
                                                                       lx,
                                                                       ly,
                                                                       20 * aguix->getCharWidth(),
                                                                       ( push_string != NULL ) ? push_string : "",
                                                                       0 ) );
  lx += pssg->getWidth();
  Button *psb = (Button*)push_win->add( new Button( aguix,
                                                    lx,
                                                    ly,
                                                    "F",
                                                    1,
                                                    0,
                                                    0 ) );
  psb->resize( psb->getWidth(), pssg->getHeight() );
  AGUIX::centerElementsY( ttext, pssg );
  
  lx = 0;
  ly += pssg->getHeight() + 5;
  
  ChooseButton *pdcb = (ChooseButton*)push_win->add( new ChooseButton( aguix,
                                                                       lx,
                                                                       ly,
                                                                       dodebug,
                                                                       catalog.getLocale( 474 ),
                                                                       LABEL_RIGHT,
                                                                       1,
                                                                       0 ) );

  push_win->maximizeX();
  push_win->maximizeY();
  
  pop_win = new AWindow( aguix );
  pop_win->create( win, ttx, tty, 10, 10, 0, "" );
  pop_win->setBorderWidth( 0 );
  lx = 0;
  ly = 0;
  ttext = (Text*)pop_win->add( new Text( aguix, lx, ly, catalog.getLocale( 472 ), 1 ) );

  ly += ttext->getHeight() + 5;

  sprintf( buf, "%d", stack_nr );
  ttext = (Text*)pop_win->add( new Text( aguix, lx, ly, catalog.getLocale( 470 ), 1 ) );
  lx += ttext->getWidth() + 5;
  popsg = (StringGadget*)pop_win->add( new StringGadget( aguix,
                                                         lx,
                                                         ly,
                                                         20 * aguix->getCharWidth(),
                                                         buf,
                                                         0 ) );
  AGUIX::centerElementsY( ttext, popsg );

  ly += popsg->getHeight() + 5;
  pop_win->maximizeX();
  pop_win->maximizeY();
  
  label_win = new AWindow( aguix );
  label_win->create( win, ttx, tty, 10, 10, 0, "" );
  label_win->setBorderWidth( 0 );
  lx = 0;
  ly = 0;

  ttext = (Text*)label_win->add( new Text( aguix, lx, ly, catalog.getLocale( 475 ), 1 ) );
  lx += ttext->getWidth() + 5;
  StringGadget *labelsg = (StringGadget*)label_win->add( new StringGadget( aguix,
                                                                          lx,
                                                                          ly,
                                                                          20 * aguix->getCharWidth(),
                                                                          ( label != NULL ) ? label : "",
                                                                          0 ) );
  AGUIX::centerElementsY( ttext, labelsg );

  label_win->maximizeX();
  label_win->maximizeY();
  
  nop_win = new AWindow( aguix );
  nop_win->create( win, ttx, tty, 10, 10, 0, "" );
  nop_win->setBorderWidth( 0 );
  lx = 0;
  ly = 0;
  ttext = (Text*)nop_win->add( new Text( aguix, lx, ly, catalog.getLocale( 476 ), 1 ) );
  nop_win->maximizeX();
  nop_win->maximizeY();
  
  end_win = new AWindow( aguix );
  end_win->create( win, ttx, tty, 10, 10, 0, "" );
  end_win->setBorderWidth( 0 );
  lx = 0;
  ly = 0;
  ttext = (Text*)end_win->add( new Text( aguix, lx, ly, catalog.getLocale( 477 ), 1 ) );
  end_win->maximizeX();
  end_win->maximizeY();
  
  if_win = new AWindow( aguix );
  if_win->create( win, ttx, tty, 10, 10, 0, "" );
  if_win->setBorderWidth( 0 );
  lx = 0;
  ly = 0;
  ttext = (Text*)if_win->add( new Text( aguix, lx, ly, catalog.getLocale( 478 ), 1 ) );
  lx += ttext->getWidth() + 5;
  Button *ifhelpb = (Button*)if_win->add( new Button( aguix, lx, ly, catalog.getLocale( 483 ), 1, 0, 0 ) );
  lx = 0;
  AGUIX::centerElementsY( ttext, ifhelpb );
  
  ly += ifhelpb->getHeight() + 5;

  ttext = (Text*)if_win->add( new Text( aguix, lx, ly, catalog.getLocale( 479 ), 1 ) );
  lx += ttext->getWidth() + 5;
  StringGadget *iftsg = (StringGadget*)if_win->add( new StringGadget( aguix,
                                                                      lx,
                                                                      ly,
                                                                      20 * aguix->getCharWidth(),
                                                                      ( if_test != NULL ) ? if_test : "",
                                                                      0 ) );
  lx += iftsg->getWidth();
  Button *iftb = (Button*)if_win->add( new Button( aguix,
                                                   lx,
                                                   ly,
                                                   "F",
                                                   1,
                                                   0,
                                                   0 ) );
  iftb->resize( iftb->getWidth(), iftsg->getHeight() );
  AGUIX::centerElementsY( ttext, iftsg );

  ly += iftsg->getHeight() + 5;
  lx = 0;

  ttext = (Text*)if_win->add( new Text( aguix, lx, ly, catalog.getLocale( 475 ), 1 ) );
  lx += ttext->getWidth() + 5;
  StringGadget *iflsg = (StringGadget*)if_win->add( new StringGadget( aguix,
                                                                      lx,
                                                                      ly,
                                                                      20 * aguix->getCharWidth(),
                                                                      ( if_label != NULL ) ? if_label : "",
                                                                      0 ) );
  AGUIX::centerElementsY( ttext, iflsg );

  lx = 0;
  ly += iflsg->getHeight() + 5;
  
  ChooseButton *idcb = (ChooseButton*)if_win->add( new ChooseButton( aguix,
                                                                     lx,
                                                                     ly,
                                                                     dodebug,
                                                                     catalog.getLocale( 474 ),
                                                                     LABEL_RIGHT,
                                                                     1,
                                                                     0 ) );

  if_win->maximizeX();
  if_win->maximizeY();
  
  set_win = new AWindow( aguix );
  set_win->create( win, ttx, tty, 10, 10, 0, "" );
  set_win->setBorderWidth( 0 );
  lx = 0;
  ly = 0;
  set_win->addTextFromString( catalog.getLocale( 487 ), lx, ly, 5, NULL, NULL, &ly );
  ly += 5;

  ChooseButton *setrcb = (ChooseButton*)set_win->add( new ChooseButton( aguix,
                                                                        lx,
                                                                        ly,
                                                                        wpu_recursive,
                                                                        catalog.getLocale( 330 ),
                                                                        LABEL_RIGHT,
                                                                        1,
                                                                        0 ) );
  ly += setrcb->getHeight() + 5;

  ChooseButton *settdcb = (ChooseButton*)set_win->add( new ChooseButton( aguix,
                                                                         lx,
                                                                         ly,
                                                                         wpu_take_dirs,
                                                                         catalog.getLocale( 513 ),
                                                                         LABEL_RIGHT,
                                                                         1,
                                                                         0 ) );
  set_win->maximizeX();
  set_win->maximizeY();
  
  win_win = new AWindow( aguix );
  win_win->create( win, ttx, tty, 10, 10, 0, "" );
  win_win->setBorderWidth( 0 );
  lx = 0;
  ly = 0;
  Button *winhelpb = (Button*)win_win->add( new Button( aguix, lx, ly, catalog.getLocale( 483 ), 1, 0, 0 ) );
  ly += winhelpb->getHeight() + 5;

  ChooseButton *cwcb = (ChooseButton*)win_win->add( new ChooseButton( aguix, lx, ly,
                                                        ( wintype == SCRIPT_WINDOW_LEAVE ) ? 0 : 1,
                                                        catalog.getLocale( 490 ), LABEL_RIGHT, 1, 0 ) );
  lx += cwcb->getWidth() + 5;
  CycleButton *cwcycb = (CycleButton*)win_win->add( new CycleButton( aguix, lx, ly, 100, 1, 0, 0 ) );
  cwcycb->addOption( catalog.getLocale( 491 ) );
  cwcycb->addOption( catalog.getLocale( 492 ) );
  cwcycb->resize( cwcycb->getMaxSize(), cwcycb->getHeight() );
  if ( wintype == SCRIPT_WINDOW_CLOSE ) cwcycb->setOption( 1 );
  else cwcycb->setOption( 0 );

  lx = 0;                                                        
  ly += cwcb->getHeight() + 5;

  BevelBox *winbb1 = (BevelBox*)win_win->add( new BevelBox( aguix, lx, ly, 10, 10, 1 ) );
  lx = 5;
  ly += 5;
  
  ChooseButton *cpcb = (ChooseButton*)win_win->add( new ChooseButton( aguix, lx, ly,
                                                        ( change_progress == true ) ? 1 : 0,
                                                        catalog.getLocale( 493 ), LABEL_RIGHT, 1, 0 ) );
  ly += cpcb->getHeight() + 5;
  lx = 15;

  ttext = (Text*)win_win->add( new Text( aguix, lx, ly, catalog.getLocale( 494 ), 1 ) );
  lx += ttext->getWidth() + 5;
  StringGadget *prosg = (StringGadget*)win_win->add( new StringGadget( aguix,
                                                                       lx,
                                                                       ly,
                                                                       20 * aguix->getCharWidth(),
                                                                       ( progress != NULL ) ? progress : "",
                                                                       0 ) );
  lx += prosg->getWidth();
  Button *prob = (Button*)win_win->add( new Button( aguix,
                                                    lx,
                                                    ly,
                                                    "F",
                                                    1,
                                                    0,
                                                    0 ) );
  prob->resize( prob->getWidth(), prosg->getHeight() );
  AGUIX::centerElementsY( ttext, prosg );
  
  lx = 15;
  ly += prosg->getHeight();

  ChooseButton *prouocb = (ChooseButton*)win_win->add( new ChooseButton( aguix, lx, ly,
                                                           ( progress_useoutput == true ) ? 1 : 0,
                                                           catalog.getLocale( 495 ), LABEL_RIGHT, 1, 0 ) );
  ly += prouocb->getHeight() + 5;
  winbb1->resize( winbb1->getWidth(), ly - winbb1->getY() );
  ly += 5;
  
  lx = 0;

  BevelBox *winbb2 = (BevelBox*)win_win->add( new BevelBox( aguix, lx, ly, 10, 10, 1 ) );
  lx = 5;
  ly += 5;
  
  ChooseButton *cwtcb = (ChooseButton*)win_win->add( new ChooseButton( aguix, lx, ly,
                                                         ( change_text == true ) ? 1 : 0,
                                                         catalog.getLocale( 496 ), LABEL_RIGHT, 1, 0 ) );
  ly += cwtcb->getHeight() + 5;
  lx = 15;

  ttext = (Text*)win_win->add( new Text( aguix, lx, ly, catalog.getLocale( 497 ), 1 ) );
  lx += ttext->getWidth() + 5;
  StringGadget *wintextsg = (StringGadget*)win_win->add( new StringGadget( aguix,
                                                                           lx,
                                                                           ly,
                                                                           20 * aguix->getCharWidth(),
                                                                           ( wintext != NULL ) ? wintext : "",
                                                                           0 ) );
  lx += wintextsg->getWidth();
  Button *wintextb = (Button*)win_win->add( new Button( aguix,
                                                        lx,
                                                        ly,
                                                        "F",
                                                        1,
                                                        0,
                                                        0 ) );
  wintextb->resize( wintextb->getWidth(), wintextsg->getHeight() );
  AGUIX::centerElementsY( ttext, wintextsg );
  
  lx = 15;
  ly += wintextsg->getHeight();

  ChooseButton *wintextuocb = (ChooseButton*)win_win->add( new ChooseButton( aguix, lx, ly,
                                                               ( wintext_useoutput == true ) ? 1 : 0,
                                                               catalog.getLocale( 498 ), LABEL_RIGHT, 1, 0 ) );
  ly += wintextuocb->getHeight() + 5;
  winbb2->resize( winbb2->getWidth(), ly - winbb2->getY() );
  ly += 5;
  
  lx = 0;

  win_win->maximizeX();
  win_win->maximizeY();
  
  goto_win = new AWindow( aguix );
  goto_win->create( win, ttx, tty, 10, 10, 0, "" );
  goto_win->setBorderWidth( 0 );
  lx = 0;
  ly = 0;

  ttext = (Text*)goto_win->add( new Text( aguix, lx, ly, catalog.getLocale( 475 ), 1 ) );
  lx += ttext->getWidth() + 5;
  StringGadget *gotosg = (StringGadget*)goto_win->add( new StringGadget( aguix,
                                                                         lx,
                                                                         ly,
                                                                         20 * aguix->getCharWidth(),
                                                                         ( label != NULL ) ? label : "",
                                                                         0 ) );
  AGUIX::centerElementsY( ttext, gotosg );

  goto_win->maximizeX();
  goto_win->maximizeY();
  
  win->maximizeX();
  win->maximizeY();
  
  tty = win->getHeight();
  ttw = win->getWidth();

  okb = (Button*)win->add( new Button( aguix,
                                       5,
                                       tty,
                                       catalog.getLocale( 11 ),
                                       1,
                                       0,
                                       0 ) );
  cb = (Button*)win->add( new Button( aguix,
                                      5,
                                      tty,
                                      catalog.getLocale( 8 ),
                                      1,
                                      0,
                                      0 ) );
  bs[0] = okb;
  bs[1] = cb;
  AGUIX::scaleElementsW( ttw, 5, 10, -1, false, false, bs, NULL, 2 );
  
  psnsg->resize( ttw - psnsg->getX() - 5 - 5, psnsg->getHeight() );
  pssg->resize( ttw - pssg->getX() - 5 - 5 - psb->getWidth(), pssg->getHeight() );
  psb->move( pssg->getX() + pssg->getWidth(), psb->getY() );
  popsg->resize( ttw - popsg->getX() - 5 - 5, popsg->getHeight() );
  labelsg->resize( ttw - labelsg->getX() - 5 - 5, labelsg->getHeight() );
  iftsg->resize( ttw - iftsg->getX() - 5 - 5 - iftb->getWidth(), iftsg->getHeight() );
  iftb->move( iftsg->getX() + iftsg->getWidth(), iftb->getY() );
  iflsg->resize( ttw - iflsg->getX() - 5 - 5, iflsg->getHeight() );
  ifhelpb->move( ttw - 5 - 5 - ifhelpb->getWidth(), ifhelpb->getY() );
  prosg->resize( ttw - prosg->getX() - 5 - 5 - prob->getWidth() - 5, prosg->getHeight() );
  prob->move( prosg->getX() + prosg->getWidth(), prob->getY() );
  winbb1->resize( ttw - 5 - 5, winbb1->getHeight() );
  wintextsg->resize( ttw - wintextsg->getX() - 5 - 5 - wintextb->getWidth() - 5, wintextsg->getHeight() );
  wintextb->move( wintextsg->getX() + wintextsg->getWidth(), wintextb->getY() );
  winbb2->resize( ttw - 5 - 5, winbb2->getHeight() );
  gotosg->resize( ttw - gotosg->getX() - 5 - 5, gotosg->getHeight() );

  push_win->maximizeX();
  pop_win->maximizeX();
  label_win->maximizeX();
  if_win->maximizeX();
  set_win->maximizeX();
  win_win->maximizeX();
  goto_win->maximizeX();
  
  tty += okb->getHeight() + 5;
  
  tth=tty;
  
  win->setDoTabCycling( true );
  win->resize(ttw,tth);
  win->setMaxSize(ttw,tth);
  win->setMinSize(ttw,tth);
  
  winlist[0] = nop_win;
  winlist[1] = push_win;
  winlist[2] = pop_win;
  winlist[3] = label_win;
  winlist[4] = if_win;
  winlist[5] = end_win;
  winlist[6] = set_win;
  winlist[7] = win_win;
  winlist[8] = goto_win;
  switch( type ) {
    case SCRIPT_PUSH:
      cur_win = 1;
      break;
    case SCRIPT_POP:
      cur_win = 2;
      break;
    case SCRIPT_LABEL:
      cur_win = 3;
      break;
    case SCRIPT_IF:
      cur_win = 4;
      break;
    case SCRIPT_END:
      cur_win = 5;
      break;
    case SCRIPT_SETTINGS:
      cur_win = 6;
      break;
    case SCRIPT_WINDOW:
      cur_win = 7;
      break;
    case SCRIPT_GOTO:
      cur_win = 8;
      break;
    default:
      cur_win = 0;
      break;
  }
  winlist[cur_win]->show();
  
  win->show();
  for(;endmode==-1;) {
    msg=aguix->WaitMessage(win);
    if(msg!=NULL) {
      switch(msg->type) {
        case AG_CLOSEWINDOW:
          if(msg->closewindow.window==win->getWindow()) endmode=1;
          break;
        case AG_BUTTONCLICKED:
          if(msg->button.button==okb) endmode=0;
          else if(msg->button.button==cb) endmode=1;
          else if ( msg->button.button == psb ) {
            tstr = getPushFlag();
            if ( tstr != NULL ) {
              pssg->insertAtCursor( tstr );
              _freesafe( tstr );
            }
          } else if ( msg->button.button == iftb ) {
            tstr = getIfFlag();
            if ( tstr != NULL ) {
              iftsg->insertAtCursor( tstr );
              _freesafe( tstr );
            }
          } else if ( msg->button.button == prob ) {
            tstr = getPushFlag();
            if ( tstr != NULL ) {
              prosg->insertAtCursor( tstr );
              _freesafe( tstr );
            }
          } else if ( msg->button.button == wintextb ) {
            tstr = getPushFlag();
            if ( tstr != NULL ) {
              wintextsg->insertAtCursor( tstr );
              _freesafe( tstr );
            }
          } else if ( msg->button.button == ifhelpb ) {
            Requester *req = new Requester( aguix );
            req->request( catalog.getLocale( 124 ), catalog.getLocale( 482 ), catalog.getLocale( 11 ) );
            delete req;
          } else if ( msg->button.button == winhelpb ) {
            Requester *req = new Requester( aguix );
            req->request( catalog.getLocale( 124 ), catalog.getLocale( 489 ), catalog.getLocale( 11 ) );
            delete req;
          }
          break;
        case AG_FIELDLV_ONESELECT:
        case AG_FIELDLV_MULTISELECT:
          if ( msg->fieldlv.lv == lv ) {
            winlist[cur_win]->hide();
            cur_win = lv->getActiveRow();
            if ( ( cur_win < 0 ) || ( cur_win > 8 ) ) cur_win = 0;
            winlist[cur_win]->show();
          }
          break;
      }
      aguix->ReplyMessage(msg);
    }
  }
  
  if ( endmode == 0 ) {
    // ok
    switch ( lv->getActiveRow() ) {
      case 1:
        type = SCRIPT_PUSH;
        if ( chb->getState() == 1 ) push_useoutput = true;
        else push_useoutput = false;
        stack_nr = atoi( psnsg->getText() );
        if ( stack_nr < 0 ) stack_nr = 0;
        setPushString( pssg->getText() );
        setDoDebug( ( pdcb->getState() == 1 ) ? true : false );
        break;
      case 2:
        type = SCRIPT_POP;
        stack_nr = atoi( popsg->getText() );
        if ( stack_nr < 0 ) stack_nr = 0;
        break;
      case 3:
        type = SCRIPT_LABEL;
        setLabel( labelsg->getText() );
        break;
      case 4:
        type = SCRIPT_IF;
        setIfTest( iftsg->getText() );
        setIfLabel( iflsg->getText() );
        setDoDebug( ( idcb->getState() == 1 ) ? true : false );
        break;
      case 5:
        type = SCRIPT_END;
        break;
      case 6:
        type = SCRIPT_SETTINGS;
        setWPURecursive( ( setrcb->getState() == 1 ) ? true : false );
        setWPUTakeDirs( ( settdcb->getState() == 1 ) ? true : false );
        break;
      case 7:
        type = SCRIPT_WINDOW;
        if ( cwcb->getState() == 1 ) {
          switch ( cwcycb->getSelectedOption() ) {
            case 1:
              setWinType( SCRIPT_WINDOW_CLOSE );
              break;
            default:
              setWinType( SCRIPT_WINDOW_OPEN );
              break;
          }
        } else {
          setWinType( SCRIPT_WINDOW_LEAVE );
        }
        if ( cpcb->getState() == 1 ) {
          setChangeProgress( true );
          setProgress( prosg->getText() );
          setProgressUseOutput( ( prouocb->getState() == 1 ) ? true : false );
        } else {
          setChangeProgress( false );
        }
        if ( cwtcb->getState() == 1 ) {
          setChangeText( true );
          setWinText( wintextsg->getText() );
          setWinTextUseOutput( ( wintextuocb->getState() == 1 ) ? true : false );
        } else {
          setChangeText( false );
        }
        break;
      case 8:
        type = SCRIPT_GOTO;
        setLabel( gotosg->getText() );
        break;
      default:
        type = SCRIPT_NOP;
        break;
    }
  }
  win->close();
  delete win;

  return endmode;
}

bool ScriptOp::save(Datei *fh)
{
  if ( fh == NULL ) return false;
  switch(type) {
    case SCRIPT_PUSH:
      WConfig::configPutPair( fh, "type", "push" );
      break;
    case SCRIPT_LABEL:
      WConfig::configPutPair( fh, "type", "label" );
      break;
    case SCRIPT_IF:
      WConfig::configPutPair( fh, "type", "if" );
      break;
    case SCRIPT_END:
      WConfig::configPutPair( fh, "type", "end" );
      break;
    case SCRIPT_POP:
      WConfig::configPutPair( fh, "type", "pop" );
      break;
    case SCRIPT_SETTINGS:
      WConfig::configPutPair( fh, "type", "settings" );
      break;
    case SCRIPT_WINDOW:
      WConfig::configPutPair( fh, "type", "window" );
      break;
    case SCRIPT_GOTO:
      WConfig::configPutPair( fh, "type", "goto" );
      break;
    default:
      WConfig::configPutPair( fh, "type", "nop" );
      break;
  }
  WConfig::configPutPairBool( fh, "pushuseoutput", push_useoutput );
  WConfig::configPutPairBool( fh, "dodebug", dodebug );
  WConfig::configPutPairBool( fh, "wpurecursive", wpu_recursive );
  WConfig::configPutPairBool( fh, "wputakedirs", wpu_take_dirs );
  WConfig::configPutPairNum( fh, "stacknr", stack_nr );
  if ( label != NULL ) {
    WConfig::configPutPairString( fh, "label", label );
  }
  if ( push_string != NULL ) {
    WConfig::configPutPairString( fh, "pushstring", push_string );
  }
  if ( if_test != NULL ) {
    WConfig::configPutPairString( fh, "iftest", if_test );
  }
  if ( if_label != NULL ) {
    WConfig::configPutPairString( fh, "iflabel", if_label );
  }

  switch ( wintype ) {
    case SCRIPT_WINDOW_OPEN:
      WConfig::configPutPair( fh, "wintype", "open" );
      break;
    case SCRIPT_WINDOW_CLOSE:
      WConfig::configPutPair( fh, "wintype", "close" );
      break;
    default:
      WConfig::configPutPair( fh, "wintype", "leave" );
      break;
  }
  WConfig::configPutPairBool( fh, "changeprogress", change_progress );
  WConfig::configPutPairBool( fh, "changetext", change_text );
  WConfig::configPutPairBool( fh, "progressuseoutput", progress_useoutput );
  WConfig::configPutPairBool( fh, "wintextuseoutput", wintext_useoutput );
  if ( progress != NULL ) {
    WConfig::configPutPairString( fh, "progress", progress );
  }
  if ( wintext != NULL ) {
    WConfig::configPutPairString( fh, "wintext", wintext );
  }

  return true;
}

char *ScriptOp::getPushFlag()
{
  OwnOp *o;
  char *tstr;
  
  o = new OwnOp();
  tstr = o->getFlag();
  delete o;
  return tstr;
}

char *ScriptOp::getIfFlag()
{
  AGUIX *aguix = Worker::getAGUIX();
  Button *okb,*cb;
  AWindow *win;
  FieldListView *lv;
  int w,h,x,y;
  AGMessage *msg;
  int endmode=-1;
  char *returnstr;
  Button *b2[2];
  int i, trow, nr, nr2;
  
  w=400;
  h=300;
  x=y=5;
  win=new AWindow(aguix);
  win->create(NULL,10,10,w,h,0,catalog.getLocale(338));

  win->addTextFromString( catalog.getLocale( 481 ), x, y, 5, NULL, NULL, &y );
  y += 5;

  lv = (FieldListView*)win->add( new FieldListView( aguix,
                                                    x,
                                                    y,
                                                    w - 2 * x,
                                                    200,
                                                    0 ) );
  y+=lv->getHeight()+5;
  lv->setHBarState(2);
  lv->setVBarState(2);

  lv->setNrOfFields( 3 );
  lv->setFieldWidth( 1, 3 );
  
  nr = sizeof( scriptop_flags ) / sizeof( scriptop_flags[0] );
  nr2 = sizeof( ownop_flags ) / sizeof( ownop_flags[0] );
  
  for ( i = 0; i < nr; i++ ) {
    trow = lv->addRow();
    lv->setText( trow, 0, scriptop_flags[i].flag );
    lv->setText( trow, 2, catalog.getLocaleFlag( scriptop_flags[i].catcode ) );
    lv->setPreColors( trow, FieldListView::PRECOLOR_ONLYACTIVE );
    lv->setData( trow, i );
  }
  
  for( i = 0; i < nr2; i++ ) {
    trow = lv->addRow();
    lv->setText( trow, 0, ownop_flags[ i ].flag );
    lv->setText( trow, 2, catalog.getLocaleFlag( ownop_flags[ i ].catcode ) );
    lv->setPreColors( trow, FieldListView::PRECOLOR_ONLYACTIVE );
    lv->setData( trow, i + nr );
  }

  okb=(Button*)win->add(new Button(aguix,
                                   5,
                                   y,
                                   catalog.getLocale(11),
                                   1,
                                   0,
                                   0));
  cb=(Button*)win->add(new Button(aguix,
                                  5,
                                  y,
                                  catalog.getLocale(8),
                                  1,
                                  0,
                                  0));
  y+=okb->getHeight()+5;

  b2[0] = okb;
  b2[1] = cb;
  AGUIX::scaleElementsW( w, 5, 10, -1, false, false, (GUIElement**)b2, NULL, 2 );
  
  h=y;
  win->setDoTabCycling( true );
  win->resize(w,h);
  win->setMinSize(w,h);
  win->show();
  for(;endmode==-1;) {
    msg=aguix->WaitMessage(win);
    if(msg!=NULL) {
      switch(msg->type) {
        case AG_CLOSEWINDOW:
          if(msg->closewindow.window==win->getWindow()) endmode=1;
          break;
        case AG_BUTTONCLICKED:
          if(msg->button.button==okb) endmode=0;
          else if(msg->button.button==cb) endmode=1;
          break;
        case AG_SIZECHANGED:
          AGUIX::scaleElementsW( msg->size.neww, 5, 10, -1, false, false, (GUIElement**)b2, NULL, 2 );
          for ( i = 0; i < 2; i++ ) {
            b2[i]->move( b2[i]->getX(), msg->size.newh - 5 - b2[i]->getHeight() );
          }
          lv->resize( msg->size.neww - 10, b2[0]->getY() - 5 - lv->getY() );
          break;
      }
      aguix->ReplyMessage(msg);
    }
  }
  
  returnstr=NULL;
  if(endmode==0) {
    // ok
    trow = lv->getActiveRow();
    if ( lv->isValidRow( trow ) == true ) {
      i = lv->getData( trow );
      if ( i >= nr ) {
        returnstr = dupstring( ownop_flags[i - nr].flag );
      } else {
        returnstr = dupstring( scriptop_flags[i].flag );
      }
    }
  }
  
  win->close();
  delete win;

  return returnstr;
}

void ScriptOp::setDoDebug( bool nv )
{
  dodebug = nv;
}

bool ScriptOp::getDoDebug()
{
  return dodebug;
}

void ScriptOp::setWPURecursive( bool nv )
{
  wpu_recursive = nv;
}

bool ScriptOp::getWPURecursive()
{
  return wpu_recursive;
}

void ScriptOp::setWPUTakeDirs( bool nv )
{
  wpu_take_dirs = nv;
}

bool ScriptOp::getWPUTakeDirs()
{
  return wpu_take_dirs;
}

void ScriptOp::setWinType( scriptop_window_t nv )
{
  wintype = nv;
}

ScriptOp::scriptop_window_t ScriptOp::getWinType()
{
  return wintype;
}

void ScriptOp::setChangeProgress( bool nv )
{
  change_progress = nv;
}

bool ScriptOp::getChangeProgress()
{
  return change_progress;
}

void ScriptOp::setChangeText( bool nv )
{
  change_text = nv;
}

bool ScriptOp::getChangeText()
{
  return change_text;
}

void ScriptOp::setProgress( const char *str )
{
  if ( progress != NULL ) _freesafe( progress );
  if ( str != NULL ) {
    progress = dupstring( str );
  } else {
    progress = NULL;
  }
}

const char *ScriptOp::getProgress()
{
  return progress;
}

void ScriptOp::setProgressUseOutput( bool nv )
{
  progress_useoutput = nv;
}

bool ScriptOp::getProgressUseOutput()
{
  return progress_useoutput;
}

void ScriptOp::setWinText( const char *str )
{
  if ( wintext != NULL ) _freesafe( wintext );
  if ( str != NULL ) {
    wintext = dupstring( str );
  } else {
    wintext = NULL;
  }
}

const char *ScriptOp::getWinText()
{
  return wintext;
}

void ScriptOp::setWinTextUseOutput( bool nv )
{
  wintext_useoutput = nv;
}

bool ScriptOp::getWinTextUseOutput()
{
  return wintext_useoutput;
}

