/************************************************************************

WorldPrint aka wprint

This is the wprint program that allows Netscape users to print web
pages written in (at least) UNICODE, BIG5, SJIS and all ISO-8859-*
charsets provided a suitable TTF font.

Copyright (C) 2000  Eduardo Trapani (etrapani@unesco.org.uy)

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/
#include <stdio.h>
#include <iconv.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <ctype.h>
#include <sys/stat.h>
#include "wprint.h"

struct tt_info	ttglobalinfo;

static struct fontInfo fnc[MAXFONTS] = {
{"/Times-Roman","",NULL,NULL,72,72,14.0},
{"/Times-Bold","",NULL,NULL,72,72,14.0},
{"/Times-Italic","",NULL,NULL,72,72,14.0},
{"/Times-BoldItalic","",NULL,NULL,72,72,14.0},
{"/Courier","",NULL,NULL,72,72,14.0},
{"/Courier-Bold","",NULL,NULL,72,72,14.0},
{"/Courier-Oblique","",NULL,NULL,72,72,14.0},
{"/Courier-BoldOblique","",NULL,NULL,72,72,14.0},
{"/Helvetica","",NULL,NULL,72,72,14.0},
{"/Helvetica-Bold","",NULL,NULL,72,72,14.0},
{"/Helvetica-Oblique","",NULL,NULL,72,72,14.0},
{"/Helvetica-BoldOblique","",NULL,NULL,72,72,14.0},
{"/Symbol","",NULL,NULL,72,72,14.0},
{"","",NULL,NULL,72,72,14.0}
};

static char *sfi_htmldoc[] = { "/FC", "/FCB", "/FCI", "/FCBI", "/FT", "/FTB", "/FTI", "/FTBI", "/FH", "/FHB", "/FHI", "/FHBI", "/FS", "/FSB", "/FSI", "/FSBI", NULL};
static char *sfi_htmldoc1_8[] = { "/F0", "/F1", "/F2", "/F3", "/F4", "/F5", "/F6", "/F7","/F8","/F9","/F10","/F11","/F12", NULL };

static char *sfi_netscape[] = { "f0", "f1", "f2", "f3", "f4", "f5", "f6", "f7", "of", NULL };

static char *sfi_mozilla[] = { "f0", "f1", "f2", "f3", "f4", "f5", "f6", "f7", "f8", "f9", "f10", "f11", "f12", NULL };

static unsigned char pschars64[] =
	{"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/"};

static char* psnames64[] =
	{"A","B","C","D","E","F","G","H",
	 "I","J","K","L","M","N","O","P",
	 "Q","R","S","T","U","V","W","X",
	 "Y","Z","a","b","c","d","e","f",
	 "g","h","i","j","k","l","m","n",
	 "o","p","q","r","s","t","u","v",
	 "w","x","y","z","zero","one","two","three",
	 "four","five","six","seven","eight","nine","plus","slash"};

static unsigned char pscharsiso[] =
	{"ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/"
         "\xa1\xa2\xa3\xa4\xa5\xa6\xa7\xa8\xa9\xaa\xab\xac\xad\xae\xaf\xb0"
         "\xb1\xb2\xb3\xb4\xb5\xb6\xb7\xb8\xb9\xba\xbb\xbc\xbd\xbe\xbf\xc0"
         "\xc1\xc2\xc3\xc4\xc5\xc6\xc7\xc8\xc9\xca\xcb\xcc\xcd\xce\xcf\xd0"
         "\xd1\xd2\xd3\xd4\xd5\xd6\xd7\xd8\xd9\xda\xdb\xdc\xdd\xde\xdf\xe0"
         "\xe1\xe2\xe3\xe4\xe5\xe6\xe7\xe8\xe9\xea\xeb\xec\xed\xee\xef\xf0"
         "\xf1\xf2\xf3\xf4\xf5\xf6\xf7\xf8\xf9\xfa\xfb\xfc\xfd\xfe\xff"};
static char* psnamesiso[] =
	{"A","B","C","D","E","F","G","H",
	 "I","J","K","L","M","N","O","P",
	 "Q","R","S","T","U","V","W","X",
	 "Y","Z","a","b","c","d","e","f",
	 "g","h","i","j","k","l","m","n",
	 "o","p","q","r","s","t","u","v",
	 "w","x","y","z","zero","one","two","three",
	 "four","five","six","seven","eight","nine","plus","slash",
	 "exclamdown", "cent", "sterling", "currency", "yen", "brokenbar",
	 "section", "dieresis", "copyright", "ordfeminine",
	 "guillemotleft", "logicalnot", "hyphen", "registered", "macron",
	 "degree", "plusminus", "twosuperior", "threesuperior", "acute",
	 "mu", "paragraph", "periodcentered", "cedilla", "onesuperior",
	 "ordmasculine", "guillemotright", "onequarter", "onehalf",
	 "threequarters", "questiondown", "Agrave", "Aacute",
	 "Acircumflex", "Atilde", "Adieresis", "Aring", "AE",
	 "Ccedilla", "Egrave", "Eacute", "Ecircumflex", "Edieresis",
	 "Igrave", "Iacute", "Icircumflex", "Idieresis", "Eth", "Ntilde",
	 "Ograve", "Oacute", "Ocircumflex", "Otilde", "Odieresis",
	 "multiply", "Oslash", "Ugrave", "Uacute", "Ucircumflex",
	 "Udieresis", "Yacute", "Thorn", "germandbls", "agrave",
	 "aacute", "acircumflex", "atilde", "adieresis", "aring", "ae",
	 "ccedilla", "egrave", "eacute", "ecircumflex", "edieresis",
	 "igrave", "iacute", "icircumflex", "idieresis", "eth", "ntilde",
	 "ograve", "oacute", "ocircumflex", "otilde", "odieresis",
	 "divide", "oslash", "ugrave", "uacute", "ucircumflex",
	 "udieresis", "yacute", "thorn", "ydieresis"
	};

char		**psnames = psnames64;
unsigned char	*pschars = pschars64;
static Font	*fontroot = NULL;
static Map	charmap;
int	verbose = 0;
static int	nchars = 64;

Source		*sourceglobal;

int strwse2cmp(unsigned char *str1,unsigned char *str2)
{
int	i,c,res;

	c = -1;
	for (i = 0;str2[i] != 0;i++)
	{
		if (isspace(str2[i]))
		{
			c = str2[i];
			str2[i] = 0;
			break;
		}
	}
	res = strcmp(str1,str2);
	if (c != -1)
		str2[i] = c;

	return(res);
}
/* sourceInit */
int	sourceInitMozillaNetscape(Source *s,char *encoding)
{
static struct extraConv ec;

	s->extraData = &ec;

	ec.convorig = iconv_open("UNICODEBIG",encoding);
	ec.convascii = iconv_open("UNICODEBIG","ISO_8859-1");
	ec.convlocal = iconv_open("UNICODEBIG","UNICODELITTLE");

	if (ec.convorig == (iconv_t) -1 || ec.convascii == (iconv_t) -1
	|| ec.convlocal == (iconv_t) -1)
	{
		perror("Impossible conversion");
		return(-1);
	}
	else
		return(0);
}
int	sourceFinalizeMozillaNetscape(Source *s)
{
struct extraConv *ec;

	ec = (struct extraConv *) s->extraData;
	iconv_close(ec->convorig);
	iconv_close(ec->convascii);
	iconv_close(ec->convlocal);
	ec->convorig = (iconv_t) -1;
	ec->convascii = (iconv_t) -1;
	ec->convlocal = (iconv_t) -1;
	return(0);
}
int	sourceInitHtmlDoc(Source *s,char *encoding)
{
static struct extraConv ec;

	s->extraData = &ec;

	if (s->creator.version >= 0x010800)
	{
		return(sourceInitMozillaNetscape(s,encoding));
	}
	else
	{
		ec.convorig = iconv_open("UNICODEBIG",encoding);

		if (ec.convorig == (iconv_t) -1)
		{
			perror("Impossible conversion");
			return(-1);
		}
		else
			return(0);
	}
}
int	sourceFinalizeHtmlDoc(Source *s)
{
struct extraConv *ec;

	if (s->creator.version >= 0x010800)
	{
		return(sourceFinalizeMozillaNetscape(s));
	}
	else
	{
		ec = (struct extraConv *) s->extraData;
		iconv_close(ec->convorig);

		ec->convlocal = (iconv_t) -1;
		return(0);
	}
}
/* sourceConversion */
iconv_t sourceConversionMozillaNetscape(Source *s,char *text)
{
struct extraConv *ec;

	ec = (struct extraConv *) s->extraData;

	if (s->creator.id == NETSCAPE)
	{
		if (strwse2cmp("show",wordMemory(0,NULL)) == 0)
			return(ec->convorig);
		else
			return(ec->convascii);
	}
	else if (s->creator.id == MOZILLA)
	{
		if (strwse2cmp("show",wordMemory(0,NULL)) == 0)
			return(ec->convascii);
		else
			return(ec->convlocal);
	}
	return((iconv_t) -1);
}
iconv_t sourceConversionHtmlDoc(Source *s,char *text)
{
struct extraConv *ec;

	ec = (struct extraConv *) s->extraData;

	return(ec->convorig);
}
/* sourceGetFonts */
int sourceGetFontsMozillaNetscape(Source *s,char *word)
{
static char *sfm_netscape[] = { "/Times-Roman", "/Times-Bold", "/Times-Italic", "/Times-BoldItalic", "/Courier", "/Courier-Bold", "/Courier-Oblique", "/Courier-BoldOblique", "/Ryumin-Light-RKSJ-H", NULL};

static char *sfm_mozilla[] = { "/Times-Roman", "/Times-Bold", "/Times-BoldItalic", "/Times-Italic", "/Helvetica", "/Helvetica-Bold", "/Helvetica-BoldOblique", "/Helvetica_Oblique", "/Courier", "/Courier-Bold", "/Courier-BoldOblique", "/Courier-Oblique", "/Symbol", NULL };
int	i;
char	**search;

	if (s->creator.id == MOZILLA)
		search = sfm_mozilla;
	else if (s->creator.id == NETSCAPE)
		search = sfm_netscape;
	else
		return(-1);

	for (i = 0;search[i] != NULL && strwse2cmp(search[i],word) != 0;i++)
		;

	if (search[i] == NULL)
		return(-1);
	else
		return(i);
}
int sourceGetFontsHtmlDoc(Source *s,char *word)
{
static char *sfm_htmldoc[] = { "/Courier","/Courier-Bold", "/Courier-Oblique", "/Courier-BoldOblique", "/Times-Roman", "/Times-Bold", "/Times-Italic", "/Times-BoldItalic", "/Helvetica", "/Helvetica-Bold", "/Helvetica-Oblique", "/Helvetica-BoldOblique", "/Symbol", "/Symbol", "/Symbol", "/Symbol", NULL};
int	i;
char	**search;

	if (s->creator.id == HTMLDOC)
		search = sfm_htmldoc;
	else
		return(-1);

	for (i = 0;search[i] != NULL && strwse2cmp(search[i],word) != 0;i++)
		;
	if (search[i] == NULL)
		return(-1);
	else
		return(i);
}
/* sourceFontId */
int	sourceFontIdMozillaNetscape(Source *s,char *text)
{
char	**search;
int	i;

	if (s->creator.id == NETSCAPE)
		search = sfi_netscape;
	else if (s->creator.id == MOZILLA)
		search = sfi_mozilla;
	else
		return(-1);

	for (i = 0;search[i] != NULL && strwse2cmp(search[i],text) != 0;i++)
		;
	if (search[i] == NULL)
		return(-1);
	else
		return(i);
}
int	sourceFontIdHtmlDoc(Source *s,char *text)
{
char	**search;
int	i;

	if ((s->creator.id == HTMLDOC) && s->creator.version > 0x010700)
		search = sfi_htmldoc1_8;
	else if ((s->creator.id == HTMLDOC) && s->creator.version < 0x010800)
		search = sfi_htmldoc;
	else
		return(-1);

	for (i = 0;search[i] != NULL && strwse2cmp(search[i],text) != 0;i++)
		;

	if (search[i] == NULL)
		return(-1);
	else
		return(i);
}
/* sourceCommand */
char	*sourceCommandMozillaNetscape(Source *s,int type,int id)
{
char	*retorno = NULL;

	switch (type)
	{
		case FONT:
				/* FIXME: check for upper bound */
				if (s->creator.id == MOZILLA)
					retorno = sfi_mozilla[id];
				else if (s->creator.id == NETSCAPE)
					retorno = sfi_netscape[id];
				break;
		case COMMAND:
				switch(id)
				{
					case SHOW:
						if (s->creator.id == MOZILLA)
							retorno = "show";
						else if (s->creator.id == NETSCAPE)
							retorno = "show";
						break;
				}
				break;
	}
	return(retorno);
}
char	*sourceCommandHtmlDoc(Source *s,int type,int id)
{
char	*retorno = NULL;

	switch (type)
	{
		case FONT:
				/* FIXME: check for upper bound */
				if ((s->creator.id == HTMLDOC))
					retorno = sfi_netscape[id];
				break;
		case COMMAND:
				switch(id)
				{
					case SHOW:
						if ((s->creator.id == HTMLDOC))
							retorno = "S";
						break;
				}
				break;
	}
	return(retorno);
}

/* sourceConvertibleTextContext */
int sourceConvertibleTextContextMozillaNetscape(char *s)
{
	if (isdigit(*s))
		return(0);
	else
		return(1);
}
int sourceConvertibleTextContextHtmlDoc(char *s)
{
	return(1);
}

Source *openSource(char *text)
{
static Source source;

	source.extraData = NULL;
	source.sourceInit = NULL;
	source.sourceConversion = NULL;
	source.sourceGetFonts = NULL;
	source.sourceFontId = NULL;
	source.sourceCommand = NULL;
	source.sourceFinalize = NULL;
	source.sourceConvertibleTextContext = NULL;

	source.creator.major = source.creator.minor = 0;
	strcpy(source.creator.name,"");
	strcpy(source.creator.desc,"");
	strncpy(source.creator.desc,text,MAXNAME);
	sscanf(text,"%*s%s%d.%d",source.creator.name,&source.creator.major,&source.creator.minor);
	source.creator.version = versionfromparts(source.creator.major,source.creator.minor);

	if (verbose)
		fprintf(stderr,"Source application %s %d.%d [%s]\n",
		source.creator.name,source.creator.major,source.creator.minor,text);

	if (strcasecmp(source.creator.name,"mozilla") == 0)
	{
		source.creator.id = AUTOMOZILLA;
		source.sourceInit = sourceInitMozillaNetscape;
		source.sourceConversion = sourceConversionMozillaNetscape;
		source.sourceGetFonts = sourceGetFontsMozillaNetscape;
		source.sourceFontId = sourceFontIdMozillaNetscape;
		source.sourceCommand = sourceCommandMozillaNetscape;
		source.sourceConvertibleTextContext = sourceConvertibleTextContextMozillaNetscape;
		source.sourceFinalize = sourceFinalizeMozillaNetscape;
	}
	else if (strcasecmp(source.creator.name,"htmldoc") == 0 && source.creator.version >= 0x010800)
	{
		source.creator.id = HTMLDOC;
		source.sourceInit = sourceInitHtmlDoc;
		source.sourceConversion = sourceConversionHtmlDoc;
		source.sourceGetFonts = sourceGetFontsHtmlDoc;
		source.sourceFontId = sourceFontIdHtmlDoc;
		source.sourceCommand = sourceCommandHtmlDoc;
		source.sourceConvertibleTextContext = sourceConvertibleTextContextHtmlDoc;
		source.sourceFinalize = sourceFinalizeHtmlDoc;
	}
	else if (strcasecmp(source.creator.name,"htmldoc") == 0 && source.creator.version < 0x010800)
	{
		source.creator.id = HTMLDOC;
		source.sourceInit = sourceInitHtmlDoc;
		source.sourceConversion = sourceConversionHtmlDoc;
		source.sourceGetFonts = sourceGetFontsHtmlDoc;
		source.sourceFontId = sourceFontIdHtmlDoc;
		source.sourceCommand = sourceCommandHtmlDoc;
		source.sourceConvertibleTextContext = sourceConvertibleTextContextHtmlDoc;
		source.sourceFinalize = sourceFinalizeHtmlDoc;
	}
	else
	{
		source.creator.id = UNKNOWN;
		return(NULL);
	}
	return(&source);
}

int	closeSource(Source *s)
{
	return(0);
}

void	hexdump(unsigned char *c,int n)
{
int	i;

	for (i = 0;i < n;i++)
	{
		fprintf(stderr,"%02X%c ",c[i],c[i]);
	}
	fprintf(stderr,"\n");
}
char	*wordMemory(int id,char *word)
{
const int	howmany = 9;
static char	lastwords[9][MAXWORD + 1];
static int	head = 0,tail = 0;

	if (id == 1) /* Add word */
	{
		strcpy(lastwords[tail],word);
		tail++;
		if (tail == howmany)
			tail = 0;
		if (tail == head)
		{
			head++;
			if (head == howmany)
				head = 0;
		}
		return(word);
	}
	else if (id <= 0)
	{
	int	which;

		for (which = tail;id < 1;id++)
		{
			which--;
			if (which < 0)
				which = howmany - 1;
		}
		return(lastwords[which]);
	}
	return(NULL);
}
int	processFontList(char *fontlist,const char *path)
{
int	i;
char	*from,*comma;

	from = fontlist;
	for (i = 0;i < MAXFONTS;i++)
	{
		if (from != NULL)
		{
		char	*p;

			fnc[i].xr = 72;
			fnc[i].yr = 72;
			fnc[i].pt = 14.0;

			comma = strchr(from,',');
			if (comma != NULL)
				*comma = 0;
			p = strchr(from,'=');
			if (p == NULL)
				fnc[i].iconv = NULL;
			else
			{
			char	enco[MAXENCODING + 1];

				*p = 0;
				if (strchr(p + 1,';') != NULL)
					sscanf(p + 1,"%[^;];%d;%d;%lf",enco,
					&fnc[i].xr,&fnc[i].yr,&fnc[i].pt);
				else
					strcpy(enco,p + 1);

				if (strcmp (enco,"UNICODEBIG") != 0)
				{
					fnc[i].iconv = iconv_open(enco,"UNICODEBIG");
					if (fnc[i].iconv == (iconv_t) -1)
						fnc[i].iconv = NULL;
				}
				else
					fnc[i].iconv = NULL;
			}
			strcpy(fnc[i].filename,path);
			strcat(fnc[i].filename,from);
			if (comma != NULL)
				from = comma + 1;
			else
				from = NULL;
		}
		else
		{
			strcpy(fnc[i].filename,fnc[i - 1].filename);
			fnc[i].iconv = fnc[i - 1].iconv;
			fnc[i].xr = fnc[i - 1].xr;
			fnc[i].yr = fnc[i - 1].yr;
			fnc[i].pt = fnc[i - 1].pt;
		}
	}
	return(i);
}

int	convertFromHex(char *buf,int len)
{
int	i,j,r;

	for (i = j = 0;i < len;i += 2,j++)
	{
		sscanf(buf + i,"%2x",&r);
		buf[j] = r;
	}
	buf[j] = 0;
	return(j);
}
void FTError(char *string, int error)
{
  fprintf(stderr, "FreeType error: %s: 0x%04lx\n", string, error);
  exit(2);
}

void Error(char *string)
{
  fprintf(stderr, "Error: %s\n", string);
  exit(2);
}
int	translated_char(Font *font,unsigned short unicode)
{
int	i;

	for (i = 0;i < font->nextchar;i++)
		if (font->map[i] == unicode)
			return(pschars[i]);

	return('!'); /* it is not defined */
}
Font	*addChar(unsigned short unicode,int fontid)
{
Font	*aux;
unsigned short	eid;
unsigned short	row,col;

	row = unicode >> 8;
	col = unicode  & 0x0ff;
	if (charmap.rows[row] == NULL)
	{
		charmap.rows[row] = (unsigned char *) malloc(256 * sizeof(char));
		if (charmap.rows[row] == NULL)
		{
			fprintf(stderr,"Out of memory\n");
			exit(-1);
		}
		memset(charmap.rows[row],0,256);
	}
	if (charmap.rows[row][col] & (1 << fontid))
	{
	Font	*search;
	int	i;

		for (search = fnc[fontid].firstpage;search != NULL;search=search->next)
		{
			for (i = 0;i < search->nextchar;i++)
				if (search->map[i] == unicode)
					return(search);
		}
		return(NULL);
	}
	else
		charmap.rows[row][col] |= (1 << fontid);

	eid = 0;
	for (aux = fontroot;aux != NULL;aux = aux->next)
	{
		if (aux->id == fontid)
		{
			if (aux->nextchar < nchars)
				break;
			else
				eid++;
		}
	}

	if (aux == NULL)
	{
		aux = newFont(fontid,eid);
	}
	aux->map[aux->nextchar] = unicode;
	aux->nextchar++;
	return(aux);
}

Font	*newFont(unsigned short fid,unsigned short feid)
{
Font	*newfont;

	if (verbose)
		fprintf(stderr,"New font %d %d\n",fid,feid);
	newfont = (Font *) malloc(sizeof(Font));
	if (newfont == NULL)
	{
		fprintf(stderr,"Out of memory\n");
		exit(-1);
	}
	newfont->map = (unsigned short int *) malloc(sizeof(unsigned short int) * nchars);
	if (newfont->map == NULL)
	{
		fprintf(stderr,"Out of memory\n");
		exit(-1);
	}
	newfont->next = NULL;
	newfont->nexteid = NULL;
	newfont->nextchar = 0;
	newfont->eid = feid;
	newfont->id = fid;

	if (fontroot == NULL)
		fontroot = newfont;
	else
	{
	Font	*aux;

		for (aux = fontroot;aux->next != NULL;aux = aux->next)
			;
		aux->next = newfont;
	}
	if (feid == 0)
	{
		fnc[fid].firstpage = newfont;
	}
	else
	{
	Font	*aux;

		for (aux = fontroot;aux->next != NULL && !(aux->id == fid && aux->eid == feid - 1);aux = aux->next)
			;
		if (aux != NULL)
			aux->nexteid = newfont;
	}
	return(newfont);
}

FILE	*processConfiguration(FILE *in,char *entry,char *encoding)
{
char	fontprefix[MAXFILENAME + 1];
char	fontlist[MAXLINE + 1];
char	line[MAXLINE + 1];
FILE	*out;
int	i;
char	*p;

	out = NULL;
	strcpy(fontprefix,"");
	while (out == NULL && fgets(line,MAXLINE,in) != NULL)
	{
		i = strlen(line);
		if (i < 2)
			continue;
		if (line[i - 1] == '\n')
			line[i - 1] = 0;

		p = strtok(line,":");
		if (p == NULL)
			continue;
		else if (strcmp(p,"fontpath") == 0)
		{
			p = strtok(NULL,":");
			if (p != NULL)
				strcpy(fontprefix,p);
		}
		else if (strcmp(p,entry) == 0)
		{
			p = strtok(NULL,":");
			if (p != NULL)
				strcpy(encoding,p);
			p = strtok(NULL,":");
			if (p != NULL)
			{
				strcpy(fontlist,p);
				processFontList(fontlist,fontprefix);
			}
			p = strtok(NULL,":");
			if (p != NULL)
			{
				if (*p != '|')
					out = fopen(p,"w+b");
				else
					out = popen(p + 1,"w");
			}
		}
	}
	return(out);
}

void	printUsage()
{
	printf("wprint -p filename.ps -i encoding -f font_list -v -x -c conffile -l entry -u\n");
	printf("\n");
	printf("\t-c conffile\tThe filenam to the config file.  This file is not read\n");
	printf("\t\t\tunless you use the -l option.  The default filename\n");
	printf("\t\t\tis /etc/wprint.conf.\n");
	printf("\n");
	printf("\t-i encoding\tThe encoding of the file, KOI8-R or ISO_8859-3 or UTF-8\n");
	printf("\t\t\tor whatever encoding is supported by iconv (the list is\n");
	printf("\t\t\trather impressive).  The default is UTF-8.  NONE means\n");
	printf("\t\t\tthat no conversion should be made.\n");
	printf("\n");
	printf("\t-l entry\tThe entry in the config file.  The default entry is\n");
	printf("\t\t\t'default'.\n");
	printf("\n");
	printf("\t-f font_list\tYou can either write the *filename* of a unicode font\n");
	printf("\t\t\tsuch a cyberbit or you can type a list in the form:\n");
	printf("\t\t\t\tfont1[=encoding1[;xr;yr;pt]],\n");
	printf("\t\t\t\tfont2[=encoding2[;xr;yr;pt]], ...\n");
	printf("\t\t\tIf you don't specify the font encoding the unicode map\n");
	printf("\t\t\twill be used.  The values xr, yr and pt default to\n");
	printf("\t\t\t72, 72 and 14.0.\n");
	printf("\n");
	printf("\t-p filename\tThe name of the file you want to print.  If you don't\n");
	printf("\t\t\tspecify it standard input will be used instead.\n");
	printf("\n");
	printf("\t-u\t\tPrints the usage and exits.\n");
	printf("\n");
	printf("\t-v\t\tJust adds some diagnostic output to stderr.\n");
	printf("\n");
	printf("\t-x\t\tBy default the font tables are constructed using\n");
	printf("\t\t\ta 64 character map.  You can also use a 159 character\n");
	printf("\t\t\tmap that lead to fewer subfonts.  It is optional\n");
	printf("\t\t\tbecause I have not tested it enough.\n");
}

int main(int argc,char *argv[])
{
extern char *optarg;
extern int optind,opterr,optopt;
FILE	*tmp;
char	encoding[MAXENCODING + 1];
char	file[MAXFILENAME + 1];
char	conf[MAXFILENAME + 1];
char	section[MAXNAME + 1];
FILE	*in,*out;
int	readconf = 0;
int	readfile = 0;
int	givenfont = 0;
int	conffile = 0;
int	c,i;
Font	*aux;

	strcpy(file,"");
	strncpy(encoding,"NONE",MAXENCODING);
	strcpy(conf,"/etc/wprint.conf");
	strcpy(section,"default");
	out = stdout;
	while ((c=getopt(argc, argv, "l:c:p:i:f:vxu")) != EOF)
	{
		switch(c)
		{
			case 'i':	strncpy(encoding,optarg,MAXENCODING);
					break;
			case 'p':	strncpy(file,optarg,MAXFILENAME);
					readfile = 1;
					break;
			case 'c':	strncpy(conf,optarg,MAXFILENAME);
					conffile = 1;
					break;
			case 'l':	strncpy(section,optarg,MAXNAME);
					readconf = 1;
					break;
			case 'f':	processFontList(optarg,"");
					givenfont = 1;
					break;
			case 'v':	verbose = 1;
					break;
			case 'x':	pschars = pscharsiso;
					psnames = psnamesiso;
					nchars = strlen((const char*) pschars);
					break;
			case 'u':	printUsage();
					break;
		}
	}
	
	in = fopen(conf,"rb");

	if (in != NULL && (optind == 1 || (optind == 3 && readfile == 1)))
		readconf = 1;

	if (readconf == 0 && conffile == 0)
	{
		if (givenfont == 0 && strcmp(encoding,"NONE") != 0)
		{
			perror("You have to specify a font with -f if there is no configuration file and encoding is not NONE");
			exit(-1);
		}
		out = stdout;
	}
	else if ((in = fopen(conf,"rb")) == NULL)
	{
		perror("Unable to read configuration file");
		exit(-1);
	}
	else
	{
		if ((out = processConfiguration(in,section,encoding)) == NULL)
		{
			fprintf(stderr,"Unable to process configuration file\n");
			exit(-1);
		}
	}
	if (in != NULL)
		fclose(in);
	if (strlen(file) != 0)
	{
		in = fopen(file,"rb");
		if (in == NULL)
		{
			perror("Unable to open postscript file");
			exit(-1);
		}
	}
	else
		in = stdin;

	if (strcmp(encoding,"NONE") == 0)
	{
	int	c;
		while ((c = fgetc(in)) != EOF)
			fputc(c,out);
		exit(0);
	}

	for (i = 0;i < MAXFONTS;i++)
	{
	struct stat state;
	int	ret;

		ret = stat(fnc[i].filename,&state);
		if (ret != 0 || !(S_ISREG(state.st_mode)))
		{
			fprintf(stderr,"Check your fontpath setting, font file %s does not exist\n",fnc[i].filename);
			exit(-1);
		}
	}
	tmp = tmpfile();
	if (tmp == NULL)
	{
		perror("Opening temporary file");
		exit(0);
	}
	convertPsText(encoding,in,tmp);

	if (initTT(&ttglobalinfo) != 0)
	{
		fprintf(stderr,"Error initializing true type engine/library\n");
		exit(1);
	}
	for (i = 0;i < MAXFONTS;i++)
	{
		if (fnc[i].firstpage != NULL)
		{
			for (aux = fnc[i].firstpage;aux != NULL;aux = aux->nexteid)
			{
				ttglobalinfo.font.xr = fnc[i].xr;
				ttglobalinfo.font.yr = fnc[i].yr;
				ttglobalinfo.font.charsize = fnc[i].pt;
				setFont(&ttglobalinfo,fnc[i].filename);
				generateFont(out,aux,&ttglobalinfo);
				unsetFont(&ttglobalinfo);
			}
		}
	}
	finalizeTT(&ttglobalinfo);
	printTranslatedPs(tmp,out);
	if (readconf == 1)
		fclose(out);
	if (readfile == 1)
		fclose(in);
	fclose(tmp);
	return(0);
}
int printTranslatedPs(FILE *tmp,FILE *dest)
{
int	c;
char	word[MAXWORD + 1];
int	len,r,i,keepon;
int	def;
long	lastWordAt = 0;
FILE	*out;

	word[0] = 0;
	def = 0;
	keepon = 1;

	out = tmpfile();
	if (out == NULL)
	{
		perror("Opening temporary file");
		exit(0);
	}

	while (keepon && (r = nextWord(tmp,word,MAXWORD,&len,NULL)) != EOF)
	{
		if (r == NEXTWORDTEXT)
		{
			fputs("(",out);
			fputs(word,out);
			fputs(")",out);
			continue;
		}
		else if (strwse2cmp("%%EndProlog",word) == 0)
		{
			fputs(word,out);
			keepon = 0;
		}
		else if (word[0] == '/')
		{
			lastWordAt = ftell(out);
			fputs(word,out);
		}
		else if (strwse2cmp("findfont",word) == 0)
		{
			i = sourceglobal->sourceGetFonts(sourceglobal,wordMemory(-1,NULL));

			if (i != -1)
			{
				def = i;
				if (def < MAXFONTS)
				{
					strcpy(fnc[def].original,wordMemory(-1,NULL) + 1);
					fseek(out,lastWordAt,SEEK_SET);
					fprintf(out," /TF%d ",composedFont(def,0));
					def++;
				}
			}

			fputs(word,out);
		}
		else
			fputs(word,out);
	}

	while ((c = fgetc(tmp)) != EOF)
		fputc(c,out);
	rewind(out);

	while ((c = fgetc(out)) != EOF)
		fputc(c,dest);

	fclose(out);

	return(0);
}
int generateFont(FILE *out,Font *aux,struct tt_info *ti)
{
int	i;
unsigned char	*p;
char	*filename;
unsigned short index;
time_t	now;
double	sfactor;

	filename = fnc[aux->id].filename;
	if (verbose)
		fprintf(stderr,"Creating font (%s) %d %d\n",fnc[aux->id].filename,aux->id,aux->eid);
	fputs("%!PS-Adobe-3.0\n",out);
	fprintf(out,"%%%%Creator: wprint %s Copyright Eduardo Trapani\n",VERSION);
	time(&now);
	fprintf(out,"%%%%CreationDate: %s\n",ctime(&now));
	fputs("%%EndComments\n",out);
	fputs("\n",out);
	/*fputs("%%BeginProlog\n",out);*/
	fputs("\n",out);
	fputs("%%BeginResource: fonts\n",out);
	fprintf(out,"%%%% Converted from TrueType font %s\n",ti->font.names[TN_FONTNAME]);
	fputs("% Font Segments 1\n",out);
	fputs("20 dict begin\n",out);
	fputs("/FontType 3 def\n",out);
	fprintf(out,"/FontName (TF%d) def\n",(aux->id << 8) + aux->eid);
	fputs("/FontInfo 9 dict dup begin\n",out);
	fprintf(out,"/FullName (%s page id %d) def\n",
	ti->font.names[TN_FULLNAME],(aux->id << 8) + aux->eid);
	fprintf(out,"/Notice (%s) def\n",ti->font.names[TN_NOTICE]);
	fprintf(out,"/FamilyName (%s) def\n",ti->font.names[TN_FAMILY]);
	fprintf(out,"/Weight (%s) def\n",ti->font.names[TN_WEIGHT]);
	fprintf(out,"/version (%s) def\n",ti->font.names[TN_VERSION]);

	if (ti->font.ttpostscript != NULL)
	{
		fprintf(out,"/italicAngle %9.7f def\n"
		,(double)ti->font.ttpostscript->italicAngle);
		fprintf(out,"/underlineThickness %d def\n",
		ti->font.ttpostscript->underlineThickness);
		fprintf(out,"/underlinePosition %d def\n",
		ti->font.ttpostscript->underlinePosition);
		fprintf(out,"/isFixedPitch %s def end def\n",
		ti->font.ttpostscript->isFixedPitch ? "true" : "false");
	}
	else
	{
		fprintf(out,"/italicAngle %9.7f def\n",0.0);
		fprintf(out,"/underlineThickness %d def\n",100);
		fprintf(out,"/underlinePosition %d def\n",0);
		fprintf(out,"/isFixedPitch false def end def\n");
	}

	sfactor = 1000.0 / ti->font.ttheader->Units_Per_EM;
	sfactor /= 1000.0;

	fprintf(out,"/FontMatrix [%9.7f 0 0 %9.7f 0 0] def\n",sfactor,sfactor);
	fprintf(out,"/FontBBox [%d %d %d %d] def\n",
	ti->font.ttheader->xMin,
	ti->font.ttheader->yMin,
	ti->font.ttheader->xMin,
	ti->font.ttheader->yMin
	);
	fputs("/Encoding [\n",out);
	for (i = 0;i < 256;i++)
	{
		if ((p = (unsigned char *)strchr(pschars,i)) == NULL ||
		((int)(p - pschars)) >= aux->nextchar)
			fputs("/.notdef ",out);
		else
			fprintf(out,"/%s ",psnames[(int) (p - pschars)]);
		if ((i % 8) == 0)
			fputs("\n",out);
	}
	fputs("] def\n",out);
	fprintf(out,"/CharProcs %d dict def CharProcs begin\n",aux->nextchar + 1);
	fputs("/.notdef { % Empty glyph\n",out);
	fputs("1426 0 -1214 -524 2640 2146 setcachedevice\n",out);
	fputs("115 -244 moveto 115 1916 lineto stroke closepath\n",out);
	fputs("newpath 115 1916 moveto 1426 1916 lineto stroke closepath\n",out);
	fputs("newpath 1426 1916 moveto 1426 -244 lineto stroke closepath\n",out);
	fputs("newpath 1426 -244 moveto 115 -244 lineto stroke closepath\n",out);
	fputs("} bind def\n",out);
	for (i = 0;i < aux->nextchar;i++)
	{
		index = aux->map[i];
		if (fnc[aux->id].iconv != NULL)
		{
		unsigned char uni[2],other[1];
		size_t luni,lother;
		char *puni,*pother;
		unsigned short *er;

			er = (unsigned short *) uni;
			*er = htons(aux->map[i]);
			luni = 2; lother = 1;
			puni = (char *) uni; pother = (char *) other;
			if (iconv(fnc[aux->id].iconv,(char **)&puni,&luni,&pother,&lother) == -1)
				fprintf(stderr,"Error switching font mapping\n");
			else
				index = *other;
		}
		createGlyphs(out,ti,index,psnames[i]);
	}
	fputs("end\n",out);
	fputs("/BuildGlyph {\n",out);
	fputs("exch /CharProcs get exch\n",out);
	fputs("2 copy known not {pop /.notdef} if get exec } bind def\n",out);
	fputs("/BuildChar { 1 index /Encoding get exch get\n",out);
	fputs("1 index /Encoding get exec } bind def\n",out);
	fprintf(out,"currentdict end /TF%d exch definefont pop\n",(aux->id << 8) + aux->eid);
	fputs("%%EndResource: fonts\n",out);
	return(0);
}
int	nextWord(FILE *in,char *buf,int maxlength,int *actuallength,char **orig)
{
int	c;
int	i;
int	ret;
static char originalNumber[MAXWORD];

	if (orig != NULL)
		*orig = NULL;

	/* Skip whitespace */
	while ((c = fgetc(in)) != EOF && isspace(c))
		;

	if (c == EOF)
		return(EOF);
	i = 0;

	if (c == '%')
	{
	int	keepon = 1;

		buf[i] = c;
		i++;
		while (keepon && (c = fgetc(in)) != EOF)
		{
			buf[i] = c;
			i++;
			if (c == '\n')
				keepon = 0;
		}
		buf[i] = 0;
		ret = NEXTWORDPS;
	}
	else if (c == '(')
	{
	int	escape = 0;
	int	keepon = 1;

		while (keepon && (c = fgetc(in)) != EOF)
		{
			if (escape)
			{
				if (c >= '0' && c <= '7')
				{
					c = (c - '0') * 64 + (fgetc(in) - '0')
					* 8 + (fgetc(in) - '0');
				}
				buf[i] = c;
				i++;
				escape = 0;
			}
			else if (c == ')')
			{
				keepon = 0;
				buf[i] = 0;
			}
			else if (c == '\\')
				escape = 1;
			else
			{
				buf[i] = c;
				i++;
			}
		}
		ret = NEXTWORDTEXT;
	}
	else if (c == '<')
	{
	int	count = 0;
	int	keepon = 1;
	char	x[3];
	int	n;

		/* FIXME: '<<' should be handled better */
		c = fgetc(in);

		if (c == '<')
		{
			buf[0] = '<';
			buf[1] = '<';
			buf[2] = 0;
			i = 2;
			wordMemory(1,buf);
			ret = NEXTWORDPS;
		}
		else
		{
			/* FIXME: '<<' should be handled better */
			ungetc(c,in);

			x[2] = 0;
			while (keepon && (c = fgetc(in)) != EOF)
			{
				originalNumber[count + 1] = c;
				x[count % 2] = c;
				count++;
				if ((count %2) == 0)
				{
					sscanf(x,"%2x",&n);
					buf[i] = n;
					i++;
				}
				else if (c == '>')
					keepon = 0;
			}
			originalNumber[0] = '<';
			originalNumber[count + 1] = 0;
			if (orig != NULL)
				*orig = originalNumber;
			ret = NEXTWORDTEXT;
		}
	}
	else
	{
	int	keepon = 1;

		buf[i] = c;
		i++;
		while (keepon && (c = fgetc(in)) != EOF)
		{
			buf[i] = c;
			i++;
			if (isspace(c))
				keepon = 0;
			else if (c == '(' || c == '/')
			{
				i--;
				ungetc(c,in);
				keepon = 0;
			}
		}
		buf[i] = 0;
		wordMemory(1,buf);
		ret = NEXTWORDPS;
	}
	*actuallength = i;
	return(ret);
}
int	convertPsText(const char *encoding,FILE *in,FILE *tmp)
{
char	pstext[MAXPSTEXT];
int	i_pstext;
char	psunicodetext[MAXPSTEXT * 2];
char	word[MAXWORD];
char	line[MAXLINE + 1];
float	fontsize = 24.0;
int	i_word;
int	last_id;
iconv_t	conv;
int	len,len1,r;
size_t	iin,iout;
int	f,g;
const char	*sin;
char		*sout;
unsigned short	*uni;
Font		*font;
int		fontid = 0;
char		*p;
int		aux;
char		*show = NULL;
Source		*source = NULL;
char		*textnumber;

	i_pstext = 0;
	i_word = 0;

	while (fgets(line,MAXLINE,in) != NULL)
	{
		fputs(line,tmp);
		if (strncmp(line,"%%EndProlog",11) == 0)
			break;
		else if (strncmp(line,"%%Creator:",10) == 0)
		{
			source = openSource(line);
			sourceglobal = source;
			if (source == NULL)
				return(-1);
			else
			{
				source->sourceInit(source,encoding);
			}
		}
	}
	while ((r = nextWord(in,pstext,MAXPSTEXT,&len,&textnumber)) != EOF)
	{
		if (r == NEXTWORDPS)
		{
			fprintf(tmp,"%s",pstext);
			continue;
		}
		else if (source->sourceConvertibleTextContext(wordMemory(0,NULL)) == 0)
		{
			fprintf(tmp,"%s ",textnumber);
			continue;
		}
		/* Text read in, get source and set font and fontsize */
		if (source->creator.id == AUTOMOZILLA)
		{
			p = wordMemory(0,NULL);
			if (strncmp(p,"moveto",6) == 0)
				source->creator.id = MOZILLA;
			else
				source->creator.id = NETSCAPE;
		}
		if (show == NULL)
			show = source->sourceCommand(source,COMMAND,SHOW);
		if (source->creator.id == MOZILLA)
		{
			p = wordMemory(-3,NULL);
			aux = source->sourceFontId(source,p);
			if (aux != -1)
			{
				fontid = aux;
				fontsize = atof(wordMemory(-4,NULL));
			}
		}
		else if (source->creator.id == NETSCAPE)
		{
			p = wordMemory(0,NULL);
			aux = source->sourceFontId(source,p);
			if (aux != -1)
			{
				fontid = aux;
				fontsize = atof(wordMemory(-1,NULL));
			}
		}
		else if ((source->creator.id == HTMLDOC))
		{
		int where = 1;

			/* Check the type of line */
			if (*(wordMemory(-4,NULL)) == '/')
				where = -4;
			else if (*(wordMemory(-1,NULL)) == '/')
				where = -1;
			if (where <= 0)
			{
				/* FIXME: bold check, maybe I sould go back
				until I find a suitable fontsize? */

				if (isdigit(*wordMemory(where - 1,NULL)))
					fontsize = atof(wordMemory(where - 1,NULL));
				else if (isdigit(*wordMemory(where - 2,NULL)))
					fontsize = atof(wordMemory(where - 2,NULL));
				p = wordMemory(where,NULL);
				fontid = source->sourceFontId(source,p);
			}
		}
		last_id = composedFont(fontid,0);
		/* FIX:  Assumes that next word is NEXTWORDPS */
		r = nextWord(in,word,MAXWORD,&len1,NULL);

		conv = source->sourceConversion(source,wordMemory(0,NULL));
		/* Converts and outputs text */
		sin = pstext;
		sout = psunicodetext;
		iin = len;
		iout = MAXPSTEXT * 2;
		if (iconv(conv,(char **) &sin,&iin,&sout,&iout) == -1)
		{
			perror("Warning while converting");
		}
		g = ((MAXPSTEXT * 2) - iout) / 2;
		uni = (unsigned short *) psunicodetext;
		fputc('(',tmp);
		for (f = 0;f < g;f++)
		{
			font = addChar(ntohs(uni[f]),fontid);
			if (font == NULL)
			{
				fprintf(stderr,"No font!\n");
				exit(0);
			}
			if (last_id == composedFont(font->id,font->eid))
			fputc(translated_char(font,ntohs(uni[f])),tmp);
			else
			{
				fprintf(tmp,") %s /TF%d findfont %f scalefont setfont (%c",show,composedFont(font->id,font->eid),fontsize,translated_char(font,ntohs(uni[f])));
				last_id = composedFont(font->id,font->eid);
			}
		}
		fprintf(tmp,") %s\n",show);
	}
	source->sourceFinalize(source);
	closeSource(source);
	rewind(tmp);
	return(0);
}

