/*-
# X-BASED ABACUS
#
#  AbacusD.c
#
###
#
#  Copyright (c) 1999 - 2005	David Albert Bagley, bagleyd@tux.org
#
#  Abacus demo and neat pointers from
#  Copyright (c) 1991 - 1998	Luis Fernandes, elf@ee.ryerson.ca
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "useful",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/* Methods demo file for Abacus */

#include "AbacusP.h"

#ifndef WINVER
static void InitializeAbacusDemo(Widget request, Widget renew);
static void ExposeAbacusDemo(Widget renew, XEvent * event, Region region);

#if 0
static void ResizeAbacusDemo(AbacusWidget w);

#endif
static void DestroyAbacusDemo(Widget old);
static Boolean SetValuesAbacusDemo(Widget current, Widget request, Widget renew);

static void QuitAbacusDemo(AbacusWidget w, XEvent * event, char **args, int nArgs);
static void HideAbacusDemo(AbacusWidget w, XEvent * event, char **args, int nArgs);
static void SelectAbacusDemo(AbacusWidget w, XEvent * event, char **args, int nArgs);
static void ClearAbacusDemo(AbacusWidget w);
static void DemoAbacusDemo(AbacusWidget w, XEvent * event, char **args, int nArgs);
static void NextAbacusDemo(AbacusWidget w, XEvent * event, char **args, int nArgs);
static void RepeatAbacusDemo(AbacusWidget w, XEvent * event, char **args, int nArgs);
static void MoreAbacusDemo(AbacusWidget w, XEvent * event, char **args, int nArgs);

static char defaultTranslationsAbacusDemo[] =
"<KeyPress>q: quit()\n\
 Ctrl<KeyPress>C: quit()\n\
 <KeyPress>osfCancel: hide()\n\
 <KeyPress>Escape: hide()\n\
 <KeyPress>osfEscape: hide()\n\
 Ctrl<KeyPress>[: hide()\n\
 <KeyPress>0x1B: hide()\n\
 <Btn1Down>: select()\n\
 <KeyPress>c: select()\n\
 <KeyPress>o: demo()\n\
 <KeyPress>n: next()\n\
 <KeyPress>r: repeat()\n\
 <KeyPress>0x20: more()\n\
 <KeyPress>KP_Space: more()\n\
 <KeyPress>Return: more()";

/* KP_Space does not work here
   0x20 is SP (' ') in ASCII  (DP in EBCDIC)
   0x40 is SP (' ') in EBCDIC ('@' in ASCII)
 <KeyPress>0x27: hide()\n\
 <KeyPress>0x40: more()\n\
 */

static XtActionsRec actionsListAbacusDemo[] =
{
	{(char *) "quit", (XtActionProc) QuitAbacusDemo},
	{(char *) "hide", (XtActionProc) HideAbacusDemo},
	{(char *) "select", (XtActionProc) SelectAbacusDemo},
	{(char *) "demo", (XtActionProc) DemoAbacusDemo},
	{(char *) "next", (XtActionProc) NextAbacusDemo},
	{(char *) "repeat", (XtActionProc) RepeatAbacusDemo},
	{(char *) "more", (XtActionProc) MoreAbacusDemo}
};

static XtResource resourcesAbacusDemo[] =
{
	{XtNwidth, XtCWidth, XtRDimension, sizeof (Dimension),
	 XtOffset(AbacusWidget, core.width),
	 XtRString, (caddr_t) "436"},
	{XtNheight, XtCHeight, XtRDimension, sizeof (Dimension),
	 XtOffset(AbacusWidget, core.height),
	 XtRString, (caddr_t) "98"},
	{XtNmono, XtCMono, XtRBoolean, sizeof (Boolean),
	 XtOffset(AbacusWidget, abacus.mono),
	 XtRString, (caddr_t) "FALSE"},
	{XtNreverse, XtCReverse, XtRBoolean, sizeof (Boolean),
	 XtOffset(AbacusWidget, abacus.reverse),
	 XtRString, (caddr_t) "FALSE"},
	{XtNdemoPath, XtCDemoPath, XtRString, sizeof (String),
	 XtOffset(AbacusWidget, abacusDemo.path),
	 XtRString, (caddr_t) DEMOPATH},
	{XtNdemoFont, XtCFont, XtRString, sizeof (String),
	 XtOffset(AbacusWidget, abacusDemo.font),
	 XtRString, (caddr_t) "-*-times-*-r-*-*-*-180-*-*-*-*"},
	{XtNdemoForeground, XtCForeground, XtRPixel, sizeof (Pixel),
	 XtOffset(AbacusWidget, abacusDemo.foreground),
	 XtRString, (caddr_t) XtDefaultForeground},
	{XtNdemoBackground, XtCBackground, XtRPixel, sizeof (Pixel),
	 XtOffset(AbacusWidget, abacusDemo.background),
	 XtRString, (caddr_t) XtDefaultBackground},
	{XtNdeck, XtCDeck, XtRInt, sizeof (int),
	 XtOffset(AbacusWidget, abacus.deck),
	 XtRString, (caddr_t) "-1"},
	{XtNrail, XtCRail, XtRInt, sizeof (int),
	 XtOffset(AbacusWidget, abacus.rail),
	 XtRString, (caddr_t) "0"},
	{XtNnumber, XtCNumber, XtRInt, sizeof (int),
	 XtOffset(AbacusWidget, abacus.number),
	 XtRString, (caddr_t) "0"},
	{XtNframed, XtCFramed, XtRBoolean, sizeof (Boolean),
	 XtOffset(AbacusWidget, abacusDemo.framed),
	 XtRString, (caddr_t) "FALSE"},
	{XtNmode, XtCMode, XtRInt, sizeof (int),
	 XtOffset(AbacusWidget, abacus.mode),
	 XtRString, (caddr_t) "5"},
	{XtNselectCallback, XtCCallback, XtRCallback, sizeof (caddr_t),
	 XtOffset(AbacusWidget, abacus.select),
	 XtRCallback, (caddr_t) NULL}
};

static AbacusClassRec abacusDemoClassRec =
{
	{
		(WidgetClass) & widgetClassRec,		/* superclass */
		(char *) "AbacusDemo",	/* class name */
		sizeof (AbacusRec),	/* widget size */
		NULL,		/* class initialize */
		NULL,		/* class part initialize */
		FALSE,		/* class inited */
		(XtInitProc) InitializeAbacusDemo,	/* initialize */
		NULL,		/* initialize hook */
		XtInheritRealize,	/* realize */
		actionsListAbacusDemo,	/* actions */
		XtNumber(actionsListAbacusDemo),	/* num actions */
		resourcesAbacusDemo,	/* resources */
		XtNumber(resourcesAbacusDemo),	/* num resources */
		NULLQUARK,	/* xrm class */
		TRUE,		/* compress motion */
		TRUE,		/* compress exposure */
		TRUE,		/* compress enterleave */
		TRUE,		/* visible interest */
		(XtWidgetProc) DestroyAbacusDemo,	/* destroy */
#if 0
		(XtWidgetProc) ResizeAbacusDemo,	/* resize */
#else
		NULL,		/* resize */
#endif
		(XtExposeProc) ExposeAbacusDemo,	/* expose */
		(XtSetValuesFunc) SetValuesAbacusDemo,	/* set values */
		NULL,		/* set values hook */
		XtInheritSetValuesAlmost,	/* set values almost */
		NULL,		/* get values hook */
		NULL,		/* accept focus */
		XtVersion,	/* version */
		NULL,		/* callback private */
		defaultTranslationsAbacusDemo,	/* tm table */
		NULL,		/* query geometry */
		NULL,		/* display accelerator */
		NULL		/* extension */
	},
	{
		0		/* ignore */
	}
};

WidgetClass abacusDemoWidgetClass = (WidgetClass) & abacusDemoClassRec;

static void
SetAllColors(AbacusWidget w, Boolean init)
{
	XGCValues   values;
	XtGCMask    valueMask;

	valueMask = GCForeground | GCBackground;
	if (w->abacus.mono) {
		if (w->abacus.reverse) {
			values.background = w->abacus.foreground;
			values.foreground = w->core.background_pixel;
		} else {
			values.foreground = w->abacus.foreground;
			values.background = w->core.background_pixel;
		}
	} else {
		if (w->abacus.reverse) {
			values.foreground = w->abacusDemo.background;
			values.background = w->abacusDemo.foreground;
		} else {
			values.background = w->abacusDemo.background;
			values.foreground = w->abacusDemo.foreground;
		}
	}
	if (!init)
		XtReleaseGC((Widget) w, w->abacusDemo.foregroundGC);
	w->abacusDemo.foregroundGC = XtGetGC((Widget) w, valueMask, &values);

	if (w->abacus.mono) {
		if (w->abacus.reverse) {
			values.foreground = w->abacus.foreground;
			values.background = w->core.background_pixel;
		} else {
			values.background = w->abacus.foreground;
			values.foreground = w->core.background_pixel;
		}
	} else {
		if (w->abacus.reverse) {
			values.background = w->abacusDemo.background;
			values.foreground = w->abacusDemo.foreground;
		} else {
			values.foreground = w->abacusDemo.background;
			values.background = w->abacusDemo.foreground;
		}
	}
	if (!init)
		XtReleaseGC((Widget) w, w->abacusDemo.inverseGC);
	w->abacusDemo.inverseGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->abacusDemo.fontInfo)
                XSetFont(XtDisplay(w), w->abacusDemo.foregroundGC,
			w->abacusDemo.fontInfo->fid);
}

static void
loadFont(AbacusWidget w)
{
	/*const char *fontname = "-*-times-*-r-*-*-*-180-*"; */
	const char       *altfontname = "8x13";
	char buf[512];

	/* Access font */
	if ((w->abacusDemo.fontInfo = XLoadQueryFont(XtDisplay(w),
					      w->abacusDemo.font)) == NULL) {
		(void) sprintf(buf,
			"Cannot open %s font.\nAttempting %s font as alternate\n",
			w->abacusDemo.font, altfontname);
		DISPLAY_WARNING(buf);
		if ((w->abacusDemo.fontInfo = XLoadQueryFont(XtDisplay(w),
						     altfontname)) == NULL) {
			(void) sprintf(buf,
				"Cannot open %s alternate font.\nUse the -demofont option to specify a font to use.\n",
				altfontname);
			DISPLAY_ERROR(buf);
		}
	}
	w->abacusDemo.fontHeight = w->abacusDemo.fontInfo->max_bounds.ascent +
		w->abacusDemo.fontInfo->max_bounds.descent;
}

#endif

#define SetAbacusDemo(w,r) SetAbacus(w,r)

typedef enum {
	INTRO, QUERY, DISPLAY, CONCL
} TextType;

static const char *introFramelessText[LINES] =
{
	"Place this window below the abacus, then click & leave",
	"the pointer in the abacus window to begin the demo.",
	"During the demo, use the Space-bar to step.",
	"Type `q' to quit the demo.",
};

static const char *introFramedText[LINES] =
{
	"Click & leave the pointer in the abacus window",
	"to begin the demo.",
	"During the demo, use the Space-bar to step.",
	"Type `q' to quit the demo.",
};

static const char *queryText[LINES] =
{
	"Type:",
	"  `n' for the Next lesson",
	"  `r' to Repeat previous lesson",
	"  `q' to Quit the demo",
};

static const char *conclText[LINES] =
{
	"Here Endth The Lesson",
	"",
	"",
	"",
};

/* This array represents 4 lines of text that are to be displayed in the
 * demo window; the module drawDemoWindow() will be passed a pointer to this
 * array */
static char displayText[LINES][CHARS];

static void
drawText(AbacusWidget w, TextType textType)
{
	static Boolean firstTime = True;
	int line;
	Pixmap dr = 0;

	if (!firstTime) {
#ifdef WINVER
		FILLRECTANGLE(w, dr, w->abacus.inverseGC,
			0, w->core.height, w->core.width,
			w->core.height + LINES * w->abacusDemo.fontHeight - 1);
#else
		FILLRECTANGLE(w, dr, w->abacusDemo.inverseGC,
			0, 0, w->core.width, w->core.height);
#endif
	} else {
		firstTime = False;
	}

	for (line = 0; line < w->abacusDemo.lines; line++) {
		const char * temp;

		switch (textType) {
			case INTRO:
				if (w->abacusDemo.framed)
					temp = introFramedText[line];
				else
					temp = introFramelessText[line];
#ifdef DEBUG
				(void) fprintf(stderr, " INTRO\n");
#endif
				break;
			case CONCL:
				w->abacusDemo.started = False;
				temp = conclText[line];
#ifdef DEBUG
				(void) fprintf(stderr, " CONCL\n");
#endif
				break;
			case QUERY:
				w->abacusDemo.query = True;
				temp = queryText[line];
#ifdef DEBUG
				(void) fprintf(stderr, " QUERY\n");
#endif
				break;
			default:
				temp = displayText[line];
#ifdef DEBUG
				(void) fprintf(stderr, " DISPLAY\n");
#endif
				break;
		}
#ifdef DEBUG
		(void) printf("%d, %d %s\n",
			      w->abacusDemo.fontHeight, line, temp);
#endif
#ifdef WINVER
		(void) SetTextColor(w->core.hDC, w->abacusDemo.foregroundGC);
		(void) SetBkMode(w->core.hDC, TRANSPARENT);
		(void) TextOut(w->core.hDC,
			1,
			w->core.height + w->abacusDemo.fontHeight * line - 1,
			temp, (signed) strlen(temp));
#else
		(void) XDrawString(XtDisplay(w), XtWindow(w),
			w->abacusDemo.foregroundGC,
			1, w->abacusDemo.fontHeight +
			(w->abacusDemo.fontHeight * line),
			temp, strlen(temp));
		XFlush(XtDisplay(w));
#endif
	}
}

static char *
postfixType(int mode)
{
	switch (mode) {
		case CHINESE:
			return (char *) ""; /* cn */
		case JAPANESE:
			return (char *) "jp";
		case KOREAN:
			return (char *) "ko";
		case ROMAN:
			return (char *) "jp"; /* ro */
		case RUSSIAN:
			return (char *) "ru";
		default:
			return (char *) "err";
	}
}

#ifndef WINVER
/* Find the lessons... this way its easy to add more */
static void
setLessons(AbacusWidget w)
{
	char fname[512], lname[512];

	w->abacusDemo.lessons = 0;
	for (;;) {
		w->abacusDemo.lessons++;
		(void) sprintf(fname, "%s%sAbacus%s%d.les", w->abacusDemo.path,
			FINALDELIM, postfixType(w->abacus.mode),
			w->abacusDemo.lessons);
		if ((w->abacusDemo.fp = fopen(fname, "r")) == NULL) {
			/* Try current directory in case its not installed. */
			(void) sprintf(lname, "%sAbacus%s%d.les", CURRENTDELIM,
				postfixType(w->abacus.mode),
				w->abacusDemo.lessons);
			if ((w->abacusDemo.fp = fopen(lname, "r")) == NULL) {
				w->abacusDemo.lessons--;
#ifdef DEBUG
				(void) printf("lessons = %d\n",
					w->abacusDemo.lessons);
#endif
				return;
			}
		}
		(void) fclose(w->abacusDemo.fp);
	}
}
#endif

static void
startLesson(AbacusWidget w)
{
	char fname[512], lname[512], buf[512];

	(void) sprintf(fname, "%s%sAbacus%s%d.les", w->abacusDemo.path,
		FINALDELIM, postfixType(w->abacus.mode),
		w->abacusDemo.lessonCount + 1);

#ifdef DEBUG
	(void) printf("Lesson name = %s\n", fname);
#endif
	if ((w->abacusDemo.fp = fopen(fname, "r")) == NULL) {

		/* Try current directory in case its not installed. */
		(void) sprintf(lname, "%sAbacus%s%d.les", CURRENTDELIM,
		       postfixType(w->abacus.mode),
		       w->abacusDemo.lessonCount + 1);
		if ((w->abacusDemo.fp = fopen(lname, "r")) == NULL) {
			(void) sprintf(buf,
				"Could not open lesson script-file: %s\n",
				fname);
			DISPLAY_ERROR(buf);
		}
	}
	w->abacusDemo.lessonLength = 0;
	while (fscanf(w->abacusDemo.fp, "%d %d %d %d %d\n",
		  &(w->abacusDemo.aux),
		  &(w->abacusDemo.deck), &(w->abacusDemo.rail),
		  &(w->abacusDemo.number), &(w->abacusDemo.lines)) > 0) {
		int         line;

		if (w->abacusDemo.lines > LINES)
			w->abacusDemo.lines = LINES;
		for (line = 0; line < w->abacusDemo.lines; line++) {
			(void) fgets(displayText[line], 64, w->abacusDemo.fp);
			displayText[line][strlen(displayText[line]) - 1] =
				'\0';
		}
		++w->abacusDemo.lessonLength;
	}
	(void) fclose(w->abacusDemo.fp);
	/* rewind */
	if ((w->abacusDemo.fp = fopen(fname, "r")) == NULL) {
		/* Try current directory in case its not installed yet. */
		(void) sprintf(lname, "%sAbacus%s%d.les", CURRENTDELIM,
			postfixType(w->abacus.mode),
			w->abacusDemo.lessonCount + 1);
		if ((w->abacusDemo.fp = fopen(lname, "r")) == NULL) {
			(void) sprintf(buf,
				"Could not open lesson script-file: %s\n",
				fname);
			DISPLAY_ERROR(buf);
		}
	}
#ifdef DEBUG
	(void) printf("lessonLength = %d\n", w->abacusDemo.lessonLength);
#endif
}

static void
doLesson(AbacusWidget w)
{
	int         line;

	/* load first instance */
	(void) fscanf(w->abacusDemo.fp, "%d %d %d %d %d\n",
		&(w->abacusDemo.aux),
		&(w->abacusDemo.deck), &(w->abacusDemo.rail),
		&(w->abacusDemo.number), &(w->abacusDemo.lines));
#ifdef DEBUG
	(void) printf(
		"aux %d, deck = %d, rail = %d, number = %d, lines = %d\n",
		w->abacusDemo.aux,
		w->abacusDemo.deck, w->abacusDemo.rail,
		w->abacusDemo.number, w->abacusDemo.lines);
#endif

	if (w->abacusDemo.lines > LINES)
		w->abacusDemo.lines = LINES;
	for (line = 0; line < w->abacusDemo.lines; line++) {
		(void) fgets(displayText[line], 64, w->abacusDemo.fp);
		displayText[line][strlen(displayText[line]) - 1] = '\0';
	}

	/* A '0' in demo number signifies that only the text is to be
	 * displayed (i.e. nothing is to be added or subtracted).
	 */

	if (w->abacusDemo.lessonLength) {
		SetAbacusMove(w, ABACUS_MOVE, w->abacusDemo.aux,
			w->abacusDemo.deck, w->abacusDemo.rail,
			w->abacusDemo.number);
	}
	if (--w->abacusDemo.lessonLength < 0) {
		(void) fclose(w->abacusDemo.fp);
		w->abacusDemo.lines = LINES;
		drawText(w, CONCL);
		Sleep(1000);
		drawText(w, QUERY);
	} else {
		drawText(w, DISPLAY);
	}
}

#ifndef WINVER
static void
clearDemo(AbacusWidget w)
{
        SetAbacusDemo(w, ABACUS_CLEAR);
	ClearAbacusDemo(w);
}
#endif

static void
queryDemo(AbacusWidget w, Boolean advance)
{
	if (w->abacusDemo.query) {
		w->abacusDemo.started = True;
		w->abacusDemo.query = False;
		if (w->abacusDemo.lessons && advance)
			w->abacusDemo.lessonCount =
				(w->abacusDemo.lessonCount + 1) %
				w->abacusDemo.lessons;
		SetAbacusDemo(w, ABACUS_CLEAR);
#ifdef WINVER
		ClearAbacus(w);
#endif
		startLesson(w);
		doLesson(w);
	}
}

static void
incDemo(AbacusWidget w)
{
	if (w->abacusDemo.started && !w->abacusDemo.query) {
		w->abacusDemo.query = False;
		doLesson(w);
	}
}

#ifndef WINVER
static
#endif
void
InitializeAbacusDemo(
#ifdef WINVER
	AbacusWidget w
#else
	Widget request, Widget renew
#endif
	)
{
#ifndef WINVER
	AbacusWidget w = (AbacusWidget) renew;

	w->abacus.mono = (DefaultDepthOfScreen(XtScreen(w)) < 2 ||
		w->abacus.mono);
	loadFont(w);
	SetAllColors(w, True);
#if 0
	ResizeAbacusDemo(w);
#endif
#else
#if 0
	hBrush = CreateSolidBrush(w->abacusDemo.background);
	SETBACK(w->core.hWnd, hBrush);
#endif
#endif
	w->abacusDemo.started = False;
	w->abacusDemo.query = False;
	w->abacusDemo.lessonCount = 0;
}

#ifndef WINVER
static      Boolean
SetValuesAbacusDemo(Widget current, Widget request, Widget renew)
{
	AbacusWidget c = (AbacusWidget) current, w = (AbacusWidget) renew;
	Boolean     redraw = False;
#if 0
	Boolean     redrawText = False;
#endif

	if (w->abacus.mode != c->abacus.mode) {
		setLessons(w);
		redraw = True;
	}
	if (w->core.background_pixel != c->core.background_pixel ||
	    w->abacus.foreground != c->abacus.foreground ||
	    w->abacusDemo.background != c->abacusDemo.background ||
	    w->abacusDemo.foreground != c->abacusDemo.foreground ||
	    w->abacus.reverse != c->abacus.reverse ||
	    w->abacus.mono != c->abacus.mono) {
		SetAllColors(w, False);
		redraw = True;
#if 0
		redrawText = True;
#endif
	}
#if 0
	if (!redraw && XtIsRealized(renew) && renew->core.visible) {
		/* Redraw text */
	}
#endif
	if (w->abacus.deck == ABACUS_CLEAR) {
		w->abacus.deck = ABACUS_IGNORE;
		clearDemo(w);
	}
	if (w->abacus.deck == ABACUS_NEXT) {
		w->abacus.deck = ABACUS_IGNORE;
		queryDemo(w, True);
	}
	if (w->abacus.deck == ABACUS_REPEAT) {
		w->abacus.deck = ABACUS_IGNORE;
		queryDemo(w, False);
	}
	if (w->abacus.deck == ABACUS_MORE) {
		w->abacus.deck = ABACUS_IGNORE;
		incDemo(w);
	}
	return (redraw);
}

static void
QuitAbacusDemo(AbacusWidget w, XEvent * event, char **args, int nArgs)
{
	XtCloseDisplay(XtDisplay(w));
	exit(0);
}

static void
DestroyAbacusDemo(Widget old)
{
	AbacusWidget w = (AbacusWidget) old;

	XtReleaseGC(old, w->abacusDemo.foregroundGC);
	XtReleaseGC(old, w->abacusDemo.inverseGC);
	XtRemoveCallbacks(old, XtNselectCallback, w->abacus.select);
	XFreeFont(XtDisplay(w), w->abacusDemo.fontInfo);
}

#else
#if 0
void
DestroyAbacusDemo(HBRUSH hBrush)
{
	(void) DeleteObject(hBrush);
	PostQuitMessage((int) NULL);
}
#endif
#endif

#if 0
static void
ResizeAbacusDemo(AbacusWidget w)
{
}
#endif

#ifndef WINVER
static
#endif
void
ExposeAbacusDemo(
#ifdef WINVER
AbacusWidget w
#else
Widget renew, XEvent * event, Region region
#endif
)
{
	Pixmap dr = 0;

#ifdef WINVER
	FILLRECTANGLE(w, dr, w->abacus.inverseGC,
		0, w->core.height, w->core.width,
		w->core.height + LINES * w->abacusDemo.fontHeight - 1);
#else
	AbacusWidget w = (AbacusWidget) renew;

	if (!w->core.visible)
		return;
	FILLRECTANGLE(w, dr, w->abacusDemo.inverseGC,
		0, 0, w->core.width, w->core.height);
#endif
	if (w->abacusDemo.started) {
		drawText(w, DISPLAY);
	} else if (w->abacusDemo.query) {
		w->abacusDemo.lines = LINES;
		drawText(w, QUERY);
	} else {
		w->abacusDemo.lines = LINES;
		drawText(w, INTRO);
	}
}

#ifndef WINVER
static
void
HideAbacusDemo(AbacusWidget w
, XEvent * event, char **args, int nArgs
	)
{
        SetAbacusDemo(w, ABACUS_HIDE);
}

static
void
SelectAbacusDemo(AbacusWidget w
, XEvent * event, char **args, int nArgs
	)
{
	clearDemo(w);
}
#endif

#ifndef WINVER
static
#endif
void
ClearAbacusDemo(AbacusWidget w)
{
	w->abacusDemo.started = True;
	w->abacusDemo.query = False;
	startLesson(w);
	doLesson(w);
}

#ifndef WINVER
static
#endif
void
DemoAbacusDemo(AbacusWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
	)
{
#ifdef WINVER
	w->abacus.demo = !w->abacus.demo;
	if (w->abacus.demo)
        	SetAbacusDemo(w, ABACUS_DEMO_DEFAULT);
#else
        SetAbacusDemo(w, ABACUS_DEMO);
#endif
}

#ifndef WINVER
static
#endif
void
NextAbacusDemo(AbacusWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
	)
{
	queryDemo(w, True);
}

#ifndef WINVER
static
#endif
void
RepeatAbacusDemo(AbacusWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
	)
{
	queryDemo(w, False);
}

#ifndef WINVER
static
#endif
void
MoreAbacusDemo(AbacusWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
	)
{
	incDemo(w);
}
