/*
 * Xmmsfind - A small plugin similar to the built-in "Jump to file" util. 
 * Copyright (C) 2001 Isak Savo <iso01001@student.mdh.se>
 * 
 * This program is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public Licensse as published by the
 * Free Software Foundation; either version 2 of the License, or (at your
 * option) any later version.  This program is distributed in the hope
 * that it will be useful, but WITHOUT ANY WARRANTY; without even the
 * implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.  You
 * should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc., 
 * 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA. 
 */
#include <stdio.h>
#include <stdlib.h>

#include <signal.h>

#include <string.h>
#include <unistd.h>
#include <ctype.h>
#include <time.h>
#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>
#include <xmms/xmmsctrl.h>
#include <xmms/titlestring.h>
#include <regex.h>

#include "config.h"
#include "globals.h"
#include "xmmsfind.h"

gboolean do_enqueue_this_time = FALSE, do_enqueue_entire_clist = FALSE;
gint xmms_playlist_max, enqueue_offs;;
static GtkWidget *mwin_clist;
static GtkWidget *mwin_txt_search, *mwin_scrollwin;
static GtkWidget *mwin_chk_title, *mwin_chk_file;

int SECOND=10; /*alarm each 10 seconds*/
gint enqueue_position;
static gint song_played_at_first_enqueue = -1;

void create_xmmsfind_main_window ()
{
    static GtkWidget *mwin_btn_go, *mwin_btn_cfg, *mwin_btn_refresh;
    static GtkWidget *mwin_btn_enqs, *mwin_btn_enql, *mwin_btn_hbox;
    static GtkWidget *mwin_hbox, *mwin_vbox, *mwin_cfg_hbox, *mwin_cfg_frame;
    
#ifdef DEBUG
    g_print ("xmmsfind.c: (create_main_win): start\n");
#endif
    xmmsfind_do_read_config_file ();
    if (xmmsfind_main_window)
    {
#ifdef DEBUG
	g_print ("xmmsfind.c: (create_main_win): main_window already created\n");
#endif
	gtk_widget_destroy (GTK_WIDGET (xmmsfind_main_window));
    }
    else {
	 /* If enqueue, where should we move the enqueued file (1 means song after current) */
	enqueue_offs = 0;
    }
    
    xmmsfind_main_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
    gtk_widget_set_usize (GTK_WIDGET (xmmsfind_main_window), 300, 400);
    gtk_window_set_default_size(GTK_WINDOW(xmmsfind_main_window), cfg_x.size_x, cfg_x.size_y);
    gtk_container_border_width (GTK_CONTAINER (xmmsfind_main_window), 10);
    if (cfg_x.center)
    	gtk_window_set_position (GTK_WINDOW (xmmsfind_main_window), GTK_WIN_POS_CENTER);
    gtk_window_set_title (GTK_WINDOW (xmmsfind_main_window), "Xmmsfind (v" VERSION ")" );
    gtk_signal_connect (GTK_OBJECT (xmmsfind_main_window), "delete_event",
			GTK_SIGNAL_FUNC (xmmsfind_do_hide_main_window), NULL);

    gtk_widget_realize (GTK_WIDGET (xmmsfind_main_window));
    
    mwin_clist = gtk_clist_new (1);
    gtk_signal_connect (GTK_OBJECT (mwin_clist), "key_press_event",
			GTK_SIGNAL_FUNC (mwin_clist_key_pressed), NULL);
    gtk_signal_connect (GTK_OBJECT (mwin_clist), "select_row",
			GTK_SIGNAL_FUNC (mwin_clist_row_selected), NULL);
    gtk_clist_set_selection_mode (GTK_CLIST (mwin_clist), GTK_SELECTION_BROWSE);

    mwin_scrollwin = gtk_scrolled_window_new (NULL, NULL);
    gtk_container_add (GTK_CONTAINER (mwin_scrollwin),mwin_clist);
    gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (mwin_scrollwin),
				    GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
    
    /* Create the textentry and the buttons  */
    mwin_btn_refresh = gtk_button_new_with_label ("Refresh");
    mwin_btn_go = gtk_button_new_with_label ("Go To Song");
    mwin_btn_enqs = gtk_button_new_with_label ("Enqueue Song");
    mwin_btn_enql = gtk_button_new_with_label ("Enqueue List");
    mwin_btn_cfg = gtk_button_new_with_label ("Options");
    mwin_txt_search = gtk_entry_new_with_max_length (SEARCHSTR_MAX);
    
    gtk_signal_connect (GTK_OBJECT (mwin_btn_go), "clicked",
			GTK_SIGNAL_FUNC (xmmsfind_do_change_song), NULL);
    gtk_signal_connect (GTK_OBJECT (mwin_btn_enqs), "clicked",
			GTK_SIGNAL_FUNC (xmmsfind_do_enqueue_song), NULL);
    gtk_signal_connect (GTK_OBJECT (mwin_btn_enql), "clicked",
			GTK_SIGNAL_FUNC (xmmsfind_do_enqueue_list), NULL);
    gtk_signal_connect (GTK_OBJECT (mwin_btn_cfg), "clicked",
			GTK_SIGNAL_FUNC (xmmsfind_config), GINT_TO_POINTER (1));
    gtk_signal_connect (GTK_OBJECT (mwin_btn_refresh), "clicked",
			GTK_SIGNAL_FUNC (xmmsfind_do_refresh_clist), NULL);

    gtk_signal_connect (GTK_OBJECT (mwin_txt_search), "key_press_event",
			GTK_SIGNAL_FUNC (mwin_txt_search_key_pressed), mwin_clist);
    gtk_signal_connect (GTK_OBJECT (mwin_txt_search), "changed",
			GTK_SIGNAL_FUNC (xmmsfind_do_search), NULL);
    
    /* Create the checkbuttons */
    mwin_chk_title = gtk_check_button_new_with_label ("Title/Filename");
	gtk_signal_connect (GTK_OBJECT (mwin_chk_title), "clicked", 
			GTK_SIGNAL_FUNC (mwin_chk_clicked), GINT_TO_POINTER (OPT_MATCH_TITLE));
    mwin_chk_file = gtk_check_button_new_with_label ("Directory Path");
    gtk_signal_connect (GTK_OBJECT (mwin_chk_file), "clicked", 
			GTK_SIGNAL_FUNC (mwin_chk_clicked), GINT_TO_POINTER (OPT_MATCH_FILENAME));
    gtk_widget_show (mwin_chk_title);
    gtk_widget_show (mwin_chk_file);
    
    /* Start packing things in a boxes and then put the boxes into the main window  */
    mwin_cfg_frame = gtk_frame_new ("Match Options:");
    mwin_cfg_hbox = gtk_hbox_new (FALSE, 3);
    gtk_box_pack_start (GTK_BOX (mwin_cfg_hbox), mwin_chk_title,  TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (mwin_cfg_hbox), mwin_chk_file,  TRUE, TRUE, 0);
    gtk_box_pack_end (GTK_BOX (mwin_cfg_hbox), mwin_btn_cfg, TRUE, TRUE, 3);
    gtk_container_add (GTK_CONTAINER (mwin_cfg_frame), mwin_cfg_hbox);
    
    mwin_btn_hbox = gtk_hbox_new (FALSE, 3);
    gtk_box_pack_start (GTK_BOX (mwin_btn_hbox), mwin_btn_enql, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (mwin_btn_hbox), mwin_btn_enqs,  TRUE, TRUE, 0);
    gtk_box_pack_end (GTK_BOX (mwin_btn_hbox), mwin_btn_go, TRUE, TRUE, 0);
    
    mwin_vbox = gtk_vbox_new (FALSE, 5);
    mwin_hbox = gtk_hbox_new (FALSE, 5);
    gtk_box_pack_start (GTK_BOX (mwin_hbox), mwin_txt_search, TRUE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (mwin_hbox), mwin_btn_refresh, FALSE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (mwin_vbox), mwin_hbox, FALSE, TRUE, 0);
    gtk_box_pack_start (GTK_BOX (mwin_vbox), mwin_scrollwin, TRUE, TRUE, 0);
    
    gtk_box_pack_start (GTK_BOX (mwin_vbox), mwin_btn_hbox, FALSE, TRUE, 2);
    gtk_box_pack_start (GTK_BOX (mwin_vbox), mwin_cfg_frame, FALSE, TRUE, 2);

    gtk_container_add (GTK_CONTAINER (xmmsfind_main_window), mwin_vbox);

    xmmsfind_do_fill_clist (TRUE);
    
    gtk_signal_connect (GTK_OBJECT (xmmsfind_main_window), "focus_in_event",
			GTK_SIGNAL_FUNC (main_window_focus_in_event), NULL);
    gtk_widget_show_all (xmmsfind_main_window);

    xmmsfind_clist_move_to_current_song (xmms_remote_get_playlist_pos(session));

#ifdef DEBUG
    g_print ("xmmsfind.c: (create_main_win): ok\n");
#endif
}

void xmmsfind_clist_move_to_current_song (gint current_row)
{
#ifdef DEBUG
    g_print ("xmmsfind.c: (clist_move_to_current_song) (%d): start\n", current_row);
#endif
    if (current_row && (GTK_CLIST (mwin_clist)->rows >= current_row))
    {
	gtk_clist_moveto(GTK_CLIST (mwin_clist),  current_row, 0, 0.2, 0);
	gtk_clist_select_row (GTK_CLIST (mwin_clist), current_row, 0);
	GTK_CLIST (mwin_clist)->focus_row = GPOINTER_TO_INT (GTK_CLIST (mwin_clist)->selection->data);
    }
    else 
      current_row = -1;	/* To indicate that something went wrong! Dubugging purpose only */
#ifdef DEBUG
    g_print ("xmmsfind.c: (clist_move_to_current_song) (%d): ok\n", current_row);
#endif
}

void xmmsfind_do_refresh_clist (GtkWidget *btn, gpointer data)
{
    xmmsfind_do_fill_clist (TRUE);
    xmmsfind_clist_move_to_current_song (xmms_remote_get_playlist_pos(session));
}


void xmmsfind_do_fill_clist (gboolean force)
{
    gint i;
    gchar *display;
#ifdef DEBUG
    g_print ("xmmsfind.c: (xmmsfind_do_fill_clist): start\n");
#endif
    gtk_clist_freeze (GTK_CLIST (mwin_clist));

    if (force)
	xmmsfind_playlist = get_playlist_from_xmms ();
    
    gtk_clist_clear (GTK_CLIST (mwin_clist));
    
    for (i = 0; i < xmms_playlist_max; i++)
    {
	 if(cfg_x.show_title)
	      display = xmmsfind_playlist[i].title;
	 else
	      display = xmmsfind_playlist[i].fname;
	 gtk_clist_append (GTK_CLIST (mwin_clist), 
			   &(display));
	/* The row data is used to assign a songnumber to a row in the clist */
	gtk_clist_set_row_data (GTK_CLIST (mwin_clist), i, GINT_TO_POINTER (xmmsfind_playlist[i].number));
    }
    
    gtk_clist_set_column_width (GTK_CLIST (mwin_clist), 0, 
		    gtk_clist_optimal_column_width (GTK_CLIST (mwin_clist), 0));
    gtk_clist_thaw (GTK_CLIST (mwin_clist));
    
    xmmsfind_do_search ();	/* research incase the list was refreshed mid search */
#ifdef DEBUG
    g_print ("xmmsfind.c: (xmmsfind_do_fill_clist) (%d songs): ok\n", xmms_playlist_max);
#endif

}

/* search - function that updates the searchlist with the currently entered searchstring
 * This function is called _every_ time that something is entered into the searchbox */
void xmmsfind_do_search ()
{
    gchar *search_str, **search_arr;
    gint arr_max, ss_index, pl_index;
    gboolean match;
    regex_t *rx;
        
#ifdef DEBUG
    g_print ("xmmsfind.c: (xmmsfind_do_search): start\n");
#endif
    
    search_str = gtk_entry_get_text (GTK_ENTRY (mwin_txt_search));
    /* Some checks first: if txt_search was empty or the playlist is empty 
     * (shouldn't happen since we checked it previosly, but who can know for sure)   */
    if ((xmms_playlist_max < 1))
      return;
    
    gtk_clist_freeze (GTK_CLIST (mwin_clist));
    gtk_clist_clear (GTK_CLIST (mwin_clist));

    /* If the search-string was empty, we'll wan't to display all songs. */
    if (!search_str || *search_str == 0)
    {
	arr_max = 1;
	search_arr = g_malloc (sizeof (gchar *));
	search_arr[0] = g_strdup ("");
    }
    else
    {
	search_arr = str2arr (search_str, &arr_max);
	if (!search_arr)
	{
	    gtk_clist_thaw (GTK_CLIST (mwin_clist));
	    g_warning ("Xmmsfind: (str2arr) Unable to allocate memory for array\n");
		    return;
	}
    }
    
    /* Compile the regular expressions for the words we will be matching against. */
    rx = (regex_t *) g_malloc (sizeof (regex_t) * arr_max);
    for (ss_index = 0; ss_index < arr_max; ss_index++)
    	{
	    /*TODO: Use a more sophisticated response to an invalid search string? */
	    /*Note: basicly what happens here is that if a word contains regexp special
	            characters, but the word itself is not a valid expression, the word 
		    is ignored. However, this also allows valid regexperssions to be used*/
	    if (regcomp (&rx[ss_index], search_arr[ss_index], REG_EXTENDED | REG_ICASE))
	        regcomp (&rx[ss_index], " ", REG_EXTENDED | REG_ICASE);
	}

    /* We don't need the memory for the search-words anymore. */
    for (ss_index = 0; ss_index < arr_max; ss_index++)
	g_free (search_arr [ss_index]);
    g_free (search_arr);

    /* Loop round the playlist (located in memory to enhance speed) and
     * finds any matches to the searchstr 
     * Note that a title must match _ALL_ of the words in the searchbox
     */
    for (pl_index = 0; pl_index < xmms_playlist_max; pl_index++)
    {
	for (ss_index = 0; ss_index < arr_max; ss_index++)
	{
	    match = FALSE;
	    if (cfg_x.match_title && (regexec (&rx[ss_index], xmmsfind_playlist[pl_index].title, 0, NULL, 0) == 0))
		match = TRUE;
	    else if (cfg_x.match_title && (regexec (&rx[ss_index], xmmsfind_playlist[pl_index].fname, 0, NULL, 0) == 0))
		match = TRUE;
	    else if (cfg_x.match_path && (regexec (&rx[ss_index], xmmsfind_playlist[pl_index].fpath, 0, NULL, 0) == 0))
        match = TRUE;
	    else
		break;
	}
	if (match == TRUE)	/* Only add files that match _ALL_ searchwords */
	{
	     gchar *display;
	     gint row;
	     
	     if (cfg_x.show_title)
		  display = xmmsfind_playlist[pl_index].title;
	     else
		  display = xmmsfind_playlist[pl_index].fname;
	     
	     row = gtk_clist_append (GTK_CLIST (mwin_clist), 
				     &(display));
	     gtk_clist_set_row_data (GTK_CLIST (mwin_clist), row, 
				    GINT_TO_POINTER (xmmsfind_playlist[pl_index].number));
	}
    }

    gtk_clist_thaw (GTK_CLIST (mwin_clist));

    for (ss_index = 0; ss_index < arr_max; ss_index++)
    {
	    regfree (&rx[ss_index]);
	}
    g_free (rx);

#ifdef DEBUG
    g_print ("xmmsfind.c: (xmmsfind_do_search): ok\n");
#endif

} /* xmmsfind_do_search */

/* Turns shuffle mode off if its on, and if the user wants it to be off */
void xmmsfind_do_random_off_if_needed (void) {
    if ((cfg_x.force_rdm_off || (do_enqueue_this_time || do_enqueue_entire_clist))
	&& xmms_remote_is_shuffle (session))
      xmms_remote_toggle_shuffle (session);
}

void ALARMhandler(int sig) 
{
    gint current_pos = xmms_remote_get_playlist_pos(session);
    signal (SIGALRM, SIG_IGN);

#ifdef DEBUG
    printf("xmmsfind.c: ALARMhandler\n");
#endif

    /* Turn on random on again if last enqueued song is played for 2 minutes (TODO: should be better...)*/
    if (enqueue_position == current_pos && !xmms_remote_is_shuffle (session) && 
	xmms_remote_get_output_time (session) > 120000)
    {
	//printf("Random is turned on again\n");
	xmms_remote_toggle_shuffle(session);
    }

    /* Re-initialize everything */
    if (current_pos < song_played_at_first_enqueue || current_pos >= enqueue_position) 
    {
	//printf("Reschedule\n");
	song_played_at_first_enqueue = xmms_remote_get_playlist_pos(session);
	enqueue_offs=0;
    }

    /*re-install alarm*/
    alarm(SECOND);
    signal(SIGALRM,ALARMhandler);
}

/* Enqueues the song number "sel_song" */
void xmmsfind_do_list_enqueue_song (gint sel_song) {
    gchar* filename;
	
#ifdef DEBUG
    g_print ("xmmsfind.c: (xmmsfind_do_list_enqueue_song) start\n");
#endif

    if (cfg_x.smart_enqueue) 
    {
	/*only executed first time*/
	if (song_played_at_first_enqueue == -1) 
	    song_played_at_first_enqueue = xmms_remote_get_playlist_pos(session);

	/*install alarm*/
	signal(SIGALRM,ALARMhandler);
	alarm(SECOND);

	/* If we delete a song before the current, the playlist numbers will change */
        if (sel_song < xmms_remote_get_playlist_pos(session)) {
	    song_played_at_first_enqueue--;
        }

	//Why the heck does xmms_remote not work here?
	//xmms_remote_playlist_delete(session, sel_song);
	playlist_delete_index ((glong) sel_song);
	enqueue_offs++;
	
        enqueue_position = song_played_at_first_enqueue + enqueue_offs;
    } 
    else { /* User turned off smart equeuing...  */
	//Why the heck does xmms_remote not work here?
	//xmms_remote_playlist_delete(session, sel_song);
	playlist_delete_index ((glong) sel_song);
	
	/* Will someone please tell me why this doesn't work:
	enqueue_position = xmms_remote_get_playlist_pos(session)+1;
	but this does: */
	enqueue_position = xmms_remote_get_playlist_pos(session);
	enqueue_position++;
    }
    
    filename = g_strjoin("/", xmmsfind_playlist[sel_song].fpath,
	        xmmsfind_playlist[sel_song].fname, NULL);
    xmms_remote_playlist_ins_url_string(session, filename, enqueue_position);
    g_free(filename);
    xmmsfind_do_fill_clist (TRUE);
    
#ifdef DEBUG
    g_print ("xmmsfind.c: (xmmsfind_do_list_enqueue_song) ok\n");
#endif
}

void xmmsfind_do_list_enqueue_all (void)
{
    GList *filenames = NULL;
    gint file_pos, row = 0, current_pos;
    gchar *fname;
    
#ifdef DEBUG
    g_print ("xmmsfind.c: (xmmsfind_do_list_enqueue_all) start\n");
#endif
    current_pos = xmms_remote_get_playlist_pos(session);
    file_pos = GPOINTER_TO_INT (gtk_clist_get_row_data (GTK_CLIST (mwin_clist), row));
    while (gtk_clist_get_selectable (GTK_CLIST (mwin_clist), row))
    {
	fname = xmms_remote_get_playlist_file(session, file_pos);
	/* We don't want to move the song that is currently playing */
	if (file_pos != current_pos)
	  filenames = g_list_append (filenames, fname);
	row++;
	file_pos = GPOINTER_TO_INT (gtk_clist_get_row_data (GTK_CLIST (mwin_clist), row));
    }
    playlist_delete_filenames (filenames);
    row = xmms_remote_get_playlist_pos(session) + 1;
    while (filenames)
    {
	playlist_ins ((gchar *) filenames->data, row);
	row++;
	g_free (filenames->data);
	filenames = filenames->next;
    }
    g_list_free (filenames);
    xmmsfind_do_fill_clist (TRUE);
#ifdef DEBUG
    g_print ("xmmsfind.c: (xmmsfind_do_list_enqueue_all) ok\n");
#endif
    return;
}

void xmmsfind_do_xmms_enqueue_all (void)
{
    gint song_pos, row = 0, current_pos;
    
    current_pos = xmms_remote_get_playlist_pos(session);
    song_pos = GPOINTER_TO_INT (gtk_clist_get_row_data (GTK_CLIST (mwin_clist), row));
    while (gtk_clist_get_selectable (GTK_CLIST (mwin_clist), row))
    {
	/* We don't want to queue the song that is currently playing */
	if (song_pos != current_pos)
	   playlist_queue_position (song_pos);
	row++;
	song_pos = GPOINTER_TO_INT (gtk_clist_get_row_data (GTK_CLIST (mwin_clist), row));
    }
}

/* get selected song, song: is assigned the numer of the song, returns false if fails */
gboolean xmmsfind_get_song (gint* song)
{
    gint row;

#ifdef DEBUG
    g_print ("xmmsfind.c: (xmmsfind_get_song): start\n");
#endif

    if (GTK_CLIST (mwin_clist)->selection)
    {
	    row = (int) GTK_CLIST (mwin_clist)->selection->data;
	    *song = GPOINTER_TO_INT (gtk_clist_get_row_data (GTK_CLIST (mwin_clist), row));
        return TRUE;
    }
    else
    {
#ifdef DEBUG
	    g_print ("xmmsfind.c: (xmmsfind_get_song): no selection\n");
#endif
        return FALSE;
    }

#ifdef DEBUG
    g_print ("xmmsfind.c: (xmmsfind_get_song): done\n");
#endif
}

/* Change directly to the song in xmms */
void xmmsfind_do_change_song ()
{
    gint song;
#ifdef DEBUG
    g_print ("xmmsfind.c: (xmmsfind_do_change_song): start\n");
#endif
    
    if (xmmsfind_get_song(&song))
    {
        xmms_remote_stop (session);
        xmms_remote_set_playlist_pos (session, song);
        xmms_remote_play (session);
    /* Hide the main window. */
        if (!cfg_x.persistent)
            xmmsfind_do_hide_main_window ();
    }
    
#ifdef DEBUG
    g_print ("xmmsfind.c: (xmmsfind_do_change_song): stop\n");
#endif
}
   
/* enqueue the song  */
void xmmsfind_do_enqueue_song ()
{
    gint song;
#ifdef DEBUG
    g_print ("xmmsfind.c: (xmmsfind_do_enqueue_song): start\n");
#endif
    
    if (xmmsfind_get_song(&song))
    {
	    if (cfg_x.xmms_enqueue)
	    	playlist_queue_position (song);
	    else 
	    {
	        xmmsfind_do_random_off_if_needed ();
	        xmmsfind_do_list_enqueue_song (song);
	    }
        /* Hide the main window. */
        if (!cfg_x.persistent)
            xmmsfind_do_hide_main_window ();
    }

#ifdef DEBUG
    g_print ("xmmsfind.c: (xmmsfind_do_enqueue_song): stop\n");
#endif
}
   
/* enqueue the list  */
void xmmsfind_do_enqueue_list ()
{
    gint song;
#ifdef DEBUG
    g_print ("xmmsfind.c: (xmmsfind_do_enqueue_list): start\n");
#endif
    
    if (xmmsfind_get_song(&song))
    {
	    if (cfg_x.xmms_enqueue)
	    	xmmsfind_do_xmms_enqueue_all();
	    else
	    {
	    	xmmsfind_do_random_off_if_needed ();
	    	xmmsfind_do_list_enqueue_all ();
	    }
        /* Hide the main window. */
        if (!cfg_x.persistent)
            xmmsfind_do_hide_main_window ();
    }
    
#ifdef DEBUG
    g_print ("xmmsfind.c: (xmmsfind_do_enqueue_list): stop\n");
#endif
}
   
/* This function tells xmms to change (or enqueue) song, then hide the main window */
void xmmsfind_do_song ()
{
#ifdef DEBUG
    g_print ("xmmsfind.c: (xmmsfind_do_song): start\n");
#endif

	if ((cfg_x.enqueue || do_enqueue_this_time) && (!do_enqueue_entire_clist))
	{
	    /* Enqueue a single song */
	    xmmsfind_do_enqueue_song();
        do_enqueue_this_time = FALSE;
	}
	else if (do_enqueue_entire_clist)
	{
	    /* Enqueue all items in the clist */
	    xmmsfind_do_enqueue_list();
        do_enqueue_entire_clist = FALSE;
	}
	else
	{
	    /* Just play the damn song.. */
        xmmsfind_do_change_song();
	}

#ifdef DEBUG
    g_print ("xmmsfind.c: (xmmsfind_do_song): done\n");
#endif

} /* change_song () */

/*
 * Grabs the playlist from xmms and stores it in a _pl_struct (decl. in global.h)
 * 
 * Allocates the neccessary space and returns a pointer to it 
 */
_pl_struct *get_playlist_from_xmms ()
{
    _pl_struct *plist;
    gint i;

#ifdef DEBUG
    g_print ("xmmsfind.c: (get_playlist_from_xmms): start\n");
#endif
    xmmsfind_do_free_playlist ();
    if ((xmms_playlist_max = xmms_remote_get_playlist_length(session)) < 1)
    {
	xmms_playlist_max = 0;
	g_warning ("\nPlaylist empty (%d songs)\n", xmms_playlist_max);
	return NULL;
    }
    plist = (_pl_struct *) g_malloc (xmms_playlist_max * sizeof (_pl_struct));
    for (i = 0; i < xmms_playlist_max; i++)
    {
    gchar* fullpath = playlist_get_filename(i); //use internal call for speed
	plist[i].number = i;
	plist[i].fname = g_strdup(g_basename (fullpath));
    plist[i].fpath = g_dirname (fullpath);
    g_free(fullpath);
	/* we dont want xmms to read the file if we dont want
	 * to search the songtitle - FASTER for huge playlists */
	if (cfg_x.show_title) 
    {
	    gchar* tmptitle = playlist_get_songtitle(i); //use internal call for speed
        plist[i].title = g_strdup(tmptitle); 
        g_free(tmptitle);
    }
	else
	  plist[i].title = g_strdup(plist[i].fname);
    }
#ifdef DEBUG
    g_print ("xmmsfind.c: (get_playlist_from_xmms): ok\n");
#endif
    return plist;
} /* get_playlist_from_xmms */

void xmmsfind_do_free_playlist (void)
{
    gint pl_index;
#ifdef DEBUG
    g_print ("xmmsfind.c: (xmmsfind_do_free_playlist): start\n");
#endif
    
    if (xmmsfind_playlist == NULL)
    {
	#ifdef DEBUG
	g_print ("xmmsfind.c: (xmmsfind_do_free_playlist): playlist is NULL, aborting\n");
	#endif
	return;
    }
    
    for (pl_index = 0; pl_index < xmms_playlist_max; pl_index++)
    {
        g_free (xmmsfind_playlist[pl_index].title);
        g_free (xmmsfind_playlist[pl_index].fname);
        g_free (xmmsfind_playlist[pl_index].fpath);
    }
    g_free (xmmsfind_playlist);
    xmmsfind_playlist = NULL;
#ifdef DEBUG
    g_print ("xmmsfind.c: (xmmsfind_do_free_playlist): ok\n");
#endif
} /* xmmsfind_do_free_playlist */

gint xmmsfind_do_hide_main_window ()
{
    
#ifdef DEBUG
    g_print ("xmmsfind.c: (xmmsfind_do_hide_main_window): start\n");
#endif
    gdk_window_get_size(xmmsfind_main_window->window, &(cfg_x.size_x), &(cfg_x.size_y));
    cfg_write_changes(cfg_x);
    gtk_widget_hide (xmmsfind_main_window);
#ifdef DEBUG
    g_print ("xmmsfind.c: (xmmsfind_do_hide_main_window): ok\n");
#endif
    return TRUE;
} 



/* Some callbackfunctions  */

void main_window_focus_in_event (GtkWidget *widget, GdkEvent *event)
{
#ifdef DEBUG
    g_print ("xmmsfind.c: (main_window_focus_in_event): start\n");
#endif
    gtk_widget_grab_focus (GTK_WIDGET (mwin_txt_search));
    /* To make sure that the checkboxes indicate the correct values */
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (mwin_chk_title), cfg_x.match_title);
    gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (mwin_chk_file), cfg_x.match_path);
#ifdef DEBUG
    g_print ("xmmsfind.c: (main_window_focus_in_event): ok\n");
#endif

}

/* Called when a checkbutton is clicked in the main window */
void mwin_chk_clicked (GtkWidget *chk_btn, gpointer data)
{
    gboolean state = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (chk_btn));
#ifdef DEBUG
    g_print ("xmmsfind.c: (mwin_chk_clicked): (%d) start\n", GPOINTER_TO_INT (data));
#endif
    
    switch (GPOINTER_TO_INT (data))
    {
      case OPT_MATCH_TITLE:
	cfg_x.match_title = state;
	break;
      case OPT_MATCH_FILENAME:
	cfg_x.match_path = state;
	break;
    }
    cfg_write_changes (cfg_x);
    xmmsfind_do_search ();
    gtk_widget_grab_focus (GTK_WIDGET (mwin_txt_search));
    xmmsfind_clist_move_to_current_song (xmms_remote_get_playlist_pos(session));
#ifdef DEBUG
    g_print ("xmmsfind.c: (mwin_chk_clicked): (%d) ok\n", GPOINTER_TO_INT (data));
#endif
}

void mwin_clist_row_selected (GtkWidget * widget, gint row, gint column, GdkEventButton * event, gpointer data)
{
    if (event) 
      if (event->button == 1 && event->type == GDK_2BUTTON_PRESS)
	xmmsfind_do_song ();
}

/* called by gtk when a key was pressed in the clist. It (currently) only
 * checks for <Return> or <Escape> */
void mwin_clist_key_pressed (GtkWidget * widget, GdkEventKey * event, gpointer data)
{
#ifdef DEBUG
    printf ("xmmsfind.c: (clist_key_pressed): start\n");
#endif
  switch (event->keyval)
    {
    case GDK_Return:
	xmmsfind_do_song ();
	break;
    case GDK_Escape:
	xmmsfind_do_hide_main_window ();
	break;
      case GDK_F5:
	xmmsfind_do_refresh_clist (NULL, NULL);
	break;
    }
#ifdef DEBUG
    printf ("xmmsfind.c: (clist_key_pressed): ok\n");
#endif

}

/* Called when a key is pressed in the searchbox */
void mwin_txt_search_key_pressed (GtkWidget * txtEntry, GdkEventKey * event, gpointer data)
{
    GtkCList *clist = GTK_CLIST (data);
    gboolean stop = FALSE;
#ifdef DEBUG
    printf ("xmmsfind.c: (mwin_txt_search_key_pressed): start\n");
#endif

    switch (event->keyval)
    {
      case GDK_Escape:
        xmmsfind_do_hide_main_window ();
	break;
      case GDK_Return:
	if ((event->state) & GDK_CONTROL_MASK)
	  do_enqueue_this_time = TRUE;
	else if ((event->state) & GDK_SHIFT_MASK)
	  do_enqueue_entire_clist = TRUE;
	xmmsfind_do_song ();
	stop = TRUE;
	break;
      case GDK_Up:
      case GDK_Down:
      case GDK_Page_Up:
      case GDK_Page_Down:
	gtk_widget_event (GTK_WIDGET (clist), (GdkEvent *) event);
	stop = TRUE;
	break;
      case GDK_r:
	if (!((event->state) & GDK_CONTROL_MASK))
	  break;
      case GDK_F5:
	xmmsfind_do_refresh_clist (NULL, NULL);
	break;
      default:
	break;
    }
    /* Make the focus remain in the textentry  */
    if (stop)
      gtk_signal_emit_stop_by_name (GTK_OBJECT (mwin_txt_search), "key_press_event");
#ifdef DEBUG
    printf ("xmmsfind.c: (mwin_txt_search_key_pressed): done\n");
#endif

} /* mwin_txt_search_key_pressed */

/* Converts a string with words separeted by one or more whitespaces (" ", 
 * "\t" etc.) into an array of strings.
 * The array is allocated memory for, and a pointer is returned to the
 * first element */
gchar **str2arr (gchar * str, gint * elements)
{
    gint i = 0;
    gint e_count = 0, dest_count;
    gchar **destarr, *ptr;

#ifdef DEBUG
    g_print ("xmmsfind.c: (str2arr) (%s): start\n", str);
#endif
    
    /* Remove leading and trailing whitespaces  */
    g_strstrip (str);
    ptr = str;
    /* remove unneccessary whitespaces (= more than one in a row)  */
    while (ptr < &str[strlen (str)])
    {
      if (isspace (*ptr))
	{
	    str[i] = *ptr;	/* We'll allow one space but no more than that */
	    ptr++;
	    i++;
	    while (isspace (*ptr) && (ptr < &str[strlen (str)]))	
	      ptr++;
	}
	else
	{			/* Okay, not a whitespace: just copy the char */
	    str[i] = *ptr;
	    ptr++;
	    i++;
	}
    }
    str[i] = 0;			/* Null-terminate the new string */
    /* Now the str should consist of strings separated by _one_ whitespace  */
    *elements = count_words (str);
    destarr = (gchar **) g_malloc (*elements * sizeof (gchar *));
    if (destarr == NULL)
    {
	g_warning ("Xmmsfind: (str2arr) Unable to allocate memory\n");
	return NULL;
    }
    /* Now let's split the string into an array  */
    ptr = str;
    for (e_count = 0; e_count < *elements; e_count++)
    {
	dest_count = 0;
	destarr[e_count] = (gchar *) g_malloc0 (SEARCHSTR_MAX + 1);
	while ((!isspace (*ptr)) && (ptr < &str[strlen (str)]))
	{
	    destarr[e_count][dest_count] = *ptr;
	    dest_count++;
	    ptr++;
	}
	destarr[e_count][dest_count + 1] = 0;	/* Okay, whitespace reached, Null-terminate  the string */
	ptr++;
    }
#ifdef DEBUG
    g_print ("xmmsfind.c: (str2arr): ok\n");
#endif

    return destarr;
} /* str2arr */

/*
 * Function used to determine hom many words there are in a string.  The
 * string must _not_ have spaces at the beginning or in the end, and all
 * words must be separated by _ONE_ whitspace! Returns the number of
 * whitespaces + 1 
 */
gint count_words (gchar * s)
{
    gint count = 0, i;
    
    for (i = 0; i < strlen (s); i++)
      if (isspace (s[i]))
	count++;
    return count + 1;
}
