#ifndef _PrefPackageClient_H
#define _PrefPackageClient_H


/* 
Terminology:  
"main"/"pref package client" - the thread calling/using services from others; the main program
"pref package" - the complete set of calls and data structures associated with the pref suite (ie, the code libraray)
"pref window" - the instance of an OS window that contains editable GUI elements
"pref glue" - calls in this API implemented by the pref package
"main glue" - calls in this API implemented by main
"true" - non-zero integer

*/

class PrefPackage;


class PrefPackageClient {


	public:
		/*  
		  This tells the caller that the pref window has been created and is currently running.  Calling
		this says to main that the pref package is ready to support all its calls in its API.  Likewise,
		PrefWindowReady() must be called before anything in the main glue can be called.
		Pre: Calls to PrefWindowReady() will always be separated by calls to PrefWindowGone()
		 */ 
		virtual void				PrefWindowReady() = 0;


		/* 
		   This tells this PrefPackageClient the pref window has exited and there will be no further calls to the 
		main glue.  The pref package should call this when its shut down the pref window.  This also tells 
		this PrefPackageClient that the PrefPackage is ready to be safely/cleanly deleted.
		Pre: PrefWindowReady() was called earlier. 
		Pre: <inSaveChanges> is true iff calls to SetParam() should persist/be saved. If <inSaveChanges> is false,
		all the params in the client will be restored to the value they had before PrefWindowReady(). */
		virtual void				PrefWindowGone( int inSaveChanges ) = 0;

		/*
		   Sets the value of a pref field/param in main.    
		A param is either a string, integer, or boolean (defined by specification). If the 
		param is a boolean/checkbox, the empty string signifies false/cleared and non-empty
		signifies true/checked.    
		Pre: <inParamID> intentified the param to be set
		Pre: <inParam> points to a char buffer with an end signified with a NUL byte. */
		virtual void				SetParam( long inParamID, const char* inParam ) = 0;

		/* Retrieves the value of a pref field/param in main.
		Pre: <inParamID> intentified the param to be fetched
		Pre: <inOriginalParam> is true the initial/reverted value of the param will be fetched (ie, the value of the param
		     before PrefWindowReady() was first called).  If it's false, the most recent value (via SetParam() or its 
		     initial value) will be fetched.
		Pre: <inBufSize> is the dimmed size of <inParamBuf> in bytes
		Pre: <ioParamBuf> points a buffer of at least size <inBufSize>
		Post: The param indentified by <inParamID> is returned in <ioParamBuf> iff Main_GetParam returns true.
		Retruns: true iff <ioParamBuf> was set to the value of the param identidied by <inParamID> */
		virtual int					GetParam( long inParamID, int inOriginalParam, char* ioParamBuf, int inBufSize ) = 0;
};





#endif