/* rgbtxt.c - xpm2wico - .xpm to Windows .ico format converter
 *
 * Copyright (C) 2002 Wolfgang Sourdeau
 *
 * Author: Wolfgang Sourdeau <wolfgang@contre.com>
 *
 * This file is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This file is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "rgbtxt.h"

char *rgb_txt_file;

extern int verbose;

static int db_loaded = 0;
static int colour_count;
static RgbDictEntry *rgbdict = NULL;
static RgbDictEntry *last_entry = NULL;

struct _FirstIntReturn
{
  int integer;
  char *nextchar;
};

static char *
first_char (char *line)
{
  char *character;

  character = line;
  while ((*character == ' '
	  || *character == '\t')
	 && *character != 0)
    character++;

  return character;
}

static struct _FirstIntReturn
first_int (char *line)
{
  char *character;
  struct _FirstIntReturn new_int;

  character = line;

  while (*character != 0 && (*character < '0' || *character > '9'))
    character++;
  if (*character == 0)
    new_int.integer = -1;
  else
    {
      new_int.integer = 0;
      while (*character >= '0' && *character <= '9')
	{
	  new_int.integer *= 10;
	  new_int.integer += (int) (*character - 48);
	  character++;
	}
    }

  new_int.nextchar = character;

  return new_int;
}

static char *
isolate_name (char *string)
{
  char *name, *charptr, *initial_char;

  charptr = first_char (string);
  initial_char = charptr;
  while (*charptr && *charptr != '\n' && *charptr != '!')
    charptr++;

  while (charptr > initial_char
	 && (*charptr == '\n'
	     || *charptr == '!'
	     || *charptr == ' '
	     || *charptr == '\t'))
    {
      *charptr = 0;
      charptr--;
    }

  charptr = first_char (string);
  name = strdup (charptr);

  return name;
}

static void
push_rgbtxtline (char *line)
{
  char *character;
  int red, green, blue;
  char *name;
  struct _FirstIntReturn first_int_return;
  RgbDictEntry *new_entry;

  character = first_char (line);
  if (*character != '!')
    {
      first_int_return = first_int (character);
      red = first_int_return.integer;
      first_int_return = first_int (first_int_return.nextchar);
      green = first_int_return.integer;
      first_int_return = first_int (first_int_return.nextchar);
      blue = first_int_return.integer;
      name = isolate_name (first_int_return.nextchar);

      if (red != -1 && green != -1 && blue != -1
	  && strlen (name))
	{
	  new_entry = malloc (sizeof (RgbDictEntry));
	  new_entry->red = red;
	  new_entry->green = green;
	  new_entry->blue = blue;
	  new_entry->name = name;
	  new_entry->next = NULL;
	}
      else
	new_entry = NULL;
    }
  else
    new_entry = NULL;

  if (new_entry)
    {
      colour_count++;

      if (last_entry)
	last_entry->next = new_entry;
      else
	rgbdict = new_entry;

      last_entry = new_entry;
    }
}

static void
read_rgbtxt ()
{
  FILE *rgbtxt;
  char *line;
  char buffer[256];

  if (verbose)
    printf ("Reading colour database from file \"%s\"...\n",
	    rgb_txt_file);

  rgbtxt = fopen (rgb_txt_file, "r");
  if (rgbtxt)
    {
      colour_count = 0;

      while (!feof (rgbtxt))
	{
	  line = fgets (buffer, 255, rgbtxt);
	  if (line)
	    push_rgbtxtline (line);
	}
      fclose (rgbtxt);
    }
  else
    {
      perror (rgb_txt_file);
      fprintf (stderr, "You might want to specify a different path with the '-r' parameter.\n");
    }

  db_loaded = 1;

  if (verbose)
    printf ("%d colour(s) loaded\n", colour_count);
}

RgbDictEntry *
seek_dict_entry (char *name)
{
  RgbDictEntry *cur_entry;

  if (!db_loaded)
    read_rgbtxt ();

  if (verbose)
    printf ("Seeking entry for colour \"%s\"...\n", name);

  cur_entry = rgbdict;

  while (cur_entry && strcasecmp (cur_entry->name, name))
    cur_entry = cur_entry->next;

  return cur_entry;
}
