/*  
  Copyright 2002, Andreas Rottmann

  This library is free software; you can redistribute it and/or
  modify it under the terms of the GNU Lesser General Public
  License as published by the Free Software Foundation; either
  version 2.1 of the License, or (at your option) any later version.

  This library is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  Lesser General Public License for more details.

  You should have received a copy of the GNU Lesser General Public
  License along with this library; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA
*/
#include <Python.h>

extern "C" {
  // this is from pythons compile.h
  struct _node; /* Declare the existence of this type */
  PyObject *PyNode_Compile(struct _node *, char *);
  
  // this is from pythons node.h
  void PyNode_Free(struct _node *n);
}

#include <dirent.h>
#include <sys/stat.h>

#include <yehia/plugin.h>

#include "python-loader.h"
#include "python-script.h"

using namespace SigC;
using namespace SigCX;
using namespace Yehia;

namespace Yehia
{

namespace {

Plugin *call_plugin_constructor(PyObject *module, std::string name, 
                                PluginManager *mgr)
{
  PyObject *args = 0;
  PyObject *init_func = 0;
  PyObject *result = 0;
  PyObject *pluginmodule = 0;
  PyObject *mgr_class = 0;
  PyObject *mgr_instance = 0;
  PyObject *ptr_str = 0;
  PyObject *py_mgr = 0;

  SigC::Object *obj = 0;
  Plugin *plugin = 0;
  
  // use only last part of name
  std::string::size_type pos = name.find_last_of('.');
  if (pos != std::string::npos)
    name = name.substr(pos + 1);
  
  init_func = PyObject_GetAttrString(module,
                                     (char *)(name + "Plugin").c_str());
      
  if (init_func && PyCallable_Check(init_func))
  {
    args = PyTuple_New(0);
    if (!args ||
        !(pluginmodule = PyImport_ImportModule("yehia")) ||
        !(mgr_class = PyObject_GetAttrString(pluginmodule,
                                             "PluginManager")) ||
        !(mgr_instance = PyObject_GetAttrString(mgr_class, "instance")))
    {
      goto cleanup;
    }
    
    if (!(py_mgr = PyObject_CallObject(mgr_instance, args)))
    {
      goto cleanup;
    }
    Py_DECREF(args);
    args = PyTuple_New(1);
    Py_INCREF(py_mgr);
    PyTuple_SetItem(args, 0, py_mgr);
    if ((result = PyObject_CallObject(init_func, args)) != 0)
      goto cleanup;

    ptr_str = PyObject_GetAttrString(result, "__yehiainstance__");
    if (!ptr_str || !PyString_Check(ptr_str) ||
        sscanf(PyString_AsString(ptr_str), "%p", &obj) != 1)
    {
      goto cleanup;
    }
    if (!(plugin = dynamic_cast<Plugin *>(obj)))
    {
      mgr->set_error("invalid plugin class");
      goto cleanup;
    }
  }
 cleanup:
  Py_XDECREF(ptr_str);
  Py_XDECREF(result);
  Py_XDECREF(pluginmodule);
  Py_XDECREF(mgr_class);
  Py_XDECREF(mgr_instance);
  Py_XDECREF(init_func);
  Py_XDECREF(args);

  return plugin;
}

Plugin *load_module(PluginManager *mgr, PluginLoader *instance, 
                    const std::string& name)
{
  // Note: we can simply access mgr and instance, even if we run in our
  // own thread since we are executed synchronously
  
  // Note: This function is a bit hakish; the clean way to 
  PyObject *module, *o;
  Plugin *plugin = 0;
  std::string pathname = name;
  
  // s/-/_/g  s|\.|/|
  for (std::string::iterator it = pathname.begin(); it != pathname.end(); ++it)
  {
    if (*it == '-')
      *it = '_';
    else if (*it == '.')
      *it = '/';
  }
  
  std::list<std::string>::const_iterator it;
  for (it = mgr->arch_indep_paths().begin(); 
       it != mgr->arch_indep_paths().end(); ++it)
  {
    std::string fname = *it + "/" + pathname + ".py";
    FILE *fp = fopen(fname.c_str(), "r");
    if (fp != NULL)
    {
      char *c_fname = const_cast<char *>(fname.c_str());
#if 1
      // Read in and parse file
      struct _node *n = PyParser_SimpleParseFile(fp, c_fname, Py_file_input);
      if (!n)
        break;
      // Compile it
      PyObject *co = (PyObject *)PyNode_Compile(n, c_fname);
      PyNode_Free(n);
#else
      PyObject *co = Py_CompileString("", "foobar.py", Py_file_input);
#endif
      if (!co)
        break;
      // Execute in new module
      module = PyImport_ExecCodeModuleEx(const_cast<char *>(name.c_str()), co,
                                         c_fname);
      if (module)
      {
        PyObject *mod_dict = PyModule_GetDict(module);
        PyDict_DelItemString(mod_dict, "__name__"); // hack, hack
        Py_XDECREF(o);
        plugin = call_plugin_constructor(module, name, mgr);
      }
      break;
    }
    else
      mgr->set_error("no such file: '" + fname + "'");
  }
  
  if (!plugin)
  {
    PyObject *type, *value, *tracebk, *errstr, *tracebkstr;

    PyErr_Fetch(&type, &value, &tracebk);
    PyErr_NormalizeException(&type, &value, &tracebk);

    if (type && PyErr_GivenExceptionMatches(type, PyExc_Exception) &&
        value != 0 && ((errstr = PyObject_Str(value)) != 0)) //&&
      //tracebk && ((tracebk = PyObject_Str(tracebk)) != 0))

    {
      mgr->set_error(PyString_AsString(errstr));
    }
    
    Py_XDECREF(type);
    Py_XDECREF(value);
    Py_XDECREF(tracebk);
  }
  return plugin;
}

} // anon namespace

pythonPluginLoader::pythonPluginLoader(PluginManager& mgr)
    : PluginLoader(&mgr), Plugin(mgr)
{
}

pythonPluginLoader::~pythonPluginLoader()
{
}


Plugin *pythonPluginLoader::load(PluginManager& mgr, const std::string& name)
{
  Plugin *plugin = 0;
  
  Script::Language *lang = 
    Script::LanguageManager::instance().language("python");
  
  if (lang)
  {
    plugin = tunnel<Plugin *, PluginManager *, PluginLoader *, 
      const std::string&>(
            slot(load_module), &mgr, static_cast<PluginLoader *>(this), name, 
            lang->tunnel(), true);
  }
  if (plugin)
    plugin->reference();

  return plugin;
}

namespace
{

void do_scan(PluginManager& mgr, const std::string& path, 
             const std::string& basename)
{
  DIR *dir;
  struct dirent *file;

  if ((dir = opendir(path.c_str())) != NULL)
  {
    while ((file = readdir(dir)) != NULL)
    {
      int len;
      std::string fullname = path + file->d_name;
      struct stat statinfo;
      
      if (file->d_name[0] != '.' &&
          stat(fullname.c_str(), &statinfo) == 0 && S_ISDIR(statinfo.st_mode))
        do_scan(mgr,
                path + file->d_name + "/", basename + file->d_name + ".");
      
      len = strlen(file->d_name);
      if (file->d_name[0] == '_') 
        continue; // ignore files with leading underscore
      
      if (len > 3 && strcmp(file->d_name + len - 3, ".py") == 0)
        mgr.plugin_found(basename + 
                          std::string(file->d_name).substr(0, len - 3));
      else if (len > 4 && strcmp(file->d_name + len - 4, ".pyc") == 0)
        mgr.plugin_found(basename + 
                          std::string(file->d_name).substr(0, len - 4));
    }
    closedir(dir);
  }
}

}

void pythonPluginLoader::scan(PluginManager& mgr) const
{
  const std::list<std::string>& paths = mgr.arch_indep_paths();
  std::list<std::string>::const_iterator it;

  for (it = paths.begin(); it != paths.end(); ++it)
    do_scan(mgr, (*it), "");
}


}
