/* This is for Emacs: -*-Mode: Text;-*- */
%{

#include <stdlib.h>

#include "tree.h"
#include "process.h"
#include "format.h"
#include "lexer.h"

#define YYDEBUG 1

static void yyerror(const char *msg)
{
  process_error(msg);
}

%}

%union { Tree tree; TreeCode code; int itype; }

/* Operators */
%token POINTSAT SCOPE

/* Keyords */
%token NAMESPACE VALUE CLASS CONSTRUCTOR METHOD GETTER SETTER SIGNAL 
%token WRAP CONST USING VIRTUAL

/* meta-instructions */
%token SHARP_PLUGIN

/* Constants */
%token STRING_CONSTANT INTEGER_CONSTANT C_CODE

%token IDENTIFIER TYPESPEC


%type <itype> class_wrap_opt const_opt attr_kind virtual_opt purespec_opt

%type <tree> file stmts stmt meta_stmt
%type <tree> used_module_list_opt module_list_nonempty module_name
%type <tree> namespace_stmt namespace_members namespace_member
%type <tree> value_stmt class_stmt class_decl
%type <tree> value_spec
%type <tree> defby_type_opt defby_opt class_supers_opt 
%type <tree> class_supers class_members class_member
%type <tree> signature signature_opt retspec_opt
%type <tree> param_list param_list_notempty argspec
%type <tree> type_list_opt type_list typespec typespec0 typespec1
%type <tree> typename typename_sub template_spec_opt
%type <tree> constant
%type <tree> IDENTIFIER STRING_CONSTANT INTEGER_CONSTANT TYPESPEC C_CODE


%right SCOPE

%%

file: stmts
        { process_toplevel_stmts($1); }
    ;

stmts: /* empty */
         { $$ = NULL_TREE; }
     | stmt stmts
         { $$ = tree_cons($1, $2); }
     ;

stmt: namespace_stmt
    | meta_stmt
    | C_CODE
    ;

meta_stmt: SHARP_PLUGIN IDENTIFIER STRING_CONSTANT used_module_list_opt ';'
	     { $$ = tree_build_meta_plugin($2, $3, $4); }
	 ;

used_module_list_opt: /* empty */
		        { $$ = NULL_TREE; }
		    | ':' module_list_nonempty
		        { $$ = $2; }
		    ;

module_list_nonempty: module_name 
		        { $$ = tree_cons($1, NULL); }
		    | module_name ',' module_list_nonempty
		        { $$ = tree_cons($1, $3); }
		    ;

module_name: IDENTIFIER
	       { $$ = tree_cons($1, NULL_TREE); }
	   | IDENTIFIER '.' module_name
	       { $$ = tree_cons($1, $3); }
	   ;

namespace_stmt: NAMESPACE IDENTIFIER '{' namespace_members '}'
                  { $$ = tree_build_namespace($2, $4); }
              ;

namespace_members: /* empty */
		     { $$ = NULL_TREE; }
	         | namespace_member namespace_members
	             { $$ = tree_cons($1, $2); }
	         ;

namespace_member: C_CODE
		| value_stmt
		| class_decl
	        | class_stmt
		| namespace_stmt
	        | METHOD IDENTIFIER defby_opt retspec_opt ';'
		    { $$ = tree_build_attribute(ATTR_METHOD, $2, $3, 
		                                NULL_TREE, $4, FALSE, FALSE); }
	        ;

value_stmt: VALUE IDENTIFIER value_spec ';'
	      { $$ = tree_build_value($2, $3); }
	  ;

value_spec: constant
	  | IDENTIFIER
	  ;

class_decl: USING CLASS IDENTIFIER defby_type_opt ';'
	      { $$ = tree_build_class_decl($3, $4); }
	  ;
 
class_stmt: CLASS class_wrap_opt IDENTIFIER defby_opt class_supers_opt '{' class_members '}'
	      { $$ = tree_build_class($3, $2, $4, $5, $7); }
	  ;

class_wrap_opt: /* empty */
		  { $$ = 0; }
	      | WRAP
		  { $$ = 1; }
	      ;

defby_type_opt: /* empty */
	     { $$ = NULL_TREE; }
	 | POINTSAT typename
	     { $$ = $2; }
	 ;

defby_opt: defby_type_opt
	 | POINTSAT C_CODE
	     { $$ = $2; }
	 ;

class_supers_opt: /* empty */
		    { $$ = NULL_TREE; }
		| ':' class_supers
		    { $$ = $2; }
		;

class_supers: typename
	        { $$ = tree_cons($1, NULL_TREE); }
	    | typename ',' class_supers
	      { $$ = tree_cons($1, $3); }
	    ;
class_members: /* empty */
	         { $$ = NULL_TREE; }
	     | class_member class_members
	         { $$ = tree_cons($1, $2); }
	     ;

class_member: CONSTRUCTOR signature_opt defby_opt ';'
	       { $$ = tree_build_attribute(
	           ATTR_CONSTRUCTOR, NULL_TREE, $3, $2, NULL_TREE, FALSE, 
		   FALSE); }
	    | VIRTUAL METHOD IDENTIFIER signature const_opt ':' typespec defby_opt 
	        purespec_opt ';'
	       { $$ = tree_build_attribute(ATTR_METHOD, $3, $8, $4, $7,
	                                   TRUE, $9); 
		 TREE_ATTR_CONST($$) = $5; }
	    | attr_kind IDENTIFIER signature_opt retspec_opt defby_opt ';'
	       { $$ = tree_build_attribute($1, $2, $5, $3, $4, FALSE, FALSE); }
	    | C_CODE
	    ;

purespec_opt: /* empty */
	        { $$ = 0; }
	    | '=' INTEGER_CONSTANT
	        { if (TREE_INT_CST($2) != 0 || !TREE_UNSIGNED($2))
		    process_error("invalid pure specifier '%s'", 
				  format_value($2));
		  $$ = 1; }
	    ;

attr_kind: METHOD
	     { $$ = ATTR_METHOD; }
	 | CLASS METHOD
	     { $$ = ATTR_CLASS_METHOD; }
	 | GETTER
	     { $$ = ATTR_GETTER; }
	 | SETTER
	     { $$ = ATTR_SETTER; }
	 | SIGNAL
	     { $$ = ATTR_SIGNAL; }
	 ;

virtual_opt: /* empty */
	       { $$ = 0; }
	   | VIRTUAL
	       { $$ = 1; }
	   ;

signature_opt: /* empty */
	         { $$ = NULL_TREE; }
	     | signature
	     ;

retspec_opt: /* empty */
	       { $$ = NULL_TREE; }
	   | ':' typespec
	       { $$ = $2; }
	   ;

signature: '(' param_list ')'
	     { $$ = $2; }
	 ;

param_list: /* empty == void */
	    { $$ = tree_cons(tree_build_paramdescr(NULL_TREE, NULL_TREE), 
	                     NULL_TREE); }
	| param_list_notempty
	;

param_list_notempty: argspec
	             { $$ = tree_cons($1, NULL_TREE); }
	         | argspec ',' param_list_notempty
		     { $$ = tree_cons($1, $3); }
		 ;

argspec: typespec IDENTIFIER
	    { $$ = tree_build_paramdescr($1, $2); }
       | typespec
            { $$ = tree_build_paramdescr($1, NULL_TREE); }
       ;

type_list_opt: /* empty */
	         { $$ = NULL_TREE; }
	    | type_list
	    ;

type_list: typespec
	     { $$ = tree_cons($1, NULL_TREE); }
	 | typespec ',' type_list
	    { $$ = tree_cons($1, $3); }
	;

typespec: const_opt typespec0
	    { $$ = tree_build_typedescr($1, $2, TYPEDESCR_NORMAL); }
	| const_opt typespec0 '&' 
	    { $$ = tree_build_typedescr($1, $2, TYPEDESCR_REF); }
	| const_opt typespec0 '*'
	    { $$ = tree_build_typedescr($1, $2, TYPEDESCR_PTR); }
	;

const_opt: /* empty */
	     { $$ = 0; }
	 | CONST
	     { $$ = 1; }
	 ;

typespec0: typespec1
	 | typename
	 ;

typespec1: TYPESPEC
	     { $$ = tree_cons($1, NULL_TREE); }
	 | TYPESPEC typespec1
	     { $$ = tree_cons($1, $2); }
	 ;

typename: SCOPE typename_sub
	    { $$ = tree_cons(NULL_TREE, $2); }
	| typename_sub
	;

typename_sub: IDENTIFIER template_spec_opt
	         { $$ = tree_cons(tree_build_typename($1, $2), NULL_TREE); }
	    | IDENTIFIER template_spec_opt SCOPE typename_sub
	         { $$ = tree_cons(tree_build_typename($1, $2), $4); }
            ;

template_spec_opt: /* empty */
		     { $$ = NULL_TREE; }
	         |  '<' type_list '>'
		     { $$ = $2; }
		 ;

constant: STRING_CONSTANT
	| INTEGER_CONSTANT
	;
