#charset "us-ascii"

/* Copyright (c) 2000, 2002 by Michael J. Roberts.  All Rights Reserved. */
/*
 *   TADS 3 Library - scoring
 *   
 *   This module defines objects related to keeping track of the player's
 *   score, which indicates the player's progress through the game.  
 */

/* include the library header */
#include "adv3.h"


/* ------------------------------------------------------------------------ */
/*
 *   An Achievement is an object used to award points in the score.  For
 *   most purposes, an achievement can be described simply by a string,
 *   but the Achievement object provides more flexibility in describing
 *   combined scores when a set of similar achievements are to be grouped.
 */
class Achievement: object
    /* 
     *   The number of times the achievement has been awarded.  Each time
     *   the achievement is passed to addToScore(), this is incremented.
     *   Note that this is distinct from the number of points.  
     */
    scoreCount = 0
    
    /* 
     *   Describe the achievement - this must display a string explaining
     *   the reason the points associated with this achievement were
     *   awarded.
     *   
     *   Note that this description can make use of the scoreCount
     *   information to show different descriptions depending on how many
     *   times the item has scored.  For example, an achievement for
     *   finding various treasure items might want to display "finding a
     *   treasure" if only one treasure was found and "finding five
     *   treasures" if five were found.
     *   
     *   In some cases, it might be desirable to keep track of additional
     *   custom information, and use that information in generating the
     *   description.  For example, the game might keep a list of
     *   treasures found with the achievement, adding to the list each
     *   time the achievement is scored, and displaying the contents of
     *   the list when the description is shown.
     */
    desc = ""

    /* 
     *   the number of points awarded for the achievement; if this
     *   achievement has been accomplished multiple times, this reflects
     *   the aggregate number of points awarded for all of the times it
     *   has been accomplished 
     */
    totalPoints = 0
;

/*
 *   Generic text achievement.  When we add an achievement to the full
 *   score list and the achievement is a simple string description, we'll
 *   create one of these to encapsulate the achievement.  
 */
class SimpleAchievement: Achievement
    /* create dynamically with a given string as our description */
    construct(str) { desc_ = str; }

    /* show my description */
    desc { say(desc_); }

    /* my description string */
    desc_ = ''
;

/*
 *   List interface for showing the full score list 
 */
fullScoreLister: Lister
    showListPrefixTall(itemCount, pov, parent)
    {
        /* showt he full score list intro message */
        libMessages.showFullScorePrefix;
    }

    /* every achievement is listed */
    isListed(obj) { return true; }

    /* ignore 'seen' markings */
    markAsSeen(obj, pov) { }

    /* achievements have no containment hierarchy */
    getContents(obj) { return []; }
    getListedContents(obj, infoTab) { return []; }
    showContentsList(pov, obj, options, indent, infoTab) { }
    showInlineContentsList(pov, obj, options, indent, infoTab) { }

    /* show an item */
    showListItem(obj, options, pov, infoTab)
    {
        /* show the number of points for the item */
        libMessages.fullScoreItemPoints(obj.totalPoints);

        /* show the achievement's description */
        obj.desc;
    }
;
    

/*
 *   Score notification daemon handler.  We'll receive a
 *   checkNotification() call each turn; we'll display a notification
 *   message each time the score has changed since the last time we ran.  
 */
scoreNotifier: PreinitObject
    /* the score as it was the last time we displayed a notification */
    lastScore = static (libScore.totalScore)

    /* we've never generated a notification about the score before */
    everNotified = nil

    /* daemon entrypoint */
    checkNotification()
    {
        /* 
         *   if the score has changed since the last time we checked,
         *   possibly generate a notification 
         */
        if (libScore.totalScore != lastScore)
        {
            /* only show a message if we're allowed to */
            if (libScore.scoreNotify)
            {
                local delta;
            
                /* calculate the change since the last notification */
                delta = libScore.totalScore - lastScore;
                
                /* 
                 *   generate the first or non-first notification, as
                 *   appropriate 
                 */
                if (everNotified)
                    libMessages.scoreChange(delta);
                else
                    libMessages.firstScoreChange(delta);
            
                /* 
                 *   note that we've ever generated a score change
                 *   notification, so that we don't generate the more
                 *   verbose first-time message on subsequent
                 *   notifications 
                 */
                everNotified = true;
            }

            /* 
             *   Remember the current score, so that we don't generate
             *   another notification until the score has changed again.
             *   Note that we note the new score even if we aren't
             *   displaying a message this time, because we don't want to
             *   generate a message upon re-enabling notifications.  
             */
            lastScore = libScore.totalScore;
        }
    }

    /* execute pre-initialization */
    execute()
    {
        /* initialize the score change notification daemon */
        new PromptDaemon(self, &checkNotification);
    }
;

/* 
 *   Add points to the total score.  This is a convenience function that
 *   simply calls libScore.addToScore_().
 */
addToScore(points, desc)
{
    /* simply call the libScore method to handle it */
    libScore.addToScore_(points, desc);
}

/*
 *   The main game score object.  
 */
libScore: PreinitObject
    /*
     *   Add to the score.  'points' is the number of points to add to the
     *   score, and 'desc' is a string describing the reason the points
     *   are being awarded, or an Achievement object describing the points.
     *   
     *   We keep a list of each unique achievement.  If 'desc' is already
     *   in this list, we'll simply add the given number of points to the
     *   existing entry for the same description.
     *   
     *   Note that, if 'desc' is an Achievement object, it will match a
     *   previous item only if it's exactly the same Achievement instance.
     */
    addToScore_(points, desc)
    {
        local idx;
        
        /* 
         *   if the description is a string, encapsulate it in a
         *   SimpleAchievement object 
         */
        if (dataType(desc) == TypeSString)
        {
            local newDesc;
            
            /* 
             *   look for an existing SimpleAchievement in our list with
             *   the same descriptive text - if we find one, reuse it,
             *   since this is another instance of the same group of
             *   achievements and thus can be combined into the same
             *   achievement object 
             */
            newDesc = libScore.fullScoreList.valWhich(
                { x: x.ofKind(SimpleAchievement) && x.desc_ == desc });

            /* 
             *   if we didn't find it, create a new simple achievement to
             *   wrap the descriptive text 
             */
            if (newDesc == nil)
                newDesc = new SimpleAchievement(desc);

            /* 
             *   for the rest of our processing, use the wrapper simple
             *   achievement object instead of the original text string 
             */
            desc = newDesc;
        }

        /* increase the use count for the achievement */
        desc.scoreCount++;
        
        /* add the points to the total */
        libScore.totalScore += points;
        
        /* try to find a match in our list of past achievements */
        idx = libScore.fullScoreList.indexOf(desc);
        
        /* if we didn't find it, add it to the list */
        if (idx == nil)
            libScore.fullScoreList.append(desc);

        /* 
         *   combine the points awarded this time into the total for this
         *   achievement 
         */
        desc.totalPoints += points;
    }

    /*
     *   Show the simple score 
     */
    showScore()
    {
        /* show the basic score statistics */
        libMessages.showScoreMessage(libScore.totalScore, libScore.maxScore,
                                     libGlobal.totalTurns);

        /* show the score ranking */
        showScoreRank(libScore.totalScore);
    }

    /* 
     *   show the score rank message 
     */
    showScoreRank(points)
    {
        local idx;
        
        /* if there's no rank table, skip the ranking */
        if (libMessages.scoreRankTable == nil)
            return;
        
        /*
         *   find the last item for which our score is at least the
         *   minimum - the table is in ascending order of minimum score,
         *   so we want the last item for which our score is sufficient 
         */
        idx = libMessages.scoreRankTable.lastIndexWhich({x: points >= x[1]});
    
        /* if we didn't find an item, use the first by default */
        if (idx == nil)
            idx = 1;
    
        /* show the description from the item we found */
        libMessages.showScoreRankMessage(libMessages.scoreRankTable[idx][2]);
    }

    /*
     *   Display the full score 
     */
    showFullScore()
    {
        /* show the basic score statistics */
        showScore();
        
        /* list the achievements in 'tall' mode */
        fullScoreLister.showListAll(libScore.fullScoreList.toList(),
                                    ListTall, 0);
    }

    /*
     *   Vector for the full score achievement list.  This is a list of
     *   all of the Achievement objects awarded for accomplishments so
     *   far.  
     */
    fullScoreList = static new Vector(32)

    /* the total number of points scored so far */
    totalScore = 0

    /* 
     *   the maximum number of points possible - the game should set this
     *   to the appropriate value at startup 
     */
    maxScore = 100

    /* 
     *   current score notification status - if true, we'll show a message
     *   at the end of each turn where the score changes, otherwise we
     *   won't mention anything 
     */
    scoreNotify = true

    /* execute pre-initialization */
    execute()
    {
        /* register as the global score handler */
        libGlobal.scoreObj = self;
    }
;
