#charset "us-ascii"

/* Copyright (c) 2000, 2002 Michael J. Roberts.  All Rights Reserved. */
/*
 *   TADS 3 Library - travel
 *   
 *   This module defines the parts of the simulation model related to
 *   travel: rooms and other locations, directions, passages.  
 */

/* include the library header */
#include "adv3.h"


/* ------------------------------------------------------------------------ */
/*
 *   Directions.  Each direction is represented by an instance of
 *   Direction.
 *   
 *   A Direction object encapsulates information about a travel direction.
 *   By using an object to represent each possible travel direction, we
 *   make it possible for a game or library extension to add new custom
 *   directions without having to change the basic library.  
 */
class Direction: object
    /*
     *   The link property for the direction.  This is a property pointer
     *   that we use to ask an actor's location for a TravelConnector when
     *   the actor attempts to travel in this direction.  
     */
    dirProp = nil

    /* 
     *   The default TravelConnector when an actor attempts travel in this
     *   direction, but the actor's location does not define the dirProp
     *   property for this direction.  This is almost always a connector
     *   that goes nowhere, such as noTravel, since this is used only when
     *   a location doesn't have a link for travel in this direction.  
     */
    defaultConnector = noTravel

    /*
     *   Initialize.  We'll use this routine to add each Direction
     *   instance to the master direction list (Direction.allDirections)
     *   during pre-initialization.  
     */
    initializeDirection()
    {
        /* add myself to the master direction list */
        Direction.allDirections.append(self);
    }

    /*
     *   Class initialization - this is called once on the class object.
     *   We'll build our master list of all of the Direction objects in
     *   the game, and then sort the list using the sorting order.  
     */
    initializeDirectionClass()
    {
        /* initialize each individual Direction object */
        forEachInstance(Direction, { dir: dir.initializeDirection() });

        /* 
         *   sort the direction list according to the individual Directin
         *   objects' defined sorting orders 
         */
        allDirections.sort(SortAsc, {a, b: a.sortingOrder - b.sortingOrder});
    }

    /* 
     *   Our sorting order in the master list.  We use this to present
     *   directions in a consistent, aesthetically pleasing order in
     *   listings involving multiple directions.  The sorting order is
     *   simply an integer that gives the relative position in the list;
     *   the list of directions is sorted from lowest sorting order to
     *   highest.  Sorting order numbers don't have to be contiguous,
     *   since we simply put the directions in an order that makes the
     *   sortingOrder values ascend through the list.  
     */
    sortingOrder = 1

    /* 
     *   A master list of the defined directions.  We build this
     *   automatically during initialization to include each Direction
     *   instance.  Any operation that wants to iterate over all possible
     *   directions can run through this list.
     *   
     *   By using this master list to enumerate all possible directions
     *   rather than a hard-coded set of pre-defined directions, we can
     *   ensure that any new directions that a game or library extension
     *   adds will be incorporated automatically into the library simply
     *   by virtue of the existence Direction instances to represent the
     *   new directions.  
     */
    allDirections = static new Vector(10)
;

/*
 *   The base class for compass directions (north, south, etc).  We don't
 *   add anything to the basic Direction class, but we use a separate
 *   class for compass directions to allow language-specific
 *   customizations for all of the directions and to allow travel commands
 *   to treat these specially when needed.  
 */
class CompassDirection: Direction
;

/*
 *   The base class for shipboard directions (port, aft, etc).
 */
class ShipboardDirection: Direction
    defaultConnector = noShipTravel
;

/*
 *   The base class for vertical directions (up, down) 
 */
class VerticalDirection: Direction
;

/*
 *   The base class for "relative" directions (in, out) 
 */
class RelativeDirection: Direction
;

/*
 *   Individual compass directions.  
 *   
 *   Our macro defines a direction object with a name based on the
 *   direction's room travel link property and the base class.  So,
 *   DefineDirection(north, Compass) defines a direction object called
 *   'northDirection' based on the CompassDirection class, with the link
 *   property 'north' and default travel connector 'noTravel'.
 *   
 *   Note that we define a sorting order among the default directions as
 *   follows.  First, we define several groups of related directions,
 *   which we put in a relative order: the compass directions, then the
 *   vertical directions, then the "relative" (in/out) directions, and
 *   finally the shipboard directions.  Then, we order the directions
 *   within these groups.  For the sortingOrder values, we use arbitrary
 *   integers with fairly wide separations, to leave plenty of room for
 *   custom game-specific directions to be added before, between, after,
 *   or within these pre-defined groups.  
 */
#define DefineDirection(prop, base, order) \
    prop##Direction: base##Direction \
    dirProp = &prop \
    sortingOrder = order

DefineDirection(north, Compass, 1000);
DefineDirection(south, Compass, 1100);
DefineDirection(east, Compass,  1200);
DefineDirection(west, Compass,  1300);
DefineDirection(northeast, Compass, 1400);
DefineDirection(northwest, Compass, 1500);
DefineDirection(southeast, Compass, 1600);
DefineDirection(southwest, Compass, 1700);

DefineDirection(up, Vertical, 3000);
DefineDirection(down, Vertical, 3100);

DefineDirection(in, Relative, 5000)
    defaultConnector = noTravelIn
;

DefineDirection(out, Relative, 5100)
    defaultConnector = noTravelOut
;

DefineDirection(fore, Shipboard, 7000);
DefineDirection(aft, Shipboard, 7100);
DefineDirection(port, Shipboard, 7200);
DefineDirection(starboard, Shipboard, 7300);


/* ------------------------------------------------------------------------ */
/*
 *   A Traveler is an object that can travel.  In general, the two main
 *   kinds of travelers are Actors and Vehicles.
 *   
 *   This class is intended to be multiply inherited, since it is not
 *   based on any simulation object class.  
 */
class Traveler: object
    /*
     *   Check, using pre-condition rules, that the traveler is directly
     *   in the given room.  We'll attempt to implicitly remove the actor
     *   from any nested rooms within the given room.  
     */
    checkDirectlyInRoom(dest, allowImplicit)
    {
        /* ask the destination to ensure the traveler is in it directly */
        return dest.checkTravelerDirectlyInRoom(self, allowImplicit);
    }

    /*
     *   Check, using pre-condition rules, that the traveler is in the
     *   given room, moving the traveler to the room if possible. 
     */
    checkMovingTravelerInto(room, allowImplicit)
    {
        /* subclasses must override */
    }

    /*
     *   Get the travel preconditions specific to the traveler.  By
     *   default, we return no extra conditions.  
     */
    travelerPreCond = []

    /*
     *   Show my departure message - this is shown when I'm leaving a room
     *   where the player character can see me for another room where the
     *   player character cannot see me.  
     */
    describeDeparture(dest, connector)
    {
        /* 
         *   If we're visible to the player character, describe the
         *   departure.  Ask the connector to describe the travel, since
         *   it knows the direction of travel and can describe special
         *   things like climbing stairs.
         *   
         *   Don't bother to describe the departure if the traveler is the
         *   player character, or the player character is inside the
         *   traveler.  The PC obviously can't leave the presence of
         *   itself.  
         */
        if (libGlobal.playerChar != self
            && !libGlobal.playerChar.isIn(self))
            connector.describeDeparture(location, dest);
    }

    /*
     *   Show my arrival message - this is shown when I'm entering a room
     *   in view of the player character from a location where the player
     *   character could not see me.  'backConnector' is the connector in
     *   the destination location from which we appear to be emerging.  
     */
    describeArrival(origin, backConnector)
    {
        /*
         *   Add an intra-report separator, to visually separate the
         *   description of the arrival from any result text that came
         *   before.  (For example, if we performed an implied command
         *   before we could begin travel, this will visually separate the
         *   results of the implied command from the arrival message.)  
         */
        say(libMessages.intraCommandSeparator);

        /*
         *   If the traveler is the player character, or the player
         *   character is inside the traveler, describe the arrival from
         *   the point of view of the player character by showing the
         *   "look around" description of the new location.  Otherwise,
         *   describe the traveler's arrival. 
         */
        if (libGlobal.playerChar == self || libGlobal.playerChar.isIn(self))
        {
            /* 
             *   The player character is traveling - show the travel from
             *   the PC's perspective.  If we're seeing the location for
             the first time, 
             */
            libGlobal.playerChar.lookAround(libGlobal.verboseMode, nil);
        }
        else if (libGlobal.playerChar.canSee(self))
        {
            /* 
             *   The player character isn't traveling, but the PC can see
             *   me now that I'm in my new location, so describe the
             *   arrival of the traveler.  If we know the connector back,
             *   use its arrival message; otherwise, use a generic arrival
             *   message.  
             */
            if (backConnector != nil)
                backConnector.describeArrival(origin, location);
            else
                libMessages.sayArriving(self);
        }
    }

    /*
     *   Travel to a new location.  Moves the traveler to a new location
     *   via the given connector, triggering any side effects of the
     *   travel.
     *   
     *   Note that this routine is not normally called directly; in most
     *   cases, the actor's travelTo is called, and it in turn invokes
     *   this method in the appropriate traveler.
     *   
     *   'dest' is the new location to which we're traveling.  'connector'
     *   is the TravelConnector we're traversing from the source location
     *   to reach the new destination; the connector is normally the
     *   invoker of this routine.  'backConnector' is the connector in the
     *   destination from which the actor will appear to emerge on the
     *   other end of the travel.  
     */
    travelerTravelTo(dest, connector, backConnector)
    {
        local origin;
        local isDescribed;

        /* 
         *   Remember my departure original location - this is the
         *   location where the traveler was before we moved. 
         */
        origin = location;

        /*
         *   Determine if we're describing the travel.  We'll describe it
         *   if we're actually changing locations, or if the connector is
         *   explicitly a "circular" passage, which means that it connects
         *   back to the same location but involves a non-trivial passage,
         *   as sometimes happens in settings like caves.  
         */
        isDescribed = (dest != origin || connector.isCircularPassage);

        /* tell the old room we're leaving, if we changed locations */
        if (origin != nil && isDescribed)
            origin.travelerLeaving(self, dest, connector);

        /* notify all interested actors so they can track us for 'follow' */
        sendTrackFollowInfo(connector);
        
        /* move to the destination */
        moveIntoForTravel(dest);

        /* 
         *   we've successfully completed the travel, so remember it for
         *   each actor involved in the travel 
         */
        foreach (local cur in getTravelerActors())
        {
            /* ask the actor to remember the travel */
            cur.rememberTravel(origin, dest, backConnector);

            /* remember the destination of the connector for this actor */
            connector.rememberTravel(origin, cur, dest);

            /* 
             *   If there's a back-connector, also remember travel in the
             *   other direction.  The actor can reasonably be expected to
             *   assume that the connector through which it's arriving
             *   connects back to the source location, so remember the
             *   association.  
             */
            if (backConnector != nil)
                backConnector.rememberTravel(dest, cur, origin);
        }

        /*
         *   Recalculate the global sense context for message generation
         *   purposes, since we've moved to a new location.  It's possible
         *   that we've arrived at the player character's location, in
         *   which case we might have been suppressing messages (due to
         *   our being in a different room) but should now show messages
         *   (because we're now near the player character).  
         */
        if (gAction != nil)
            gAction.recalcSenseContext();

        /* tell the new room we're arriving, if we changed locations */
        if (location != nil && isDescribed)
            location.travelerArriving(self, origin, connector, backConnector);
    }

    /*
     *   Send tracking information to each interested actor for a change
     *   in effective follow location. 
     */
    sendTrackFollowInfo(connector)
    {
        local tab;

        /* 
         *   get the connection table - this is the maximum set of objects
         *   that could have any sense connection to us 
         */
        tab = connectionTable();
        
        /* send the tracking information to each connected actor */
        tab.forEachAssoc(new function(obj, val)
        {
            /* if it's an Actor, notify it of the travel */
            if (obj.ofKind(Actor))
                obj.trackFollowInfo(self, connector);
        });
    }

    /*
     *   Get the list of actors taking part in the travel.  When an actor
     *   is the traveler, this list simply contains the actor itself; for
     *   a vehicle or other composite traveler that moves more than one
     *   actor at a time, this should return the list of all of the actors
     *   involved in the travel.  
     */
    getTravelerActors = []
;


/* ------------------------------------------------------------------------ */
/*
 *   A Travel Connector is a special connection interface that allows for
 *   travel from one location to another.  Most actor movement, except for
 *   movement between locations related by containment (such as from a room
 *   to sitting in a chair within the room) are handled through travel
 *   connector objects.
 *   
 *   Travel connectors are used in the directional link properties in rooms
 *   - north, south, east, west, in, out, etc.  A room direction link
 *   property is always set to a travel connector - but note that a room is
 *   itself a travel connector, so a travel link in one room can simply be
 *   set to point directly to another room.  In many cases, rooms
 *   themselves serve as travel connectors, so that one room can point a
 *   direction link property directly to another room.
 *   
 *   Some travel connectors are physical objects in the simulation, such as
 *   doors or stairways; other connectors are just abstract objects that
 *   represent connections, but don't appear as manipulable objects in the
 *   game.
 *   
 *   A travel connector provides several types of information about travel
 *   through its connection:
 *   
 *   - For actors actually traveling, the connector provides a method that
 *   moves an actor through the connector.  This method can trigger any
 *   side effects of the travel.  
 *   
 *   - For automatic map builders, actor scripts, and other callers who
 *   want to learn what can be known about the link without actually
 *   traversing it, the connector provides an "apparent destination"
 *   method.  This method returns the destination of travel through the
 *   connector that a given actor would expect just by looking at the
 *   connector.  The important thing about this routine is that it doesn't
 *   trigger any side effects, but simply indicates whether travel is
 *   apparently possible, and if so what the destination of the travel
 *   would be.  
 */
class TravelConnector: object
    /*
     *   Get any connector-specific pre-conditions for travel via this
     *   connector.  By default, we impose no special conditions for
     *   travel.  
     */
    connectorTravelPreCond = []

    /*
     *   Barrier or barriers to travel.  This property can be set to a
     *   single TravelBarrier object or to a list of TravelBarrier
     *   objects.  checkTravelBarriers() checks each barrier specified
     *   here.  
     */
    travelBarrier = []

    /*
     *   Check barriers.  The TravelVia check() routine must call this to
     *   enforce barriers.  
     */
    checkTravelBarriers()
    {
        local traveler;
        local lst;

        /* get the traveler */
        traveler = gActor.getTraveler();

        /* check any travel conditions we apply directly */
        if (!canTravelerPass(traveler))
        {
            /* explain why the traveler can't pass */
            explainTravelBarrier(traveler);

            /* terminate the command */
            exit;
        }

        /* get the barrier list */
        lst = travelBarrier;

        /* if it's just a single object, make it a list of one element */
        if (!lst.ofKind(Collection))
            lst = [lst];
        
        /* check each item in our barrier list */
        foreach (local cur in lst)
        {
            /* if this barrier doesn't allow travel, we cannot travel */
            if (!cur.canTravelerPass(traveler))
            {
                /* ask the barrier to explain why travel isn't possible */
                cur.explainTravelBarrier(traveler);

                /* terminate the command */
                exit;
            }
        }
    }

    /*
     *   Check to see if the Traveler object is allowed to travel through
     *   this connector.  Returns true if travel is allowed, nil if not.
     *   
     *   This is called from checkTravelBarriers() to check any conditions
     *   coded directly into the TravelConnector.  By default, we simply
     *   return true; subclasses can override this to apply special
     *   conditions.
     *   
     *   If an override wants to disallow travel, it should return nil
     *   here, and then provide an override for explainTravelBarrier() to
     *   provide a descriptive message explaining why the travel isn't
     *   allowed.
     *   
     *   Conditions here serve essentially the same purpose as barrier
     *   conditions.  The purpose of providing this additional place for
     *   the same type of conditions is simply to improve the convenience
     *   of defining travel conditions for cases where barriers are
     *   unnecessary.  The main benefit of using a barrier is that the same
     *   barrier object can be re-used with multiple connectors, so if the
     *   same set of travel conditions apply to several different
     *   connectors, barriers allow the logic to be defined once in a
     *   single barrier object and then re-used easily in each place it's
     *   needed.  However, when a particular condition is needed in only
     *   one place, creating a barrier to represent the condition is a bit
     *   verbose; in such cases, the condition can be placed in this method
     *   more conveniently.  
     */
    canTravelerPass(traveler) { return true; }

    /*
     *   Explain why canTravelerPass() returned nil.  This is called to
     *   display an explanation of why travel is not allowed by
     *   self.canTravelerPass().
     *   
     *   Since the default canTravelerPass() always allows travel, the
     *   default implementation of this method does nothing.  Whenever
     *   canTravelerPass() is overridden to return nil, this should also be
     *   overridden to provide an appropriate explanation.  
     */
    explainTravelBarrier(traveler) { }

    /*
     *   Determine if the travel connection is apparent - as a travel
     *   connector - to the actor in the given origin location.  This
     *   doesn't indicate whether or not travel is possible, or where
     *   travel goes, or that the actor can tell where the passage goes;
     *   this merely indicates whether or not the actor should realize
     *   that the passage exists at all.
     *   
     *   A closed door, for example, would return true, because even a
     *   closed door makes it clear that travel is possible in the
     *   direction, even if it's not possible currently.  A secret door,
     *   on the other hand, would return nil while closed, because it
     *   would not be apparent to the actor that the object is a door at
     *   all.  
     */
    isConnectorApparent(origin, actor)
    {
        /* by default, passages are apparent */
        return true;
    }

    /*
     *   Get the apparent destination of travel by the actor to the given
     *   origin.  This returns the location to which the connector
     *   travels, AS FAR AS THE ACTOR KNOWS.  If the actor does not know
     *   and cannot tell where the connector leads, this should return nil.
     *   
     *   Note that this method does NOT necessarily return the actual
     *   destination, because we obviously can't know the destination for
     *   certain until we traverse the connection.  Rather, the point of
     *   this routine is to return as much information as the actor is
     *   supposed to have.  This can be used for purposes like
     *   auto-mapping, where we'd want to show what the player character
     *   knows of the map, and NPC goal-seeking, where an NPC tries to
     *   figure out how to get from one point to another based on the
     *   NPC's knowledge of the map.  In these sorts of applications, it's
     *   important to use only knowledge that the actor is supposed to
     *   have within the parameters of the simulation.
     *   
     *   Callers should always test isConnectorApparent() before calling
     *   this routine.  This routine does not check to ensure that the
     *   connector is apparent, so it could return misleading information
     *   if used independently of isConnectorApparent(); for example, if
     *   the connector *formerly* worked but has now disappeared, and the
     *   actor has a memory of the former destination, we'll return the
     *   remembered destination.
     *   
     *   The actor can know the destination by a number of means:
     *   
     *   1.  The location is familiar to the character.  For example, if
     *   the setting is the character's own house, the character would
     *   obviously know the house well, so would know where you'd end up
     *   going east from the living room or south from the kitchen.  We
     *   use the origin method actorKnowsDestination() to determine this.
     *   
     *   2.  The destination is readily visible from the origin location,
     *   or is clearly marked.  For example, in an outdoor setting, it
     *   might be clear that going east from the field takes you to the
     *   hilltop.  In an indoor setting, an open passage might make it
     *   clear that going east from the living room takes you to the
     *   dining room.  We use the origin method actorKnowsDestination() to
     *   determine this.
     *   
     *   3. The actor has been through the connector already in the course
     *   of the game, and so remembers the connection by virtue of recent
     *   experience.  If our travelMemory class property is set to a
     *   non-nil lookup table object, then we'll automatically use the
     *   lookup table to remember the destination each time an actor
     *   travels via a connector, and use this information by default to
     *   provide apparent destination information.  
     */
    getApparentDestination(origin, actor)
    {
        local dest;
        
        /* 
         *   Ask the origin if the actor knows the destination for the
         *   given connector.  If so, and we can determine our
         *   destination, then return the destination.  
         */
        if (origin.actorKnowsDestination(actor, self)
            && (dest = getDestination(origin, actor)) != nil)
            return dest;

        /*
         *   If we have a travelMemory table, look to see if the traversal
         *   of this actor via this connector from this origin is recorded
         *   in the table, and if so, assume that the destination is the
         *   same as it was last time.
         *   
         *   Note that we ignore our memory of travel if we never saw the
         *   destination of the travel (which would be the case if the
         *   destination was dark every time we've been there, so we've
         *   never seen any details about the location).  
         */
        if (travelMemory != nil
            && (dest = travelMemory[[actor, origin, self]]) != nil
            && dest.seenBy(actor))
        {
            /* we know the destination from past experience */
            return dest;
        }

        /* we don't know the destination */
        return nil;
    }

    /*
     *   Get our destination, given the actor and the origin location.
     *   
     *   If it is not possible to determine the destination without
     *   actually traversing the connector, this can simply return nil.
     *   Note that a nil return does NOT indicate that travel is not
     *   possible - it simply means that the destination cannot be
     *   determined at present.
     *   
     *   This method should avoid side effects, because it can be called
     *   in the course of operations that don't actually involve travel
     *   (such as building a map or trying to find a path between two
     *   points).
     *   
     *   This method isn't required to return the same thing every time
     *   it's called, nor is it required to return the actual destination
     *   that the actor would reach by traversing the connector (since
     *   that might change by the time an actual traversal is attempted).
     *   Because of this, callers MUST NOT assume that the destination
     *   will remain stable
     *   
     *   This method is intended for internal use only.  Code outside this
     *   class and subclasses should use one of the higher-level routines,
     *   such as getApparentDestination(), for auto-mapping and the like.  
     */
    getDestination(origin, actor) { return nil; }

    /*
     *   Service routine: add a memory of a successful traversal of a
     *   travel connector.  If we have a travel memory table, we'll add
     *   the traversal to the table, so that we can find it later.
     *   
     *   This is called from Traveler.travelerTravelTo() on successful
     *   travel.  We're called for each actor participating in the travel.
     */
    rememberTravel(origin, actor, dest)
    {
        /* 
         *   If we have a travelMemory table, add this traversal.  Store
         *   the destination, keyed by the combination of the actor,
         *   origin, and connector object (i.e., self) - this will allow
         *   us to remember the destination we reached last time if we
         *   want to know where the same route goes in the future.  
         */
        if (TravelConnector.travelMemory != nil)
            TravelConnector.travelMemory[[actor, origin, self]] = dest;
    }

    /*
     *   Our "travel memory" table.  If this contains a non-nil lookup
     *   table object, we'll store a record of each successful traversal
     *   of a travel connector here - we'll record the destination keyed
     *   by the combination of actor, origin, and connector, so that we
     *   can later check to see if the actor has any memory of where a
     *   given connector goes from a given origin.
     *   
     *   We keep this information by default, which is why we statically
     *   create the table here.  Keeping this information does involve
     *   some overhead, so some authors might want to get rid of this
     *   table (by setting the property to nil) if the game doesn't make
     *   any use of the information.  Note that this table is stored just
     *   once, in the TravelConnector class itself - there's not a
     *   separate table per connector.  
     */
    travelMemory = static new LookupTable(256, 512)

    /*
     *   Is this a "circular" passage?  A circular passage is one that
     *   explicitly connects back to its origin, so that traveling through
     *   the connector leaves us where we started.  When a passage is
     *   marked as circular, we'll describe travel through the passage
     *   exactly as though we had actually gone somewhere.  By default, if
     *   traveling through a passage leaves us where we started, we assume
     *   that nothing happened, so we don't describe any travel.
     *   
     *   Circular passages don't often occur in ordinary settings; these
     *   are mostly useful in disorienting environments, such as twisty
     *   cave networks, where a passage between locations can change
     *   direction and even loop back on itself.  
     */
    isCircularPassage = nil

    /*
     *   Describe an actor's departure through the connector from the
     *   given origin to the given destination.  This description is from
     *   the point of view of another actor in the origin location.  
     */
    describeDeparture(origin, dest)
    {
        /* 
         *   By default, show the generic departure message.  Subclasses
         *   can override this to provide more specific descriptions of
         *   traversing the connector; for example, a stairway might want
         *   to override this to say that the actor climbed up or down the
         *   stairs.  
         */
        libMessages.sayDeparting(gActor.getTraveler());
    }

    /*
     *   Describe an actor's arrival through the connector from the given
     *   origin into the given destination.  This description is from the
     *   point of view of another actor in the destination.
     *   
     *   Note that this is called on the connector that reverses the
     *   travel, NOT on the connector the actor is actually traversing.
     *   So, if we have two sides to a door, and the actor traverses the
     *   first side, this will be called on the second side - the one that
     *   links the destination back to the origin.  
     */
    describeArrival(origin, dest)
    {
        /* 
         *   By default, show the generic arrival message.  Subclasses can
         *   override this to provide more specific descriptions of
         *   traversing the connector. 
         */
        libMessages.sayArriving(gActor.getTraveler());
    }

    /*
     *   Find a connector in the destination location that connects back
     *   as the source of travel from the given connector when traversed
     *   from the source location.  Returns nil if there is no such
     *   connector.  This must be called while the actor is still in the
     *   source location; we'll attempt to find the connector back to the
     *   actor's current location.
     *   
     *   The purpose of this routine is to identify the connector by which
     *   the actor arrives in the new location.  This can be used, for
     *   example, to generate a connector-specific message describing the
     *   actor's emergence from the connector (so we can say one thing if
     *   the actor arrives via a door, and another if the actor arrives by
     *   climing up a ladder).
     *   
     *   By default, we'll try to find a travel link in the destination
     *   that links us back to this same connector, in which case we'll
     *   return 'self' as the connector from which the actor emerges in
     *   the new location.  Failing that, we'll look for a travel link
     *   whose apparent source is the origin location.
     *   
     *   This should be overridden for any connector with an explicit
     *   complementary connector.  For example, it is common to implement
     *   a door using a pair of objects, one representing each side of the
     *   door; in such cases, each door object would simply return its
     *   twin here.  Note that a complementary connector doesn't actually
     *   have to go anywhere, since it's still useful to have a connector
     *   back simply for describing actors arriving on the connector.
     *   
     *   This *must* be overridden when the destination location doesn't
     *   have a simple connector whose apparent source is this connector,
     *   because in such cases we won't be able to find the reverse
     *   connector with our direction search.  
     */
    connectorBack(actor, dest)
    {
        local origin;

        /* if there's no destination, there's obviously no connector */
        if (dest == nil)
            return nil;

        /* 
         *   get the origin location - this is the immediate location of
         *   the object that's actually doing the traveling (which is the
         *   actor, unless the actor is in a vehicle, in which case it's
         *   the vehicle; in any case we can just ask the actor) 
         */
        origin = actor.getTraveler().location;
        
        /* 
         *   First, try to find a link back to this same connector - this
         *   will handle simple symmetrical links with the same connector
         *   object shared between two rooms.
         *   
         *   We try to find the actual connector before looking for a
         *   connector back to the origin - it's possible that there are
         *   several ways back to the starting point, and we want to make
         *   sure we pick the one that was actually traversed if possible.
         */
        foreach (local dir in Direction.allDirections)
        {
            /* 
             *   if this direction link from the destination is linked
             *   back to this same connector, we have a symmetrical
             *   connection, so we're the connector back 
             */
            if (dest.getTravelConnector(dir, actor) == self)
            {
                /* the same connector goes in both directions */
                return self;
            }
        }

        /*
         *   we didn't find a link back to the same connector, so try to
         *   find a link from the destination whose apparent source is the
         *   origin 
         */
        foreach (local dir in Direction.allDirections)
        {
            local conn;
            
            /* 
             *   if this link from the destination has an apparent source
             *   of our origin, the actor appears to be arriving from this
             *   link 
             */
            if ((conn = dest.getTravelConnector(dir, actor)) != nil
                && conn.fixedSource(dest, actor) == origin)
            {
                /* 
                 *   this direction has an apparent source of the origin
                 *   room - it's not necessarily the same link they
                 *   traversed, but at least it appears to come from the
                 *   same place they came from, so it'll have to do 
                 */
                return conn;
            }
        }

        /* we couldn't find any link back to the origin */
        return nil;
    }

    /*
     *   Get the "fixed" source for actors emerging from this connector,
     *   if possible.  This can return nil if the connector does not have
     *   a fixed relationship with another connector.
     *   
     *   The purpose of this routine is to find complementary connectors
     *   for simple static map connections.  This is especially useful for
     *   direct room-to-room connections.
     *   
     *   When a connector relationship other than a simple static mapping
     *   exists, the connectors must generally override connectorBack(),
     *   in which case this routine will not be needed (at least, this
     *   routine won't be needed as long as the overridden connectorBack()
     *   doesn't call it).  Whenever it is not clear how to implement this
     *   routine, don't - implement connectorBack() instead.  
     */
    fixedSource(dest, actor)
    {
        /* by default, return nothing */
        return nil;
    }

    /*
     *   Can the given actor see this connector in the dark, looking from
     *   the given origin?  Returns true if so, nil if not.
     *   
     *   This is used to determine if the actor can travel from the given
     *   origin via this connector when the actor (in the origin location)
     *   is in darkness.
     *   
     *   By default, we implement the usual convention, which is that
     *   travel from a dark room is possible only when the destination is
     *   lit.  If we can't determine our destination, we will assume that
     *   the connector is not visible.
     */
    isConnectorVisibleInDark(origin, actor)
    {
        local dest;
        
        /* 
         *   Get my destination - if we can't determine our destination,
         *   then assume we're not visible. 
         */
        if ((dest = getDestination(origin, actor)) == nil)
            return nil;

        /*
         *   Check the ambient illumination level in the destination.  If
         *   it's 2 or higher, then it's lit; otherwise, it's dark.  If
         *   the destination is lit, consider the connector to be visible,
         *   on the theory that the connector lets a little bit of the
         *   light from the destination leak into the origin room - just
         *   enough to make the connection itself visible without making
         *   anything else in the origin room visible.  
         */
        return (dest.wouldBeLitFor(actor));
    }

    /*
     *   Handle travel in the dark.  Specifically, this is called when an
     *   actor attempts travel from one dark location to another dark
     *   location.  (We don't invoke this in any other case:
     *   light-to-light, light-to-dark, and dark-to-light travel are all
     *   allowed without any special checks.)
     *   
     *   By default, we will prohibit dark-to-dark travel by calling the
     *   location's darkTravel handler.  Individual connectors can
     *   override this to allow such travel or apply different handling.  
     */
    darkTravel(actor, dest)
    {
        /* 
         *   by default, simply call the actor's location's darkTravel
         *   handler 
         */
        actor.location.darkTravel(actor);
    }

    /*
     *   Action handler for the internal "TravelVia" action.  This is not a
     *   real action, but is instead a pseudo-action that we implement
     *   generically for travel via the connector.  Subclasses that want to
     *   handle real actions by traveling via the connector can use
     *   asDobjFor(TravelVia) to implement the real action handlers.  
     */
    dobjFor(TravelVia)
    {
        preCond()
        {
            /* 
             *   For our preconditions, use the traveler's preconditions,
             *   plus the location's preconditions, plus any special
             *   connector-specific preconditions we supply. 
             */
            return gActor.getTraveler().travelerPreCond()
                + gActor.location.roomTravelPreCond()
                + connectorTravelPreCond();
        }
        verify()
        {
            /*
             *   Verify travel for the current command's actor through this
             *   connector.  This performs normal action verify processing.
             *   
             *   The main purpose of this routine is to allow the connector
             *   to flag obviously dangerous travel to allow a caller to
             *   avoid such travel as an implicit or scripted action.  In
             *   most cases, there's no need to make travel illogical
             *   because there's generally no potential ambiguity involved
             *   in analyzing a travel verb.
             *   
             *   Note that this routine must check with the actor to
             *   determine if the actor or a vehicle will actually be
             *   performing the travel, by calling gActor.getTraveler(), if
             *   the routine cares about the difference.  
             */
        }
        check()
        {
            local dest;

            /*
             *   Check the travel. 
             *   
             *   This routine should take into account the light levels at
             *   the source and destination locations, if travel between
             *   dark rooms is to be disallowed.  
             */
            
            /* get my destination */
            dest = getDestination(gActor.getTraveler().location, gActor);
            
            /* check dark-to-dark travel */
            gActor.checkDarkTravel(dest, self);

            /* enforce barriers */
            checkTravelBarriers();
        }
        
        action()
        {
            local dest;

            /*
             *   Execute the travel, moving the command's actor through the
             *   travel connection: we carry out any side effects of the
             *   travel and deliver the actor (if appropriate) to the
             *   destination of the connector.
             *   
             *   Note that this routine must check with the actor to
             *   determine if the actor or a vehicle will actually be
             *   performing the travel, by calling gActor.getTraveler(), if
             *   the routine cares about the difference.  In most cases,
             *   the routine won't care: most implementations of this
             *   routine will (if they effect any travel at all) eventually
             *   call gActor.travelTo() to carry out the travel, and that
             *   routine will always route the actual movement to the
             *   vehicle if necessary.  
             */
            
            /* get my destination */
            dest = getDestination(gActor.getTraveler().location, gActor);

            /* travel to my destination */
            gActor.travelTo(dest, self, connectorBack(gActor, dest));            
        }
    }
;

/*
 *   A "probe" object, for testing light levels in rooms.  This is a dummy
 *   object that we use for what-if testing - it's not actually part of
 *   the simulation.  
 */
lightProbe: Thing;

/*
 *   A TravelBarrier can be attached to a TravelConnector, via the
 *   travelBarrier property, to form a conditional barrier to travel.
 */
class TravelBarrier: object
    /*
     *   Determine if this barrier blocks the given traveler.  By default,
     *   we don't block anyone.  This doesn't make us much of a barrier,
     *   so subclasses should override this with a more specific condition.
     */
    canTravelerPass(traveler) { return nil; }

    /*
     *   Explain why travel isn't allowed.  This should generate an
     *   appropriate failure report explaining the problem.  This is
     *   invoked when travel is attempted and canTravelerPass returns nil.
     *   Subclasses must override this.  
     */
    explainTravelBarrier(traveler) { }
;


/*
 *   A travel connector that doesn't allow any travel - if travel is
 *   attempted, we simply use the origin's cannotTravel method to display
 *   an appropriate message.  
 */
noTravel: TravelConnector
    isConnectorApparent(origin, actor)
    {
        /* it is obvious that is no passage this way */
        return nil;
    }

    dobjFor(TravelVia)
    {
        /* 
         *   we know that no travel will occur, so we don't need to satisfy
         *   any preconditions 
         */
        preCond = []

        /* likewise, no checks are necessary */
        check() { }

        action()
        {
            /* we can't go this way - use the origin's default message */
            gActor.location.cannotTravel();
        }
    }
;

/*
 *   A default travel connector for going in.  When travel in the relative
 *   direction "in" isn't allowed, we'll try recasting the command as an
 *   "enter" command with an unknown direct object. 
 */
noTravelIn: TravelConnector
    dobjFor(TravelVia)
    {
        /* 
         *   no preconditions or checks are needed, as we'll try to recast
         *   the command into an entirely new action
         */
        preCond = []
        check() { }
        action() { askForDobj(Enter); }
    }

    /* 
     *   since this is a default connector for all locations, indicate that
     *   no travel is apparently possible in this direction 
     */
    isConnectorApparent(origin, actor) { return nil; }
;

/*
 *   A default travel connector for going out.  When travel in the
 *   relative direction "out" isn't allowed, we'll try recasting the
 *   command as an "get out of" command with an unknown direct object. 
 */
noTravelOut: TravelConnector
    dobjFor(TravelVia)
    {
        preCond = []
        check() { }
        action() { askForDobj(GetOutOf); }
    }

    /* indicate that no travel is apparently possible */
    isConnectorApparent(origin, actor) { return nil; }
;

/*
 *   A special travel connector for 'down' that recasts the command as a
 *   "get off of" command.  This can be used for platforms and the like,
 *   where a 'down' command should usually be taken to mean "get off
 *   platform" rather than "down from enclosing room". 
 */
noTravelDown: TravelConnector
    dobjFor(TravelVia)
    {
        preCond = []
        check() { }
        action() { askForDobj(GetOffOf); }
    }

    /* indicate that no travel is apparently possible */
    isConnectorApparent(origin, actor) { return nil; }
;

/*
 *   A travel connector for going in that explicitly redirects the command to
 *   "enter" with an unknown direct object.  This behaves the same way as
 *   noTravelIn, but explicitly makes the inward travel apparent; this can be
 *   used to override the noTravelIn default for locations where travel in is
 *   explicitly allowed. 
 */
redirectTravelIn: noTravelIn
    isConnectorApparent(origin, actor) { return true; }
;

/* explicitly redirect travel out to "get out of" */
redirectTravelOut: noTravelOut
    isConnectorApparent(origin, actor) { return true; }
;

/* explicitly redirect travel down to "get off of" */
redirectTravelDown: noTravelDown
    isConnectorApparent(origin, actor) { return true; }
;

/*
 *   "No Ship" travel connector.  This is used as the default connector
 *   for the shipboard directions for the base Room class.  This connector
 *   displays a special message indicating that the room is not a boat.  
 */
noShipTravel: noTravel
    dobjFor(TravelVia)
    {
        action()
        {
            /* simply indicate that this direction isn't applicable here */
            libMessages.notOnboardShip();
        }
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   A simple connector that displays a message when the connector is
 *   traversed.
 */
class TravelMessage: TravelConnector
    /*
     *   My message to display when the player character traverses the
     *   connector.  This should be overridden with the custom message for
     *   the connector.  
     */
    travelDesc = ""

    /*
     *   My message to display when any non-player character traverses the
     *   connector.  If this is not overridden, no message will be
     *   displayed when an NPC travels through the connector.  
     */
    npcTravelDesc = ""

    /*
     *   Display my message.  By default, we show one message for the
     *   player character and another message for NPC's.  
     */
    showTravelDesc()
    {
        if (gActor.isPlayerChar())
            travelDesc;
        else
            npcTravelDesc;
    }

    /* my destination location */
    destination = nil

    /* get my destination */
    getDestination(origin, actor) { return destination; }

    /* our source is the same as our destination */
    fixedSource(dest, actor) { return destination; }

    /* carry out travel via this connector */
    dobjFor(TravelVia)
    {
        action()
        {
            /* display my message */
            showTravelDesc();

            /* inherit the default handling */
            inherited();
        }
    }
;

/*
 *   A travel connector that can't be traversed, and which shows a custom
 *   failure message when traversal is attempted.  Instances should define
 *   travelDesc to the message to display when travel is attempted.
 *   
 *   Travel is not apparently possible in this direction, so this type of
 *   connector will not show up in automatic exit lists or maps.  This
 *   class is designed for connections that are essentially the same as no
 *   connection at all, but where it's desirable to use a special message
 *   to describe why travel can't be accomplished.  
 */
class NoTravelMessage: TravelMessage
    dobjFor(TravelVia)
    {
        /* as in noTravel, we need no preconditions or checks */
        preCond = []
        check() { }
        action()
        {
            /* simply show my message - that's all we do */
            showTravelDesc();
        }
    }

    /* 
     *   Because no travel is possible, we want a non-empty message for
     *   NPC's as well as for the PC; by default, use the same message for
     *   all actors by using travelDesc for NPC's.  
     */
    npcTravelDesc = (travelDesc)

    /* travel is not apparently possible in this direction */
    isConnectorApparent(origin, actor) { return nil; }
;

/*
 *   A "fake" connector.  This is a connector that doesn't actually allow
 *   travel, but acts like it *could*.  We simply show a special message
 *   when travel is attempted, but we don't move the actor.
 *   
 *   This is a subclass of NoTravelMessage, so instances should customize
 *   travelDes with the special failure message when travel is attempted.
 *   
 *   That this type of connector by default is *apparently* an exit, even
 *   though it doesn't actually go anywhere.  This is useful for "soft
 *   boundaries," where the game's map is meant to appear to the player to
 *   continue but beyond which no more locations actually exist in the map.
 *   This is an oft-used device meant to create an illusion that the game's
 *   map exists in a larger world even though the larger world is not
 *   modeled.  The message we display should in such cases attribute the
 *   actor's inability to traverse the connector to a suitable constraint
 *   within the context of the game world; for example, the actor could be
 *   said to be unwilling to go beyond this point because the actor knows
 *   or suspects there's nothing for a long way in this direction, or
 *   because the actor's goals require staying within the modeled map, or
 *   because the actor is afraid of what lies beyond.  
 */
class FakeConnector: NoTravelMessage
    /* 
     *   travel is *apparently* possible in this direction (even though
     *   it's not *actually* possible) 
     */
    isConnectorApparent(origin, actor) { return true; }
;

/* ------------------------------------------------------------------------ */
/*
 *   A direct room-to-room connector.  In most cases, it is not necessary
 *   to create one of these objects, because rooms can serve as their own
 *   connectors.  These objects are needed in certain cases, though, such
 *   as when a room-to-room connection requires a travel barrier.  
 */
class RoomConnector: TravelConnector
    /* the two rooms we link */
    room1 = nil
    room2 = nil

    /* 
     *   get the destination, given the origin: this is the one of the two
     *   rooms we link that we're not in now 
     */
    getDestination(origin, actor)
    {
        if (origin == room1)
            return room2;
        else if (origin == room2)
            return room1;
        else
            return nil;
    }

    fixedSource(origin, actor)
    {
        /* 
         *   we're a symmetrical two-way connector, so our source from the
         *   perspective of one room is the same as the destination from
         *   that same perspective 
         */
        return getDestination(origin, actor);
    }

    describeDeparture(origin, dest)
    {
        local dir;
        
        /* 
         *   See if we can find the direction of the link from the origin
         *   to the destination.  If so, describe the departure using the
         *   direction; otherwise, describe it using the generic departure
         *   message.  Note that we are the connector in this context.  
         */
        if ((dir = origin.directionForConnector(self, gActor)) != nil)
        {
            /*
             *   We found a direction linked to the connector, so this
             *   must have been the way they traveled.  Describe the
             *   departure as being in that direction.
             *   
             *   Note that it's possible that more than one direction is
             *   linked to the same connector.  In such cases, we'll just
             *   take the first link we find, because it's equally
             *   accurate to say that the actor went in any of the
             *   directions linked to the same connector.  
             */
            dir.sayDeparting(gActor.getTraveler());
        }
        else
        {
            /*
             *   We didn't find any direction out of the origin linking to
             *   this connector, so we don't know how what direction they
             *   went.  Show the generic departure message.  
             */
            libMessages.sayDeparting(gActor.getTraveler());
        }
    }

    describeArrival(origin, dest)
    {
        local dir;
        
        /*
         *   See if we can find the direction of the link back to the
         *   origin from our destination.  If so, describe the arrival
         *   using the direction; otherwise, describe it using a generic
         *   arrival message.  Note that we are the back-connector in this
         *   context.  
         */
        if ((dir = dest.directionForConnector(self, gActor)) != nil)
        {
            /* 
             *   we found a direction linked to this back connector, so
             *   describe the arrival as coming from the direction we
             *   found 
             */
            dir.sayArriving(gActor.getTraveler());
        }
        else
        {
            /* 
             *   we didn't find any direction links, so use a generic
             *   arrival message 
             */
            libMessages.sayArriving(gActor.getTraveler());
        }
    }
;

/*
 *   A one-way room connector.  This works like an ordinary room
 *   connector, but connects only in one direction.  To use this class,
 *   simply define the 'destination' property to point to the room we
 *   connect to.  
 */
class OneWayRoomConnector: RoomConnector
    /* my destination - instances must define this */
    destination = nil

    /* we always have a fixed destination */
    getDestination(origin, actor) { return destination; }
;


/* ------------------------------------------------------------------------ */
/*
 *   Base class for passages between rooms.  This can be used for a
 *   passage that not only connects the rooms but also exists as an object
 *   in its own right within the rooms it connects.
 *   
 *   In most cases, two passage objects will exist - one on each side of
 *   the passage, so one object in each room connected.  One of the
 *   objects should be designated as the "master"; the other is the
 *   "slave."  The master object is the one which should implement all
 *   special behavior involving state changes (such as opening or closing
 *   a door).
 *   
 *   This basic passage is not designed to be opened and closed; use Door
 *   for a passage that can be opened and closed.  
 */
class Passage: Fixed, TravelConnector
    /*
     *   Our "master object."  The slave object should set this to point
     *   to the other side.  
     */
    masterObject() { return self; }

    /*
     *   Our destination - this is where the actor ends up when traveling
     *   through the passage (assuming the passage is open).  By default, we
     *   return the location of our other side; in cases where our other
     *   side is not directly in our destination room (for example, the
     *   other side is part of some larger object structure), this
     *   property should be overridden to specify the actual destination.
     *   
     *   If our otherSide is nil, the passage doesn't go anywhere.  This
     *   can be useful to create things that look and act like passages,
     *   but which don't actually go anywhere - in other words,
     *   passage-like decorations.  
     */
    destination = (otherSide != nil ? otherSide.location : nil)

    /* get my destination - just return my 'destination' property */
    getDestination(origin, actor) { return destination; }

    /* assume we're open */
    isOpen = true

    /*
     *   Initialize.  If we're a slave, we'll set up the otherSide
     *   relationship between this passage and our master passage.  
     */
    initializeThing()
    {
        /* inherit default handling */
        inherited();

        /* 
         *   if we have a master side, initialize our relationship with
         *   the master side
         */
        if (masterObject != self)
        {
            /* set our otherSide to point to the master */
            otherSide = masterObject;

            /* set the master's otherSide to point to us */
            masterObject.otherSide = self;
        }
    }

    /*
     *   Our corresponding passage object on the other side of the
     *   passage.  This will be set automatically during initialization
     *   based on the masterObject property of the slave - it is not
     *   generally necessary to set this manually.  
     */
    otherSide = nil

    /*
     *   Display our message when we don't allow the actor to travel
     *   through the passage because the passage is closed.  By default,
     *   we'll simply display the default cannotTravel message for the
     *   actor's location, but this can be overridden to provide a more
     *   specific report of the problem.  
     */
    cannotTravel()
    {
        /* use the actor's location's cannotTravel handling */
        gActor.location.cannotTravel();
    }

    fixedSource(origin, actor)
    {
        /* 
         *   our source is always our destination, since we have a
         *   one-to-one relationship with our comlementary passage in the
         *   destination location (we point to it, it points to us) 
         */
        return destination;
    }

    connectorBack(actor, dest)
    {
        /* if we have a complementary side, it's the connector back */
        if (otherSide != nil)
            return otherSide;

        /* we don't have a complementary side, so use the default handling */
        return inherited(actor, dest);
    }

    /* carry out travel via this connector */
    dobjFor(TravelVia)
    {
        /* check travel */
        check()
        {
            /* 
             *   Move the actor only if we're open; if we're not, use the
             *   standard no-travel handling for the actor's location.
             *   Note that we don't try to implicitly open the passage,
             *   because the basic passage is not openable; if we're
             *   closed, it means we're impassable for some reason that
             *   presumably cannot be remedied by a simple "open" command.
             */
            if (!masterObject.isOpen)
            {
                /* we're closed, so use our no-travel handling */
                cannotTravel();
                exit;
            }
            else
            {
                /* inherit the default checks */
                inherited();
            }
        }
    }
;

/*
 *   A passage that an actor can travel through (with a "go through" or
 *   "enter" command).  A "go through" command applied to the passage
 *   simply makes the actor travel through the passage as though using the
 *   appropriate directional command.
 *   
 *   We describe actors arriving or departing via the passage using
 *   "through" descriptions ("Bob arrives through the door," etc).  
 */
class ThroughPassage: Passage
    describeDeparture(origin, dest)
    {
        /* describe the actor traveling through this passage */
        libMessages.sayDepartingByPassage(gActor.getTraveler(), self);
    }

    describeArrival(origin, dest)
    {
        /* describe the actor arriving through this passage */
        libMessages.sayArrivingByPassage(gActor.getTraveler(), self);
    }

    /*
     *   Treat "go through self" as travel through the passage 
     */
    dobjFor(GoThrough) asDobjFor(TravelVia)

    /* "enter" is the same as "go through" for this type of passage */
    dobjFor(Enter) asDobjFor(GoThrough)

    /* 
     *   Explicitly map the indirect object handlers for the push-travel
     *   commands corresponding to the regular travel commands we provide.
     *   This isn't strictly necessary except when we're mixed in to
     *   something like a Decoration, in which case explicitly defining
     *   the mapping here is important because it will give the mapping
     *   higher precedence than a catch-all handlers overriding the same
     *   mapping we inherit from Thing.  
     */
    iobjFor(PushTravelThrough) asDobjWithoutActionFor(GoThrough)
    iobjFor(PushTravelEnter) asDobjWithoutActionFor(Enter)
;

/* ------------------------------------------------------------------------ */
/*
 *   The exit portal of a one-way passage.  This isn't a fully functional
 *   passage, but rather an object that acts as the receiving end of a
 *   passage that can only be traversed in one direction.
 *   
 *   This can be used for various purposes: the underside of a trap door,
 *   the bottom of a chute, the exit of a wormhole.  
 */
class ExitOnlyPassage: ThroughPassage
    dobjFor(TravelVia)
    {
        action()
        {
            /* 
             *   Show our default failure message.  This can be overridden
             *   to provide a more customized description of why the
             *   passage cannot be entered from this side.  
             */
            reportFailure(&cannotEnterExitOnly, self);
        }
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Stairway - this is a special kind of passage that is used for
 *   vertical connections, such as stairways and ladders.  This type of
 *   passage doesn't allow "enter" or "go through," but does allow "climb".
 *   
 *   The basic Stairway should generally not be used, as it doesn't know
 *   where it is relative to connected stairs.  Instead, the more specific
 *   StairwayUp and StairwayDown should be used in most cases.
 *   
 *   Note that at midpoints along long stairways (landings, for example),
 *   separate stairway objects should normally be used: one going up and
 *   one going down.  
 */
class Stairway: Passage
    /*
     *   Treat "climb self" as travel through the passage 
     */
    dobjFor(Climb) asDobjFor(TravelVia)
;

/*
 *   A stairway going up from here. 
 */
class StairwayUp: Stairway
    describeArrival(origin, dest)
    {
        /* 
         *   describe the actor arriving by coming down these stairs (they
         *   leave up, so they arrive down) 
         */
        libMessages.sayArrivingDownStairs(gActor.getTraveler(), self);
    }

    describeDeparture(origin, dest)
    {
        /* describe the actor leaving up these stairs */
        libMessages.sayDepartingUpStairs(gActor.getTraveler(), self);
    }

    /* "climb up" is the specific direction for "climb" here */
    dobjFor(ClimbUp) asDobjFor(Climb)

    /* cannot climb down from here */
    dobjFor(ClimbDown)
    {
        verify() { illogical(&stairwayNotDown); }
    }
;

/*
 *   A stairway going down from here. 
 */
class StairwayDown: Stairway
    /* "climb down" is the specific direction for "climb" here */
    dobjFor(ClimbDown) asDobjFor(Climb)

    describeArrival(origin, dest)
    {
        /* 
         *   describe the actor arriving by coming up these stairs (they
         *   leave down, so they arrive up) 
         */
        libMessages.sayArrivingUpStairs(gActor.getTraveler(), self);
    }

    describeDeparture(origin, dest)
    {
        /* describe the actor traveling down these stairs */
        libMessages.sayDepartingDownStairs(gActor.getTraveler(), self);
    }

    /* cannot climb up from here */
    dobjFor(ClimbUp)
    {
        verify() { illogical(&stairwayNotUp); }
    }
;


/* ------------------------------------------------------------------------ */
/*
 *   Door.  This is a travel connector that can be opened or closed, and
 *   which has two sides.  Each side of the door is associated with the
 *   other, so that opening or closing one side makes the same change to
 *   the other side.  
 */
class Door: Openable, ThroughPassage
    /* make us initially closed */
    initiallyOpen = nil

    /* in case we're a locked door, make us initially locked */
    initiallyLocked = true

    /* 
     *   Open/close the door.  If we have a complementary door object
     *   representing the other side, we'll remark in the sensory context
     *   of its location that it's also opening or closing. 
     */
    makeOpen(stat)
    {
        /* inherit the default behavior */
        inherited(stat);

        /* 
         *   if our new status is in effect, notify the other side so that
         *   it can generate a message in its location 
         */
        if (otherSide != nil && stat == masterObject.isOpen)
            otherSide.noteRemoteOpen(stat);
    }

    /*
     *   Note a "remote" change to our open/close status.  This is an
     *   open/close operation performed on our complementary object
     *   representing the other side of the door.  We'll remark on the
     *   change in the sensory context of this side, but only if we're
     *   suppressing output in the current context - if we're not, then
     *   the player will see the message generated by the side that we
     *   directly acted upon, so we don't need a separate report for the
     *   other side.  
     */
    noteRemoteOpen(stat)
    {
        /* 
         *   If I'm not visible to the player character in the current
         *   sense context, where the action is actually taking place,
         *   switch to my own sensory context and display a report.  This
         *   way, if the player can see this door but not the other side,
         *   and the action is taking place on the other side, we'll still
         *   see a note about the change.  We only need to do this if
         *   we're not already visible to the player, because if we are,
         *   we'll generate an ordinary report of the door opening in the
         *   action's context.  
         */
        if (senseContext.isBlocking)
        {
            /* show a message in my own sensory context */
            callWithSenseContext(self, sight,
                {: libMessages.sayOpenDoorRemotely(self, stat) });
        }
    }

    /* the door must be open before we can travel this way */
    connectorTravelPreCond()
    {
        return inherited() + [new ObjectPreCondition(self, doorOpen)];
    }

    /* carry out travel through the door */
    dobjFor(TravelVia)
    {
        action()
        {
            /* move the actor to our destination */
            inherited();

            /* remember that this is the last door the actor traversed */
            gActor.rememberLastDoor(self);
        }
    }

    /*
     *   When attempting to "go through" me, we must be open as a
     *   condition of the travel 
     */
    dobjFor(GoThrough)
    {
        preCond = [objOpen]
    }

    dobjFor(Close)
    {
        verify()
        {
            /* inherit default handling */
            inherited();

            /* boost the likelihood if they just traveled through us */
            boostLikelihoodOnTravel();
        }
    }

    dobjFor(Lock)
    {
        verify()
        {
            /* inherit default handling */
            inherited();

            /* boost the likelihood if they just traveled through us */
            boostLikelihoodOnTravel();
        }
    }

    /*
     *   Boost the likelihood that a command is referring to us if we just
     *   traveled through the door; this is meant to be called from
     *   verify() routines.  For a few commands (close, lock), the most
     *   likely door being referred to is the door just traversed.  
     */
    boostLikelihoodOnTravel()
    {
        /* 
         *   if this (or our other side) is the last door the actor
         *   traversed, boost the likelihood that they're referring to us 
         */
        if (gActor.lastDoorTraversed == self
            || (otherSide != nil && gActor.lastDoorTraversed == otherSide))
            logicalRank(120, 'last door traversed');
    }

;

/* ------------------------------------------------------------------------ */
/*
 *   Secret Door.  This is a special kind of door that gives no hint of
 *   its being a door when it's closed.  This can be used for objects that
 *   serve as a secret passage but are otherwise normal objects, such as a
 *   bookcase that conceals a passage.  
 */
class SecretDoor: ThroughPassage
    /* a secret passage usually starts off secret (i.e., closed) */
    isOpen = nil

    isConnectorApparent(origin, actor)
    {
        /* 
         *   A secret passage is not apparent as a pasage unless it's
         *   open. 
         */
        if (masterObject.isOpen)
        {
            /* the passage is open - use the inherited handling */
            return inherited(origin, actor);
        }
        else
        {
            /* 
             *   the passage is closed, so it's not apparently a passage
             *   at all 
             */
            return nil;
        }
    }
;

/*
 *   Hidden Door.  This is a secret door that is invisible when closed.
 *   This can be used when the passage isn't even visible when it's
 *   closed, such as a door that seamlessly melds into the wall. 
 */
class HiddenDoor: SecretDoor
    /*
     *   When we're closed, we're completely invisible, so we have no
     *   sight presence.  When we're open, we have our normal visual
     *   presence.  
     */
    sightPresence = (masterObject.isOpen)
;

/* ------------------------------------------------------------------------ */
/*
 *   Automatically closing door.  This type of door closes behind an actor
 *   as the actor traverses the connector. 
 */
class AutoClosingDoor: Door
    dobjFor(TravelVia)
    {
        action()
        {
            /* inherit the default handling */
            inherited();

            /* close the door, and remark on its closing */
            makeOpen(nil);
            mainReport(&doorClosesBehind, self);
        }
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   An Enterable is an object that exists in one location, and which can
 *   be entered to take an actor to another location.  Enterables are used
 *   for things such as the outsides of buildings, so that the building
 *   can have a presence on its outside and can be entered via a command
 *   like "go into building".
 *   
 *   An Enterable isn't a connector, but points to a connector.  This type
 *   of object is most useful when there's already a connector that exists
 *   as a separate object, such as the door to a house: the house object
 *   can be made an Enterable that simply points to the door object.  
 */
class Enterable: Fixed
    /* the travel connector used to enter the object */
    connector = nil

    /*
     *   "Enter" action - this simply causes the actor to travel via the
     *   connector.  
     */
    dobjFor(Enter) remapTo(TravelVia, connector)

    /* 
     *   the internal "TravelVia" action is equivalent to a "TravelVia" on
     *   our connector 
     */
    dobjFor(TravelVia) remapTo(TravelVia, connector)

    /* 
     *   define the push-travel indirect object mappings for our travel
     *   actions 
     */
    iobjFor(PushTravelEnter) asDobjWithoutActionFor(Enter)
;


/* ------------------------------------------------------------------------ */
/*
 *   A "TravelPushable" is an object that can't be taken, but can be moved
 *   from one location to another via commands of the form "push obj dir,"
 *   "push obj through passage," and the like.
 *   
 *   TravelPushables tend to be rather rare, and we expect that instances
 *   will almost always be special cases that require additional
 *   specialized code.  This is therefore only a general framework for
 *   pushability.  
 */
class TravelPushable: Fixed
    cannotTakeMsg = &cannotTakePushable
    cannotMoveMsg = &cannotMovePushable
    cannotPutMsg = &cannotPutPushable

    /*
     *   Move the object to a new location as part of a push-travel
     *   operation.  By default, this simply uses moveInto to move the
     *   object, but subclasses can override this to apply conditions to
     *   the pushing, put the object somewhere else in some cases, display
     *   extra messages, or do anything else that's necessary.
     *   
     *   Our special PushTraveler calls this routine after the underlying
     *   real traveler has finished its travel to the new location, so the
     *   traveler's location will indicate the destination of the travel.
     *   Note that this routine is never called if the traveler ends up in
     *   its original location after the travel, so this routine isn't
     *   called when travel isn't allowed for the underlying traveler.  
     */
    movePushable(traveler)
    {
        /* move me to the traveler's new location */
        moveInto(traveler.location);

        /* describe what we're doing */
        describeMovePushable(traveler);
    }

    /*
     *   Describe the actor pushing the object into the new location.
     *   This is called from movePushable; we pull this out as a separate
     *   routine so that the description of the pushing can be overridden
     *   without having to override all of movePushable.  
     */
    describeMovePushable(traveler)
    {
        /* 
         *   If the actor is the player character, mention that we're
         *   pushing the object with us.  For an NPC, show nothing,
         *   because the normal travel message will mention that the NPC
         *   is pushing the object as part of the normal travel report
         *   (thanks to PushTraveler's name override). 
         */
        if (gActor.isPlayerChar)
            mainReport(&okayPushTravel, self);
    }

    dobjFor(PushTravel)
    {
        verify() { }
        action()
        {
            local newTrav;
            local oldTrav;

            /*
             *   Create a push traveler to coordinate the travel.  The
             *   push traveler performs the normal travel, and also moves
             *   the object being pushed.  
             */
            newTrav = pushTravelerClass.createInstance(
                self, gActor.getTraveler(), gActor.specialTraveler);
            
            /* set the actor's special traveler to our push traveler */
            oldTrav = gActor.setSpecialTraveler(newTrav);
            try
            {
                /* 
                 *   Now that we've activated our special push traveler
                 *   for the actor, simply perform the ordinary travel
                 *   command.  Use the special internal "travel via
                 *   connector" action, since the connector is the only
                 *   aspect of the travel that is common to all kinds of
                 *   room-to-room travel.  
                 */
                nestedAction(TravelVia, gAction.getConnector());
            }
            finally
            {
                /* restore the actor's old special traveler on the way out */
                gActor.setSpecialTraveler(oldTrav);
            }
        }
    }

    /* 
     *   The class we create for our special push traveler - by default,
     *   this is PushTraveler, but we parameterize this via this property
     *   to allow special PushTraveler subclasses to be created; this
     *   could be useful, for example, to customize the traveler name
     *   messages. 
     */
    pushTravelerClass = PushTraveler
;

/*
 *   A special Traveler class for travel involving pushing an object from
 *   one room to another.  This class encapsulates the object being pushed
 *   and the actual Traveler performing the travel.
 *   
 *   For the most part, we refer Traveler methods to the underlying
 *   Traveler.  We override a few methods to provide special handling.  
 */
class PushTraveler: object
    construct(obj, traveler, oldSpecial)
    {
        /* remember the object being pushed and the real traveler */
        obj_ = obj;
        traveler_ = traveler;
    }

    /* the object being pushed */
    obj_ = nil

    /* 
     *   the underlying Traveler - this is the real Traveler that will
     *   move to a new location 
     */
    traveler_ = nil

    /*
     *   Travel to a new location.  We'll run the normal travel routine
     *   for the underlying real traveler; then, if we ended up in a new
     *   location, we'll move the object being pushed to the traveler's
     *   new location.  
     */
    travelerTravelTo(dest, connector, backConnector)
    {
        local origin;
        
        /* remember the traveler's origin, so we can tell if we moved */
        origin = traveler_.location;

        /* 
         *   Call the traveler's normal travel routine, to perform the
         *   actual movement of the underlying traveler.  
         */
        traveler_.travelerTravelTo(dest, connector, backConnector);

        /* 
         *   if we moved to a new location, tell the object being pushed
         *   to move there as well 
         */
        if (traveler_.location != origin)
            obj_.movePushable(traveler_);
    }

    /* by default, send everything to the underlying Traveler */
    propNotDefined(prop, [args]) { return traveler_.(prop)(args...); }
;

/*
 *   A PushTravelBarrier is a TravelConnector that allows regular travel,
 *   but not travel that involves pushing something.  By default, we block
 *   all push travel, but subclasses can customize this so that we block
 *   only specific objects.  
 */
class PushTravelBarrier: TravelBarrier
    /*
     *   Determine if the given pushed object is allowed to pass.  Returns
     *   true if so, nil if not.  By default, we'll return nil for every
     *   object; subclasses can override this to allow some objects to be
     *   pushed through the barrier but not others. 
     */
    canPushedObjectPass(obj) { return nil; }

    /* explain why an object can't pass */
    explainTravelBarrier(traveler)
    {
        reportFailure(&cannotPushObjectThatWay, traveler.obj_);
    }

    /*
     *   Determine if the given traveler can pass through this connector.
     *   If the traveler isn't a push traveler, we'll allow the travel;
     *   otherwise, we'll block the travel if our canPushedObjectPass
     *   routine says the object being pushed can pass. 
     */
    canTravelerPass(traveler)
    {
        /* if it's not a push traveler, it can pass */
        if (!traveler.ofKind(PushTraveler))
            return true;

        /* it can pass if we can pass the object being pushed */
        return canPushedObjectPass(traveler.obj_);
    }
;


/* ------------------------------------------------------------------------ */
/*
 *   A basic location - this is the base class for locations that can
 *   contain actors.
 */
class BasicLocation: Thing
    /*
     *   Get the listing group for an actor in this room in the given
     *   posture.  By default, we'll return nil, since we don't normally
     *   group actors in the room.  This is used to list actors within a
     *   nested room as part of the "examine" description for the nested
     *   room object; for example, it might group actors who are sitting
     *   in a chair.  
     */
    listWithActorIn(posture) { return nil; }

    /*
     *   Check the ambient illumination level in the room for the given
     *   actor's senses to determine if the actor would be able to see if
     *   the actor were in this room without any additional light sources.
     *   Returns true if the room is lit for the purposes of the actor's
     *   visual senses, nil if not.
     *   
     *   Note that if the actor is already in this location, there's no
     *   need to run this test, since we could get the information from
     *   the actor directly.  The point of this test is to determine the
     *   light level in this location without the actor having to be
     *   present.  
     */
    wouldBeLitFor(actor)
    {
        /*
         *   Check for a simple, common case before running the more
         *   expensive full what-if test.  Most rooms provide their own
         *   illumination in the actor's 'sight' sense; if this room does
         *   provide its own interior 'brightness' of at least 2, and the
         *   actor has 'sight' among its visual senses, then the room is
         *   lit.  Note that we must use transSensingOut() to determine
         *   how we transmit our own brightness to our interior, since
         *   that gives the transparency for looking from within this room
         *   at objects outside the room; our own intrinsic brightness is
         *   defined as the brightness of our exterior surface.  
         */
        if (actor.sightlikeSenses.indexOf(sight) != nil
            && brightness > 1
            && transSensingOut(sight) == transparent)
            return true;

        /*
         *   We can't determine for sure that the location is lit with a
         *   simple test, so run a full what-if test, checking what
         *   ambient light level our "probe" object would see if it were
         *   moved to this location.  Return true if the ambient light
         *   level is higher than the "self-lit" value of 1, nil if not.  
         */
        return (lightProbe.whatIf(
            {: senseAmbientMax(actor.sightlikeSenses) },
            &moveInfo, self) > 1);
    }

    /* 
     *   The location's first description.  This is the description we'll
     *   display the first time we see the room when it isn't dark (it's
     *   the first time if seen is nil).  By default, we'll just display
     *   the normal long description, but a room can override this if it
     *   wants a special description on the first non-dark arrival.  
     */
    firstDesc { desc; }

    /*
     *   Show a list of exits from this room as part of failed travel
     *   ("you can't go that way"). 
     */
    cannotGoShowExits(actor)
    {
        /* if we have an exit lister, ask it to show exits */
        if (gExitLister != nil)
            gExitLister.cannotGoShowExits(actor, self);
    }

    /* show the exit list in the status line */
    showStatuslineExits()
    {
        /* if we have a global exit lister, ask it to show the exits */
        if (gExitLister != nil)
            gExitLister.showStatuslineExits();
    }

    /* 
     *   Get the estimated height, in lines of text, of the exits display's
     *   contribution to the status line.  This is used to calculate the
     *   extra height we need in the status line, if any, to display the
     *   exit list.  If we're not configured to display exits in the status
     *   line, this should return zero. 
     */
    getStatuslineExitsHeight()
    {
        if (gExitLister != nil)
            return gExitLister.getStatuslineExitsHeight();
        else
            return 0;
    }

    /*
     *   Show the room's basic description for the given actor and
     *   illumination status.  This displays the main room-specific
     *   portion of the room's description; it does not display the status
     *   line name or anything about the dynamic contents of the room.  
     */
    lookAroundWithinDesc(actor, illum)
    {
        /* 
         *   check for illumination - we must have at least dim ambient
         *   lighting (level 2) to see the room's long description 
         */
        if (illum > 1)
        {
            /* 
             *   display the normal description of the room - use the
             *   firstDesc if this is the first time in the room,
             *   otherwise the desc.  
             */
            if (seenBy(actor))
            {
                /* we've seen it already - show the regular description */
                desc;
            }
            else
            {
                /* 
                 *   we've never seen this location before - show the
                 *   first-time description 
                 */
                firstDesc;
            }
        }
        else
        {
            /* display the in-the-dark description of the room */
            roomDarkDesc;
        }
    }

    /*
     *   Get the outermost containing room.  By default, we simply return
     *   'self', since the basic location is not designed to be nested.  
     */
    getOutermostRoom() { return self; }

    /*
     *   Make the actor stand up from this location.  By default, we'll
     *   simply change the actor's posture to "standing," and show a
     *   default success report.
     *   
     *   Subclasses might need to override this.  For example, a chair
     *   will set the actor's location to the room containing the chair
     *   when the actor stands up from the chair.  
     */
    makeStandingUp()
    {
        /* simply set the actor's new posture */
        gActor.makePosture(standing);

        /* issue a default report of the change */
        defaultReport(&okayStand);
    }

    /* 
     *   Default posture for an actor in the location.  This is the
     *   posture assumed by an actor when moving out of a nested room
     *   within this location. 
     */
    defaultPosture = standing

    /* failure report we issue when we can't return to default posture */
    mustDefaultPostureProp = &mustBeStanding

    /* run the appropriate implied command to achieve our default posture */
    tryMakingDefaultPosture()
    {
        return defaultPosture.tryMakingPosture(self);
    }

    /*
     *   Check this object as a staging location.  We're a valid location,
     *   so we allow this. 
     */
    checkStagingLocation(dest)
    {
        /* we've valid, so we don't need to do anything */
    }

    /*
     *   Try moving the actor into this location.
     */
    checkMovingActorInto(allowImplicit)
    {
        /* 
         *   If the actor isn't somewhere within us, we can't move the
         *   actor here implicitly - we have no generic way of causing
         *   implicit travel between top-level locations.  Note that some
         *   rooms might want to override this when travel between
         *   adjacent locations makes sense as an implicit action; we
         *   expect that such cases will be rare, so we don't attempt to
         *   generalize this possibility.  
         */
        if (!gActor.isIn(self))
        {
            reportFailure(&cannotDoFromHere);
            exit;
        }

        /*
         *   if the actor is already directly in me, simply check to make
         *   sure the actor is in the default posture for the room - if
         *   not, try running an appropriate implied command to change the
         *   posture 
         */
        if (gActor.isDirectlyIn(self))
        {
            /* if the actor's already in the default posture, we're okay */
            if (gActor.posture == defaultPosture)
                return nil;
            
            /* run the implied command to stand up (or whatever) */
            if (allowImplicit && tryMakingDefaultPosture())
            {
                /* make sure we're in the proper posture now */
                if (gActor.posture != defaultPosture)
                    exit;
                
                /* note that we ran an implied command */
                return true;
            }
            
            /* we couldn't get into the default posture - give up */
            reportFailure(mustDefaultPostureProp);
            exit;
        }

        /*
         *   The actor is within a nested room within me.  Find our
         *   immediate child containing the actor, and remove the actor
         *   from the child. 
         */
        foreach (local cur in contents)
        {
            /* if this is the one containing the actor, remove the actor */
            if (gActor.isIn(cur))
                return cur.checkActorOutOfNested(allowImplicit);
        }

        /* we didn't find the nested room with the actor, so give up */
        reportFailure(&cannotDoFromHere);
        exit;
    }

    /*
     *   Check, using pre-condition rules, that the actor is ready to
     *   enter this room as a nested location.  By default, we do nothing,
     *   since we're not designed as a nested location.  
     */
    checkActorReadyToEnterNestedRoom(allowImplicit)
    {
        return nil;
    }

    /*
     *   Check that the traveler is directly in the given room, using
     *   pre-condition rules.  'nested' is the nested location immediately
     *   within this room that contains the actor (directly or
     *   indirectly).  
     */
    checkTravelerDirectlyInRoom(traveler, allowImplicit)
    {
        /* if the actor is already directly in this room, we're done */
        if (traveler.isDirectlyIn(self))
            return nil;

        /* try moving the actor here */
        return traveler.checkMovingTravelerInto(self, allowImplicit);
    }

    /*
     *   Check, using pre-condition rules, that the actor is removed from
     *   this nested location and moved to its immediate location.  By
     *   default, we're not a nested location, so there's nothing for us
     *   to do.  
     */
    checkActorOutOfNested(allowImplicit)
    {
        /* we're not a nested location, so there's nothing for us to do */
        return nil;
    }

    /*
     *   An actor is attempting to disembark this location.  By default,
     *   we'll simply turn this into an "exit" command.  
     */
    disembarkRoom()
    {
        /* treat this as an 'exit' command */
        replaceAction(Out);
    }

    /*
     *   Get the apparent location of one of our room parts (the floor, the
     *   ceiling, etc).  By default, we don't have any room parts, so we
     *   return nil.  
     */
    getRoomPartLocation(part) { return nil; }

    /*
     *   The destination for objects explicitly dropped by an actor within
     *   this room.  By default, we'll return self, because items dropped
     *   should simply go in the room itself.  Some types of rooms will
     *   want to override this; for example, a room that represents the
     *   middle of a tightrope would probably want to set the drop
     *   destination to the location below the tightrope. 
     */
    getDropDestination(objToDrop)
    {
        /* by default, objects dropped in this room end up in this room */
        return self;
    }

    /* 
     *   The nominal drop destination - this is the location where objects
     *   are *reported* to go when dropped by an actor in this location.
     *   By default, we simply return 'self'.
     *   
     *   The difference between the actual drop location and the nominal
     *   drop location is that the nominal drop location is used only for
     *   reporting messages, while the actual drop location is the
     *   location where objects are moved on 'drop' or equivalent actions.
     *   Rooms, for example, want to report that a dropped object lands on
     *   the floor (or the ground, or whatever), even though the room
     *   itself is the location where the object actually ends up.  We
     *   distinguish between the nominal and actual drop location to allow
     *   these distinctions in reported messages.  
     */
    getNominalDropDestination() { return self; }

    /*
     *   The nominal location for actors sitting/lying within this room.
     *   Normally this is the same as the nominal drop destination, but
     *   these can be overridden separately as needed. 
     */
    getNominalStandingContainer() { return self; }
    getNominalSittingContainer() { return self; }
    getNominalLyingContainer() { return self; }

    /*
     *   Get any extra items in scope for an actor in this location.
     *   These are items that are to be in scope even if they're not
     *   reachable through any of the normal sense paths (so they'll be in
     *   scope even in the dark, for example).
     *   
     *   By default, this returns nothing.  Subclasses can override as
     *   necessary to include additional items in scope.  For example, a
     *   chair would probably want to include itself in scope, since the
     *   actor presumably knows he or she is sitting in a chair even if
     *   it's too dark to see the chair.  
     */
    getExtraScopeItems(actor) { return []; }
    
    /*
     *   Receive notification that we're about to perform a command within
     *   this location.  This is called on the outermost room first, then
     *   on the nested rooms, from the outside in, until reaching the room
     *   directly containing the actor performing the command.  
     */
    roomBeforeAction()
    {
    }

    /*
     *   Receive notification that we've just finished a command within
     *   this location.  This is called on the room immediately containing
     *   the actor performing the command, then on the room containing
     *   that room, and so on to the outermost room. 
     */
    roomAfterAction()
    {
    }

    /*
     *   Get my notification list - this is a list of objects on which we
     *   must call beforeAction and afterAction when an action is
     *   performed within this room.
     *   
     *   We'll also include any registered notification items for all of
     *   our containing rooms up to the outermost container.
     *   
     *   The general notification mechanism always includes in the
     *   notification list all of the objects connected by containment to
     *   the actor, so objects that are in this room need not register for
     *   explicit notification.  
     */
    getRoomNotifyList()
    {
        local lst;
        
        /* start with our explicitly registered list */
        lst = roomNotifyList;

        /* add notification items for our immediate locations  */
        forEachContainer(
            {cont: lst = lst.appendUnique(cont.getRoomNotifyList())});

        /* return the result */
        return lst;
    }

    /*
     *   Add an item to our registered notification list for actions in
     *   the room.
     *   
     *   Items can be added here if they must be notified of actions
     *   performed by within the room even when the items aren't in the
     *   room at the time of the action.  All items connected by
     *   containment with the actor performing an action are automatically
     *   notified of the action; only items that must receive notification
     *   even when not connected by containment need to be registered
     *   here.  
     */
    addRoomNotifyItem(obj)
    {
        roomNotifyList += obj;
    }

    /* remove an item from the registered notification list */
    removeRoomNotifyItem(obj)
    {
        roomNotifyList -= obj;
    }

    /* our list of registered notification items */
    roomNotifyList = []

    /*
     *   Receive notification of travel among nested rooms.  When an actor
     *   moves between two locations related directly by containing (such
     *   as from a chair to the room containing the chair, or vice versa),
     *   we first call this routine on the origin of the travel, then we
     *   move the actor, then we call this same routine on the destination
     *   of the travel.
     *   
     *   This routine is used any time an actor is moved with
     *   travelWithin().  This is not used when an actor travels between
     *   locations related by a TravelConnector object rather than by
     *   direct containment.
     *   
     *   We do nothing by default.  Locations can override this if they
     *   wish to perform any special handling during this type of travel.  
     */
    actorTravelingWithin(origin, dest)
    {
    }

    /*
     *   Get the object that's actually going to change location when an
     *   actor within this location performs a travel command.  By
     *   default, this is simply the actor.  Vehicles can override this to
     *   indicate the vehicle that will move.  
     */
    getTraveler(actor) { return actor; }

    /*
     *   Determine if the given actor has "intrinsic" knowledge of the
     *   destination of the given travel connector leading away from this
     *   location.  This knowledge is independent of any memory the actor
     *   has of actual travel through the connector in the course of the
     *   game, which we track separately via the TravelConnector's travel
     *   memory mechanism.
     *   
     *   There are two main reasons an actor would have intrinsic
     *   knowledge of a connector's destination:
     *   
     *   1. The actor is supposed to be familiar with the location and its
     *   surroundings, within the context of the game.  For example, if
     *   part of the game is the player character's own house, the PC
     *   would probably know where all of the connections among rooms go.
     *   
     *   2. The destination location is plainly visible from this location
     *   or is clearly marked (such as with a sign).  For example, if the
     *   current location is an open field, a nearby hilltop to the east
     *   might be visible from here, so we could see from here where we'll
     *   end up by going east.  Alternatively, if we're in a lobby, and
     *   the passage to the west is marked with a sign reading "electrical
     *   room," an actor would have good reason to think an electrical
     *   room lies to the west.
     *   
     *   We handle case (1) automatically through our actorIsFamiliar()
     *   method: if the actor is familiar with the location, we assume by
     *   default that the actor knows where all of the connectors from
     *   here go.  We don't have any default handling for case (2), so
     *   individual rooms (or subclasses) must override this method if
     *   they want to specify intrinsic knowledge for any of their
     *   outgoing connectors.  
     */
    actorKnowsDestination(actor, conn)
    {
        /* 
         *   if the actor is familiar with this location, then the actor
         *   by default knows where all of the outgoing connections go 
         */
        if (actorIsFamiliar(actor))
            return true;

        /* there's no other way the actor would know the destination */
        return nil;
    }

    /*
     *   Is the actor familiar with this location?  In other words, is the
     *   actor supposed to know the location well at the start of the game?
     *   
     *   This should return true if the actor is familiar with this
     *   location, nil if not.  By default, we return nil, since actors
     *   are not by default familiar with any locations.
     *   
     *   The purpose of this routine is to determine if the actor is meant
     *   to know the location well, within the context of the game, even
     *   before the game starts.  For example, if an area in the game is
     *   an actor's own house, the actor would naturally be familiar,
     *   within the context of the game, with the locations making up the
     *   house.
     *   
     *   Note that this routine doesn't need to "learn" based on the
     *   events of the game.  The familiarity here is meant only to model
     *   the actor's knowledge as of the start of the game.  
     */
    actorIsFamiliar(actor) { return nil; }

    /* 
     *   The default "you can't go that way" message for travel within
     *   this location in directions that don't allow travel.  This is
     *   shown whenever an actor tries to travel in one of the directions
     *   we have set to point to noTravel.  A room can override this to
     *   produce a different, customized message for unset travel
     *   directions - this is an easy way to change the cannot-travel
     *   message for several directions at once.
     *   
     *   If it's dark in the current location, we'll call our
     *   cannotTravelDark routine to handle it.  
     */
    cannotTravel()
    {
        /* check for darkness */
        if (!gActor.isLocationLit())
        {
            /* the actor is in the dark - use our cannotTravelDark */
            cannotGoThatWayInDark();
        }
        else
        {
            /* use the standard "can't go that way" routine */
            cannotGoThatWay();
        }
    }

    /*
     *   Receive notification of travel from one dark location to another.
     *   This is called before the actor is moved from the source
     *   location, and can cancel the travel if desired by using 'exit' to
     *   terminate the command.
     *   
     *   By default, we'll simply display the same handler we do when the
     *   player attempts travel in a direction with no travel possible in
     *   the dark (cannotGoThatWayInDark), and then use 'exit' to cancel
     *   the command.  This default behavior provides the player with no
     *   mapping information in the dark, since the same message is
     *   generated whether or not travel would be possible in a given
     *   direction were light present.  
     */
    darkTravel(actor)
    {
        /* 
         *   show the same message we would show if we attempted travel in
         *   the dark in a direction with no exit 
         */
        cannotGoThatWayInDark();

        /* terminate the command */
        exit;
    }

    /* 
     *   Show the default "you can't go that way" message for this
     *   location.  By default, we show a generic message, but individual
     *   rooms might want to override this to provide a more specific
     *   description of why travel isn't allowed.  
     */
    cannotGoThatWay()
    {
        /* "you can't go that way" */
        reportFailure(&cannotGoThatWay);

        /* show a list of exits, if appropriate */
        cannotGoShowExits(gActor);
    }

    /*
     *   Show a version of the "you can't go that way" message for travel
     *   while in the dark.  This is called when the actor is in the dark
     *   (i.e., there's no ambient light at the actor) and attempts to
     *   travel in a direction that doesn't allow travel.  By default, we
     *   show a generic "you can't see where you're going in the dark"
     *   message.
     *   
     *   This routine is essentially a replacement for the
     *   cannotGoThatWay() routine that we use when the actor is in the
     *   dark.  
     */
    cannotGoThatWayInDark()
    {
        /* "it's too dark; you can't see where you're going */
        reportFailure(&cannotGoThatWayInDark);
    }

    /*
     *   Find a direction linked to a given connector.  Returns the first
     *   direction (as a Direction object) we find linked to the
     *   connector, or nil if we don't find any direction linked to it.  
     */
    directionForConnector(conn, actor)
    {
        /* search all known directions */
        foreach (local dir in Direction.allDirections)
        {
            /* if this direction is linked to the connector, return it */
            if (self.getTravelConnector(dir, actor) == conn)
                return dir;
        }

        /* we didn't find any directions linked to the connector */
        return nil;
    }

    /*
     *   Get preconditions for travel for an actor in this location.
     *   These preconditions should be applied by any command that will
     *   involve travel from this location.
     *   
     *   By default, the traveler (be it the actor or the vehicle the
     *   actor is using to move) must be in the outermost containing room
     *   where there are nested rooms, and must be standing.  
     */
    roomTravelPreCond = [new ObjectPreCondition(self, travelerDirectlyInRoom)]

    /*
     *   Get the effective location of an actor directly within me, for
     *   the purposes of a "follow" command.  To follow someone, we must
     *   have the same effective follow location that the target had when
     *   we last observed the target leaving.
     *   
     *   For most rooms, this is simply the room itself.  
     */
    effectiveFollowLocation = (self)

    /*
     *   My "atmosphere" list.  This can be set to a TextList object to
     *   provide atmosphere messages while the player character is within
     *   this room.  The default roomDaemon will show one message from
     *   this TextList (by calling the TextList's doScript() method) on
     *   each turn the player character is in this location.  
     */
    atmosphereList = nil

    /*
     *   Room daemon - this is invoked on the player character's immediate
     *   location once per turn in a daemon.
     */
    roomDaemon()
    {
        /* 
         *   if we have an atmosphere message list, display the next
         *   message 
         */
        if (atmosphereList != nil)
            atmosphereList.doScript();
    }

    /*
     *   Dispatch the room daemon.  This is a daemon routine invoked once
     *   per turn; we in turn invoke roomDaemon on the current player
     *   character's current location.  
     */
    dispatchRoomDaemon()
    {
        /* call roomDaemon on the player character's location */
        gPlayerChar.location.roomDaemon();
    }
;

/* ------------------------------------------------------------------------ */
/*
 *   Room: the basic class for top-level game locations (that is, game
 *   locations that aren't inside any other simulation objects, but are at
 *   the top level of the containment hierarchy).  This is the smallest
 *   unit of movement; we do not distinguish among locations within a
 *   room, even if a Room represents a physically large location.  If it
 *   is necessary to distinguish among different locations in a large
 *   physical room, simply divide the physical room into sections and
 *   represent each section with a separate Room object.
 *   
 *   A Room is not necessarily indoors; it is simply a location where an
 *   actor can be located.  This peculiar usage of "room" to denote any
 *   atomic location, even outdoors, was adopted by the authors of the
 *   earliest adventure games, and has been the convention ever since.
 *   
 *   A room's contents are the objects contained directly within the room.
 *   These include fixed features of the room as well as loose items in
 *   the room, which are effectively "on the floor" in the room.
 *   
 *   The Room class implements the Travel Connector interface in such a
 *   way that travel from one room to another can be established simply by
 *   setting a direction property (north, south, etc) in the origin room
 *   to point to the destination room.  This type of travel link has no
 *   side effects and is unconditional.
 *   
 *   A room is by default an indoor location; this means that it contains
 *   walls, floor, and ceiling.  An outdoor location should be based on
 *   OutdoorDoor rather than Room.  
 */
class Room: Fixed, BasicLocation, RoomConnector
    /* 
     *   Initialize 
     */
    initializeThing()
    {
        /* inherit default handling */
        inherited();

        /* add my room parts to my contents list */
        contents += roomParts;
    }

    /* 
     *   we're a "top-level" location: we don't have any other object
     *   containing us, but we're nonetheless part of the game world, so
     *   we're at the top level of the containment tree 
     */
    isTopLevel = true

    /*
     *   we generally do not want rooms to be included when a command
     *   refers to 'all' 
     */
    hideFromAll(action) { return true; }

    /*
     *   Most rooms provide their own implicit lighting.  We'll use
     *   'medium' lighting (level 3) by default, which provides enough
     *   light for all activities, but is reduced to dim light (level 2)
     *   when it goes through obscuring media or over distance.  
     */
    brightness = 3

    /*
     *   Get my "destination name," as seen by the given actor from the
     *   given origin location.  This gives the name we can use to
     *   describe this location from the perspective of an actor in an
     *   adjoining location looking at a travel connector from that
     *   location to here.
     *   
     *   By default, we simply return our destName property.  This default
     *   behavior can be overridden if it's necessary for a location to
     *   have different destination names in different adjoining
     *   locations, or when seen by different actors.
     *   
     *   If this location's name cannot or should not be described from an
     *   adjoining location, this should simply return nil.  
     */
    getDestName(actor, origin) { return destName; }

    /* 
     *   Our destination name, if we have one.  By default, we make this
     *   nil, which means the room cannot be described as a destination of
     *   connectors from adjoining locations. 
     */
    destName = nil

    /* 
     *   The nominal drop destination.  By default, we always return the
     *   first element of the roomParts list, which represents our floor
     *   or appropriate equivalent.  
     */
    getNominalDropDestination() { return roomParts[1]; }

    /*
     *   The nominal sitting/lying containers.  By default, the nominal
     *   standing container is the same as the nominal drop destination,
     *   which is in turn usually the room's floor or equivalent, and the
     *   nominal sitting and lying containers are the same as the standing
     *   container.
     *   
     *   These can all be overridden as needed.  The arrangement is
     *   designed so that, if all three are to be overridden to the same
     *   alternative container, only the standing container actually needs
     *   to be changed, since the sitting and lying containers default to
     *   that value.  
     */
    getNominalStandingContainer() { return getNominalDropDestination(); }
    getNominalSittingContainer() { return getNominalStandingContainer(); }
    getNominalLyingContainer() { return getNominalSittingContainer(); }

    /*
     *   A traveler is about to leave the room - take appropriate action.
     *   Note that this routine should generally not display a routine
     *   message describing the departure; that's handled in the arrival
     *   at our new location, because we generally only want to describe
     *   the change if it actually causes a change in our visibility.
     *   
     *   'traveler' is the object actually traveling.  In most cases this
     *   is simply the actor; but when the actor is in a vehicle, this is
     *   the vehicle instead.  
     */
    travelerLeaving(traveler, dest, connector)
    {
        /* describe the departure */
        if (dest != location)
            traveler.describeDeparture(dest, connector);
    }

    /*
     *   A traveler has just entered the room - take appropriate action.
     *   By default, we describe the arrival.  
     */
    travelerArriving(traveler, origin, connector, backConnector)
    {
        /* describe the arrival */
        traveler.describeArrival(origin, backConnector);
    }

    /*
     *   Get the apparent location of one of our room parts.
     *   
     *   In most cases, we use generic objects (defaultFloor,
     *   defaultNorthWall, etc.) for the room parts.  There's only one
     *   instance of each of these generic objects in the whole game -
     *   there's only one floor, one north wall, and so on - so these
     *   instances can't have specific locations the way normal objects do.
     *   Thus the need for this method: this tells us the *apparent*
     *   location of one of the room part objects as perceived from this
     *   room.
     *   
     *   If the part isn't in this location, we'll return nil.  
     */
    getRoomPartLocation(part)
    {
        /* 
         *   if this part is in our part list, then its apparent location
         *   is 'self' 
         */
        if (roomParts.indexOf(part) != nil)
            return self;

        /* we don't have the part */
        return nil;
    }

    /*
     *   Get the list of extra parts for the room.  An indoor location has
     *   walls, floor, and ceiling; these are all generic objects that are
     *   included for completeness only.
     *   
     *   A room with special walls, floor, or ceiling should override this
     *   to eliminate the default objects from appearing in the room.
     *   
     *   If the room has any kind of floor, it should always be the first
     *   element of this list, because we use the first element in other
     *   places.  If the room doesn't have a floor, or if for some reason
     *   the floor object is not the first element of this list, the room
     *   MUST override getNominalDropDestination and getExtraScopeItems so
     *   that they return something appropriate.  
     */
    roomParts = [defaultFloor, defaultCeiling,
                 defaultNorthWall, defaultSouthWall,
                 defaultEastWall, defaultWestWall]

    /*
     *   Get any extra items in scope in this location.  These are items
     *   that are to be in scope even if they're not reachable through any
     *   of the normal sense paths (so they'll be in scope even in the
     *   dark, for example).
     *   
     *   By default this returns our floor (i.e., the first element of the
     *   roomParts list), because an actor is presumed to be in contact
     *   with the floor whenever in the room.  In some rooms, it's
     *   desirable to have certain objects in scope because they're
     *   essential features of the room; for example, a location that is
     *   part of a stairway would probably have the stairs in scope by
     *   virtue of the actor standing on them.  
     */
    getExtraScopeItems(actor)
    {
        /* 
         *   return our floor, which is generally the first element of our
         *   roomParts list 
         */
        return [roomParts[1]];
    }

    /* -------------------------------------------------------------------- */
    /*
     *   Override some parts of the RoomConnector implementation we
     *   inherit.
     *   
     *   We implement the TravelConnector interface in such a way that one
     *   room can point to another directly through a direction link
     *   property (north, south, etc) to specify travel in the given
     *   direction takes an actor to the given room.  
     *   
     *   Suppose that roomA.north = roomB.  This means that if an actor is
     *   in roomA, and executes a "north" command, we'll execute a
     *   TravelVia action on room B.  So, the destination of travel in
     *   roomB.moveInto is simply roomB.  
     */

    /* we are our own destination */
    getDestination(origin, actor) { return self; }
;

/*
 *   A dark room, which provides no light of its own 
 */
class DarkRoom: Room
    /* 
     *   turn off the lights 
     */
    brightness = 0
;

/*
 *   An outdoor location.  This differs from an indoor location in that it
 *   has ground and sky rather than floor and ceiling, and has no walls.  
 */
class OutdoorRoom: Room
    /* an outdoor room has ground and sky, but no walls */
    roomParts = [defaultGround, defaultSky]
;

/* ------------------------------------------------------------------------ */
/*
 *   Room Part - base class for "parts" of rooms, such as floors and walls.
 *   Room parts are unusual in a couple of ways.
 *   
 *   First, room parts are frequently re-used widely throughout a game.  We
 *   define a single instance of each of several parts that are found in
 *   typical rooms, and then re-use those instances in all rooms with those
 *   parts.  For example, we define one "default floor" object, and then
 *   use that object in most or all rooms that have floors.  We do this for
 *   efficiency, to avoid creating hundreds of essentially identical copies
 *   of the common parts.
 *   
 *   Second, because room parts are designed to be re-used, things that are
 *   in or on the parts are actually represented in the containment model
 *   as being directly in their containing rooms.  For example, an object
 *   that is said to be "on the floor" actually has its 'location' property
 *   set to its immediate room container, and the 'contents' list that
 *   contains the object is that of the room, not of the floor object.  We
 *   must therefore override some of the normal handling for object
 *   locations within room parts, in order to make it appear (for the
 *   purposes of command input and descriptive messages) that the things
 *   are in/on their room parts, even though they're not really represented
 *   that way in the containment model.  
 */
class RoomPart: Fixed
    /* 
     *   Don't include room parts in 'all'.  Room parts are so ubiquitous
     *   that we never want to assume that they're involved in a command
     *   except when it is specifically so stated.  
     */
    hideFromAll(action) { return true; }

    /* describe the status - shows the things that are in/on the part */
    examineStatus()
    {
        /* show the contents of the room part */
        examinePartContents(&descContentsLister);
    }

    /* show our contents */
    examinePartContents(listerProp)
    {
        local loc;
        
        /* 
         *   Get my location, as perceived by the actor - this is the room
         *   that contains this part.  If I don't have a location as
         *   perceived by the actor, then we can't show any contents.  
         */
        loc = gActor.location.getRoomPartLocation(self);
        if (loc == nil)
            return;

        /* 
         *   create a copy of the lister customized for this part, if we
         *   haven't already done so 
         */
        if (self.(listerProp).part_ == nil)
        {
            self.(listerProp) = self.(listerProp).createClone();
            self.(listerProp).part_ = self;
        }

        /* show the contents of the containing location */
        self.(listerProp).showList(gActor, self, loc.contents, 0, 0,
                                   gActor.visibleInfoTable(), nil);
    }

    /* 
     *   show our special contents - this shows objects with special
     *   descriptions that are specifically in this room part 
     */
    examineSpecialContents()
    {
        local infoTab;
        local lst;
        
        /* get the actor's list of visible items */
        infoTab = gActor.visibleInfoTable();

        /* 
         *   get the list of special description items, using only the
         *   subset that uses special descriptions in this room part 
         */
        lst = specialDescList(infoTab,
                              {obj: obj.useSpecialDescInRoomPart(self)});

        /* show the list */
        specialDescLister.showList(gActor, nil, lst, 0, 0, infoTab, nil);
    }

    /* 
     *   Get the destination for a thrown object that hits me.  Since we
     *   don't have a real location, we must ask the actor for our room
     *   part location, and then use its hit-and-fall destination.  
     */
    getHitFallDestination(thrownObj, prvCont)
    {
        /* 
         *   get our effective location from the actor, and use its
         *   destination for thrown objects
         */
        return gActor.location.getRoomPartLocation(self)
            .getDropDestination(thrownObj);
    }

    /* consider me to be in any room of which I'm a part */
    isIn(loc)
    {
        local rpl;

        /* 
         *   get the room-part location of this room part, from the
         *   perspective of the prospective location we're asking about 
         */
        if (loc != nil && (rpl = loc.getRoomPartLocation(self)) != nil)
        {
            /* 
             *   We indeed have a room part location in the given
             *   location, so we're a part of the overall location.  We
             *   might be directly in the location (i.e., 'rpl' could
             *   equal 'loc'), or we might be somewhere relative to 'loc'
             *   (for example, we could be within a nested room within
             *   'loc', in which case 'rpl' is a nested room unequal to
             *   'loc' but is within 'loc').  So, if 'rpl' equals 'loc' or
             *   is contained in 'loc', I'm within 'loc', because I'm
             *   within 'rpl'.  
             */
            if (rpl == loc || rpl.isIn(loc))
                return true;
        }

        /* 
         *   we don't appear to be in 'loc' on the basis of our special
         *   room-part location; fall back on the inherited handling 
         */
        return inherited(loc);
    }

    /* our contents listers */
    contentsLister = roomPartContentsLister
    descContentsLister = roomPartDescContentsLister
    lookInLister = roomPartLookInLister

    /* look in/on: show our contents */
    dobjFor(LookIn)
    {
        verify() { }
        action()
        {
            /* show my contents */
            examinePartContents(&lookInLister);

            /* show my special contents */
            examineSpecialContents();
        }
    }
;


/*
 *   Base class for the default floor and the default ground.  The floor
 *   and ground are where things usually go when dropped, and they're the
 *   locations where actors within a room are normally standing.  
 */
class Floor: RoomPart
    /* 
     *   'put x on floor' equals 'drop x'.  Add a precondition that the
     *   drop destination is the main room, since otherwise we could have
     *   strange results if we dropped something inside a nested room.  
     */
    iobjFor(PutOn)
    {
        preCond()
        {
            /* 
             *   require that the drop destination for the direct object
             *   is an outermost room 
             */
            return [new ObjectPreCondition(
                gDobj, dropDestinationIsOuterRoom)];
        }
        verify() { }
        action() { replaceAction(Drop, gDobj); }
    }

    /* 'throw x at floor' - suggest just putting it down instead */
    iobjFor(ThrowAt)
    {
        check()
        {
            mainReport(&shouldNotThrowAtFloor);
            exit;
        }
    }

    /* 'sit on floor' causes the actor to sit in the containing room */
    dobjFor(SitOn)
    {
        preCond =
            [new ObjectPreCondition(gActor.location.getOutermostRoom(),
                                    travelerDirectlyInRoom)]
        verify()
        {
            /* if we're already sitting in my location, can't do it again */
            if (gActor.posture == sitting && location == gActor.location)
                illogicalNow(&alreadySittingOn);
        }
        action()
        {
            /* 
             *   make the actor sitting; there's no need for actually
             *   moving the actor, since an actor standing in a room is
             *   always on the floor anyway 
             */
            gActor.makePosture(sitting);

            /* report success */
            defaultReport(&okaySitOn);
        }
    }

    /* 'lie on floor' causes the actor to lie down in the room */
    dobjFor(LieOn)
    {
        preCond =
            [new ObjectPreCondition(gActor.location.getOutermostRoom(),
                                    travelerDirectlyInRoom)]
        verify()
        {
            /* if we're already sitting in my location, can't do it again */
            if (gActor.posture == lying && location == gActor.location)
                illogicalNow(&alreadyLyingOn);
        }
        action()
        {
            /* make the actor lying; there's no need to move the actor */
            gActor.makePosture(lying);

            /* report success */
            defaultReport(&okayLieOn);
        }
    }

    /* 
     *   Our messages for when actor is standing on me.  Since standing is
     *   the default posture for an actor, and since the floor (or
     *   equivalent) is the default place to be standing, we don't add any
     *   special mention of an actor standing on a floor.  
     */
    statusStanding(actor) { ""; }
    okayStand(actor) { defaultReport(&okayStand); }
    actorStandingDesc(actor) { ""; }
    actorStandingHere(actor) { libMessages.actorHereDesc(actor); }
    listActorStanding(actor) { ""; }

    /* our messages for when an actor is sitting on me */
    statusSitting(actor) { libMessages.statusSittingOn(self); }
    okaySit(actor) { defaultReport(&okaySitOnObj, self); }
    actorSittingDesc(actor) { libMessages.actorSittingOnDesc(actor, self); }
    actorSittingHere(actor) { libMessages.actorHereSittingOn(actor, self); }
    listActorSitting(actor) { libMessages.listActorSitting(actor); }

    /* our room description message when an actor is sitting/lying on me */
    statusLying(actor) { libMessages.statusLyingOn(self); }
    okayLie(actor) { defaultReport(&okayLieOnObj, self); }
    actorLyingDesc(actor) { libMessages.actorLyingOnDesc(actor, self); }
    actorLyingHere(actor) { libMessages.actorHereLyingOn(actor, self); }
    listActorLying(actor) { libMessages.listActorLying(actor); }
;

/* ------------------------------------------------------------------------ */
/*
 *   Define the default room parts. 
 */

/*
 *   the default floor, for indoor locations 
 */
defaultFloor: Floor
    /* 
     *   the prepositional phrase for objects being put into the floor is
     *   special - it's not just an ordinary surface 
     */
    putDestMessage = &putDestFloor

;

/*
 *   the default ceiling, for indoor locations 
 */
defaultCeiling: RoomPart
;

/*
 *   The default walls, for indoor locations.  We provide a north, south,
 *   east, and west wall in each indoor location by default. 
 */
class DefaultWall: RoomPart
;

defaultNorthWall: DefaultWall
;

defaultSouthWall: DefaultWall
;

defaultEastWall: DefaultWall
;

defaultWestWall: DefaultWall
;

/*
 *   the default sky, for outdoor locations 
 */
defaultSky: Distant, RoomPart
;

/*
 *   The default ground, for outdoor locations.
 */
defaultGround: Floor
    /* 
     *   the prepositional phrase for objects being put onto the ground is
     *   special - it's not just an ordinary surface 
     */
    putDestMessage = &putDestFloor
;


/* ------------------------------------------------------------------------ */
/*
 *   A Nested Room is any object that isn't a room but which can contain
 *   an actor: chairs, beds, platforms, vehicles, and the like.
 *   
 *   An important property of nested rooms is that they're not
 *   full-fledged rooms for the purposes of actor arrivals and departures.
 *   Specifically, an actor moving from a room to a nested room within the
 *   room does not trigger an actor.travelTo invocation, but simply moves
 *   the actor from the containing room to the nested room.  Moving from
 *   the nested room to the containing room likewise triggers no
 *   actor.travelTo invocation.  The travelTo method is not applicable for
 *   intra-room travel because no TravelConnector objects are traversed in
 *   such travel; we simply move in and out of contained objects.  To
 *   mitigate this loss of notification, we instead call
 *   actor.travelWithin() when moving among nested locations.
 *   
 *   By default, an actor attempting to travel from a nested location via
 *   a directional command will simply attempt the travel as though the
 *   actor were in the enclosing location.  
 */
class NestedRoom: BasicLocation
    /*
     *   Our interior room name.  This is the status line name we display
     *   when an actor is within this object and can't see out to the
     *   enclosing room.  Since we can't rely on the enclosing room's
     *   status line name if we can't see the enclosing room, we must
     *   provide one of our own.
     *   
     *   By default, we'll use our regular name.  
     */
    roomName = (name)

    /*
     *   Show our interior room description.  We use this to generate the
     *   long "look" description for the room when an actor is within the
     *   room and cannot see the enclosing room.
     *   
     *   Note that this is used ONLY when the actor cannot see the
     *   enclosing room - when the enclosing room is visible (because the
     *   nested room is something like a chair that doesn't enclose the
     *   actor, or can enclose the actor but is open or transparent), then
     *   we'll simply use the description of the enclosing room instead,
     *   adding a note to the short name shown at the start of the room
     *   description indicating that the actor is in the nested room.
     *   
     *   By default, we'll show the appropriate "actor here" description
     *   for the posture, so we'll say something like "You are sitting on
     *   the red chair" or "You are in the phone booth."  Instances can
     *   override this to customize the description with something more
     *   detailed, if desired.  
     */
    roomDesc { gActor.listActorPosture(gActor); }

    /*
     *   The maximum bulk the room can hold.  We'll define this to a large
     *   number by default so that bulk isn't a concern.
     *   
     *   Lower numbers here can be used, for example, to limit the seating
     *   capacity of a chair.  
     */
    bulkCapacity = 10000

    /*
     *   Get the nested room list grouper for an actor in the given
     *   posture directly in this room.  This is used when we're listing
     *   the actors within the nested room as 
     *   
     *   By default, we maintain a lookup table, and store one nested
     *   actor grouper object for each posture.  This makes it so that we
     *   show one group per posture in this room; for example, if we
     *   contain two sitting actors and three standing actors, we'll say
     *   something like "bill and bob are sitting on the stage, and jill,
     *   jane, and jack are standing on the stage."  This can be
     *   overridden if a different arrangement of groups is desired; for
     *   example, an override could simply return a single grouper to list
     *   everyone in the room together, regardless of posture.  
     */
    listWithActorIn(posture)
    {
        /* if we don't have a lookup table for this yet, create one */
        if (listWithActorInTable == nil)
            listWithActorInTable = new LookupTable(5, 5);

        /* if this posture isn't in the table yet, create a grouper for it */
        if (listWithActorInTable[posture] == nil)
            listWithActorInTable[posture] =
                new NestedActorGrouper(self, posture);

        /* return the grouper for this posture */
        return listWithActorInTable[posture];
    }

    /* 
     *   our listWithActorIn table - this gets initialized to a
     *   LookupTable as soon as we need one (in listWithActorIn) 
     */
    listWithActorInTable = nil

    /* 
     *   The lister object for showing the actors within this nested
     *   location.  By default, we use the standard nested actor lister;
     *   this can be overridden to use a different lister.  This is used
     *   to process "examine." 
     */
    nestedActorListerObj = nestedActorLister

    /*
     *   Check for ownership.  For a nested room, an actor can be taken to
     *   own the nested room by virtue of being inside the room - the
     *   chair Bob is sitting in can be called "bob's chair".
     *   
     *   If we don't have an explicit owner, and the potential owner 'obj'
     *   is in me and can own me, we'll report that 'obj' does in fact own
     *   me.  Otherwise, we'll defer to the inherited implementation.  
     */
    isOwnedBy(obj)
    {
        /* 
         *   if we're not explicitly owned, and 'obj' can own me, and
         *   'obj' is inside me, consider us owned by 'obj' 
         */
        if (owner == nil && obj.isIn(self) && obj.canOwn(self))
            return true;

        /* defer to the inherited definition of ownership */
        return inherited(obj);
    }

    /*
     *   Get the extra scope items within this room.  Normally, the
     *   immediately enclosing nested room is in scope for an actor in the
     *   room.  
     */
    getExtraScopeItems(actor) { return [self]; }

    /*
     *   Our atmospheric message list.  By default, if our container is
     *   visible to us, we'll use our container's atmospheric messages.
     *   This can be overridden to provide our own atmosphere list when
     *   the player character is in this nested room.  
     */
    atmosphereList()
    {
        if (location != nil && gPlayerChar.canSee(location))
            return location.atmosphereList;
        else
            return nil;
    }

    /*
     *   An actor is attempting to "get out" while in this location.  By
     *   default, we'll treat this as getting out of this object.  This
     *   can be overridden if "get out" should do something different.  
     */
    disembarkRoom()
    {
        /* run the appropriate command to get out of this nested room */
        removeFromNested();
    }

    /*
     *   Get the apparent location of one of our room parts (the floor,
     *   the ceiling, etc).  By default, we'll simply ask our container
     *   about it, since a nested room by default doesn't have any of the
     *   standard room parts.  
     */
    getRoomPartLocation(part)
    {
        if (location != nil)
            return location.getRoomPartLocation(part);
        else
            return nil;
    }

    /*
     *   Get the drop destination - this is the location where an object
     *   goes when an actor is in this room and drops the object.  By
     *   default, objects dropped in a nested room land in the room.  Some
     *   types of nested rooms might want to override this; for example, a
     *   nested room that doesn't enclose its actor, such as a chair,
     *   might want to send dropped items to the enclosing room.  
     */
    getDropDestination(objToDrop)
    {
        return self;
    }

    /*
     *   Get the object that's actually going to change location when an
     *   actor within this location performs a travel command.  By
     *   default, we'll indicate what our containing room indicates.  (The
     *   enclosing room might be a vehicle or an ordinary room; in any
     *   case, it'll know what to do, so we merely have to ask it.)
     *   
     *   We defer to our enclosing room by default because this allows for
     *   things like a seat in a car: the actor is sitting in the seat and
     *   starts traveling in the car, so the seat calls the enclosing
     *   room, which is the car, and the car returns itself, since it's
     *   the car that will be traveling.  
     */
    getTraveler(actor)
    {
        /* 
         *   ask our location if we have one; otherwise, it's the actor
         *   that moves by default 
         */
        return (location != nil ? location.getTraveler(actor) : actor);
    }

    /*
     *   Get the outermost containing room.  We return our container, if
     *   we have one, or self if not.  
     */
    getOutermostRoom()
    {
        /* return our container's outermost room, if we have one */
        return (location != nil ? location.getOutermostRoom() : self);
    }

    /*
     *   Make the actor stand up from this location.  By default, we'll
     *   cause the actor to travel (using travelWithin) to our container,
     *   and assume the appropriate posture for the container.  
     */
    makeStandingUp()
    {
        /* 
         *   Set the actor's posture to the default for the new location.
         *   Do this before effecting the actual travel, so that the
         *   destination can change this default if it wants. 
         */
        gActor.makePosture(location.defaultPosture);

        /* 
         *   move the actor to our container, traveling entirely within
         *   nested locations 
         */
        gActor.travelWithin(location);

        /* generate the appropriate default for the new location */
        gActor.okayPostureChange();
    }

    /*
     *   Try an implied command to move the actor from outside of this
     *   nested room into this nested room.  This must be overridden in
     *   subclasses to carry out the appropriate implied command.  Returns
     *   the result of tryImplicitAction().
     *   
     *   This is called when we need to move an actor into this location
     *   as part of an implied command.  We use an overridable method
     *   because different kinds of nested rooms have different commands
     *   for entering: SIT ON CHAIR, LIE ON BED, GET IN CAR, RIDE BIKE,
     *   and so on.  This should be normally be overridden imply by
     *   calling tryImplicitAction() with the appropriate command for the
     *   specific type of nested room, and returning the result.  
     */
    tryMovingIntoNested()
    {
        /* do nothing by default - subclasses must override */
        return nil;
    }

    /* 
     *   message property to use for reportFailure when
     *   tryMovingIntoNested fails 
     */
    mustMoveIntoProp = nil

    /*
     *   Try an implied command to remove an actor from this location and
     *   place the actor in my immediate containing location.  This must
     *   be overridden in subclasses to carry out the appropriate implied
     *   command.  Returns the result of tryImplicitAction().
     *   
     *   This is essentially the reverse of tryMovingIntoNested(), and
     *   should in most cases be implemented by calling
     *   tryImplicitAction() with the appropriate command to get out of
     *   the room, and returning the result.  
     */
    tryRemovingFromNested()
    {
        /* do nothing by default - subclasses must override */
        return nil;
    }

    /*
     *   Replace the current action with one that removes the actor from
     *   this nested room.  This is used to implement the GET OUT command
     *   when the actor is directly in this nested room.  In most cases,
     *   this should simply be implemented with a call to replaceAction()
     *   with the appropriate command.  
     */
    removeFromNested()
    {
        /* subclasses must override */
    }

    /*
     *   Try moving the actor into this location.  This is used to move
     *   the actor into this location as part of meeting preconditions,
     *   and we use the normal precondition check protocol: we return nil
     *   if the condition (actor is in this room) is already met; we
     *   return true if we successfully execute an implied command to meet
     *   the condition; and we report a failure message and terminate the
     *   command with 'exit' if we don't know how to meet the condition or
     *   the implied command we try to execute fails or fails to satisfy
     *   the condition.
     *   
     *   This does not normally need to be overridden in subclasses.  
     */
    checkMovingActorInto(allowImplicit)
    {
        /* if the actor is within me, use default handling */
        if (gActor.isIn(self))
            return inherited(allowImplicit);

        /* try an implied command to move the actor into this nested room */
        if (allowImplicit && tryMovingIntoNested())
        {
            /* if we didn't succeed, terminate the command */
            if (!gActor.isDirectlyIn(self)
                || gActor.posture != defaultPosture)
                exit;

            /* tell the caller we executed an implied command */
            return true;
        }

        /* can't perform the implicit command - report the failure */
        reportFailure(mustMoveIntoProp, self);
        exit;
    }

    /*
     *   Check, using pre-condition rules, that the actor is removed from
     *   this nested location and moved to its immediate location.  This
     *   is used to enforce a precondition that the actor is in the
     *   enclosing location.
     *   
     *   This isn't normally overridden in subclasses.  
     */
    checkActorOutOfNested(allowImplicit)
    {
        /* try removing the actor from this nested location */
        if (allowImplicit && tryRemovingFromNested())
        {
            /* 
             *   make sure we managed to move the actor to the desired
             *   location 
             */
            if (!gActor.isDirectlyIn(location))
                exit;

            /* indicate that we carried out an implied command */
            return true;
        }

        /* we can't carry out our implied departure plan - fail */
        reportFailure(&cannotDoFrom, self);
        exit;
    }

    /*
     *   Check, using pre-condition rules, that the actor is ready to
     *   enter this room as a nested location.
     *   
     *   This isn't normally overridden in subclasses.  
     */
    checkActorReadyToEnterNestedRoom(allowImplicit)
    {
        /* 
         *   If the actor is directly in this room, we obviously need do
         *   nothing, as the actor is already in this nested room.  
         */
        if (gActor.isDirectlyIn(self))
            return nil;

        /*
         *   If the actor isn't within us (directly or indirectly), we
         *   must move the actor to a valid "staging location," so that
         *   the actor can move from the staging location into us.  (A
         *   staging location is simply any location from which we can
         *   move directly into this nested room without any intervening
         *   travel in or out of other nested rooms.)  
         */
        if (!gActor.isIn(self))
            return checkActorInStagingLocation(allowImplicit);

        /*
         *   The actor is within us, but isn't directly within us, so
         *   handle this with the normal routine to move the actor into
         *   this room. 
         */
        return checkMovingActorInto(allowImplicit);
    }

    /*
     *   Check, using precondition rules, that the actor is in a valid
     *   "staging location" for entering this nested room.  We'll ensure
     *   that the actor is directly in one of the locations in our
     *   stagingLocations list, running an appropriate implicit command to
     *   move the actor to the first item in that list if the actor isn't
     *   in any of them.
     *   
     *   This isn't normally overridden in subclasses.
     */
    checkActorInStagingLocation(allowImplicit)
    {
        local lst;
        local target;

        /* get the list of staging locations */
        lst = stagingLocations;

        /* if there are no valid staging locations, we can't move here */
        if (lst.length() == 0)
        {
            cannotMoveActorToStagingLocation();
            exit;
        }
        
        /*
         *   Try each of the locations in our staging list, to see if the
         *   actor is directly in any of them. 
         */
        foreach (local cur in lst)
        {
            /* 
             *   if the actor is directly in this staging location, then
             *   the actor can reach the destination with no additional
             *   intervening travel - simply return nil in this case to
             *   indicate that no implicit commands are needed before the
             *   proposed nested room entry 
             */
            if (gActor.isDirectlyIn(cur))
                return nil;
        }

        /*
         *   The actor isn't directly in any staging location, so we must
         *   move the actor to an appropriate staging location before we
         *   can proceed.  Choose a staging location based on the actor's
         *   current location.  
         */
        if ((target = chooseStagingLocation()) != nil)
        {
            /* 
             *   We've chosen a target staging location.  First, check to
             *   make sure the location we've chosen is valid - we might
             *   have chosen a default (such as the nested room's
             *   immediate container) that isn't usable as a staging
             *   location, so we need to check with it first to make sure
             *   it's willing to allow this.  
             */
            target.checkStagingLocation(self);

            /* 
             *   The check routine didn't abort the command, so try an
             *   appropriate implicit command to move the actor into the
             *   chosen staging location. 
             */
            return target.checkMovingActorInto(allowImplicit);
        }

        /*
         *   There's no apparent intermediate staging location given the
         *   actor's current location.  We thus cannot proceed with the
         *   command; simply report that we can't get there from here.  
         */
        cannotMoveActorToStagingLocation();
        exit;
    }

    /*
     *   Choose an intermediate staging location, given the actor's
     *   current location.  This routine is called when the actor is
     *   attempting to move into 'self', but isn't in any of the allowed
     *   staging locations for 'self'; this routine's purpose is to choose
     *   the staging location that the actor should implicitly try to
     *   reach on the way to 'self'.
     *   
     *   By default, we'll attempt to find the first of our staging
     *   locations that indirectly contains the actor.  (We know none of
     *   the staging locations directly contains the actor, because if one
     *   did, we wouldn't be called in the first place - we're only called
     *   when the actor isn't already directly in one of our staging
     *   locations.)  This approach is appropriate when nested rooms are
     *   related purely by containment: if an actor is in a nested room
     *   within one of our staging locations, we can reach that staging
     *   location by having the actor get out of the more deeply nested
     *   room.
     *   
     *   However, this default approach is not appropriate when nested
     *   rooms are related in some way other than simple containment.  We
     *   don't have any general framework for other types of nested room
     *   relationships, so this routine must be overridden in such a case
     *   with special-purpose code defining the special relationship.
     *   
     *   If we fail to find any staging location indirectly containing the
     *   actor, we'll return the result of defaultStagingLocation().  
     */
    chooseStagingLocation()
    {
        /* look for a staging location indirectly containing the actor */
        foreach (local cur in stagingLocations)
        {
            /* 
             *   if the actor is indirectly in this staging location,
             *   choose it as the target intermediate staging location
             */
            if (gActor.isIn(cur))
                return cur;
        }
            
        /*
         *   We didn't find any locations in the staging list that
         *   indirectly contain the actor, so use the default staging
         *   location.  
         */
        return defaultStagingLocation();
    }

    /*
     *   The default staging location for this nested room.  This is the
     *   staging location we'll attempt to reach implicitly if the actor
     *   isn't in any of the rooms in the stagingLocations list already.
     *   We'll return the first element of our stagingLocations list for
     *   which isStagingLocationKnown returns true.  
     */
    defaultStagingLocation()
    {
        local lst;
        
        /* get the list of valid staging locations */
        lst = stagingLocations;

        /* find the first element which is known to the actor */
        foreach (local cur in lst)
        {
            /* if this staging location is known, take it as the default */
            if (isStagingLocationKnown(cur))
                return cur;
        }

        /* we didn't find any known staging locations - there's no default */
        return nil;
    }

    /*
     *   Report that we are unable to move an actor to any staging
     *   location for this nested room.  By default, we'll generate the
     *   message "you can't do that from here," but this can overridden to
     *   provide a more specific if desired.  
     */
    cannotMoveActorToStagingLocation()
    {
        /* report the standard "you can't do that from here" message */
        reportFailure(&cannotDoFromHere);
    }

    /*
     *   The valid "staging locations" for this nested room.  This is a
     *   list of the rooms from which an actor can DIRECTLY reach this
     *   nested room; in other words, the actor will be allowed to enter
     *   'self', with no intervening travel, if the actor is directly in
     *   any of these locations.
     *   
     *   If the list is empty, there are no valid staging locations.
     *   
     *   The point of listing staging locations is to make certain that
     *   the actor has to go through one of these locations in order to
     *   get into this nested room.  This ensures that we enforce any
     *   conditions or trigger any side effects of moving through the
     *   staging locations, so that a player can't bypass a puzzle by
     *   trying to move directly from one location to another without
     *   going through the required intermediate steps.  Since we always
     *   require that an actor go through one of our staging locations in
     *   order to enter this nested room, and since we carry out the
     *   travel to the staging location using implied commands (which are
     *   just ordinary commands, entered and executed automatically by the
     *   parser), we can avoid having to code any checks redudantly in
     *   both the staging locations and any other nearby locations.
     *   
     *   By default, an actor can only enter a nested room from the room's
     *   direct container.  For example, if a chair is on a stage, an
     *   actor must be standing on the stage before the actor can sit on
     *   the chair.  
     */
    stagingLocations = [location]

    /*
     *   Our exit destination.  This is where an actor ends up when the
     *   actor is immediately inside this nested room and uses a "get out
     *   of" or equivalent command to exit the nested room.  By default,
     *   when we leave a nested room, we end up in the enclosing room.  
     */
    exitDestination = (location)

    /*
     *   Is the given staging location "known"?  This returns true if the
     *   staging location is usable as a default, nil if not.  If this
     *   returns true, then the location can be used in an implied command
     *   to move the actor to the staging location in order to move the
     *   actor into self.
     *   
     *   If this returns nil, no implied command will be attempted for
     *   this possible staging location.  This doesn't mean that an actor
     *   gets a free pass through the staging location; on the contrary,
     *   it simply means that we won't try any automatic command to move
     *   an actor to the staging location, hence travel from a non-staging
     *   location to this nested room will simply fail.  This can be used
     *   when part of the puzzle is to figure out that moving to the
     *   staging location is required in the first place: if we allowed an
     *   implied command in such cases, we'd give away the puzzle by
     *   solving it automatically.
     *   
     *   By default, we'll treat all of our staging locations as known.  
     */
    isStagingLocationKnown(loc) { return true; }

    /*
     *   Get the travel preconditions for an actor in this location.  By
     *   default, we return the travel preconditions for our container. 
     */
    roomTravelPreCond()
    {
        local ret;

        /* 
         *   If we can see out to our location, use the location's
         *   conditions, since by default we'll try traveling from the
         *   location; if we can't see out to our location, we won't be
         *   attempting travel through our location's connectors, so use
         *   our own preconditions instead. 
         */
        if (location != nil && gActor.canSee(location))
            ret = location.roomTravelPreCond();
        else
            ret = inherited();

        /* return the results */
        return ret;
    }

    /*
     *   Show our special contents.  By default, we'll list any actors
     *   occupying the nested room.  
     */
    examineSpecialContents()
    {
        /* list the actors occupying us */
        examineNestedRoomActors();
    }

    /*
     *   As part of Examine action processing, list the actors occupying
     *   this nested room.  This is separated from the main Examine
     *   processing so that it can be used by subclasses directly, without
     *   subclasses having to inherit our entire Examine processing.  
     */
    examineNestedRoomActors()
    {
        local infoTab;
        local actorList;

        /* 
         *   get the list of all objects that the actor can sense using
         *   sight-like senses 
         */
        infoTab = gActor.visibleInfoTable();
            
        /* 
         *   get the list of actors who are visible and within this nested
         *   location, other than the actor doing the looking 
         */
        actorList = senseInfoTableSubset(infoTab,
            {obj, info: obj.isActor && obj != gActor && obj.isIn(self)});

        /* list the actors */
        nestedActorListerObj.showList(gActor, nil, actorList, 0, 0,
                                      infoTab, nil);
    }

    /*
     *   We cannot take a nested room that the actor is occupying 
     */
    dobjFor(Take)
    {
        verify()
        {
            /* it's illogical to take something that contains the actor */
            if (gActor.isIn(self))
                illogicalNow(&cannotTakeLocation);

            /* inherit the default handling */
            inherited();
        }
    }

    /*
     *   "get out of" action - exit the nested room
     */
    dobjFor(GetOutOf)
    {
        preCond()
        {
            return [new ObjectPreCondition(self, actorDirectlyInRoom)];
        }
        verify()
        {
            /* 
             *   the actor must be located on the platform; but allow the
             *   actor to be indirectly on the platform, since we'll use a
             *   precondition to move the actor out of any more nested
             *   rooms within us 
             */
            if (!gActor.isIn(self))
                illogicalNow(&notOnPlatform);
        }
        action()
        {
            /* travel to our get-out-of destination */
            gActor.travelWithin(exitDestination);

            /* 
             *   set the actor's posture to the default posture for the
             *   new location 
             */
            gActor.makePosture(gActor.location.defaultPosture);

            /* issue a default report of the change */
            defaultReport(&okayNotStandingOn);
        }
    }

    /* 
     *   define the push-travel indirect object mappings for our travel
     *   actions 
     */
    iobjFor(PushTravelOutOf) asDobjWithoutActionFor(GetOutOf)
;


/* ------------------------------------------------------------------------ */
/*
 *   A "high nested room" is a nested room that is elevated above the rest
 *   of the room.  This specializes the staging location handling so that
 *   it generates more specific messages.  
 */
class HighNestedRoom: NestedRoom
    /* report that we're unable to move to a staging location */
    cannotMoveActorToStagingLocation()
    {
        reportFailure(&nestedRoomTooHigh, self);
    }

    /*
     *   Staging locations.  By default, we'll return an empty list,
     *   because a high location is not usually reachable directly from
     *   its containing location.
     *   
     *   Note that puzzles involving moving platforms will have to manage
     *   this list dynamically, which could be done either by writing a
     *   method here that returns a list of currently valid staging
     *   locations, or by adding objects to this list as they become valid
     *   staging locations and removing them when they cease to be.  For
     *   example, if we have an air vent in the ceiling that we can only
     *   reach when a chair is placed under the vent, this property could
     *   be implemented as a method that returns a list containing the
     *   chair only when the chair is in the under-the-vent state.  
     */
    stagingLocations = []
;


/* ------------------------------------------------------------------------ */
/*
 *   A chair is an item that an actor can sit on.  When an actor is
 *   sitting on a chair, the chair contains the actor.
 *   
 *   We define the "BasicChair" as something that an actor can sit on, and
 *   then subclass this with the standard "Chair", which adds surface
 *   capabilities.  
 */
class BasicChair: NestedRoom
    /* 
     *   Show the appropriate status message addendum when we're seated on
     *   this chair. 
     */
    statusSitting(actor) { libMessages.statusSittingOn(self); }

    /* acknowledge sitting here */
    okaySit(actor) { defaultReport(&okaySitOnObj, self); }

    /*
     *   Show the appropriate message as part of an actor's "examine"
     *   description when the actor is sitting on me. 
     */
    actorSittingDesc(actor)
        { libMessages.actorSittingOnDesc(actor, self); }

    /* 
     *   Show the appropriate message as part of a room description
     *   mentioning that an actor is sitting on me.  
     */
    actorSittingHere(actor)
        { libMessages.actorHereSittingOn(actor, self); }

    /* describe the actor sitting on me as part of an actor list */
    listActorSitting(actor)
        { libMessages.listActorSittingOn(actor, self); }

    /*
     *   A chair's effective follow location is usually its location's
     *   effective follow location, because we don't usually want to treat
     *   a chair as a separate location for the purposes of "follow."
     *   That is, if A and B are in the same room, and A sits down on a
     *   chair in the room, we don't want to count this as a move that B
     *   could follow.  
     */
    effectiveFollowLocation = (location.effectiveFollowLocation)

    /*
     *   Try an implied command to move the actor from outside of this
     *   nested room into this nested room.
     */
    tryMovingIntoNested()
    {
        /* try sitting on me */
        return tryImplicitAction(SitOn, self);
    }

    /* tryMovingIntoNested failure message is "must sit on chair" */
    mustMoveIntoProp = &mustSitOn

    /* default posture in this nested room is sitting */
    defaultPosture = sitting

    /* 
     *   by default, objects dropped while sitting in a chair go into the
     *   enclosing location's drop destination 
     */
    getDropDestination(obj)
    {
        return location != nil ? location.getDropDestination(obj) : self;
    }

    /*
     *   Remove an actor from the chair.  By default, we'll simply stand
     *   up, since this is the normal way out of a chair.  
     */
    tryRemovingFromNested()
    {
        /* try standing up */
        return tryImplicitAction(Stand);
    }

    /*
     *   Run the appropriate command to remove us from this nested
     *   container, as a replacement command. 
     */
    removeFromNested()
    {
        /* to get out of a chair, we simply stand up */
        replaceAction(Stand);
    }

    /*
     *   "sit on" action 
     */
    dobjFor(SitOn)
    {
        preCond = [new ObjectPreCondition(self, actorReadyToEnterNestedRoom)]
        verify()
        {
            /* 
             *   If the actor is already sitting on this chair, we can't
             *   sit on it again.  If we already verified okay on this
             *   point for this same action, ignore the repeated command -
             *   it must mean that we applied a precondition that did all
             *   of our work for us (such as moving us out of a nested
             *   room immediately within us).  
             */
            if (gActor.posture == sitting && gActor.isDirectlyIn(self)
                && gAction.verifiedOkay.indexOf(self) == nil)
                illogicalNow(&alreadySittingOn);
            else
                gAction.verifiedOkay += self;

            /*
             *   If there's not room for the actor's added bulk, don't
             *   allow the actor to sit on the chair.  If the actor is
             *   already within the chair, there's no need to add the
             *   actor's bulk for this change, since it's already counted
             *   as being within us.  
             */
            if (!gActor.isIn(self)
                && getBulkWithin() + gActor.getBulk() > bulkCapacity)
                illogicalNow(&noRoomToSit);

            /* we can't sit in something the actor is holding */
            if (isIn(gActor))
                illogicalNow(&cannotEnterHeld);

            /* 
             *   if the actor is already in me (but not sitting), boost
             *   the likelihood slightly 
             */
            if (gActor.isDirectlyIn(self) && gActor.posture != sitting)
                logicalRank(120, 'already in');
        }
        action()
        {
            /* 
             *   Move the actor into me - this counts as interior travel
             *   within the enclosing room.  Note that we move the actor
             *   before changing the actor's posture in case the travel
             *   fails.  
             */
            gActor.travelWithin(self);
            
            /* set the actor's posture to sitting */
            gActor.makePosture(sitting);

            /* report success */
            defaultReport(&okaySitOn);
        }
    }

    /* "get on/in" is the same as "sit on" */
    dobjFor(Board) asDobjFor(SitOn)

    /* "get off of" is the same as "get out of" */
    dobjFor(GetOffOf) asDobjFor(GetOutOf)
;

/*
 *   A Chair is a basic chair with the addition of being a Surface.
 */
class Chair: BasicChair, Surface
    /*
     *   By default, a chair has a seating capacity of one person, so use
     *   a maximum bulk that only allows one actor to occupy the chair at
     *   a time.  
     */
    bulkCapacity = 10
    
    /* show our status for "examine" */
    examineStatus()
    {
        /* 
         *   inherit the standard surface description - this will list the
         *   items on the chair 
         */
        inherited Surface();
    }

    /* show our special contents */
    examineSpecialContents()
    {
        /* inherit the surface handling first */
        inherited Surface();
        
        /* list actors occupying us */
        examineNestedRoomActors();
    }
;

/*
 *   Bed.  This is an extension of Chair that allows actors to lie on it
 *   as well as sit on it.  As with chairs, we have a basic bed, plus a
 *   regular bed that serves as a surface as well.  
 */
class BasicBed: BasicChair
    /* our status line message when an actor is lying on this object */
    statusLying(actor) { libMessages.statusLyingOn(self); }

    /* acknowledge lying here */
    okayLie(actor) { defaultReport(&okayLieOnObj, self); }

    /* our actor "examine" description when the actor is lying on me */
    actorLyingDesc(actor) { libMessages.actorLyingOnDesc(actor, self); }

    /* our room description message when an actor is lying on this object */
    actorLyingHere(actor) { libMessages.actorHereLyingOn(actor, self); }

    /* describe the actor sitting on me as part of an actor list */
    listActorLying(actor)
        { libMessages.listActorLyingOn(actor, self); }

    /*
     *   Try an implied command to move the actor from outside of this
     *   nested room into this nested room.
     */
    tryMovingIntoNested()
    {
        /* try lying on me */
        return tryImplicitAction(LieOn, self);
    }

    /* tryMovingIntoNested failure message is "must sit on chair" */
    mustMoveIntoProp = &mustLie

    /* default posture in this nested room is sitting */
    defaultPosture = lying

    /*
     *   "lie on" action 
     */
    dobjFor(LieOn)
    {
        preCond = [new ObjectPreCondition(self, actorReadyToEnterNestedRoom)]
        verify()
        {
            /* 
             *   If the actor is already lying on this chair, we can't lie
             *   on it again.  Note that, like the "sit on" processing for
             *   a chair, we check for previous verify success and ignore
             *   the repeated command in such cases.  
             */
            if (gActor.posture == lying && gActor.isDirectlyIn(self)
                && gAction.verifiedOkay.indexOf(self) == nil)
                illogicalNow(&alreadyLyingOn);
            else
                gAction.verifiedOkay += self;

            /*
             *   If there's not room for the actor's added bulk, don't
             *   allow the actor to lie on the bed.  If the actor is
             *   already within us, there's no need to add the actor's
             *   bulk for this change, since it's already counted as being
             *   within us.  
             */
            if (!gActor.isIn(self)
                && getBulkWithin() + gActor.getBulk() > bulkCapacity)
                illogicalNow(&noRoomToLie);

            /* we can't lie on something the actor is holding */
            if (isIn(gActor))
                illogicalNow(&cannotEnterHeld);

            /* 
             *   if the actor is already in me (but not lying down), boost
             *   the likelihood slightly 
             */
            if (gActor.isDirectlyIn(self) && gActor.posture != lying)
                logicalRank(120, 'already in');
        }
        action()
        {
            /* move the actor into me */
            gActor.travelWithin(self);

            /* set the actor's posture to lying */
            gActor.makePosture(lying);

            /* report success */
            defaultReport(&okayLieOn);
        }
    }

    /* "get on/in" is the same as "lie on" */
    dobjFor(Board) asDobjFor(LieOn)
;

/*
 *   A Bed is a basic bed with the addition of Surface capabilities. 
 */
class Bed: BasicBed, Surface
    /* show our status for "examine" */
    examineStatus()
    {
        /* 
         *   inherit the standard surface description - this will list the
         *   items on the bed 
         */
        inherited Surface();
    }

    /* show our special contents */
    examineSpecialContents()
    {
        /* inherit Surface handling first */
        inherited Surface();
        
        /* list actors occupying us */
        examineNestedRoomActors();
    }
;

/*
 *   A Platform is a nested room upon which an actor can stand.  In
 *   general, when you can stand on something, you can also sit and lie on
 *   it as well (it might not be comfortable, but it is usually at least
 *   possible), so we make this a subclass of Bed.  
 */
class BasicPlatform: BasicBed
    /* our various message mappings for standing on this object */
    statusStanding(actor) { libMessages.statusStandingOn(self); }
    okayStand(actor) { defaultReport(&okayStandOnObj, self); }
    actorStandingDesc(actor)
        { libMessages.actorStandingOnDesc(actor, self); }
    actorStandingHere(actor)
        { libMessages.actorHereStandingOn(actor, self); }
    listActorStanding(actor) { libMessages.listActorStanding(actor); }

    /* an actor can follow another actor onto or off of a platform */
    effectiveFollowLocation = (self)

    /*
     *   Try an implied command to move the actor from outside of this
     *   nested room into this nested room.
     */
    tryMovingIntoNested()
    {
        /* try standing on me */
        return tryImplicitAction(StandOn, self);
    }

    /* tryMovingIntoNested failure message is "must get on platform" */
    mustMoveIntoProp = &mustGetOn

    /* default posture in this nested room is sitting */
    defaultPosture = standing

    /* by default, objects dropped on a platform go onto the platform */
    getDropDestination(obj)
    {
        return self;
    }

    /*
     *   Remove an actor from the platform.  "Get off" is the normal
     *   command to leave a platform.  
     */
    tryRemovingFromNested()
    {
        /* try getting off of the platform */
        return tryImplicitAction(GetOffOf, self);
    }

    /*
     *   Replace the current action with one that removes the actor from
     *   this nested room.
     */
    removeFromNested()
    {
        /* get off of the platform */
        replaceAction(GetOffOf, self);
    }

    /*
     *   Make the actor stand up.  On a platform, "stand" does not imply
     *   "get off platform," since you can stand on the platform. 
     */
    makeStandingUp()
    {
        /* simply set the actor's new posture */
        gActor.makePosture(standing);

        /* issue a default report of the change */
        defaultReport(&okayStandOnObj, self);
    }

    /*
     *   "stand on" action 
     */
    dobjFor(StandOn)
    {
        preCond = [new ObjectPreCondition(self, actorReadyToEnterNestedRoom)]
        verify()
        {
            /* 
             *   if the actor is already lying on this chair, we can't lie
             *   on it again 
             */
            if (gActor.posture == standing && gActor.isDirectlyIn(self)
                && gAction.verifiedOkay.indexOf(self) == nil)
                illogicalNow(&alreadyStandingOn);
            else
                gAction.verifiedOkay += self;

            /*
             *   If there's not room for the actor's added bulk, don't
             *   allow the actor to stand here.  If the actor is already
             *   within us, there's no need to add the actor's bulk for
             *   this change, since it's already counted.  
             */
            if (!gActor.isIn(self)
                && getBulkWithin() + gActor.getBulk() > bulkCapacity)
                illogicalNow(&noRoomToStand);

            /* we can't stand on something the actor is holding */
            if (isIn(gActor))
                illogicalNow(&cannotEnterHeld);

            /* 
             *   if the actor is already in me (but not standing), boost
             *   the likelihood slightly 
             */
            if (gActor.isDirectlyIn(self) && gActor.posture != standing)
                logicalRank(120, 'already in');
        }
        action()
        {
            /* move the actor into me */
            gActor.travelWithin(self);

            /* set the actor's posture to standing */
            gActor.makePosture(standing);

            /* report success */
            defaultReport(&okayStandOn);
        }
    }

    /* "get off of" is the same as "stand on" */
    dobjFor(Board) asDobjFor(StandOn)

    /*
     *   Traveling 'down' from a platform should generally be taken to
     *   mean 'get off platform'. 
     */
    down = noTravelDown
;

/*
 *   A Platform is a basic platform with the addition of Surface behavior. 
 */
class Platform: BasicPlatform, Surface
    /* show our status for "examine" */
    examineStatus()
    {
        /* inherit the surface description */
        inherited Surface();
    }

    /* show our special contents */
    examineSpecialContents()
    {
        /* inherit the surface description first */
        inherited Surface();
        
        /* list actors occupying us */
        examineNestedRoomActors();
    }
;

/*
 *   A booth is a nested room that serves as a small enclosure within a
 *   larger room.  Booths can serve as regular containers as well as
 *   nested rooms, and can be made openable by addition of the Openable
 *   mix-in class.  Note that booths don't have to be fully enclosed, nor
 *   do they have to actually be closable.
 *   
 *   Examples of booths: a cardboard box large enough for an actor can
 *   stand in; a closet; a shallow pit.  
 */
class Booth: BasicPlatform, Container
    /* show our status for "examine" */
    examineStatus()
    {
        /* inherit the container description */
        inherited Container();
    }

    /* show our special contents */
    examineSpecialContents()
    {
        /* inherit default Container handling */
        inherited Container();
        
        /* list actors occupying us */
        examineNestedRoomActors();
    }

    /*
     *   Try an implied command to move the actor from outside of this
     *   nested room into this nested room.
     */
    tryMovingIntoNested()
    {
        /* try getting in me */
        return tryImplicitAction(Board, self);
    }

    /*
     *   Remove an actor from the booth.  "Get out" is the normal command
     *   to leave this type of room.  
     */
    tryRemovingFromNested()
    {
        /* try getting out of the object */
        return tryImplicitAction(GetOutOf, self);
    }

    /*
     *   Replace the current action with one that removes the actor from
     *   this nested room.
     */
    removeFromNested()
    {
        /* get out of the object */
        replaceAction(GetOutOf, self);
    }

    /*
     *   "Enter" is equivalent to "get in" (or "board") for a booth 
     */
    dobjFor(Enter) asDobjFor(Board)

    /* 
     *   define the push-travel indirect object mappings for our travel
     *   actions 
     */
    iobjFor(PushTravelEnter) asDobjWithoutActionFor(Enter)
;

/* ------------------------------------------------------------------------ */
/*
 *   A Vehicle is a special type of nested room that moves instead of the
 *   actor in response to travel commands.  When an actor in a vehicle
 *   types, for example, "go north," the vehicle moves north, not the
 *   actor.
 *   
 *   In most cases, a Vehicle should multiply inherit from one of the
 *   other nested room subclasses to make it more specialized.  For
 *   example, a bicycle might inherit from Chair, so that actors can sit
 *   on the bike.  
 */
class Vehicle: NestedRoom, Traveler
    /*
     *   When an actor is in a vehicle, and the actor performs a travel
     *   command, the vehicle is what changes location; the actor simply
     *   stays put while the vehicle moves.  
     */
    getTraveler(actor) { return self; }

    /* 
     *   Get the actors involved in the travel.  This is a list consisting
     *   of all of the actors contained within the vehicle. 
     */
    getTravelerActors = (allContents().subset({x: x.isActor}))

    /*
     *   Traveler preconditions for the vehicle.  By default, we add no
     *   preconditions of our own, but specific vehicles might want to
     *   override this.  For example, a car might want to require that the
     *   doors are closed, the engine is running, and the seatbelts are
     *   fastened before it can travel.  
     */
    travelerPreCond = []

    /*
     *   Check, using pre-condition rules, that the traveler is in the
     *   given room, moving the traveler to the room if possible. 
     */
    checkMovingTravelerInto(room, allowImplicit)
    {
        /* if we're in the desired location, we're set */
        if (isDirectlyIn(room))
            return nil;

        /* 
         *   By default, we can't move a vehicle into a room implicitly.
         *   Individual vehicles can override this when there's an obvious
         *   way of moving the vehicle in and out of nested rooms.  
         */
        reportFailure(&vehicleCannotDoFrom, self);
        exit;
    }

    /* 
     *   the lister object we use to display the list of actors aboard, in
     *   arrival and departure messages for the vehicle 
     */
    aboardVehicleListerObj = aboardVehicleLister

    /*
     *   Send tracking information to everyone interested in following us.
     *   We'll use the default handling to allow actors to follow this
     *   vehicle, and we'll also send tracking information for any actor
     *   occupants of the vehicle, so that any of the occupants can be
     *   followed as well.  
     */
    sendTrackFollowInfo(connector)
    {
        /* 
         *   send tracking information for the vehicle, using the default
         *   implementation 
         */
        inherited Traveler(connector);

        /* 
         *   send tracking information for each actor we're taking along
         *   for the ride 
         */
        foreach (local cur in allContents())
        {
            /* if this is an actor, send tracking information for it */
            if (cur.isActor)
                cur.sendTrackFollowInfo(connector);
        }
    }
;

/*
 *   A VehicleBarrier is a TravelConnector that allows actors to travel,
 *   but blocks vehicles.  By default, we block all vehicles, but
 *   subclasses can customize this so that we block only specific
 *   vehicles. 
 */
class VehicleBarrier: TravelBarrier
    /*
     *   Determine if the given traveler can pass through this connector.
     *   By default, we'll return nil for a Vehicle, true for anything
     *   else.  This can be overridden to allow specific vehicles to pass,
     *   or to filter on any other criteria.  
     */
    canTravelerPass(traveler) { return !traveler.ofKind(Vehicle); }

    /* explain why we can't pass */
    explainTravelBarrier(traveler)
    {
        reportFailure(&cannotGoThatWayInVehicle, traveler);
    }
;

