/***************************************************************************
 *   Copyright (C) 2005-2006 Gao Xianchao                                  *
 *                 2007 Gao Xianchao gnap_an linux_lyb ahlongxp            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

/*
 * Author:	gxc
 * Create data:	2005-10-12 20:09
 */

#include <unistd.h>
#include <sys/epoll.h>
#include "errno.h"
#include "EpollReactor.h"
#include "log.h"
#include "utils.h"


#define EVENT_COUNT 150

CEpollReactor::CEpollReactor()
: _epollHandle(-1)
, _toExit(false)
, _freeTimerID(1)
{
}

CEpollReactor::~CEpollReactor()
{
}

bool CEpollReactor::addSocket(ISocket* socket)
{
	struct epoll_event ee;
	
	ee.events = 0;
	if(socket->maskRead())
	{
		ee.events = ee.events | EPOLLIN;
	}
	if(socket->maskWrite())
	{
		ee.events = ee.events | EPOLLOUT;
	}
	
	ee.data.ptr = socket;

	epoll_ctl(_epollHandle, EPOLL_CTL_ADD, socket->getHandle(), &ee);
	return true;
}
 
void CEpollReactor::removeSocket(ISocket* socket)
{
	epoll_ctl(_epollHandle, EPOLL_CTL_DEL, socket->getHandle(), NULL);
}
 
bool CEpollReactor::start()
{
	LOG_INFO("CEpollReactor starting");
	
	_toExit = false;
	
	_epollHandle = epoll_create(1024);
	if(_epollHandle == -1)
	{
		return false;
	}
	
	_epollEvents = new struct epoll_event[EVENT_COUNT];
	
	LOG_INFO("CEpollReactor started");
	return true;
}
 
void CEpollReactor::stop()
{
	LOG_INFO("CEpollReactor stopping");
	
	_toExit = true;
	
	close(_epollHandle);
	_epollHandle = -1;
	
	_timerList.clear();
	_timerAddList.clear();
	
	delete[] _epollEvents;
	
	LOG_INFO("CEpollReactor stopped");
}

void CEpollReactor::update()
{
	int epret = epoll_wait(_epollHandle, _epollEvents, EVENT_COUNT, 200);
	if(epret == -1)
	{
		LOG_ERROR("epoll_wait error errno="<<errno);
		//break;
		return;
	}

	if(epret == 0)
	{
		//timeout
	}

	if(epret > 0)
	{
		for(int i=0; i<epret; ++i)
		{
			ISocket* socket = (ISocket*)(_epollEvents[i].data.ptr);
			
			if(_epollEvents[i].events & (EPOLLERR | EPOLLHUP))
			{
				//LOG_DEBUG("EPOLLERR on fd "<<socket->getHandle());
				socket->handleClose();
				socket->setReactor(NULL);
				socket->close();
				continue;
			}
			
			int ret = 0;
			if(_epollEvents[i].events & EPOLLIN)
			{
				//LOG_DEBUG("EPOLLIN on fd "<<socket->getHandle());
				ret = socket->handleRead();
			}
			if(ret == -1)
			{
				socket->handleClose();
				socket->setReactor(NULL);
				socket->close();
				continue;
			}
			
			if(_epollEvents[i].events & EPOLLOUT)
			{
				//LOG_DEBUG("EPOLLOUT on fd "<<socket->getHandle());
				ret = socket->handleWrite();
			}
			if(ret == -1)
			{
				socket->handleClose();
				socket->setReactor(NULL);
				socket->close();
				continue;
			}
		}
	}
	
	updateTimerList();	
}

void CEpollReactor::updateMask(ISocket* socket)
{
	struct epoll_event ee;

	ee.events = 0;
	if(socket->maskRead())
	{
		ee.events = ee.events | EPOLLIN;
	}
	if(socket->maskWrite())
	{
		ee.events = ee.events | EPOLLOUT;
	}

	ee.data.ptr = socket;

	epoll_ctl(_epollHandle, EPOLL_CTL_MOD, socket->getHandle(), &ee);	
}

unsigned int CEpollReactor::addTimer(ITimerCallback* callback, unsigned int interval,  bool oneShot)
{
	if(callback == NULL)
	{
		return 0;
	}
	
	TTimerInfo timerInfo;
	
	if(_freeTimerIDList.size() == 0)
	{
		timerInfo.id = _freeTimerID;
		_freeTimerID++;
	}
	else
	{
		timerInfo.id = _freeTimerIDList.front();
		_freeTimerIDList.pop_front();
	}
	
	timerInfo.interval = interval;
	timerInfo.oneShot = oneShot;
	timerInfo.remove = false;
	timerInfo.lastShotTick = GetTickCount();
	timerInfo.callback = callback;
	
	
	_timerAddList.push_back(timerInfo);
	
	return timerInfo.id;
}

void CEpollReactor::removeTimer(unsigned int id)
{
	TTimerList::iterator iter = _timerList.begin();
	for(; iter!=_timerList.end(); ++iter)
	{
		if(iter->id == id)
		{
			iter->remove = true;			
			return;
		}
	}
	
	TTimerList::iterator iter2 = _timerAddList.begin();
	for(; iter2!=_timerAddList.end(); ++iter2)
	{
		if(iter2->id == id)
		{
			_freeTimerIDList.push_back(id);
			_timerAddList.erase(iter2);			
			return;
		}
	}	
}

void CEpollReactor::updateTimerList()
{
	TTimerList::iterator iter = _timerList.begin();
	for(; iter!=_timerList.end();)
	{
		if(iter->remove == true)
		{
			//LOG_DEBUG("timer " << iter->id <<" removed");
			_freeTimerIDList.push_back(iter->id);
			iter = _timerList.erase(iter);				
			continue;			
		}
		
		if(GetTickCount() >= iter->lastShotTick + iter->interval)
		{
			iter->callback->onTimer(iter->id);
			iter->lastShotTick = GetTickCount();
			
			if(iter->oneShot)
			{
				//LOG_DEBUG("timer " << iter->id <<" removed");
				_freeTimerIDList.push_back(iter->id);
				iter = _timerList.erase(iter);				
				continue;
			}
		}
		
		++iter;
	}
	
	TTimerList::iterator iter2 = _timerAddList.begin();
	for(; iter2!=_timerAddList.end(); ++iter2)
	{
		//LOG_DEBUG("timer " << iter2->id <<" added");
		_timerList.push_back(*iter2);
	}
	
	_timerAddList.clear();
}
