/** \brief segment info parser and helper functions

   mkvmerge -- utility for splicing together matroska files
   from component media subtypes

   Distributed under the GPL v2
   see the file COPYING for details
   or visit http://www.gnu.org/copyleft/gpl.html

   \file

   \author Written by Moritz Bunkus <moritz@bunkus.org> and
   \author Steve Lhomme <steve.lhomme@free.fr>.
*/

#include "common/common_pch.h"

#include <ebml/EbmlVersion.h>

#include <matroska/KaxInfo.h>
#include <matroska/KaxInfoData.h>
#include <matroska/KaxVersion.h>

#include "common/ebml.h"
#include "common/hacks.h"
#include "common/segmentinfo.h"
#include "common/strings/utf8.h"
#include "common/version.h"

using namespace libmatroska;

template<typename A> void
provide_default_for_child(EbmlMaster &master,
                          const UTFstring &default_value) {
  A *value = FindChild<A>(master);
  if (value)
    value->SetValue(value->GetValue());
  else
    GetChild<A>(master).SetValue(default_value);
}

template<typename A> void
provide_default_for_child(EbmlMaster *master,
                          const UTFstring &default_value) {
  provide_default_for_child<A>(*master, default_value);
}

/** \brief Add missing mandatory elements

   The Matroska specs and \c libmatroska say that several elements are
   mandatory. This function makes sure that they all exist by adding them
   with their default values if they're missing.

   The parameters are checked for validity.

   \param e An element that really is an \c EbmlMaster. \a e's children
     should be checked.
*/
void
fix_mandatory_segmentinfo_elements(EbmlElement *e) {
  if (!e)
    return;

  KaxInfo *info = dynamic_cast<KaxInfo *>(e);
  if (!info)
    return;

  GetChild<KaxTimecodeScale>(info);

  if (!hack_engaged(ENGAGE_NO_VARIABLE_DATA)) {
    provide_default_for_child<KaxMuxingApp>( info, to_utfstring(boost::format("libebml v%1% + libmatroska v%2%") %  EbmlCodeVersion % KaxCodeVersion));
    provide_default_for_child<KaxWritingApp>(info, to_utfstring(get_version_info("mkvmerge", static_cast<version_info_flags_e>(vif_full | vif_untranslated))));
  } else {
    provide_default_for_child<KaxMuxingApp>( info, to_utfstring("no_variable_data"));
    provide_default_for_child<KaxWritingApp>(info, to_utfstring("no_variable_data"));
  }
}
