///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
///
/// =========================================================================
//
// input basis from string or files:
//   idiststream& operator >> (idiststream&, basis&);
//
// author: Pierre.Saramito@imag.fr
//
// date: 2 october 2017
//
// Note:
//  - some technical stuff with bison & flex
//  - some others for distributed issues:
//        * lecture du fichier sur proc=0 -> spec sur proc=0
//        * lecture ecriture spec sur stringstream sur proc=0
//        * broadcast string sur ts procs
//        * lecture du stringstream sur ts les proc -> spec
//        * conversion spec en basis sur ts les procs
//

// 
#include <sstream> // flex include it, but in namespace rheolef
#include <cstring>
#include "rheolef/basis.h"

namespace rheolef {

// ================================================================================
// part 1 : basis specification
// ================================================================================

struct family_degree_option_t {
  family_degree_option_t (std::string f="", size_t d=0, basis_option o=basis_option()) 
    : family(f), degree(d), option(o) {}
  std::string       family;
  size_t            degree;
  basis_option option;
};

static family_degree_option_t _current_fdo;

} // namespace rheolef

// ================================================================================
// part 2 : read from istream and build family_degree_option_t basis specification
// ================================================================================
/* AIX requires this to be the first thing in the file.  */
#ifndef __GNUC__
# if _RHEOLEF_HAVE_ALLOCA_H
#  include <alloca.h>
# else
#  ifdef _AIX
#pragma alloca
#  else
#   ifndef alloca /* predefined by HP cc +Olibcalls */
char *alloca ();
#   endif
#  endif
# endif
#endif

namespace rheolef {

using namespace std;

typedef size_t size_type;

static size_type   basis_line_no = 1;
static size_type   basis_n_error = 0;
static std::string basis_input_string;

extern int basis_lex();
void basis_error (const char* msg) {
  std::string near;
  error_macro("invalid basis name: \"" << basis_input_string << "\"");
  basis_n_error++;
}
int basis_wrap () { return 1; }

static std::vector<std::string> symbol_table;
static const std::string& symbol (size_t i) { return symbol_table[i]; }
static size_t insert (const std::string& str) {
        size_t i = symbol_table.size();
        symbol_table.push_back (str);
        return i;
}
#define YYMALLOC ::malloc
#define YYFREE   ::free

#include "basis_yacc.cc"

// avoid re-definition of YY_NULL within flex
#ifdef YY_NULL
#undef YY_NULL
#endif
#include "basis_lex.cc"

static yyFlexLexer input_basis;

int basis_lex() { return input_basis.yylex(); }

static void basis_get_pass_2 (std::string& str);

// ================================================================================
// read from string via istrstream
// ================================================================================
// parse from string on all procs and re-build result_ptr available on all procs
static
void
build_fdo_from_string (std::string& str)
{
  std::istringstream istrstr;
  // force reading until end of string
  // otherwise skip all optional terms as "d" and "{...}" !
  std::string str_eol = str + ";";
  istrstr.str (str_eol);
  input_basis.yyrestart(istrstr);
  symbol_table.clear();
  _current_fdo = family_degree_option_t();
  basis_line_no = 1;
  basis_n_error = 0;
  basis_input_string = str; // for error msg
  if (basis_parse() != 0 || basis_n_error != 0) {
    _current_fdo = family_degree_option_t();
    error_macro ("invalid basis name: \"" << str << "\"");
  }
  symbol_table.clear();
  // then _current_fdo is available on all procs
}
template<class T>
void
basis_basic<T>::reset (std::string& str)
{
  if (str == "") {
    base::operator= (basis_rep<T>::make_ptr(str, basis_option()));
    return;
  }
  // get the fdo variable:
  build_fdo_from_string (str);
  // convert the fdo variable into a basis:
  std::string name;
  if (basis_rep<T>::have_degree_parameter (_current_fdo.family)) {
    name = basis_rep<T>::standard_naming (_current_fdo.family, _current_fdo.degree, _current_fdo.option);
  } else {
    name = _current_fdo.family;
  }
  base::operator= (basis_rep<T>::make_ptr(name,_current_fdo.option));
  _clear();
  // clear the fdo variable:
  _current_fdo = family_degree_option_t();
}
// the same for raw basis, without options:
template<class T>
void
basis_raw_basic<T>::reset (std::string& str)
{
  if (str == "") {
    base::operator= (basis_raw_rep<T>::make_ptr(str));
    return;
  }
  build_fdo_from_string (str);
  std::string name = _current_fdo.family + itos(_current_fdo.degree);
  base::operator= (basis_raw_rep<T>::make_ptr(name));
  _clear();
  _current_fdo = family_degree_option_t();
}
#ifdef TODO
// ================================================================================
// read from file
// ================================================================================
// read from file on io_proc into a string
static
void
build_fdo_from_file (idiststream& ids)
{
  typedef distributor::size_type size_type;
  std::istream& is = ids.is();
  communicator comm = ids.comm();
  size_type io_proc = odiststream::io_proc();
  size_type my_proc = comm.rank();
  std::string str;
  if (my_proc == io_proc) {
    // getline could be problematic: read after the possible end
    // of the basis specification, when there is something else
    // after in the file, e.g.
    //    P5{warburton}(square)
    // then the "(square)" is read in the file, until end of line...
    std::getline (ids.is(), str);
  }
  // then broadcast the string
#ifdef _RHEOLEF_HAVE_MPI
  mpi::broadcast (mpi::communicator(), str, io_proc); 
#endif // _RHEOLEF_HAVE_MPI
  build_fdo_from_string (str);
}
template<class T>
idiststream&
operator>> (idiststream& ids, basis_basic<T>& b)
{
  build_fdo_from_file (ids);
  // convert _current_fdo to basis
  b = basis_basic<T> (_current_fdo.family + itos(_current_fdo.degree), _current_fdo.option);
  _current_fdo = family_degree_option_t();
  return ids;
}
template<class T>
idiststream&
operator>> (idiststream& ids, basis_raw_basic<T>& b)
{
  build_fdo_from_file (ids);
  // convert _current_fdo to basis
  b = basis_raw_basic<T> (_current_fdo.family + itos(_current_fdo.degree));
  _current_fdo = family_degree_option_t();
  return ids;
}
#endif // TODO
// -----------------------------------------------------------------------------
// instanciation in library
// -----------------------------------------------------------------------------
#define _RHEOLEF_instanciate(T) 						\
template void basis_basic<T>::reset (std::string&);				\
template void basis_raw_basic<T>::reset (std::string&);				

_RHEOLEF_instanciate(Float)

#undef _RHEOLEF_instanciate

} // namespace rheolef
