/* ups_dials.c - Keeping track of the dials in gupsc

   Copyright (C) 2000  Henning Kulander <hennikul@ifi.uio.no>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include "dial.h"
#include "preferences.h"
#include "ups_dials.h"
#include "ups_fetch.h"

#define GREEN  GNOME_CANVAS_COLOR_A(0x00, 0xff, 0x00, 0xff)
#define YELLOW GNOME_CANVAS_COLOR_A(0xff, 0xff, 0x00, 0xff)
#define RED    GNOME_CANVAS_COLOR_A(0xff, 0x00, 0x00, 0xff)
#define MAX_TYPES 10

Dial *dial_herz;
Dial *dial_volt;
Dial *dial_load;

typedef struct _DialType DialType;

struct _DialType {
	gchar *name;
	gchar *description;
	gint *labels;
	gchar *format;
	void (*edge_colors)(Dial *dial);
};

gint lowxfer=0;
gint highxfer=275;

DialType *dial_types[MAX_TYPES];

gint labels_b[] = { /* Labels for the Battery meter */
	0, 20, 40, 60, 80, 100, 0
};
gint labels_l[] = { /* Labels for the Load meter */
	0, 20, 40, 60, 80, 100, 120, 0
};
gint labels_v[] = { /* Labels for the Volt meter */
	0, 50, 100, 150, 200, 250, 300, 0
};
gint labels_h[] = { /* Labels for the frequenzy meter */
	25, 35, 45, 55, 65, 75, 85, 0
};
gint labels_t[] = { /* Labels for the temperature meter */
	-20, 0, 20, 40, 60, 80, 100, 0
};
gint labels_w[] = { /* Labels for the humidity meter */
	0, 20, 40, 60, 80, 100, 0
};

/* Edge colors for battery percent dial */
static void 
edge_colors_battpct (Dial *dial)
{
	dial_area_mark (dial, RED, 0, 41.666);
	dial_area_mark (dial, YELLOW, 41.666, 83.333);
	dial_area_mark (dial, GREEN, 83.333, 100);
}

/* Edge colors for load dial */
static void 
edge_colors_loadpct (Dial *dial)
{  
	dial_area_mark (dial, YELLOW, 100, 110);
	dial_area_mark (dial, RED, 110, 120);
}

/* Edge colors for current (volt) dial */
static void 
edge_colors_utility (Dial *dial)
{
	if (lowxfer != 0) {
		dial_area_mark (dial, RED, 0, lowxfer);
		dial_area_mark (dial, YELLOW, lowxfer, lowxfer + 25);
		dial_area_mark (dial, GREEN, lowxfer + 25, highxfer - 25);
	}
	dial_area_mark (dial, YELLOW, highxfer - 25, highxfer);
	dial_area_mark (dial, RED, highxfer, 300);
}

/* Edge colors for AC-frequency dial */
static void 
edge_colors_acfreq (Dial *dial)
{
	dial_area_mark (dial, RED, 25, 40);
	dial_area_mark (dial, YELLOW, 40, 45);
	dial_area_mark (dial, GREEN, 45, 65);
	dial_area_mark (dial, YELLOW, 65, 70);
	dial_area_mark (dial, RED, 70, 85);
}

/* Edge colors for UPS temperature dial */
static void 
edge_colors_upstemp (Dial *dial)
{
	dial_area_mark (dial, RED, 60, 100);
	dial_area_mark (dial, YELLOW, 40, 60);
	dial_area_mark (dial, GREEN, -20, 40);
}

/* Edge colors for ambient humidity dial */
static void 
edge_colors_ambhumid (Dial *dial)
{
	dial_area_mark (dial, RED, 70, 100);
	dial_area_mark (dial, YELLOW, 30, 70);
	dial_area_mark (dial, GREEN, 0, 30);
}

/* Set the limits in the current dial */
void 
set_current_limits (void)
{
	gfloat low, high;
	
	low = ups_get_lowxfer ();
	lowxfer = ((gint) low / 25) * 25;
	high = ups_get_highxfer ();
	highxfer = (1+(gint) high / 25) * 25;
}

/* Make a new dialtype */
static DialType *
make_dial_type (gchar *name, gchar *description, gchar *format, 
		gint labels[], void (*edge_colors)(Dial *dial))
{
	DialType *dial_type;
	
	dial_type = g_new (DialType, 1);
	dial_type->name = name;
	dial_type->description = description;
	dial_type->format = format;
	dial_type->labels = labels;
	dial_type->edge_colors = edge_colors;
	
	return dial_type;
}

/* Make all the dial types */
void 
ups_make_dial_types (void)
{
	/* Close your eyes, you don't want to see this.. :) */
	/* Make batterypercent type */
	dial_types[0] = make_dial_type ("BATTPCT", _("Battery\nPercent"), 
					_("%4.1f%%"),labels_b,
					edge_colors_battpct);
	/* Make loadpercent type */
	dial_types[1] = make_dial_type ("LOADPCT", _("Load\nLevel"), 
					_("%4.1f%%"),labels_l, 
					edge_colors_loadpct );
	/* Make voltage utility type */
	dial_types[2] = make_dial_type ("UTILITY", _("Input\nVoltage"), 
					_("%4.1fV"), labels_v, 
					edge_colors_utility );
	/* Make frequenzy type */
	dial_types[3] = make_dial_type ("ACFREQ", _("Input\nFrequency"),
					_("%4.1fHz"),labels_h, 
					edge_colors_acfreq );
	/* Make temperature type */
	dial_types[4] = make_dial_type ("UPSTEMP", _("UPS\nTemperature"),
					_("%4.1f C"),labels_t, 
					edge_colors_upstemp );
	/* Make humidity type */
	dial_types[5] = make_dial_type ("AMBHUMID", _("UPS\nHumidity"),
					_("%4.1f%%"),labels_w, 
					edge_colors_ambhumid);
}

/* Get a description for dialtype "name" */
gchar *
ups_get_description (gchar *name)
{
	int i;
	gchar *description, *ptr;
	
	for (i = 0; i < MAX_TYPES; i++) {
		if ((dial_types[i] != NULL) 
		    && (!strcmp(dial_types[i]->name, name))) {
			description = g_strdup (dial_types[i]->description);
			ptr = strchr (description, '\n'); 
			if (ptr) 
				*ptr = ' '; /* Remove linefeeds */
			return description;
		}
	}

	return NULL;
}

/* Make a new dial */
Dial *
ups_make_dial (gchar *type, GtkWidget *canvas)
{
	Dial *dial;
	gint i;
	
	dial = g_new( Dial, 1 );
	
	for (i = 0; dial_types[i] != NULL ; i++)
		if (!strcmp( type, dial_types[i]->name)) {
			dial_make (dial, canvas, dial_types[i]->description, 
				   dial_types[i]->format,
				   preferences_get_fontname_dials (),
				   GNOME_CANVAS_COLOR_A (0xdf,0xdf,0x00,0xa0), 
				   dial_types[i]->labels);
			dial_types[i]->edge_colors (dial);
		}
	return dial;
}

