/* $Id: polling.h,v 1.5 2000/09/29 10:15:37 bergo Exp $ */

#ifndef GPS_POLLING_H
#define GPS_POLLING_H

#include <gtk/gtk.h>

/// maximum process name length
#define GPS_MAXPROCESSNAME      256
/// maximum command line length
#define GPS_MAXPROCESSLONGNAME  512
/// maximum username length
#define GPS_MAXOWNER            64
/// maximum length of a memory representation
#define GPS_MAXMEMSTRING        32
/// maximum length of a start time string
#define GPS_MAXTIMESTRING       64
/// maximum length of a hostname
#define GPS_MAXHOSTNAME         128

/* FreeBSD is a box full of surprises... */
#undef PID
#undef NAME
#undef LONGNAME
#undef OWNER
#undef MACHINE
#undef STATE
#undef CPU
#undef SIZE
#undef RSS

typedef enum { PID=0,NAME,LONGNAME,OWNER,MACHINE,
	       STATE,CPU,SIZE,RSS,NICE,
	       PRIORITY,START,
	       PPID,PGID,SID,TTY,TPGID,FLAGS,
	       SIGPENDING,SIGBLOCKED,SIGIGNORED,SIGCAUGHT,
	       RSSLIM,MINFLT,CMINFLT,MAJFLT,CMAJFLT,
	       USRJIFFIES,KRNJIFFIES,CUSRJIFFIES,CKRNJIFFIES } ProcessField;


/// represents a row in the main process list
class ProcessListItem {
 public:
  /// creates an empty instance
  ProcessListItem();

  /// use this method to set values, it checks for oversized entries.
  void setField(ProcessField f,char *value);
  
  char pid[16];
  char name[GPS_MAXPROCESSNAME];
  char longname[GPS_MAXPROCESSLONGNAME];
  char owner[GPS_MAXOWNER];
  char machine[GPS_MAXHOSTNAME];
  char state[16];
  char cpu[32];
  char size[GPS_MAXMEMSTRING];
  char rss[GPS_MAXMEMSTRING];
  char nice[16];
  char priority[16];
  char start[GPS_MAXTIMESTRING];
};

/// represents the details of a process, for use in the Details Dialog.
class ProcessItem : public ProcessListItem{
 public:
  /// creates an empty instance
  ProcessItem();
  /// use this method to set values, it checks for oversized entries.
  void setField(ProcessField f,char *value);

  char ppid[16],pgid[16],sid[16],tty[64],
    tpgid[16],flags[256];
  char sigpend[256],sigblock[256],sigign[256],sigcau[256];
  char rssl[256],minflt[32],cminflt[32],majflt[32],cmajflt[32];
  char uj[32],kj[32],cuj[32],ckj[32]; /* jiffies, user/kernel */
};

/// capabilities unique to Unix critters
class UnixHostnames {
 public:
  /// returns the local hostname
  virtual char *get_local_hostname(); /* should work under any Un*x beast */
 private:
  char hostbuffer[GPS_MAXHOSTNAME];
};

/**
 abstract representation of a poller capable of retrieving the
 list of processes in the local machine.
*/
class ProcessListPoller : public UnixHostnames {  
 public:
  /// trivial constructor
  ProcessListPoller();
  /// trivial destructor
  ~ProcessListPoller();

  /** 
      fills the process_list with the current process list of this
      machine (as a list of ProcessListItem * objects).
      Should not take more than half a second to fill the table!
      See linuxprocfs.cc / .h for a sample implementation.
      In case of trouble, returning with process_list NULL is
      acceptable.
  */
  virtual void poll()=0;
  /// clean up, called on destruction. Preferred over ~ProcessListPoller.
  virtual void terminate();

  /// the process list, as a GList (from glib) of (ProcessListItem *) nodes.
  GList *process_list;

 protected:
  /**
     deletes the current process_list, if not NULL. Usually poll()
     calls this.
  */
  void reset();
};

/**
 abstract representation of a poller capable of retrieving the
 details of a local process.
*/
class ProcessDetailsPoller : public UnixHostnames {
 public:
  /// trivial constructor
  ProcessDetailsPoller();
  /// tricial destructor
  ~ProcessDetailsPoller();

  /// abstract, must fill item with information about process whichpid.
  virtual void poll(int whichpid)=0;
  /// clean-up, preferred over ~ProcessDetailsPoller.
  virtual void terminate();

  /// the details item, filled by poll().
  ProcessItem *item;

 protected:
  /// frees item if it's not NULL.
  void reset();
};

/**
 abstract representation of a poller capable of retrieving the
 cpu usage (as a float in the range 0.0 to 1.0), memory usage
 (bytes free, bytes used) and swap usage (bytes free, bytes used).
*/
class SystemInfoProvider {
 public:
  /**
     abstract, fills the memory_*, swap_* and cpu_usage fields.
     Will be called in approximately 250 msec intervals, so it
     can't take much time to complete.
     If it can't stat SMP boxes, should call SMP_faker() before
     returning.
  */
  virtual void update()=0;

  /// returns a double in the range 0 to 1 representing memory usage.
  double mem_fraction();
  /// returns a double in the range 0 to 1 representing swap usage.
  double swap_fraction();
  
  /// clean up, called upon destruction.
  virtual void terminate();

  /// setups SMP info for one CPU only
  void SMP_faker();

  /// in bytes
  unsigned long memory_total;
  /// in bytes
  unsigned long memory_used;
  /// in bytes
  unsigned long memory_free;

  /// in bytes
  unsigned long swap_total;
  /// in bytes
  unsigned long swap_used;
  /// in bytes
  unsigned long swap_free;

  /// avg cpu load, 0 to 1. On one-cpu boxes this is the cpu load,
  /// on SMP systems this is the unweighted average among all CPUs.
  double cpu_usage;

  /// SMP extensions (max 64 CPUs)
  int CpuCount;
  double cpus_load[64];
};

#endif
