/* $Id: rgpsp_linux.cc,v 1.5 2000/02/29 18:04:50 bergo Exp $ */

/*

    GPS/RGPSP - Graphical Process Statistics / Remote GPS Poller
    Copyright (C) 1999 Felipe Paulo Guazzi Bergo
    bergo@seul.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <iostream.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <time.h>
#include <pwd.h>
#include <errno.h>
#include <dirent.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/resource.h>
#include <asm/page.h>

static char *native_id="Linux /proc";

long last_idle,last_busy;

void poller_routine();
char *native_description();
char *tb[32];
char *really_big_buffer;

int lastpid,nprocesses;
time_t btime;
char buffer[1024];
unsigned long mem_used,mem_free,mem_shd,mem_buf,swp_used,swp_free;
float cpu_usage;

/* local */

void actual_poll(int detail);
void get_last_pid();
void poll_memory();
void poll_cpu();
void count_processes();
void poll_process_list();
void poll_detailed_list();
char * power_of_two_suffix(int expon);

void poller_routine() {
  unsigned char x;
  int i,goon;
 
  btime=0;

  last_idle=-1;
  last_busy=-1;

  for(i=0;i<32;i++)
    if ((tb[i]=(char *)malloc(1024))==NULL)
      return;

  really_big_buffer=(char *)malloc(20<<10);
  if (really_big_buffer==NULL) {
    cerr << "** rgpsp ** unable to allocate memory.\n";
  }

  actual_poll(0);

  for(goon=1;goon;) {
    while(read(0,&x,1)<1) usleep(50*1000);
    switch(x) {
    case ' ': actual_poll(0); break; /* normal */
    case  12: actual_poll(1); break; /* details */
    case 'h': actual_poll(2); break; /* cpu&mem header only */
    case 'q': goon=0; break;         /* bye bye */
    }
  }
  for(i=0;i<32;i++) free(tb[i]);
  free(really_big_buffer);
  return;
}

char *native_description() {
  return(native_id);
}

void actual_poll(int detail) {
  char local[256];

  if (detail<2) {
    get_last_pid();
    count_processes();
  } else {
    lastpid=0;
    nprocesses=0;
  }
  poll_memory();
  poll_cpu();

  sprintf(local,"last pid: %d  <linux><special> rgpsp poller ! ! !\n",lastpid);
  strcpy(really_big_buffer,local);

  sprintf(local,"%d processes: no info about the state\n",nprocesses);
  strcat(really_big_buffer,local);

  sprintf(local,
	  "CPU states: %.2f%% user, 0.0%% nice, 0.0%% system, %.2f%% idle\n",
	  100.0*cpu_usage,100.0-100.0*cpu_usage);
  strcat(really_big_buffer,local);

  sprintf(local,
	  "Memory: %uK used, %uK free, %uK shd, %uK bufcch  Swap: %uK used, %uK free\n\n",
	  mem_used>>10,mem_free>>10,mem_shd>>10,
	  mem_buf>>10,swp_used>>10,swp_free>>10);
  strcat(really_big_buffer,local);
  
  sprintf(local,"PID OWNER PRI NICE SIZE RSS STATE TIME WCPU CPU COMMAND OTHERS\n");
  strcat(really_big_buffer,local);

  switch(detail) { /* 0=normal 1=detail 2=header only */
  case 0: poll_process_list(); break;
  case 1: poll_detailed_list(); break;
  }

  printf("%s\n",really_big_buffer);
  fflush(stdout);
}

void get_last_pid() {
  FILE *f;
  /* get boot time and last pid */
  f=fopen("/proc/stat","r");
  if (f==NULL) {
    lastpid=1;
    return;
  }
  while(1) {
    if (fgets(buffer,1024,f)==NULL)
      break;
    if (btime==0)
      if (strstr(buffer,"btime ")!=NULL) {
	strtok(buffer," \n\t");
	btime=atol(strtok(NULL," \n\t"));
      }
    if (strstr(buffer,"processes ")!=NULL) {
      strtok(buffer," \n\t");
      lastpid=atoi(strtok(NULL," \n\t"));
      break;
    }
  } 
  fclose(f);
}

void poll_cpu() {
  FILE *f;
  long idle,busy,da,db;
  double fa,fb;
  char *p;
  int i;

  f=fopen("/proc/stat","r");
  if (f==NULL) {
    cpu_usage=0.0;
    return;
  }

  do 
    p=fgets(buffer,1024,f);
  while((strstr(buffer,"cpu")==NULL)&&(p!=NULL));

  fclose(f);

  if (p==NULL) {
    cpu_usage=0.0;
    return;
  }

  p=strtok(buffer," \n\t");

  busy=0;
  for(i=0;i<3;i++) {
    p=strtok(NULL," \n\t");
    busy+=atol(p);
  }
  idle=atol(strtok(NULL," \n\t"));

  if (last_busy==-1) {
    last_busy=busy;
    last_idle=idle;
    cpu_usage=0.0;
    return;
  }

  if ((idle==last_idle)&&(busy==last_busy)) {
    last_busy=busy;
    last_idle=idle;
    cpu_usage=0.0;
    return;
  }

  da=busy-last_busy;
  db=idle-last_idle;
  fa=(double)da;
  fb=(double)db;
  fa/=fa+fb;
  if (fa>1.0) fa=1.0;
  
  last_busy=busy;
  last_idle=idle;
  cpu_usage=fa;  
}

void poll_memory() {
  FILE *f;
  char *p;

  f=fopen("/proc/meminfo","r");

  if (f==NULL) {
    mem_used=swp_used=0;
    mem_free=swp_free=1;
    return;
  }

  /* MEMORY */
  do 
    p=fgets(buffer,1024,f);
  while((strstr(buffer,"Mem:")==NULL)&&(p!=NULL));

  if (p==NULL) {
    mem_used=swp_used=0;
    mem_free=swp_free=1;
    return;
  }

  strtok(buffer," \n\t");
  strtok(NULL," \n\t"); /* total */
  mem_used=atol(strtok(NULL," \n\t"));
  mem_free=atol(strtok(NULL," \n\t"));
  mem_shd=atol(strtok(NULL," \n\t"));
  mem_buf=atol(strtok(NULL," \n\t")); // buffers
  mem_buf+=atol(strtok(NULL," \n\t")); // and cached

  /* SWAP */
  do 
    p=fgets(buffer,1024,f);
  while((strstr(buffer,"Swap:")==NULL)&&(p!=NULL));
  fclose(f);

  if (p==NULL) {
    swp_used=0;
    swp_free=1;
    return;
  }

  strtok(buffer," \n\t");
  strtok(NULL," \n\t");
  swp_used=atol(strtok(NULL," \n\t"));
  swp_free=atol(strtok(NULL," \n\t"));
}

void count_processes() {
  FILE *f;
  char *p1,*p2;
  DIR *slashproc;
  struct dirent *dentry;
  int i;

  /* open /proc dir */
  slashproc=opendir("/proc");
  if (slashproc==NULL) {
    nprocesses=0;
    return;
  }

  nprocesses=0;
  dentry=readdir(slashproc);
  while(dentry!=NULL) {
    i=atoi(dentry->d_name); /* is process dir ? */
    if (i<=0) {
      dentry=readdir(slashproc);
      continue;
    }
    sprintf(buffer,"/proc/%d/stat",i);
    f=fopen(buffer,"r");
    if (f==NULL) {
      dentry=readdir(slashproc);
      continue;
    }
    fclose(f);
    nprocesses++;
    dentry=readdir(slashproc); /* read next entry */
  } /* while dentry != NULL */    
  closedir(slashproc);
}

void poll_process_list() {
  FILE *f,*g;
  int i,j;
  int mem_unit,yav;
  unsigned long rtime,smem[2];
  unsigned long ulv,dlv,rib;
  double cpufrac;
  time_t ptime;
  char buffercopy[1024],b2[128],b3[512],st[64],local[128];
  char *p1,*p2;
  DIR *slashproc;
  struct dirent *dentry;
  struct passwd *ident;

  /* open /proc dir */
  slashproc=opendir("/proc");
  if (slashproc==NULL) {
    cerr << "** rgpsp: unable to open /proc, cannot read process table\n";
    return;
  }

  for(i=0;i<32;i++)
    tb[i][0]=0;

  dentry=readdir(slashproc);
  while(dentry!=NULL) {
    i=atoi(dentry->d_name); /* is process dir ? */
    if (i<=0) {
      dentry=readdir(slashproc);
      continue;
    }
    
    sprintf(buffer,"/proc/%d/stat",i);
    f=fopen(buffer,"r");
    if (f==NULL) {
      dentry=readdir(slashproc);
      continue;
    }

    /* get owner */
    sprintf(buffer,"/proc/%d/status",i);
    g=fopen(buffer,"r");
    strcpy(tb[0],"???");
    if (g!=NULL) {
      strcpy(tb[0],"???");
      while(1) {
	if (fgets(buffer,1024,g)==NULL)
	  break;
	if (strstr(buffer,"Uid:")==NULL)
	  continue;
	strtok(buffer," \t");
	j=atoi(strtok(NULL," \t"));
	fclose(g);
	ident=getpwuid(j);
	strcpy(tb[0],ident->pw_name);
	break;
      } /* while 1 */
    } /* else */

    fgets(buffer,1024,f);
    strcpy(buffercopy,buffer);
    strtok(buffer," \t\n");

    /* NAME */
    p1=strchr(buffercopy,'(');
    p2=strrchr(buffercopy,')'); /* pattern matching, the hard way */

    if ((p1!=NULL)&&(p2!=NULL)) {
       memset(b2,0,128);
       memcpy(b2,p1+1,(p2-p1)-1);
       strcpy(tb[1],b2);
    } else
	strcpy(tb[1],"<null stat>");

    /* STATE (1/2) */
    strcpy(buffer,p2+1);
    strcpy(st,strtok(buffer," \t\n"));

    for(j=0;j<10;j++)
      strtok(NULL," \t\n"); /* fix later */

    /* CPU */
    ulv=(unsigned long)atol(strtok(NULL," \t\n"));
    ulv+=(unsigned long)atol(strtok(NULL," \t\n"));
    dlv=(unsigned long)atol(strtok(NULL," \t\n"));
    dlv+=(unsigned long)atol(strtok(NULL," \t\n"));

    /* PRIORITY & NICE */
    rib=atoi(strtok(NULL," \t\n"));
    sprintf(b3,"%d",rib);
    strcpy(tb[2],b3);
    rib=atoi(strtok(NULL," \t\n"));
    sprintf(b3,"%d",rib);
    strcpy(tb[3],b3);

    for(j=0;j<2;j++)
      strtok(NULL," \t\n");

    rtime=(unsigned long)atol(strtok(NULL," \t\n"));
    ptime=btime+(time_t)(rtime/100UL);

    strcpy(b3,ctime(&ptime));
    memset(b2,0,128);
    memcpy(b2,b3+11,5);
    strcat(b2," ");
    memcpy(b2+6,b3,11);
    memcpy(b2+17,b3+20,4); // how cute

    strcpy(tb[4],b2);

    dlv=time(NULL)-ptime;
    if (dlv==0)
      dlv=1;
    cpufrac=((double)ulv)/((double)dlv);
    sprintf(b3,"%.2f",cpufrac);
    strcpy(tb[5],b3);

    /* MEM (VSIZE) */
    smem[0]=(unsigned long)atol(strtok(NULL," \t\n"));
    mem_unit=0;
    while(smem[0]>(99UL<<10)) {
      ++mem_unit;
      smem[0]>>=10;
    }
    sprintf(b3,"%lu",smem[0]);
    strcat(b3,power_of_two_suffix(mem_unit));
    strcpy(tb[6],b3);

    /* RSS */
    smem[1]=(unsigned long)atol(strtok(NULL," \t\n"));

    /* STATE (2/2) */
    strcpy(tb[7],"sleep");
    if (st[0]=='R')
      strcpy(tb[7],"run");
    if (st[0]=='Z')
      strcpy(tb[7],"zomb");

    smem[1]<<=PAGE_SHIFT;
    mem_unit=0;
    while(smem[1]>(99UL<<10)) {
      ++mem_unit;
      smem[1]>>=10;
    }
    sprintf(b3,"%lu",smem[1]);
    strcat(b3,power_of_two_suffix(mem_unit));
    strcpy(tb[8],b3);
    fclose(f);

    /* LONG NAMES */
    sprintf(buffer,"/proc/%d/cmdline",i);
    f=fopen(buffer,"r");
    if (f!=NULL) {
      memset(buffer,0,1024);
      yav=fread(buffer,1,1024,f);
      fclose(f);
      for(int yayav=0;yayav<(yav-1);yayav++)
	if (buffer[yayav]==0)
	  buffer[yayav]=0x20;
      if (strlen(buffer)>0)
	strcpy(tb[9],buffer);
      else
	strcpy(tb[9],"unavailable");
    } /* if f != NULL */

    sprintf(local,"%d %s %s %s %s %s %s 0:00 %s%% %s%% %s !%s! %c%s%c\n",
	    i,tb[0],tb[2],tb[3],tb[6],tb[8],
	    tb[7],tb[5],tb[5],tb[1],tb[4],8,tb[9],8);
    strcat(really_big_buffer,local);

    dentry=readdir(slashproc); /* read next entry */
  } /* while dentry != NULL */    

  closedir(slashproc);
}

char * power_of_two_suffix(int expon) {
  static char sarray[16]={'K',0,'M',0,'G',0,'T',0,'P',0,'<','!','>',0};
  int i;

  i=expon;
  if (i==0)
    return(&sarray[1]);
  i=((i%7)-1)<<1;
  return(&sarray[i]);
}

void poll_detailed_list() {
  FILE *f,*g;
  int i,j;
  int mem_unit,yav;
  unsigned long rtime,smem[2];
  unsigned long ulv,dlv,rib;
  double cpufrac;
  time_t ptime;
  char buffercopy[1024],b2[128],b3[512],st[64],local[128];
  char *p1,*p2;
  DIR *slashproc;
  struct dirent *dentry;
  struct passwd *ident;

  /* open /proc dir */
  slashproc=opendir("/proc");
  if (slashproc==NULL) {
    cerr << "** rgpsp: unable to open /proc, cannot read process table\n";
    return;
  }

  for(i=0;i<32;i++)
    tb[i][0]=0;

  dentry=readdir(slashproc);
  while(dentry!=NULL) {
    i=atoi(dentry->d_name); /* is process dir ? */
    if (i<=0) {
      dentry=readdir(slashproc);
      continue;
    }
    
    sprintf(buffer,"/proc/%d/stat",i);
    f=fopen(buffer,"r");
    if (f==NULL) {
      dentry=readdir(slashproc);
      continue;
    }

    /* get owner */
    sprintf(buffer,"/proc/%d/status",i);
    g=fopen(buffer,"r");
    if (g==NULL)
      strcpy(tb[0],"???");
    else {
      while(1) {
	fgets(buffer,1024,g);
	if (strstr(buffer,"Uid:")==NULL)
	  continue;
	strtok(buffer," \t");
	j=atoi(strtok(NULL," \t"));
	fclose(g);
	ident=getpwuid(j);
	strcpy(tb[0],ident->pw_name);
	break;
      } /* while 1 */
    } /* else */

    fgets(buffer,1024,f);
    strcpy(buffercopy,buffer);
    strtok(buffer," \t\n");

    /* NAME */
    p1=strchr(buffercopy,'(');
    p2=strrchr(buffercopy,')'); /* pattern matching, the hard way */

    if ((p1!=NULL)&&(p2!=NULL)) {
      memset(b2,0,128);
      memcpy(b2,p1+1,(p2-p1)-1);
      strcpy(tb[1],b2);
    } else {
      strcpy(tb[1],"<null stat>");
    }

    /* STATE (1/2) */
    strcpy(buffer,p2+1);
    strcpy(st,strtok(buffer," \t\n"));

    /*
      10 PPID
      11 PGID
      12 SID
      13 TPGID
      14 MINFLT
      15 CMINFLT
      16 MAJFLT
      17 CMAJFLT

      18 UTIME
      19 CUTIME
      20 STIME
      21 CSTIME

     */

    /* PPID PGID SID */
    strcpy(tb[10],strtok(NULL," \t\n"));
    strcpy(tb[11],strtok(NULL," \t\n"));
    strcpy(tb[12],strtok(NULL," \t\n"));
    strtok(NULL," \t\n"); // tty
    strcpy(tb[13],strtok(NULL," \t\n"));
    strtok(NULL," \t\n"); // flags
    strcpy(tb[14],strtok(NULL," \t\n"));
    strcpy(tb[15],strtok(NULL," \t\n"));
    strcpy(tb[16],strtok(NULL," \t\n"));
    strcpy(tb[17],strtok(NULL," \t\n"));

    strcpy(tb[18],strtok(NULL," \t\n"));
    strcpy(tb[19],strtok(NULL," \t\n"));
    strcpy(tb[20],strtok(NULL," \t\n"));
    strcpy(tb[21],strtok(NULL," \t\n"));

    /* CPU */
    ulv=(unsigned long)atol(tb[18]);
    ulv+=(unsigned long)atol(tb[19]);
    dlv=(unsigned long)atol(tb[20]);
    dlv+=(unsigned long)atol(tb[21]);

    /* PRIORITY & NICE */
    rib=atoi(strtok(NULL," \t\n"));
    sprintf(b3,"%d",rib);
    strcpy(tb[2],b3);
    rib=atoi(strtok(NULL," \t\n"));
    sprintf(b3,"%d",rib);
    strcpy(tb[3],b3);

    for(j=0;j<2;j++)
      strtok(NULL," \t\n");

    rtime=(unsigned long)atol(strtok(NULL," \t\n"));
    ptime=btime+(time_t)(rtime/100UL);

    strcpy(b3,ctime(&ptime));
    memset(b2,0,128);
    memcpy(b2,b3+11,5);
    strcat(b2," ");
    memcpy(b2+6,b3,11);
    memcpy(b2+17,b3+20,4); // how cute

    strcpy(tb[4],b2);

    dlv=time(NULL)-ptime;
    if (dlv==0)
      dlv=1;
    cpufrac=((double)ulv)/((double)dlv);
    sprintf(b3,"%.2f",cpufrac);
    strcpy(tb[5],b3);

    /* MEM (VSIZE) */
    smem[0]=(unsigned long)atol(strtok(NULL," \t\n"));
    mem_unit=0;
    while(smem[0]>(99UL<<10)) {
      ++mem_unit;
      smem[0]>>=10;
    }
    sprintf(b3,"%lu",smem[0]);
    strcat(b3,power_of_two_suffix(mem_unit));
    strcpy(tb[6],b3);

    /* RSS */
    smem[1]=(unsigned long)atol(strtok(NULL," \t\n"));

    /* STATE (2/2) */
    strcpy(tb[7],"sleep");
    if (st[0]=='R')
      strcpy(tb[7],"run");
    if (st[0]=='Z')
      strcpy(tb[7],"zomb");

    smem[1]<<=PAGE_SHIFT;
    mem_unit=0;
    while(smem[1]>(99UL<<10)) {
      ++mem_unit;
      smem[1]>>=10;
    }
    sprintf(b3,"%lu",smem[1]);
    strcat(b3,power_of_two_suffix(mem_unit));
    strcpy(tb[8],b3);
    fclose(f);

    /* LONG NAMES */
    sprintf(buffer,"/proc/%d/cmdline",i);
    f=fopen(buffer,"r");
    if (f!=NULL) {
      memset(buffer,0,1024);
      yav=fread(buffer,1,1024,f);
      fclose(f);
      for(int yayav=0;yayav<(yav-1);yayav++)
	if (buffer[yayav]==0)
	  buffer[yayav]=0x20;
      if (strlen(buffer)>0)
	strcpy(tb[9],buffer);
      else
	strcpy(tb[9],"unavailable");
    } /* if f != NULL */

    sprintf(local,"%d %s %s %s %s %s %s 0:00 %s%% %s%% %s ",
	    i,tb[0],tb[2],tb[3],tb[6],tb[8],
	    tb[7],tb[5],tb[5],tb[1]);
    strcat(really_big_buffer,local);
    /* won't work if the long name contains backspaces. But if it does,
       you deserve it! */
    sprintf(local,"!%s! %c%s%c %s %s %s %s %s %s %s %s %s %s %s %s\n",
	    tb[4],8,tb[9],8,tb[10],tb[11],tb[12],tb[13],
	    tb[14],tb[15],tb[16],tb[17],
	    tb[18],tb[19],tb[20],tb[21]);
    strcat(really_big_buffer,local);

    dentry=readdir(slashproc); /* read next entry */
  } /* while dentry != NULL */    

  closedir(slashproc);
}
