/*
 * $Id: server-config.c,v 0.75 2002/03/29 22:29:42 ceder Exp $
 * Copyright (C) 1991-1999, 2001-2002  Lysator Academic Computer Association.
 *
 * This file is part of the LysKOM server.
 * 
 * LysKOM is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License as published by 
 * the Free Software Foundation; either version 1, or (at your option) 
 * any later version.
 * 
 * LysKOM is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with LysKOM; see the file COPYING.  If not, write to
 * Lysator, c/o ISY, Linkoping University, S-581 83 Linkoping, SWEDEN,
 * or the Free Software Foundation, Inc., 675 Mass Ave, Cambridge, 
 * MA 02139, USA.
 *
 * Please mail bug reports to bug-lyskom@lysator.liu.se. 
 */
/*
 *  server-config.c
 *
 *  This is in a .c file to make it possible to change a value without having
 *  to recompile the entire server (or, in fact, anything!)
 *
 *  Compile with -DDEFAULT_PREFIX='"/usr/lyskom"' or something similar.
 */


#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <limits.h>
#include <stdio.h>
#include <sys/types.h>
#ifdef HAVE_STRING_H
#  include <string.h>
#endif
#include <time.h>
#include <assert.h>
#include <setjmp.h>

#include "server/smalloc.h"
#include "kom-config.h"
#include "kom-types.h"
#include "com.h"
#include "async.h"
#include "connections.h"
#include "manipulate.h"
#include "server-config.h"
#include "misc-types.h"
#include "s-string.h"
#include "kom-types.h"
#include "param.h"
#include "conf-file.h"
#include "admin.h"
#include "log.h"
#include "lyskomd.h"
#include "unused.h"

struct kom_par param;
Bool reread_param;
char *read_config_file;

static Success log_param(const char *val, const struct parameter *par);
static Success jubel(const char *val, const struct parameter *par);
static Success ident_param(const char *val, const struct parameter *par);

/* See lyskomd.texi for more info about the parameters.
   Please remember to update lyskomd.texi if you add more parameters!
   Try to keep this list and the list in lyskomd.texi in the same order. */
static const struct parameter parameters[] = {

    /* "Normal" configuration */

    {"Locale",
	 assign_string,  unassign_string,  0, 1, NULL, &param.use_locale},
    {"Force ISO 8859-1",
         assign_bool,    NULL,             0, 1, "no", &param.force_iso_8859_1},
    {"Prefix",
	 assign_string,  unassign_string,  0, 1, DEFAULT_PREFIX, &param.dbase_dir},
    {"Send async",
  	 assign_bool,    NULL,             0, 1, "1",  &param.send_async_messages},
    {"Client host",
	 assign_string,  unassign_string,  0, 1, NULL, &param.ip_client_host},
    {"Client port",
	 assign_string,  unassign_string,  1, 1, NULL, &param.ip_client_port},
    {"Presentation of conferences",
	 assign_conf_no, NULL,             0, 1, "1", &kom_info.conf_pres_conf},
    {"Presentation of persons",
	 assign_conf_no, NULL,             0, 1, "2", &kom_info.pers_pres_conf},
    {"Motd-conference",
	 assign_conf_no, NULL,             0, 1, "3", &kom_info.motd_conf},
    {"News-conference", 
	 assign_conf_no, NULL,             0, 1, "4", &kom_info.kom_news_conf},
    {"Message of the day",
	 assign_text_no, NULL,             0, 1, "0", &kom_info.motd_of_lyskom},
    {"Garb",
	 assign_bool,    NULL,             0, 1, "on", &param.garb_enable},
    {"Never save",
	 assign_bool,    NULL,             0, 1, "no", &param.never_save},
#ifdef LOGACCESSES
    {"Log accesses", 
	 assign_string,  unassign_string,  0, 1, NULL,
         &param.logaccess_file},
#endif

    /* The database files. */

    {"Data file",
	 assign_string,  unassign_string,  0, 1, "db/lyskomd-data",
         &param.datafile_name},
    {"Backup file",
	 assign_string,  unassign_string,  0, 1, "db/lyskomd-backup",
         &param.backupfile_name},
    {"Backup file 2",
	 assign_string,  unassign_string,  0, 1, "db/lyskomd-backup-prev",
         &param.backupfile_name_2},
    {"Lock file",
	 assign_string,  unassign_string,  0, 1, "db/lyskomd-lock",
         &param.lockfile_name},
    {"Text file",
	 assign_string,  unassign_string,  0, 1, "db/lyskomd-texts",
         &param.textfile_name},
    {"Text backup file",
	 assign_string,  unassign_string,  0, 1, "db/lyskomd-texts-backup",
         &param.textbackupfile_name},
    {"Backup export directory",
	 assign_string,  unassign_string,  0, 1, "exportdb",
         &param.backup_dir},

    /* Various log files */

    {"Log file",
	 assign_string,  unassign_string, 0, 1, "etc/server-log",
         &param.logfile_name},
    {"Log statistics",
	 assign_string,  unassign_string,  0, 1, "etc/lyskomd-log",
         &param.statistic_name},
    {"Pid file",
	 assign_string,  unassign_string,  0, 1, "etc/pid",
         &param.pid_name},
    {"Memory usage file",
	 assign_string,  unassign_string,  0, 1, "etc/memory-usage",
         &param.memuse_name},

    /* Other files. */

    {"Aux-item definition file",
	 assign_string,  unassign_string,  0, 1, "etc/aux-items.conf",
         &param.aux_def_file},
    {"Status file",
         assign_string,  unassign_string,  0, 1, "etc/status",
         &param.status_file},

    /* Where to dump core. */

    {"Core directory",
	 assign_string,  unassign_string,  0, 1, "cores",
         &param.core_dir},
    {"Nologin file",
         assign_string,  unassign_string,  0, 1, "/etc/nologin",
         &param.nologin_file},

    /* Stuff */

    { "Y2K Compatibility",
         assign_int,     NULL,             0, 1, "2", &param.y2k_compat },


    /* Performance tuning parameters (milliseconds) */

    {"Idle timeout",
	 assign_int,     NULL,             0, 1, "120000", &param.timeout},
    {"Garb timeout",
	 assign_int,     NULL,             0, 1, "100", &param.garbtimeout},
    {"Sync timeout",
	 assign_int,     NULL,             0, 1, "0", &param.synctimeout},

    /* Performance tuning parameters (minutes) */

    {"Garb interval",
	 assign_int,     NULL,             0, 1, "1440", &param.garb_interval},
    {"Permissive sync",
         assign_bool,    NULL,             0, 1, "off", &param.permissive_sync},
    {"Sync interval",
	 assign_int,     NULL,             0, 1, "5", &param.sync_interval},
    
    {"Sync retry interval",
	 assign_int,     NULL,             0, 1, "1",
         &param.sync_retry_interval},

    {"Saved items per call",
         assign_int,     NULL,             0, 1, "5",
         &param.saved_items_per_call},


    /* More performance tuning. */

    {"Max client transmit queue",
	 assign_int,     NULL,             0, 1, "300", &param.maxqueuedsize},
    {"Max simultaneous client replies",
	 assign_int,     NULL,             0, 1, "10", &param.maxdequeuelen},
    {"Open files",
         assign_int,     NULL,             0, 1, "-1", &param.no_files},

    /* String limits */

    {"Max conference name length",
	 assign_int,     NULL,             0, 1, "60", &param.conf_name_len},
    {"Max client data length",
         assign_int,     NULL,             0, 1, "60", &param.client_data_len},
    {"Max password length",
	 assign_int,     NULL,             0, 1, "128", &param.pwd_len},
    {"Max what am I doing length",
	 assign_int,     NULL,             0, 1, "60", &param.what_do_len},
    {"Max username length",
	 assign_int,     NULL,             0, 1, "128", &param.username_len},
    {"Max text length",
	 assign_int,     NULL,             0, 1, "131072", &param.text_len},
    {"Max aux_item length",
	 assign_int,     NULL,             0, 1, "16384", &param.aux_len},
    {"Max broadcast length",
	 assign_int,     NULL,             0, 1, "1024", &param.broadcast_len},
    {"Max regexp length",
	 assign_int,     NULL,             0, 1, "1024", &param.regexp_len},

    /* Text_stat limits */

    {"Max marks per person",
	 assign_int,     NULL,             0, 1, "2048", &param.max_marks_person},
    {"Max marks per text",
	 assign_int,     NULL,             0, 1, "1024", &param.max_marks_text},
    {"Max recipients per text",
	 assign_int,     NULL,             0, 1, "512", &param.max_recipients},
    {"Max comments per text",
	 assign_int,     NULL,             0, 1, "128", &param.max_comm},
    {"Max footnotes per text",
	 assign_int,     NULL,             0, 1, "32", &param.max_foot},
    {"Max links per text",
	 assign_int,     NULL,             0, 1, "512", &param.max_crea_misc},

    /* Other client-visible configuration */

    {"Max mark_as_read chunks",
	 assign_int,     NULL,             0, 1, "128", &param.mark_as_read_chunk},
    {"Max accept_async len",
	 assign_int,     NULL,             0, 1, "128", &param.accept_async_len},
    {"Max aux_items added per call",
         assign_int,     NULL,             0, 1, "128", &param.max_add_aux},
    {"Max aux_items deleted per call",
         assign_int,     NULL,             0, 1, "128", &param.max_delete_aux},
    {"Max super_conf loop",
	 assign_int,     NULL,             0, 1, "17", &param.max_super_conf_loop},
    {"Default garb nice",
	 assign_int,     NULL,             0, 1, "77", &param.default_nice},
    {"Default keep commented nice",
	 assign_int,     NULL,             0, 1, "77", &param.default_keep_commented},

    /* Security options */

    {"Anyone can create new persons",
	 assign_bool,    NULL,             0, 1, "yes", &param.anyone_can_create_new_persons},
    {"Anyone can create new conferences",
	 assign_bool,    NULL,             0, 1, "yes", &param.anyone_can_create_new_confs},
    {"Allow creation of persons before login",
	 assign_bool,    NULL,             0, 1, "yes", &param.create_person_before_login},
    {"Default change name capability",
	 assign_bool,    NULL,             0, 1, "on", &param.default_change_name},
    {"Add members by invitation",
        assign_bool,     NULL,             0, 1, "on", &param.invite_by_default},
    {"Allow secret memberships",
        assign_bool,     NULL,             0, 1, "on", &param.secret_memberships},
    {"Allow reinvitations",
        assign_bool,     NULL,             0, 1, "off", &param.allow_reinvite},
    {"Log login",
	 assign_bool,    NULL,             0, 1, "off", &param.log_login},
    {"Ident-authentication",
	ident_param,     NULL,             0, 1, "try", &param.authentication_level},

    /* Cache configuration */

    {"Cache conference limit",
	 assign_int,     NULL,             0, 1, "20", &param.cache_conferences},
    {"Cache person limit",
	 assign_int,     NULL,             0, 1, "20", &param.cache_persons},
    {"Cache text_stat limit",
	 assign_int,     NULL,             0, 1, "20", &param.cache_text_stats},

    /* Echo the value to the log. */

    {"Echo",
	 log_param,      NULL,             0, -1, NULL, NULL},

    /* Register a forbidden text number. */

    {"Jubel",
         jubel,          NULL,             0, -1, NULL, NULL},

    {"Max conferences",
        assign_ulong,    NULL,             1, 1, "4765", &param.max_conf},
    {"Max texts",
        assign_ulong,    NULL,             1, 1, "2000000", &param.max_text},

    /* Configuration for support programs.  */

    {"Normal shutdown time",
	 assign_int,     NULL,             0, 1, "21", &param.normal_shutdown_time},
    {"Mail after downtime",
	 assign_int,     NULL,             0, 1, "60", &param.downtime_mail_start},
    {"Mail until downtime",
	 assign_int,     NULL,             0, 1, "120", &param.downtime_mail_end},
    {"lyskomd path",
	 assign_string,  unassign_string,  0, 1, "bin/lyskomd", &param.lyskomd_path},
    {"savecore path",
	 assign_string,  unassign_string,  0, 1, "bin/savecore", &param.savecore_path},
    
    /* end marker */

    {NULL, NULL, NULL, 0, 0, NULL, NULL}};

/* Where to save things. These are used by lyskomd and dbck. */

/*
 * DEFAULT_DBASE_DIR can be overriden in the config file.  The default 
 * config file is found in DEFAULT_DBASE_DIR/CONFIG_FILE (before
 * DEFAULT_DBASE_DIR is overriden, of course).
 */

const char * DEFAULT_DBASE_DIR = DEFAULT_PREFIX;
const char *CONFIG_FILE = "etc/config";

/*
 * The following should always be true:
 * 0 <= SYNCTIMEOUT <= GARBTIMEOUT <= TIMEOUT
 * Times in milliseconds.
 */

/*
 * MAX_NO_OF_CONNECTIONS must be small enough.  See comment in kom-config.h.
 */

#if defined(HAVE_GETDTABLESIZE) || defined(HAVE_SYSCONF)
int MAX_NO_OF_CONNECTIONS = 0;	/* Initialized by main(). */
#else
const int MAX_NO_OF_CONNECTIONS = (OPEN_MAX - PROTECTED_FDS);
#endif

/* What is whitespace? */
const unsigned char *WHITESPACE = (const unsigned char *)" \t\n\r";

static Success
log_param(const char *val, const struct parameter *UNUSED(par))
{
    if (val != NULL)
	kom_log ("config: %s\n", val);
    return OK;
}

static Success
jubel(const char *val, const struct parameter *par)
{
    long a, b, c;
    int res;
    Bool public = FALSE;
    
    if (val == NULL)
        return OK;

    if (!strncmp(val, "public ", 7))
    {
	public = TRUE;
	val += 7;
    }

    res = sscanf(val, "%ld %ld %ld", &a, &b, &c);
    switch (res)
    {
    case 3:
	register_jubel(a, b, c, public);
	break;
    case 2:
	register_jubel(a, 0, b, public);
	break;
    default:
	kom_log("%s expecting [public ] x y [z]\n", par->name);
	return  FAILURE;
    }
    return OK;
}

static Success
ident_param(const char *val, const struct parameter *par)
{
    if (val == NULL)
	restart_kom("ident_param(): val == NULL\n");
    if (!strcmp(val, "off")
	|| !strcmp(val, "never"))
    {
	*(int*)par->value = 0;
    }
    else if (!strcmp(val, "on")
	     || !strcmp(val, "try"))
    {
	*(int*)par->value = 1;
    }
    else if (!strcmp(val, "require")
	     || !strcmp(val, "required"))
    {
	*(int*)par->value = 2;
    }
    else
    {
	kom_log ("%s expects \"never\", \"try\" or \"required\" as argument\n",
	     par->name);
	return FAILURE;
    }
    return OK;
}

static void
add_prefix(char **name)
{
    char *s;

    if (**name == '/')
	return;			/* Don't alter full paths. */

    s = smalloc(2 + strlen(param.dbase_dir) + strlen(*name));
    sprintf(s, "%s/%s", param.dbase_dir, *name);
    sfree(*name);
    *name = s;
}

static const char *
param_name(void *value)
{
    int ix;

    for (ix = 0; parameters[ix].name != NULL; ix++)
	if (parameters[ix].value == value)
	    return parameters[ix].name;

    restart_kom("Internal error: non-existing config param in param_name.\n");
    /* notreached */
    return NULL;
}

static Bool
check_abs_path(char **path)
{
    if (**path == '/')
	return FALSE;
	
    kom_log("Parameter '%s' must be an absolute path when 'Prefix' is empty.\n",
	param_name(path));
    return TRUE;
}

void
read_configuration(const char *conf_file)
{
    Bool err = FALSE;

    read_config(conf_file, parameters);
    
    assert(param.dbase_dir != NULL);
    assert(param.datafile_name != NULL);
    assert(param.backupfile_name != NULL);
    assert(param.backupfile_name_2 != NULL);
    assert(param.lockfile_name != NULL);
    assert(param.textfile_name != NULL);
    assert(param.textbackupfile_name != NULL);
    assert(param.backup_dir != NULL);
    assert(param.statistic_name != NULL);
    assert(param.pid_name != NULL);
    assert(param.memuse_name != NULL);
    assert(param.logfile_name != NULL);
    assert(param.aux_def_file != NULL);
    assert(param.status_file != NULL);
    assert(param.core_dir != NULL);
    assert(param.lyskomd_path != NULL);
    assert(param.savecore_path != NULL);

    if (strlen(param.dbase_dir) > 0) 
    {
	if (param.dbase_dir[0] != '/')
	{
	    kom_log("The 'Prefix' parameter must be an absolute path.\n");
	    err = TRUE;
	}

	add_prefix(&param.datafile_name);
	add_prefix(&param.backupfile_name);
	add_prefix(&param.backupfile_name_2);
	add_prefix(&param.lockfile_name);
	add_prefix(&param.textfile_name);
	add_prefix(&param.textbackupfile_name);
	add_prefix(&param.backup_dir);
	add_prefix(&param.statistic_name);
	add_prefix(&param.pid_name);
	add_prefix(&param.memuse_name);
	add_prefix(&param.logfile_name);
	add_prefix(&param.aux_def_file);
	add_prefix(&param.status_file);
	add_prefix(&param.core_dir);
	add_prefix(&param.lyskomd_path);
	add_prefix(&param.savecore_path);
    }
    else
    {
	err |= check_abs_path(&param.datafile_name);
	err |= check_abs_path(&param.backupfile_name);
	err |= check_abs_path(&param.backupfile_name_2);
	err |= check_abs_path(&param.lockfile_name);
	err |= check_abs_path(&param.textfile_name);
	err |= check_abs_path(&param.textbackupfile_name);
	err |= check_abs_path(&param.backup_dir);
	err |= check_abs_path(&param.statistic_name);
	err |= check_abs_path(&param.pid_name);
	err |= check_abs_path(&param.memuse_name);
	err |= check_abs_path(&param.logfile_name);
	err |= check_abs_path(&param.aux_def_file);
	err |= check_abs_path(&param.status_file);
	err |= check_abs_path(&param.core_dir);
	err |= check_abs_path(&param.lyskomd_path);
	err |= check_abs_path(&param.savecore_path);
    }

    if (param.saved_items_per_call < 1)
    {
	err = TRUE;
	kom_log("Parameter '%s' must be at least 1.\n",
		param_name(&param.saved_items_per_call));
    }

    /* FIXME (bug 165): Check config parameters for sanity */

    if (err)
	restart_kom("Please fix the above configuration errors.\n");
}

void
free_configuration(void)
{
    int i = 0;

    while (parameters[i].name != NULL)
    {
        if (parameters[i].freer != NULL)
        {
            (*parameters[i].freer)(&parameters[i]);
        }
        i += 1;
    }
}
