/*!---------------------------------------------------------------------
  @file           RTESys_SharedLib.cpp
  @author         JoergM
  @brief          System: Base class for loading DLL during runtime
  @see            

\if EMIT_LICENCE

    ========== licence begin  GPL
    Copyright (c) 2000-2004 SAP AG

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
    ========== licence end


\endif
---------------------------------------------------------------------*/



/*===========================================================================*
 *  INCLUDES                                                                 *
 *===========================================================================*/
#include "RunTime/System/RTESys_SharedLib.h"

/*===========================================================================*
 *  DEFINES                                                                  *
 *===========================================================================*/

#define MAX_ERROR_TEXT_SIZE 4096

/*===========================================================================*
 *  MACROS                                                                   *
 *===========================================================================*/


/*===========================================================================*
 *  CLASSES, STRUCTURES, TYPES, UNIONS ...                                   *
 *===========================================================================*/

SAPDB_Bool RTESys_SharedLib::LoadSharedLib( 
  SAPDB_Char const *sharedLibName,
  RTE_LoadMode loadMode,
  SAPDBErr_MessageList &errList)
{
    SAPDB_Bool callResult;
    SAPDB_Char errorText[MAX_ERROR_TEXT_SIZE];

    callResult = RTESys_LoadSharedLibrary( sharedLibName, 
                                           loadMode,
                                           &errorText[0],
                                           (SAPDB_Int4)sizeof(errorText),
                                           m_FullSharedLibName,
                                           m_LibHandle);

    if ( false == callResult )
    {
        errList = SAPDBErr_MessageList(RTE_CONTEXT, RTEERR_SYS_LOAD_SHARED_LIB, sharedLibName, &errorText[0]);
    }

    return callResult;
}

/*---------------------------------*/

SAPDB_Bool RTESys_SharedLib::UnloadSharedLib(SAPDBErr_MessageList *pErrList)
{
    if ( 0 == m_LibHandle )
    {
        if ( 0 != pErrList )
        {
           *pErrList = SAPDBErr_MessageList(RTE_CONTEXT, RTEINFO_SYS_CALL_DLL_FUNC, "Unload: library never loaded");
        }
        return false;
    }

    SAPDB_Bool callResult;
    SAPDB_Char errorText[MAX_ERROR_TEXT_SIZE];

    callResult = RTESys_UnloadSharedLibrary( m_LibHandle, 
                                             &errorText[0],
                                             (SAPDB_Int4)sizeof(errorText) );

    if ( false == callResult )
    {
        if ( 0 != pErrList )
        {
            *pErrList = SAPDBErr_MessageList(RTE_CONTEXT, RTEERR_SYS_UNLOAD_SHARED_LIB, m_FullSharedLibName, &errorText[0]);
        }
    }
    else
    {
        m_LibHandle = 0;
    }

    return callResult;
}

/*---------------------------------*/

SAPDB_Bool RTESys_SharedLib::GetProcAddress( 
  SAPDB_Char const * symbolicName,
  RTE_SharedLibProcAddress & procAddress,
  SAPDBErr_MessageList & errList)
{
    if ( 0 == m_LibHandle )
    {
        errList = SAPDBErr_MessageList(RTE_CONTEXT, RTEINFO_SYS_CALL_DLL_FUNC, "GetProcAddress: library never loaded");
        return false;
    }

    SAPDB_Bool callResult;
    SAPDB_Char errorText[MAX_ERROR_TEXT_SIZE];

    callResult = RTESys_GetProcAddressFromSharedLibrary( m_LibHandle,
                                                         symbolicName,
                                                         &errorText[0],
                                                         (SAPDB_Int4)sizeof(errorText),
                                                         procAddress );

    if ( false == callResult )
    {
        errList = SAPDBErr_MessageList(RTE_CONTEXT, RTEERR_SYS_GET_PROC_ADDRESS, symbolicName, m_FullSharedLibName, &errorText[0]);
    }

    return callResult;
}

/*===========================================================================*
 *  END OF CODE                                                              *
 *===========================================================================*/
