/*************************************************************************
 *
 *  $RCSfile: rmbitmap.cxx,v $
 *
 *  $Revision: 1.4 $
 *
 *  last change: $Author: ssa $ $Date: 2001/05/18 06:56:19 $
 *
 *  The Contents of this file are made available subject to the terms of
 *  either of the following licenses
 *
 *         - GNU Lesser General Public License Version 2.1
 *         - Sun Industry Standards Source License Version 1.1
 *
 *  Sun Microsystems Inc., October, 2000
 *
 *  GNU Lesser General Public License Version 2.1
 *  =============================================
 *  Copyright 2000 by Sun Microsystems, Inc.
 *  901 San Antonio Road, Palo Alto, CA 94303, USA
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License version 2.1, as published by the Free Software Foundation.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 *  MA  02111-1307  USA
 *
 *
 *  Sun Industry Standards Source License Version 1.1
 *  =================================================
 *  The contents of this file are subject to the Sun Industry Standards
 *  Source License Version 1.1 (the "License"); You may not use this file
 *  except in compliance with the License. You may obtain a copy of the
 *  License at http://www.openoffice.org/license.html.
 *
 *  Software provided under this License is provided on an "AS IS" basis,
 *  WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING,
 *  WITHOUT LIMITATION, WARRANTIES THAT THE SOFTWARE IS FREE OF DEFECTS,
 *  MERCHANTABLE, FIT FOR A PARTICULAR PURPOSE, OR NON-INFRINGING.
 *  See the License for the specific provisions governing your rights and
 *  obligations concerning the Software.
 *
 *  The Initial Developer of the Original Code is: Sun Microsystems, Inc.
 *
 *  Copyright: 2000 by Sun Microsystems, Inc.
 *
 *  All Rights Reserved.
 *
 *  Contributor(s): _______________________________________
 *
 *
 ************************************************************************/

#define _SV_RMBITMAP_CXX

#include <rtl/memory.h>
#include <osl/diagnose.h>
#include <vos/mutex.hxx>
#include <vos/xception.hxx>
#include <cppuhelper/extract.hxx>
#include <tools/stream.hxx>
#include <tools/zcodec.hxx>

#include <svdata.hxx>
#include <bitmap.hxx>
#include <bitmapex.hxx>
#include <bmpacc.hxx>
#include <outdev.hxx>
#include <svapp.hxx>
#include <impbmp.hxx>
#include <indbmp.hxx>
#include <rmbitmap.hxx>
#include <rmoutdev.hxx>
#include <salbtype.hxx>
#include <rmcache.hxx>

using namespace ::com::sun::star::uno;

// -----------
// - Defines -
// -----------

#define GLOBAL_BITMAP_CACHE_MAXSIZE 100

// ---------------------
// - RemoteBitmapCache -
// ---------------------

// Bitmap cache:
// Reuse the bitmap objects created on client side, because
// creating new objects forces two synchron UNO calls
//
// The cache is organized as single linked list. The max size
// of the cache is defined by GLOBAL_BITMAP_CACHE_MAXSIZE.
//
// Each bitmap is cleared by calling Transfer(Sequence<BYTE>())
// to free memory on client side.

// -----------
// - Statics -
// -----------

static ::vcl::InterfacePairCache< ::com::sun::star::portal::client::XRmBitmap, ::com::sun::star::portal::client::XRmJavaBitmap >* pRemoteBitmapCache = NULL;

typedef ::std::pair< ::com::sun::star::uno::Reference< ::com::sun::star::portal::client::XRmBitmap >, ::com::sun::star::uno::Reference< ::com::sun::star::portal::client::XRmJavaBitmap > > bitmapInterfacePair;

// ----------------------------
// - createRemoteBitmapCache() -
// ----------------------------

void createRemoteBitmapCache( const ::com::sun::star::uno::Sequence< ::com::sun::star::uno::Any >& aInterfaceSeq )
{
	ImplSVData* pSVData = ImplGetSVData();

	if( ! pRemoteBitmapCache )
	{
		pRemoteBitmapCache = new ::vcl::InterfacePairCache< ::com::sun::star::portal::client::XRmBitmap, ::com::sun::star::portal::client::XRmJavaBitmap >(
			pSVData->mxMultiFactory,
			aInterfaceSeq,
			::rtl::OUString::createFromAscii( "OfficeBitmap.stardiv.de" ),
			10, GLOBAL_BITMAP_CACHE_MAXSIZE );
	}
}

// ----------------------------
// - eraseRemoteBitmapCache() -
// ----------------------------

void eraseRemoteBitmapCache()
{
	if ( pRemoteBitmapCache )
	{
		delete pRemoteBitmapCache;
		pRemoteBitmapCache = NULL;
	}
}

// ------------
// - RMBitmap -
// ------------

RMBitmap::RMBitmap( Bitmap* pBmp ) :
	mpBmp				( pBmp ),
	mnGetBitCount		( 0 ),
	mpReadAccess		( NULL ),
	mpCompressor		( NULL ),
	mbGetPrepared		( FALSE )
{
	ImplInit();
}

// -----------------------------------------------------------------------------

void RMBitmap::ImplInit()
{
	bitmapInterfacePair aPair = pRemoteBitmapCache->takeInterface();
	mxRemoteBitmap		= aPair.first;
	mxJavaRemoteBitmap	= aPair.second;
}

// ------------------------------------------------------------------------

RMBitmap::~RMBitmap()
{
	if ( mxJavaRemoteBitmap.is() || mxRemoteBitmap.is() )
	{
		Sequence< sal_Int8> aSeq;

		if ( mxJavaRemoteBitmap.is() )
		{
			CHECK_FOR_RVPSYNC_NORMAL();
            try
            {
			    mxJavaRemoteBitmap->InitiateTransfer( 0, 0, 0, aSeq, 0, 0, 0 );
            }
            catch( RuntimeException &e )
            {
                rvpExceptionHandler();
            }
		}
		if ( mxRemoteBitmap.is() )
		{
			CHECK_FOR_RVPSYNC_NORMAL();
            try
            {
			    mxRemoteBitmap->InitiateTransfer( 0, 0, 0, aSeq, 0, 0, 0 );
            }
            catch( RuntimeException &e )
            {
                rvpExceptionHandler();
            }
		}
		if ( pRemoteBitmapCache )
		{
			bitmapInterfacePair aPair( mxRemoteBitmap, mxJavaRemoteBitmap );
			pRemoteBitmapCache->putInterface( aPair );
		}
	}
}

// ------------------------------------------------------------------------

BOOL RMBitmap::CreateImpl()
{
	if ( mxJavaRemoteBitmap.is() )
		ImplTransferRawData();
	else
	{
		Sequence< sal_Int8 > aPalette;
		if ( !mpReadAccess )
			mpReadAccess = mpBmp->AcquireReadAccess();

		if( mpReadAccess->HasPalette() )
		{
			const BitmapPalette& rPalette = mpReadAccess->GetPalette();
			// this depends on internal data of BitmapPalette
			aPalette = Sequence< sal_Int8 >( (sal_Int8*)(&rPalette[0]), 4 * rPalette.GetEntryCount() );
		}
		maSize = Size( mpReadAccess->Width(), mpReadAccess->Height() );
		if( mxJavaRemoteBitmap.is() )
		{
			CHECK_FOR_RVPSYNC_NORMAL();
            try
            {
			    mxJavaRemoteBitmap->InitiateTransfer(
				    mpReadAccess->Width(),
				    mpReadAccess->Height(),
				    mpReadAccess->GetBitCount(),
					aPalette,
				    mpReadAccess->GetScanlineFormat(),
				    mpReadAccess->GetScanlineSize(),
				    1 );
            }
            catch( RuntimeException &e )
            {
                rvpExceptionHandler();
            }
		}
		else if( mxRemoteBitmap.is() )
		{
			CHECK_FOR_RVPSYNC_NORMAL();
            try
            {
			    mxRemoteBitmap->InitiateTransfer(
				    mpReadAccess->Width(),
				    mpReadAccess->Height(),
				    mpReadAccess->GetBitCount(),
					aPalette,
				    mpReadAccess->GetScanlineFormat(),
				    mpReadAccess->GetScanlineSize(),
				    1 );
            }
            catch( RuntimeException &e )
            {
                rvpExceptionHandler();
            }
		}
		int nMaxBytes = mpReadAccess->GetScanlineSize();
		nMaxBytes = nMaxBytes < 0x4000 ? 0x4000 : nMaxBytes;
		mpCompressor = new ZCodec( nMaxBytes, nMaxBytes );
	}

	return TRUE;
}

// ------------------------------------------------------------------------

void RMBitmap::Create()
{
	CreateImpl();
}

// ------------------------------------------------------------------------

long RMBitmap::TransferLines( long nStartLine, BOOL bAll )
{
	long nLines = 0;
	long nLineSize = mpReadAccess->GetScanlineSize();

	// send packets of approximately 8 k size
	long nPacketLines = 8192 / nLineSize;
	if ( nPacketLines < 1 )
		nPacketLines = 1;
	else if ( nPacketLines > mpReadAccess->Height() )
		nPacketLines = mpReadAccess->Height();

	SvMemoryStream aOut;
	do
	{
		// resize for last block
		if( nStartLine + nLines + nPacketLines > mpReadAccess->Height() )
			nPacketLines = mpReadAccess->Height() - nStartLine - nLines;

		aOut.Seek( 0 );
		mpCompressor->BeginCompression( 3 );
		for( int i = 0; i < nPacketLines; i++ )
			mpCompressor->Write( aOut, mpReadAccess->GetScanline( nStartLine + nLines + i ), nLineSize );
		mpCompressor->EndCompression();

		int nPos = aOut.Tell();
		aOut.Seek( 0 );

		if( mxJavaRemoteBitmap.is() )
		{
			CHECK_FOR_RVPSYNC_NORMAL();
            try
            {
			    mxJavaRemoteBitmap->Transfer( Sequence< sal_Int8 >( (const sal_Int8*)(const void*)aOut, nPos ) );
            }
            catch( RuntimeException &e )
            {
                rvpExceptionHandler();
            }
		}
		else if( mxRemoteBitmap.is() )
		{
			CHECK_FOR_RVPSYNC_NORMAL();
            try
            {
			    mxRemoteBitmap->Transfer( Sequence< sal_Int8 >( (const sal_Int8*)(const void*)aOut, nPos ) );
            }
            catch( RuntimeException &e )
            {
                rvpExceptionHandler();
            }
		}

		nLines += nPacketLines;
	}
	while( bAll && ((nLines + nStartLine) < mpReadAccess->Height()) );

	// set mpBmp to 0 here because the bitmap can be released
	// after transfer. never touch mpBmp after transfer.
	if ( nLines + nStartLine >= mpReadAccess->Height() )
	{
		mpBmp->ReleaseAccess( mpReadAccess ), mpReadAccess = NULL, mpBmp = NULL;
		delete mpCompressor, mpCompressor = NULL;
	}

	return nLines;
}

// ------------------------------------------------------------------------

void RMBitmap::CreateGet( OutputDevice* pOut, const Point& rPt, const Size& rSz )
{
	if ( mxJavaRemoteBitmap.is() )
	{
		CHECK_FOR_RVPSYNC_NORMAL();
        try
        {
		    mxJavaRemoteBitmap->CreateBitmap( rPt.X(), rPt.Y(), rSz.Width(), rSz.Height(), pOut->mpGraphics->GetInterface() );
        }
        catch( RuntimeException &e )
        {
            rvpExceptionHandler();
        }
	}
	else if ( mxRemoteBitmap.is() )
	{
		CHECK_FOR_RVPSYNC_NORMAL();
        try
        {
		    mxRemoteBitmap->CreateBitmap( rPt.X(), rPt.Y(), rSz.Width(), rSz.Height(), pOut->mpGraphics->GetInterface() );
        }
        catch( RuntimeException &e )
        {
            rvpExceptionHandler();
        }
	}

	maSize = rSz;
	mnGetBitCount = pOut->GetBitCount();
	mnGetBitCount = ( mnGetBitCount <= 1 ) ? 1 : ( mnGetBitCount <= 4 ) ? 4 : ( mnGetBitCount <= 8 ) ? 8 : 24;
	mbGetPrepared = TRUE;
}

// ------------------------------------------------------------------------

BOOL RMBitmap::DrawImpl( OutputDevice* pOut,
						 const Point& rSrcPt, const Size& rSrcSz,
						 const Point& rDestPt, const Size& rDestSz )
{
	// Draw the bitmap in one piece, if the destination output device
	// is not a window and if the bitmap is not scaled, because
	// we doesn't know, how the client stretch the bitmap and if
	// the bitmaps isn't transferd yet
	// We also tranfer the bitmap without slices, when the client
	// is a Unix system, because the cost to draw the bitmap in slices
	// on these systems are to expensive.
	if ( (pOut->GetOutDevType() != OUTDEV_WINDOW) ||
		 (rDestSz.Height() != rSrcSz.Height()) ||
		 ((Application::GetClientSystem() & SYSTEMINFO_TYPE_SYSTEM) == SYSTEMINFO_SYSTEM_UNIX) ||
		 !mpReadAccess )
	{
		if ( mpReadAccess )
			TransferLines( 0, TRUE );
		if ( mxJavaRemoteBitmap.is() )
		{
			CHECK_FOR_RVPSYNC_NORMAL();
            try
            {
			    mxJavaRemoteBitmap->Draw( rDestPt.X(), rDestPt.Y(),
									  rDestSz.Width(), rDestSz.Height(),
									  rSrcPt.X(), rSrcPt.Y(),
									  rSrcSz.Width(), rSrcSz.Height(),
									  pOut->mpGraphics->GetInterface() );
            }
            catch( RuntimeException &e )
            {
                rvpExceptionHandler();
            }
		}
		else if ( mxRemoteBitmap.is() )
		{
			CHECK_FOR_RVPSYNC_NORMAL();
            try
            {
			    mxRemoteBitmap->Draw( rDestPt.X(), rDestPt.Y(),
								  rDestSz.Width(), rDestSz.Height(),
								  rSrcPt.X(), rSrcPt.Y(),
								  rSrcSz.Width(), rSrcSz.Height(),
								  pOut->mpGraphics->GetInterface() );
            }
            catch( RuntimeException &e )
            {
                rvpExceptionHandler();
            }
		}
	}
	else
	{
		long nY 	= rDestPt.Y();
		long nSrcY	= rSrcPt.Y();
		long nLines = 0;
		do
		{
			long nSliceLines = TransferLines( nLines, FALSE );
			if ( mxJavaRemoteBitmap.is() )
			{
				CHECK_FOR_RVPSYNC_NORMAL();
                try
                {
				    mxJavaRemoteBitmap->Draw( rDestPt.X(), nY,
										  rDestSz.Width(), nSliceLines,
										  rSrcPt.X(), nSrcY,
										  rSrcSz.Width(), nSliceLines,
										  pOut->mpGraphics->GetInterface() );
                }
                catch( RuntimeException &e )
                {
                    rvpExceptionHandler();
                }
                
			}
			else if ( mxRemoteBitmap.is() )
			{
				CHECK_FOR_RVPSYNC_NORMAL();
                try
                {
				    mxRemoteBitmap->Draw( rDestPt.X(), nY,
									  rDestSz.Width(), nSliceLines,
									  rSrcPt.X(), nSrcY,
									  rSrcSz.Width(), nSliceLines,
									  pOut->mpGraphics->GetInterface() );
                }
                catch( RuntimeException &e )
                {
                    rvpExceptionHandler();
                }
			}
			nY += nSliceLines;
			nSrcY += nSliceLines;
			nLines += nSliceLines;
		}
		while( mpReadAccess );
	}

	return TRUE;
}

void RMBitmap::Draw( OutputDevice* pOut,
					 const Point& rSrcPt, const Size& rSrcSz,
					 const Point& rDestPt, const Size& rDestSz )
{
	if ( pOut->mpGraphics->Init() )
		DrawImpl( pOut, rSrcPt, rSrcSz, rDestPt, rDestSz );
}

// ------------------------------------------------------------------------

BOOL RMBitmap::DrawExImpl( OutputDevice* pOut,
						   const Point& rSrcPt, const Size& rSrcSz,
						   const Point& rDestPt, const Size& rDestSz,
						   const Bitmap& rMaskOrAlpha, BOOL bAlpha )
{

	ImpBitmap* pImpBmp = rMaskOrAlpha.ImplGetImpBitmap();
	if ( !pImpBmp )
		return FALSE;
	if ( !pImpBmp->ImplGetRemoteBmp() )
		pImpBmp->ImplCreateRemoteBmp( rMaskOrAlpha );

	// Draw the bitmap in one piece, if the destination output device
	// is not a window and if the bitmap is not scaled, because
	// we doesn't know, how the client stretch the bitmap and if
	// the bitmaps isn't transferd yet
	RMBitmap* pRmMask = pImpBmp->ImplGetRemoteBmp();
	if ( (pOut->GetOutDevType() != OUTDEV_WINDOW) ||
		 (rDestSz.Height() != rSrcSz.Height()) ||
		 ((Application::GetClientSystem() & SYSTEMINFO_TYPE_SYSTEM) == SYSTEMINFO_SYSTEM_UNIX) ||
		 (!mpReadAccess && !pRmMask->mpReadAccess) )
	{
		if ( pRmMask->mpReadAccess )
			pRmMask->TransferLines( 0, TRUE );
		if ( mpReadAccess )
			TransferLines( 0, TRUE );

		if ( mxJavaRemoteBitmap.is() )
		{
			CHECK_FOR_RVPSYNC_NORMAL();
            try
            {
			    mxJavaRemoteBitmap->DrawEx( rDestPt.X(), rDestPt.Y(),
										rDestSz.Width(), rDestSz.Height(),
										rSrcPt.X(), rSrcPt.Y(),
										rSrcSz.Width(), rSrcSz.Height(),
										mxRemoteBitmap, pRmMask->GetRemoteInst(),
										pOut->mpGraphics->GetInterface(),
										bAlpha );
            }
            catch( RuntimeException &e )
            {
                rvpExceptionHandler();
            }
		}
		else if ( mxRemoteBitmap.is() )
		{
			CHECK_FOR_RVPSYNC_NORMAL();
            try
            {
			    mxRemoteBitmap->DrawEx( rDestPt.X(), rDestPt.Y(),
									 rDestSz.Width(), rDestSz.Height(),
									 rSrcPt.X(), rSrcPt.Y(),
									 rSrcSz.Width(), rSrcSz.Height(),
									 mxRemoteBitmap, pRmMask->GetRemoteInst(),
									 pOut->mpGraphics->GetInterface(),
									 bAlpha );
            }
            catch( RuntimeException &e )
            {
                rvpExceptionHandler();
            }
		}
	}
	else
	{
		long nY 			= rDestPt.Y();
		long nSrcY			= rSrcPt.Y();
		long nLines 		= 0;
		long nMaskLines 	= 0;
		long nLinesDrawn	= 0;
		do
		{
			if ( pRmMask->mpReadAccess )
				nMaskLines += pRmMask->TransferLines( nMaskLines, FALSE );
			if ( mpReadAccess )
				nLines += TransferLines( nLines, FALSE );
			long nDrawLines;
			if ( nLines < nMaskLines )
				nDrawLines = nLines - nLinesDrawn;
			else
				nDrawLines = nMaskLines - nLinesDrawn;

			if ( mxJavaRemoteBitmap.is() )
			{
				CHECK_FOR_RVPSYNC_NORMAL();
                try
                {
				    mxJavaRemoteBitmap->DrawEx( rDestPt.X(), nY,
											rDestSz.Width(), nDrawLines,
											rSrcPt.X(), nSrcY,
											rSrcSz.Width(), nDrawLines,
											mxRemoteBitmap, pRmMask->GetRemoteInst(),
											pOut->mpGraphics->GetInterface(),
											bAlpha );
                }
                catch( RuntimeException &e )
                {
                    rvpExceptionHandler();
                }
			}
			else if ( mxRemoteBitmap.is() )
			{
				CHECK_FOR_RVPSYNC_NORMAL();
                try
                {
				    mxRemoteBitmap->DrawEx( rDestPt.X(), nY,
										rDestSz.Width(), nDrawLines,
										rSrcPt.X(), nSrcY,
										rSrcSz.Width(), nDrawLines,
										mxRemoteBitmap, pRmMask->GetRemoteInst(),
										pOut->mpGraphics->GetInterface(),
										bAlpha );
                }
                catch( RuntimeException &e )
                {
                    rvpExceptionHandler();
                }
			}
			nY += nDrawLines;
			nSrcY += nDrawLines;
			nLinesDrawn += nDrawLines;
		}
		while( mpReadAccess || pRmMask->mpReadAccess );
	}

	return TRUE;
}

void RMBitmap::DrawEx( OutputDevice* pOut,
					   const Point& rSrcPt, const Size& rSrcSz,
					   const Point& rDestPt, const Size& rDestSz,
					   const Bitmap& rMask )
{
	if( pOut->mpGraphics->Init() )
		DrawExImpl( pOut, rSrcPt, rSrcSz, rDestPt, rDestSz, rMask, FALSE );
}

// ------------------------------------------------------------------------

void RMBitmap::DrawAlpha( OutputDevice* pOut,
						  const Point& rSrcPt, const Size& rSrcSz,
						  const Point& rDestPt, const Size& rDestSz,
						  const AlphaMask& rAlpha )
{
	if ( pOut->mpGraphics->Init() )
		DrawExImpl( pOut, rSrcPt, rSrcSz, rDestPt, rDestSz, rAlpha.GetBitmap(), TRUE );
}

// ------------------------------------------------------------------------

BOOL RMBitmap::DrawMaskImpl( OutputDevice* pOut,
							 const Point& rSrcPt, const Size& rSrcSz,
							 const Point& rDestPt, const Size& rDestSz,
							 const Color& rColor )
{
	// Draw Mask is every time complete, because there should
	// be no advantage to draw the bitmap in pieces, because
	// the bitmaps should be small (mono) and the overhead to
	// Draw the mask is much higher
	if ( mpReadAccess )
		TransferLines( 0, TRUE );
	if ( mxJavaRemoteBitmap.is() )
	{
		CHECK_FOR_RVPSYNC_NORMAL();
        try
        {
		    mxJavaRemoteBitmap->DrawMask( rDestPt.X(), rDestPt.Y(),
									  rDestSz.Width(), rDestSz.Height(),
									  rSrcPt.X(), rSrcPt.Y(),
									  rSrcSz.Width(), rSrcSz.Height(),
									  mxRemoteBitmap,
									  rColor.GetColor(),
									  pOut->mpGraphics->GetInterface() );
        }
        catch( RuntimeException &e )
        {
            rvpExceptionHandler();
        }
	}
	else if ( mxRemoteBitmap.is() )
	{
		CHECK_FOR_RVPSYNC_NORMAL();
        try
        {
		    mxRemoteBitmap->DrawMask( rDestPt.X(), rDestPt.Y(),
								  rDestSz.Width(), rDestSz.Height(),
								  rSrcPt.X(), rSrcPt.Y(),
								  rSrcSz.Width(), rSrcSz.Height(),
								  mxRemoteBitmap,
								  rColor.GetColor(),
								  pOut->mpGraphics->GetInterface() );
        }
        catch( RuntimeException &e )
        {
            rvpExceptionHandler();
        }
	}

	return TRUE;
}

void RMBitmap::DrawMask( OutputDevice* pOut,
						 const Point& rSrcPt, const Size& rSrcSz,
						 const Point& rDestPt, const Size& rDestSz,
						 const Color& rColor )
{
	if ( pOut->mpGraphics->Init() )
		DrawMaskImpl( pOut, rSrcPt, rSrcSz, rDestPt, rDestSz, rColor );
}

// ------------------------------------------------------------------------

void RMBitmap::Get( Bitmap& rBmp )
{
	if ( mbGetPrepared )
	{
		if ( mxJavaRemoteBitmap.is() )
			ImplGetRawData( rBmp );
		else if ( mxRemoteBitmap.is() )
		{
			NMSP_CLIENT::ByteSequence aSeq;

			mbGetPrepared = FALSE;
			CHECK_FOR_RVPSYNC_NORMAL();
            try
            {
			    mxRemoteBitmap->GetBitmap( aSeq );
            }
            catch( RuntimeException &e )
            {
                rvpExceptionHandler();
            }

			SvMemoryStream aMemStm( (char*)aSeq.getConstArray(), aSeq.getLength(), STREAM_READ );
			aMemStm.SetCompressMode( COMPRESSMODE_FULL );
			aMemStm >> rBmp;
		}
	}
}

// ------------------------------------------------------------------------

void RMBitmap::ImplTransferRawData()
{
	ImplServerBitmap* pSrvBmp = mpBmp->ImplGetImpBitmap()->ImplGetSalBitmap();

	if( pSrvBmp )
	{
		const long					nWidth = pSrvBmp->GetSize().Width();
		const long					nHeight = pSrvBmp->GetSize().Height();
		const long					nBitCount = pSrvBmp->GetBitCount();
		const long					nAlignedWidth = AlignedWidth4Bytes( nWidth * nBitCount );
		const BYTE* 				pData = pSrvBmp->GetRawData();
		const BitmapPalette*		pPal = pSrvBmp->GetPalette();
		const USHORT				nPalCount = ( nBitCount <= 8L && pPal ) ? pPal->GetEntryCount() : 0;
		const ULONG 				nPalSeqCount = nPalCount * 4;
		const ULONG 				nIndSeqCount = nPalCount ? ( nWidth * nHeight ) : 0;
		const ULONG 				nDirSeqCount = nPalCount ? 0 : ( nWidth * nHeight );
		NMSP_CLIENT::ByteSequence	aPalSeq( nPalSeqCount );
		NMSP_CLIENT::ByteSequence	aIndSeq( nIndSeqCount );
		NMSP_CLIENT::LongSequence	aDirSeq( nDirSeqCount );
		BYTE*						pPalSeq = (BYTE*) aPalSeq.getArray();
		BYTE*						pIndSeq = (BYTE*) aIndSeq.getArray();
		long*						pDirSeq = (long*) aDirSeq.getArray();

		// fill palette buffer if necessary
		for( USHORT i = 0; i < nPalCount; i++ )
		{
			const BitmapColor& rCol = (*pPal)[ i ];

			*pPalSeq++ = rCol.GetRed();
			*pPalSeq++ = rCol.GetGreen();
			*pPalSeq++ = rCol.GetBlue();
			*pPalSeq++ = 255L;
		}

		// fill data buffer
		for( long nY = nHeight - 1L; nY >= 0L; nY-- )
		{
			BYTE* pRow = (BYTE*) pData + nAlignedWidth * nY;

			switch( nBitCount )
			{
				case( 1 ):
				{
					for( long nX = 0L; nX < nWidth; nX++ )
						*pIndSeq++ = ( pRow[ nX >> 3 ] & ( 1 << ( 7 - ( nX & 7 ) ) ) ) ? 1 : 0;
				}
				break;

				case( 4 ):
				{
					for( long nX = 0L; nX < nWidth; nX++ )
						*pIndSeq++ = ( pRow[ nX >> 1 ] >> ( nX & 1 ? 0 : 4 ) ) & 0x0f;
				}
				break;

				case( 8 ):
				{
					rtl_copyMemory( pIndSeq, pRow, nWidth );
					pIndSeq += nWidth;
				}
				break;

				default:
				{
					for( long nX = 0L; nX < nWidth; nX++ )
					{
						*pDirSeq = 0xff000000L; 		// Alpha
						*pDirSeq |= *pRow++;			// Blue
						*pDirSeq |= *pRow++ << 8L;		// Green
						*pDirSeq++ |= *pRow++ << 16L;	// Red
					}
				}
				break;
			}
		}
		
		CHECK_FOR_RVPSYNC_NORMAL();
        try
        {
		    mxJavaRemoteBitmap->SetImageData( nBitCount, nWidth, nHeight, aPalSeq, aIndSeq, aDirSeq );
        }
        catch( RuntimeException &e )
        {
            rvpExceptionHandler();
        }
	}
}

// ------------------------------------------------------------------------

void RMBitmap::ImplGetRawData( Bitmap& rBitmap )
{
	NMSP_CLIENT::ByteSequence	aPalSeq;
	NMSP_CLIENT::ByteSequence	aIndSeq;
	NMSP_CLIENT::LongSequence	aDirSeq;
	long						nBitCount;
	long						nWidth;
	long						nHeight;
	long						nAMsk;
	long						nRMsk;
	long						nGMsk;
	long						nBMsk;
	long						nRRight, nGRight, nBRight;
	long						nRLeft, nGLeft, nBLeft;

	CHECK_FOR_RVPSYNC_NORMAL();
    try
    {
	    mxJavaRemoteBitmap->GetImageData( nBitCount, nWidth, nHeight, aPalSeq, aIndSeq, aDirSeq,
									  nAMsk, nRMsk, nGMsk, nBMsk );
    }
    catch( RuntimeException &e )
    {
        rvpExceptionHandler();
    }
    

	if( nBitCount && nWidth && nHeight )
	{
		ULONG			nAlignedWidth;
		BitmapPalette	aPal;
		BYTE*			pBuffer;
		const BYTE* 	pIndBuf;
		const long* 	pDirBuf;
		USHORT			nPalCount = (USHORT) aPalSeq.getLength() / 4;

		if( ( nBitCount <= 8L ) && nPalCount )
		{
			// try to read palette
			const sal_Int8* pPalData = aPalSeq.getConstArray();
			const sal_Int8* pPalTmp = pPalData;

			if( 2 >= nPalCount )
				nBitCount = 1;
			else if( 16 >= nPalCount )
				nBitCount = 4;
			else if( 256 >= nPalCount )
				nBitCount = 8;
			else
			{
				DBG_ERROR( "unsupported data format" );
				return;
			}

			aPal.SetEntryCount( nPalCount );

			for( USHORT i = 0; i < nPalCount; i++ )
			{
				BitmapColor& rCol = aPal[ i ];

				pPalTmp++;
				rCol.SetRed( *pPalTmp++ );
				rCol.SetGreen( *pPalTmp++ );
				rCol.SetBlue( *pPalTmp++ );
			}
		}
		else
		{
			nBitCount = 24;

			// calc shift values
			for( nRRight = 0L; ( nRRight < 32L ) && !( nRMsk & ( 1 << nRRight ) ); nRRight++ ) {;}
			for( nRLeft = nRRight + 1; ( nRLeft < 32L ) && ( nRMsk & ( 1 << nRLeft ) ); nRLeft++ ) {;}
				nRLeft = 8L - ( nRLeft - nRRight );

			for( nGRight = 0L; ( nGRight < 32L ) && !( nGMsk & ( 1 << nGRight ) ); nGRight++ ) {;}
			for( nGLeft = nGRight + 1; ( nGLeft < 32L ) && ( nGMsk & ( 1 << nGLeft ) ); nGLeft++ ) {;}
				nGLeft = 8L - ( nGLeft - nGRight );

			for( nBRight = 0L; ( nBRight < 32L ) && !( nBMsk & ( 1 << nBRight ) ); nBRight++ ) {;}
			for( nBLeft = nBRight + 1; ( nBLeft < 32L ) && ( nBMsk & ( 1 << nBLeft ) ); nBLeft++ ) {;}
				nBLeft = 8L - ( nBLeft - nBRight );
		}

		// create and fill new server bitmap
		ImplServerBitmap* pSalBmp = new ImplServerBitmap;

		pSalBmp->Create( Size( nWidth, nHeight ), (USHORT) nBitCount, aPal );
		pBuffer = (BYTE*) pSalBmp->GetRawData();
		nAlignedWidth = AlignedWidth4Bytes( nWidth * nBitCount );
		pIndBuf = (const BYTE*) aIndSeq.getConstArray();
		pDirBuf = (const long*) aDirSeq.getConstArray();

		// fill data buffer
		for( long nY = nHeight - 1L; nY >= 0L; nY-- )
		{
			BYTE* pRow = (BYTE*) pBuffer + nAlignedWidth * nY;

			switch( nBitCount )
			{
				case( 1 ):
				{
					for( long nX = 0L; nX < nWidth; nX++ )
					{
						if( ( *pIndBuf++ ) & 1 )
							pRow[ nX >> 3 ] |= 1 << ( 7 - ( nX & 7 ) );
						else
							pRow[ nX >> 3 ] &= ~( 1 << ( 7 - ( nX & 7 ) ) );
					}
				}
				break;

				case( 4 ):
				{
					for( long nX = 0L; nX < nWidth; nX++ )
					{
						BYTE& rByte = pRow[ nX >> 1 ];

						if( nX & 1 )
						{
							rByte &= 0xf0;
							rByte |= ( *pIndBuf++ ) & 0x0f;
						}
						else
						{
							rByte &= 0x0f;
							rByte |= ( *pIndBuf++ ) << 4;
						}
					}
				}
				break;

				case( 8 ):
				{
					HMEMCPY( pRow, pIndBuf, nWidth );
					pIndBuf += nWidth;
				}
				break;

				default:
				{
					for( long nX = 0L; nX < nWidth; nX++ )
					{
						const long nVal = *pDirBuf++;

						*pRow++ = (BYTE) ( ( ( nVal & nBMsk ) >> nBRight ) << nBLeft );
						*pRow++ = (BYTE) ( ( ( nVal & nGMsk ) >> nGRight ) << nGLeft );
						*pRow++ = (BYTE) ( ( ( nVal & nRMsk ) >> nRRight ) << nRLeft );
					}
				}
				break;
			}
		}

		// set Server bitmap to ImpBitmap to Bitmap
		ImpBitmap* pImpBmp = new ImpBitmap;
		pImpBmp->ImplSetSalBitmap( pSalBmp );
		rBitmap.ImplSetImpBitmap( pImpBmp );
	}
}
