# Samizdat member registration and preferences
#
#   Copyright (c) 2002-2008  Dmitry Borodaenko <angdraug@debian.org>
#
#   This program is free software.
#   You can distribute/modify this program under the terms of
#   the GNU General Public License version 2 or later.
#
# vim: et sw=2 sts=2 ts=8 tw=0

class MemberController < Controller

  # account form
  #
  def index
    check_params
    if @session.member   # change existing account
      @full_name = @session.full_name unless @full_name
      @email = rdf.get_property(@session.member, 's::email') unless @email
    end

    if @session.member
      form = []
    else   # create new account
      form = [[:label, 'login', _('Login')], [:text, 'login', @login]]
    end
    form.push(
      [:label, 'full_name', _('Full name')], [:text, 'full_name', @full_name],
      [:label, 'email', _('Email')], [:text, 'email', @email],
      [:label, 'password1', _('Password')], [:password, 'password1'],
      [:label, 'password2', _('Reenter password to confirm')],
        [:password, 'password2'],
      [:br], [:submit, 'submit']
    )

    style = _('Theme') + ': '
    config['style'].each do |s|
      style << %{<a class="action" href="member/set?style=#{s}">#{s}</a>\n}
    end
    style << '<br/><a class="action" href="member/set?nostatic=' + (
      ('yes' == @request.cookie('nostatic')) ?
        'no">' + _('Show static content on the front page') :
        'yes">' + _('Hide static content from the front page')
    ) + '</a>'
    style << '<br/><a class="action" href="member/set?ui=' + (
      @request.advanced_ui? ?
        'basic">' + _('Return to basic interface') :
        'advanced">' + _('Enable advanced interface')
    ) + '</a>'
    style = box(_('Change Appearance'), style)

    if @session.moderator?
      moderation = box(_('Moderation Facility'), (@request.moderate? ?
        %{<a class="action" href="member/set?moderate=no">} + _('Disable') + '</a>' :
        %{<a class="action" href="member/set?moderate=yes">} + _('Enable') + '</a>'))
    end

    if @session.member
      @title = _('Change Account Settings')
      action = 'change'
    else
      @title = _('Create New Account')
      action = 'create'
    end
    @content_for_layout = box(_('Member Settings'),
      box(@title, secure_form('member/' + action, *form)) +
      style + moderation.to_s)
  end

  # store UI options in cookies
  #
  def set
    %w[lang style nostatic ui].each do |param|
      value, = @request.values_at([param])
      if value
        @request.set_cookie(param, value, forever)
        @request.redirect
      end
    end

    moderate, = @request.values_at %w[moderate]
    if moderate   # cookie with timeout
      @request.set_cookie('moderate', moderate, config['timeout']['moderate'])
      @request.redirect
    end
  end

  # change existing account
  #
  def change
    changed = false
    check_params
    @request.redirect unless action_confirmed?

    db.transaction do |db|   # AutoCommit assumed off
      check_duplicates

      if @full_name and @full_name != @session.full_name
        db.do 'UPDATE Member SET full_name = ? WHERE id = ?',
          @full_name, @session.member
        cache.flush
        changed = true
      end
      if @password
        db.do 'UPDATE Member SET password = ? WHERE id = ?',
          digest(@password), @session.member
        changed = true
      end
      if @email and @email != @current_email
        if config.email_enabled?   # with confirmation
          confirm = confirm_hash(@session.login)

          prefs = Preferences.new(@session.login)
          prefs['email'] = @email
          prefs.delete('password')   # don't change password
          prefs.save(confirm)

          request_confirmation(@email, confirm,
            'Your email address was specified for an account.')

          @title = _('Confirmation Requested')
          @content_for_layout = box(@title,
            '<p>' + _('Confirmation request is sent to your new email address.') + '</p>')
          db.commit
          return
        else   # without confirmation
          db.do 'UPDATE Member SET email = ? WHERE id = ?',
            @email, @session.member
        end
        changed = true
      end
    end # transaction

    # fixme: implement proper success/failure handling
    if changed
      @title = _('Changes Accepted')
      @content_for_layout = box(@title,
        '<p>' + _("Press 'Back' button of your browser to return.") + '</p>')
    else
      @request.redirect('member')
    end
  end

  # create new account
  #
  def create
    check_params

    db.transaction do |db|   # AutoCommit assumed off
      check_duplicates

      (@login and @full_name and @email and @password) or raise UserError,
        _("You didn't fill all mandatory fields")

      # create account
      p = digest(@password)
      db.do 'INSERT INTO Member (login, full_name, email, password) VALUES
        (?, ?, ?, ?)', @login, @full_name, @email, (config.email_enabled? ? nil : p)
      if config.email_enabled?   # request confirmation over email
        confirm = confirm_hash(@login)

        prefs = Preferences.new(@login)
        prefs['email'] = @email
        prefs['password'] = p
        prefs.save(confirm)

        request_confirmation(@email, confirm,
          'Your email address was used to create an account.')
      end
    end # transaction

    if cookie = Session.start(@login, @password)
      @request.set_cookie('session', cookie, config['timeout']['last'])
      @request.redirect('')
    else
      raise RuntimeError, _('Login error: failed to open session for new account')
    end
  end

  def login
    login, password = @request.values_at %w[login password]

    if login and password
      if cookie = Session.start(login, password)
        @request.set_cookie('session', cookie, config['timeout']['last'])
        @request.redirect_when_done
      else
        @title = _('Login Failed')
        @content_for_layout = box(@title,
          '<p>'+_('Wrong login name or password. Try again.')+'</p>')
      end

    else
      unless @request.cookie('redirect_when_done')
        @request.set_cookie('redirect_when_done', @request.referer)
      end

      @title = _('Log in')
      @content_for_layout = box(@title,
        '<p>'+_('Use existing account:')+'</p>' +
        form(
          'member/login',
          [:label, 'login', _('Login')], [:text, 'login'],
          [:label, 'password', _('Password')], [:password, 'password'],
          [:br], [:submit]
        ) +
        '<p><a href="member">'+_('Create New Account')+'</a></p>' +
        ( config.email_enabled? ?
          '<p><a href="member/recover_password">'+_('Recover Lost Password')+'</a></p>' : ''
        )
      )
    end
  end

  def logout
    if @session.member
      @session.clear!
      @request.unset_cookie('session')
    end
    @request.redirect
  end

  # check in confirmed email and enable the account
  #
  def confirm
    confirm, = @request.values_at %w[hash]

    db.transaction do |db|
      login, = db.select_one('SELECT login
        FROM Member
        WHERE confirm = ?', confirm)

      login.nil? and raise UserError, _('Confirmation hash not found')
      @session.member and @session.login != login and raise UserError,
        _('This confirmation hash is intended for another user')

      # set password and email from preferences
      prefs = Preferences.new(login)
      password = prefs['password'] and db.do(
          'UPDATE Member SET password = ? WHERE login = ?', password, login)
      email = prefs['email'] and db.do(
          'UPDATE Member SET email = ? WHERE login = ?', email, login)
      prefs.delete('password')
      prefs.delete('email')
      prefs.save

      # clear confirmation hash
      db.do('UPDATE Member SET confirm = NULL WHERE login = ?', login)
    end

    @title = _('Confirmation Accepted')
    @content_for_layout = box(@title, '<p>' + _('Changes confirmed.') + '</p>')
  end

  # recover lost password
  #
  def recover_password
    config.email_enabled? or raise UserError,
      _("Sorry, password recovery not enabled on this site")
    @session.member.nil? or raise UserError,
      _('You are already logged in')

    login, = @request.values_at %w[login]

    if login and login =~ LOGIN_PATTERN
      p = ''; 1.upto(10) { p << (?a + rand(26)).chr }   # random password
      db.transaction do |db|
        email, = db.select_one 'SELECT email FROM Member WHERE login = ?', login
        email or raise UserError, _('Wrong login')
        confirm = confirm_hash(login)

        prefs = Preferences.new(login)
        prefs['password'] = digest(p)
        prefs.delete('email')   # don't change email
        prefs.save(confirm)

        request_confirmation(email, confirm,
          %{New password was generated for your account: } + p)
      end # transaction
      body = '<p>' + _('New password has been sent to you.') + '</p>'
    else
      body = form(
        'member/recover_password',
        [:label, 'login', _('Login')], [:text, 'login'],
        [:submit, 'recover'])
    end

    @title = _('Recover Lost Password')
    @content_for_layout = box(@title, body)
  end

  def block
    moderate('block') do |login, password, prefs|
      password.nil? and raise UserError, _('Account is already blocked')
      config['access']['moderators'].include? login and
        raise UserError, _('Moderator accounts can not be blocked')

      prefs['blocked_by'] = @session.member
      prefs['password'] = password   # remember password
      prefs.save
      db.do(
        'UPDATE Member SET password = NULL, session = NULL WHERE id = ?', @id)
    end
  end

  def unblock
    moderate('unblock') do |login, password, prefs|
      password.nil? or raise UserError, _('Account is not blocked')
      prefs['password'] or raise UserError, _("Can't unblock, the account is broken")

      prefs.delete('blocked_by')
      password = prefs.delete('password')   # restore password
      db.do 'UPDATE Member SET password = ? WHERE id = ?', password, @id
      prefs.save
    end
  end

  private

  def moderate(action)
    assert_moderate
    Resource.validate_id(@id) or raise ResourceNotFoundError, @id.to_s

    if @request.has_key? 'confirm' and action_confirmed?
      db.transaction do |db|
        login, password = db.select_one('SELECT login, password FROM Member
          WHERE id = ?', @id)
        login.nil? and raise ResourceNotFoundError, @id.to_s

        yield login, password, Preferences.new(login)
        log_moderation(action)
      end
      cache.flush
      @request.redirect(@id)
    else
      @title = _('Change Account Status')
      @content_for_layout = box(@title,
        '<p class="moderation">' << _('When account is blocked, the member cannot login. Please confirm that you want to change block status of this account.') << '</p>' <<
        Resource.new(@request, @id).short <<
        secure_form(nil, [:submit, 'confirm', _('Confirm')]))
    end
  end

  LOGIN_PATTERN = Regexp.new(/\A[a-z0-9]+\z/).freeze
  EMAIL_PATTERN = Regexp.new(/\A[[:alnum:]._-]+@[[:alnum:].-]+\z/).freeze

  # validate and store account parameters
  #
  def check_params
    @login, @full_name, @email, @password, password2 = \
      @request.values_at %w[login full_name email password1 password2]

    # validate password
    @password == password2 or raise UserError,
      _('Passwords do not match')

    # validate login
    if @login and @session.member.nil?
      @login == 'guest' and raise UserError,
        _('Login name you specified is reserved')
      @login =~ LOGIN_PATTERN or raise UserError,
        _('Use only latin letters and numbers in login name')
    end

    # validate email
    @email.nil? or @email =~ EMAIL_PATTERN or raise UserError,
      sprintf(_("Malformed email address: '%s'"), CGI.escapeHTML(@email))
  end

  # check full_name and email for duplicates
  #
  # run it from inside the same transaction that stores the verified values
  #
  def check_duplicates
    if @session.member
      @current_email = rdf.get_property(@session.member, 's::email')
      except_self = ' AND id != ' + @session.member.to_s
    end

    if @full_name and @full_name != @session.full_name
      id, = db.select_one(
        'SELECT id FROM Member WHERE full_name = ?' + except_self.to_s,
        @full_name)
      id.nil? or raise UserError,
        _('Full name you have specified is used by someone else')
    end
    if @email and @email != @current_email
      id, = db.select_one(
        'SELECT id FROM Member WHERE email = ?' + except_self.to_s,
        @email)
      id.nil? or raise UserError,
        _('Email address you have specified is used by someone else')
    end
    if @login and @session.member.nil?
      id, = db.select_one 'SELECT id FROM Member WHERE login = ?', @login
      id.nil? or raise UserError,
        _('Login name you specified is already used by someone else')
    end
  end

  # wrapper around sendmail
  #
  def send_mail(to, subject, body)
    if EMAIL_PATTERN =~ to
      to.untaint
    else
      raise UserError, sprintf(_("Malformed email address: '%s'"), CGI.escapeHTML(email))
    end
    message_id = config['email']['address'].sub(/^[^@]*/,
      Time.now.strftime("%Y%m%d%H%M%S." + Process.pid.to_s))
    IO.popen(config['email']['sendmail'].untaint+' '+to, 'w') do |io|
      io.write(
%{From: Samizdat <#{config['email']['address']}>
To: #{to}
Subject: #{subject}
Message-Id: <#{message_id}>

} + body)
    end
    0 == $? or raise RuntimeError, _('Failed to send email')
  end

  # generate confirmation hash
  #
  def confirm_hash(login)
    digest(login + Time.now.to_s)
  end

  # send confirmation request
  #
  def request_confirmation(email, hash, action)
    send_mail(email, 'CONFIRM ' + hash,
%{Site: #{@request.base}

#{action}

To confirm this action, visit the following web page:

#{@request.base + 'member/confirm?hash=' + hash}

To cancel this action, ignore this message.
})
  end
end
