/*
** yp_db.c - database functions for the maps
**
** This file is part of the NYS YP Server.
**
** The NYS YP Server is free software; you can redistribute it and/or
** modify it under the terms of the GNU General Public License as
** published by the Free Software Foundation; either version 2 of the
** License, or (at your option) any later version.
**
** The NYS YP Server is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
** General Public License for more details.
**
** You should have received a copy of the GNU General Public
** License along with the NYS YP Server; see the file COPYING.  If
** not, write to the Free Software Foundation, Inc., 675 Mass Ave,
** Cambridge, MA 02139, USA.
**
** Author: Thorsten Kukuk <kukuk@uni-paderborn.de>
*/

#ifndef LINUT
static const char rcsid[] = "$Id: yp_db.c,v 1.11 1997/12/30 15:22:57 kukuk Exp $";
#endif

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "system.h"

#include <stdio.h>
#include <string.h>
#include <malloc.h>
#include <sys/param.h>

#include "ypserv.h"
#include "yp_msg.h"
#include "yp_db.h"
#include "yp.h"

#if defined(HAVE_LIBGDBM)
#include <gdbm.h>
#endif

/* repleace "field" with x, if field 3 != user id and user id != 0 */
static inline int
mangle_field (datum * val, int field)
{
  int i, j, k, anz;
  char *p;

  /* allocate a little bit more memory, it's safer, because the
     field could be empty */
  if ((p = malloc (val->dsize + 3)) == NULL)
    {
      yp_msg ("ERROR: could not allocate enough memory! [%s|%d]\n",
	      __FILE__, __LINE__);
      return -1;
    }

  anz = 0;
  for (i = 0; i < val->dsize; i++)
    {
      if (val->dptr[i] == ':')
	anz++;
      if (anz + 1 == field)
	{
	  anz = i;
	  strncpy (p, val->dptr, anz);
	  p[anz] = 0;
	  /* if field == 1, we don't need a beginning ":" */
	  if (field == 1)
	    {
	      strcat (p, "x");
	      anz += 1;
	    }
	  else
	    {
	      strcat (p, ":x");
	      anz += 2;
	    }
	  for (j = anz; j < val->dsize && val->dptr[j] != ':'; j++);
	  for (k = j; k < val->dsize; k++)
	    {
	      p[anz] = val->dptr[k];
	      anz++;
	    }
	  free (val->dptr);
	  val->dptr = p;
	  val->dsize = anz;
	  return 0;
	}
    }
  free (p);
  return 0;
}

#if defined(HAVE_LIBGDBM)

/* Open a GDBM database */
static GDBM_FILE
_db_open (const char *domain, const char *map)
{
  GDBM_FILE dbp;
  char buf[MAXPATHLEN + 2];

  if (debug_flag)
    yp_msg ("\tdb_open(\"%s\", \"%s\")\n", domain, map);

  if (map[0] == '.' || strchr (map, '/'))
    {
      if (debug_flag)
	yp_msg ("\t\t->Returning 0\n");
      return 0;
    }

  if (strlen (domain) + strlen (map) < MAXPATHLEN)
    {
      sprintf (buf, "%s/%s", domain, map);

      dbp = gdbm_open (buf, 0, GDBM_READER, 0, NULL);

      if (debug_flag && dbp == NULL)
	yp_msg ("gdbm_open: GDBM Error Code #%d\n", gdbm_errno);
      else if (debug_flag)
	yp_msg ("\t\t->Returning OK!\n");
    }
  else
    {
      dbp = NULL;
      yp_msg ("Path to long: %s/%s\n", domain, map);
    }

  return dbp;
}

static inline int
_db_close (GDBM_FILE file)
{
  gdbm_close (file);
  return 0;
}

/* Get a record from a GDBM database. */
int
ypdb_read (GDBM_FILE dbp, const datum * ikey, datum * okey,
	   datum * dval, int flags, int mangle)
{
  int first_flag = 0;
  datum nkey, ckey;

  if (ikey == NULL || ikey->dptr == NULL)
    {
      if (debug_flag)
	yp_msg ("\tread_database(), gdbm_firstkey()\n");

      ckey = gdbm_firstkey (dbp);
      first_flag = 1;
    }
  else
    {
      if (debug_flag)
	yp_msg ("\tread_database(), gdbm_nextkey()\n");

      if ((flags & F_NEXT))
	ckey = gdbm_nextkey (dbp, *ikey);
      else
	ckey = *ikey;
    }

  if (ckey.dptr == NULL)
    {
      return (flags & F_NEXT) ? YP_NOMORE : YP_NOKEY;
    }

  while (1)
    {
      *dval = gdbm_fetch (dbp, ckey);
      if (dval->dptr == NULL)
	{
	  /* Free key, unless it comes from the caller! */
	  if (ikey == NULL || ckey.dptr != ikey->dptr)
	    free (ckey.dptr);

	  if (ikey && ikey->dptr != NULL)
	    {
	      return YP_NOKEY;
	    }
	  else if (first_flag)
	    return YP_BADDB;
	  else
	    return YP_FALSE;
	}

      if ((flags & F_ALL) || strncmp (ckey.dptr, "YP_", 3) != 0)
	{
	  if (okey)
	    *okey = ckey;
	  else if (ikey == NULL || ikey->dptr != ckey.dptr)
	    free (ckey.dptr);

	  if (mangle)
	    if (mangle_field (dval, mangle) < 0)
	      return YP_YPERR;

	  return YP_TRUE;
	}

      /* Free old value */
      free (dval->dptr);

      nkey = gdbm_nextkey (dbp, ckey);

      /* Free old key, unless it comes from the caller! */
      if (ikey == NULL || ckey.dptr != ikey->dptr)
	free (ckey.dptr);

      if (ckey.dptr == NULL || nkey.dptr == NULL)
	return YP_NOMORE;

      ckey = nkey;
    }
}

#else

void
ypdb_open ()
{
}

#endif

typedef struct _fopen
{
  char *domain;
  char *map;
  DB_FILE dbp;
  int flag;
}
Fopen, *FopenP;

#define F_OPEN_FLAG 1
#define F_MUST_CLOSE 2

/* MAX_FOPEN:
   big -> slow list searching, we go 3 times through the list.
   little -> have to close/open very often */
#define MAX_FOPEN 10

static int fast_open_init = -1;
static Fopen fast_open_files[MAX_FOPEN];

int
ypdb_close_all (void)
{
  int i;

  if (debug_flag)
    yp_msg ("ypdb_close_all() called\n");

  if (fast_open_init == -1)
    return 0;

  for (i = 0; i < MAX_FOPEN; i++)
    {
      if (fast_open_files[i].dbp != NULL)
	{
	  if (fast_open_files[i].flag & F_OPEN_FLAG)
	    fast_open_files[i].flag |= F_MUST_CLOSE;
	  else
	    {
	      free (fast_open_files[i].domain);
	      free (fast_open_files[i].map);
	      _db_close (fast_open_files[i].dbp);
	      fast_open_files[i].dbp = NULL;
	      fast_open_files[i].flag = 0;
	    }
	}
    }
  return 0;
}

int
ypdb_close (DB_FILE file)
{
  int i;

  if (debug_flag)
    yp_msg ("ypdb_close() called\n");

  if (fast_open_init != -1)
    {
      for (i = 0; i < MAX_FOPEN; i++)
	{
	  if (fast_open_files[i].dbp == file)
	    {
	      if (fast_open_files[i].flag & F_MUST_CLOSE)
		{
		  if (debug_flag)
		    yp_msg ("ypdb_MUST_close (%s/%s|%d)\n",
			    fast_open_files[i].domain,
			    fast_open_files[i].map, i);
		  free (fast_open_files[i].domain);
		  free (fast_open_files[i].map);
		  _db_close (fast_open_files[i].dbp);
		  fast_open_files[i].dbp = NULL;
		  fast_open_files[i].flag = 0;
		}
	      else
		{
		  fast_open_files[i].flag &= ~F_OPEN_FLAG;
                }
	      return 0;
	    }
	}
    }
  yp_msg ("ERROR: Could not close file!\n");
  return 1;
}

DB_FILE
ypdb_open (const char *domain, const char *map)
{
  int x;

  /* First call, initialize the fast_open_init struct */
  if (fast_open_init == -1)
    {
      fast_open_init = 0;
      for (x = 0; x < MAX_FOPEN; x++)
	{
	  fast_open_files[x].domain =
	    fast_open_files[x].map = NULL;
	  fast_open_files[x].dbp = (DB_FILE) NULL;
	  fast_open_files[x].flag = 0;
	}
    }

  /* Search if we have already open the domain/map file */
  for (x = 0; x < MAX_FOPEN; x++)
    {
      if (fast_open_files[x].dbp != NULL)
	{
	  if ((strcmp (domain, fast_open_files[x].domain) == 0) &&
	      (strcmp (map, fast_open_files[x].map) == 0))
	    {
	      /* The file is open and we know the file handle */
	      if (debug_flag)
		yp_msg ("Found: %s/%s (%d)\n", fast_open_files[x].domain,
			fast_open_files[x].map, x);

	      if (fast_open_files[x].flag & F_OPEN_FLAG)
                {
		  /* The file is already in use, don't open it twice.
		     I think this could never happen. */
		  yp_msg ("\t%s/%s already open.\n", domain, map);
		  return NULL;
                }
	      else
		{
		  /* Mark the file as open */
		  fast_open_files[x].flag |= F_OPEN_FLAG;
		  return fast_open_files[x].dbp;
		}
	    }
	}
    }

  /* Search for free entry. I we do not found one, close the LRU */
  for (x = 0; x < MAX_FOPEN; x++)
    {
#if 0
      /* Bad Idea. If one of them is NULL, we will get a seg.fault
	 I think it will only work with Linux libc 5.x */
      yp_msg ("Opening: %s/%s (%d) %x\n",
	      fast_open_files[x].domain,
	      fast_open_files[x].map,
	      x, fast_open_files[x].dbp);
#endif
      if (fast_open_files[x].dbp == NULL)
	{
	  /* Good, we have a free entry and don't need to close a map */
	  int j;
	  Fopen tmp;
	  fast_open_files[x].domain = strdup (domain);
	  fast_open_files[x].map = strdup (map);
	  fast_open_files[x].flag |= F_OPEN_FLAG;
	  fast_open_files[x].dbp = _db_open (domain, map);

	  if (debug_flag)
	    yp_msg ("Opening: %s/%s (%d) %x\n", domain, map, x,
		    fast_open_files[x].dbp);

	  /* LRU: put this entry at the first position, move all the other
	     one back */
	  tmp = fast_open_files[x];
	  for (j = x; j >= 1; --j)
	    fast_open_files[j] = fast_open_files[j-1];

	  fast_open_files[0] = tmp;
	  return fast_open_files[0].dbp;
	}
    }

  /* The badest thing, which could happen: no free cache entrys.
     Search the last entry, which isn't in use. */
  for (x = (MAX_FOPEN - 1); x > 0; --x)
    if ((fast_open_files[x].flag & F_OPEN_FLAG) != F_OPEN_FLAG)
      {
        int j;
	Fopen tmp;

	if (debug_flag)
	  {
	    yp_msg ("Closing %s/%s (%d)\n",
		    fast_open_files[x].domain,
		    fast_open_files[x].map, x);
	    yp_msg ("Opening: %s/%s (%d)\n", domain, map, x);
	  }
	free (fast_open_files[x].domain);
	free (fast_open_files[x].map);
	_db_close (fast_open_files[x].dbp);

	fast_open_files[x].domain = strdup (domain);
	fast_open_files[x].map = strdup (map);
	fast_open_files[x].flag = F_OPEN_FLAG;
	fast_open_files[x].dbp = _db_open (domain, map);

	/* LRU: Move the new entry to the first positon */
	tmp = fast_open_files[x];
	for (j = x; j >= 1; --j)
	  fast_open_files[j] = fast_open_files[j-1];

	fast_open_files[j] = tmp;
	return fast_open_files[j].dbp;
      }

  yp_msg ("ERROR: Couldn't find a free cache entry!\n");

  for (x = 0; x < MAX_FOPEN; x++)
    {
      yp_msg ("Open files: %s/%s (%d) %x (%x)\n",
	      fast_open_files[x].domain,
	      fast_open_files[x].map,
	      x, fast_open_files[x].dbp,
              fast_open_files[x].flag);
    }
  return NULL;
}
