/*
 * tnmIned.c --
 *
 *	Extend a Tcl command interpreter with an ined command. See the
 *	documentation of Tkined for more info about the ined command.
 *
 * Copyright (c) 1994-1996 Technical University of Braunschweig.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#include "tnmInt.h"
#include "tnmPort.h"

#define BUF_INCR 1024

/*
 * The list that is used to queue received messages.
 */

typedef struct Message {
    char *msg;
    struct Message *next;
} Message;

static Message *queue = NULL;

/*
 * The default path where Tkined is installed. This
 * is normally overwritten in the Makefile.
 */

#ifndef TKINEDLIB
#define TKINEDLIB "/usr/local/lib/tkined"
#endif

/*
 * Forward declarations for procedures defined later in this file:
 */

static void
InedInitialize		_ANSI_ARGS_((Tcl_Interp *interp));

static void 
InedFatal		_ANSI_ARGS_((Tcl_Interp *interp));

static void
InedQueue		_ANSI_ARGS_((Tcl_Interp *interp));

static void
InedFlushProc		_ANSI_ARGS_((ClientData clientData));

static void
InedFlushQueue		_ANSI_ARGS_((Tcl_Interp *));

static void 
InedAppendQueue		_ANSI_ARGS_((Tcl_Interp *interp, char *msg));

static char*
InedGets		_ANSI_ARGS_((Tcl_Interp *interp));

static int 
InedCompCmd		_ANSI_ARGS_((char *cmd, Tcl_Interp *interp, 
				     int argc, char **argv));
static void
InedReceiveProc		_ANSI_ARGS_((ClientData clientData, int mask));

/*
 * The following tkined object type definitions must be in sync 
 * with the tkined sources.
 */

#define TKINED_NONE         0
#define TKINED_ALL          1
#define TKINED_NODE         2
#define TKINED_GROUP        3
#define TKINED_NETWORK      4
#define TKINED_LINK         5
#define TKINED_TEXT         6
#define TKINED_IMAGE        7
#define TKINED_INTERPRETER  8
#define TKINED_MENU         9
#define TKINED_LOG         10
#define TKINED_REFERENCE   11
#define TKINED_STRIPCHART  12
#define TKINED_BARCHART    13
#define TKINED_GRAPH	   14
#define TKINED_HTML	   15
#define TKINED_DATA	   16
#define TKINED_EVENT	   17

static TnmTable tkiTypeTable[] = {
    { TKINED_NONE,	  "NONE" },
    { TKINED_ALL,	  "ALL" },
    { TKINED_NODE,	  "NODE" },
    { TKINED_GROUP,	  "GROUP" },
    { TKINED_NETWORK,	  "NETWORK" },
    { TKINED_LINK,	  "LINK" },
    { TKINED_TEXT,	  "TEXT" },
    { TKINED_IMAGE,	  "IMAGE" },
    { TKINED_INTERPRETER, "INTERPRETER" },
    { TKINED_MENU,	  "MENU" },
    { TKINED_LOG,	  "LOG" },
    { TKINED_REFERENCE,	  "REFERENCE" },
    { TKINED_STRIPCHART,  "STRIPCHART" },
    { TKINED_BARCHART,	  "BARCHART" },
    { TKINED_GRAPH,	  "GRAPH" },
    { TKINED_HTML,	  "HTML" },
    { TKINED_DATA,	  "DATA" },
    { TKINED_EVENT,	  "EVENT" },
    { 0, NULL }
};

/*
 *----------------------------------------------------------------------
 *
 * InedInitialize --
 *
 *	This procedure initializes the ined module. It registers
 *	the stdin channel so that we can send and receive message
 *	to/from the tkined editor. We also modify the auto_path
 *	variable so that applications are found automatically.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The stdin channel is registered in the event loop and the
 *	Tcl auto_path variable is modified.
 *
 *----------------------------------------------------------------------
 */

static void
InedInitialize(interp)
    Tcl_Interp *interp;
{
    char *path, *tmp, *p;
    Tcl_Channel channel;

    /*
     * Make stdin/stdout unbuffered and register a channel handler to
     * receive commands from the tkined editor. Make sure that an
     * already existing channel handler is removed first (make wish
     * happy).
     */
    
    channel = Tcl_GetChannel(interp, "stdout", NULL);
    if (channel == NULL) {
	return;
    }
    Tcl_SetChannelOption(interp, channel, "-buffering", "line");

    channel = Tcl_GetChannel(interp, "stdin", NULL);
    if (channel == NULL) {
	return;
    }
    Tcl_SetChannelOption(interp, channel, "-buffering", "none");
    Tcl_CreateChannelHandler(channel, TCL_READABLE, 
			     InedReceiveProc, (ClientData) interp);
    InedFlushQueue(interp);
    
    /* 
     * Adjust the auto_path to take care of the environment variable
     * TKINED_PATH, $HOME/.tkined and the default Tkined library.
     */
    
    path = Tcl_GetVar(interp, "auto_path", TCL_GLOBAL_ONLY);
    if (path) {
	path = ckstrdup(path);
    }
    
    Tcl_SetVar(interp, "auto_path", "", TCL_GLOBAL_ONLY);
    
    if ((p = getenv("TKINED_PATH"))) {
	tmp = ckstrdup(p);
	for (p = tmp; *p; p++) {
	    if (*p == ':') {
		*p = ' ';
	    }
	}
	Tcl_SetVar(interp, "auto_path", tmp, TCL_GLOBAL_ONLY);
	ckfree(tmp);
    }
    
    if ((p = getenv("HOME"))) {
	tmp = ckalloc(strlen(p) + 20);
	sprintf(tmp, "%s/.tkined", p);
	Tcl_SetVar(interp, "auto_path", tmp, 
		   TCL_APPEND_VALUE | TCL_LIST_ELEMENT | TCL_GLOBAL_ONLY);
	ckfree(tmp);
    }
    
    tmp = ckalloc(strlen(TKINEDLIB) + 20);
    sprintf(tmp, "%s/site", TKINEDLIB);
    Tcl_SetVar(interp, "auto_path", tmp,
	       TCL_APPEND_VALUE| TCL_LIST_ELEMENT | TCL_GLOBAL_ONLY);
    sprintf(tmp, "%s/apps", TKINEDLIB);
    Tcl_SetVar(interp, "auto_path", tmp,
	       TCL_APPEND_VALUE| TCL_LIST_ELEMENT | TCL_GLOBAL_ONLY);
    Tcl_SetVar(interp, "auto_path", TKINEDLIB, 
	       TCL_APPEND_VALUE | TCL_LIST_ELEMENT | TCL_GLOBAL_ONLY);
    ckfree(tmp);
    
    if (path) {
	Tcl_SetVar(interp, "auto_path", " ",
		   TCL_APPEND_VALUE | TCL_GLOBAL_ONLY);
	Tcl_SetVar(interp, "auto_path", path, 
		   TCL_APPEND_VALUE | TCL_GLOBAL_ONLY);
	ckfree(path);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * InedFatal --
 *
 *	This procedure handles errors that occur while talking to
 *	tkined. We simply print a warning and exit this process.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	The process is terminated.
 *
 *----------------------------------------------------------------------
 */

static void
InedFatal(interp)
    Tcl_Interp *interp;
{
    Tcl_Channel channel;
    char *msg = "Tnm lost connection to Tkined\n";

    channel = Tcl_GetChannel(interp, "stderr", NULL);
    if (channel) {
	Tcl_Write(channel, msg, strlen(msg));
	Tcl_Flush(channel);
    }
    Tcl_Exit(1);
}

/*
 *----------------------------------------------------------------------
 *
 * InedQueue --
 *
 *	This procedure writes a queue message to the tkined editor.
 *	No acknowledge is expected. Queue messages are used to let 
 *	tkined know how busy we are.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static void
InedQueue(interp)
    Tcl_Interp *interp;
{
    Tcl_Channel channel;
    char msg[256];
    int len = 0;
    Message *p;

    for (p = queue; p != NULL; p = p->next) len++;

    sprintf(msg, "ined queue %d\n", len);
    len = strlen(msg);

    channel = Tcl_GetChannel(interp, "stdout", NULL);
    if (channel == NULL) {
	InedFatal(interp);
	return;
    }

    if (Tcl_Write(channel, msg, strlen(msg)) < 0) {
	Tcl_Flush(channel);
	InedFatal(interp);
    }
}

/*
 *----------------------------------------------------------------------
 *
 * InedFlushProc --
 *
 *	This procedure is called from the event loop to flush the
 *	ined queue. It simply calls InedFlushQueue to do the job.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static void
InedFlushProc (clientData)
     ClientData clientData;
{
    Tcl_Interp *interp = (Tcl_Interp *) clientData;
    InedFlushQueue (interp);
}

/*
 *----------------------------------------------------------------------
 *
 * InedFlushQueue --
 *
 *	This procedure processes all queued commands.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Arbitrary Tcl commands are evaluated.
 *
 *----------------------------------------------------------------------
 */

static void
InedFlushQueue(interp)
    Tcl_Interp *interp;
{
    Message *p;

    if (queue == NULL) return;

    InedQueue(interp);
    while ((p = queue) != NULL) {
	queue = queue->next;
	if (Tcl_GlobalEval(interp, p->msg) != TCL_OK) {
	    Tcl_BackgroundError(interp);
	}
	ckfree(p->msg);
	ckfree((char *) p);
    }

    InedQueue(interp);
}

/*
 *----------------------------------------------------------------------
 *
 * InedAppendQueue --
 *
 *	This procedure appends the command given by msg to the
 *	queue of commands that need to be processed.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static void
InedAppendQueue(interp, msg)
    Tcl_Interp *interp;
    char *msg;
{
    Message *np;
    Message *p;

    if (msg == NULL) {
	return;
    }

    np = (Message *) ckalloc(sizeof(Message));
    np->msg = msg;
    np->next = NULL;

    if (queue == NULL) {
        queue = np;
        return;
    }

    for (p = queue; (p->next != (Message *) NULL); p = p->next) ;
    p->next = np;

    InedQueue(interp);
}

/*
 *----------------------------------------------------------------------
 *
 * InedGets --
 *
 *	This procedure reads a message from the Tkined editor.
 *
 * Results:
 *	A pointer to a malloced buffer containing the received 
 *	message or NULL if there was an error.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static char*
InedGets(interp)
    Tcl_Interp *interp;
{
    Tcl_Channel channel;
    Tcl_DString line;
    char *buffer = NULL;
    int len;

    channel = Tcl_GetChannel(interp, "stdin", NULL);
    if (channel == NULL) {
	InedFatal(interp);
	return NULL;
    }

    Tcl_DStringInit(&line);
    len = Tcl_Gets(channel, &line);
    if (len >= 0) {
	buffer = ckstrdup(Tcl_DStringValue(&line));
    }

    Tcl_DStringFree(&line);
    return buffer;
}

/*
 *----------------------------------------------------------------------
 *
 * InedCompCmd --
 *
 *	This procedure checks if we can evaluate a command based on
 *	the Tcl list representation of a tkined object. This allows
 *	to save some interactions between scotty and tkined.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
InedCompCmd(cmd, interp, argc, argv)
    char *cmd;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    int type = TnmGetTableKey(tkiTypeTable, argv[0]);
    if (type < 0 || (type == TKINED_NONE) || (type == TKINED_ALL)) {
	return TCL_ERROR;
    }

    if ((strcmp(cmd, "type") == 0) && (argc > 0)) {
	Tcl_SetResult(interp, argv[0], TCL_VOLATILE);
	return TCL_OK;

    } else if ((strcmp(cmd, "id") == 0) && (argc > 1)) {
	Tcl_SetResult(interp, argv[1], TCL_VOLATILE);
	return TCL_OK;

    } else if ((strcmp(cmd, "name") == 0) && (argc > 2)) {
        if ((type == TKINED_NODE) || (type == TKINED_NETWORK)
	    || (type == TKINED_BARCHART) || (type == TKINED_STRIPCHART)
	    || (type == TKINED_GROUP) || (type == TKINED_REFERENCE)
	    || (type == TKINED_MENU) || (type == TKINED_LOG) 
	    || (type == TKINED_GRAPH) || (type == TKINED_HTML)
	    || (type == TKINED_DATA) || (type == TKINED_EVENT) )
	    Tcl_SetResult(interp, argv[2], TCL_VOLATILE);
        return TCL_OK;

    } else if ((strcmp(cmd, "address") == 0) && (argc > 3)) {
        if ((type == TKINED_NODE) || (type == TKINED_NETWORK)
	    || (type == TKINED_BARCHART) || (type == TKINED_STRIPCHART)
	    || (type == TKINED_REFERENCE) || (type == TKINED_GRAPH)
	    || (type == TKINED_DATA))
	    Tcl_SetResult(interp, argv[3], TCL_VOLATILE);
        return TCL_OK;

    } else if (strcmp(cmd, "oid") == 0) {
        if ((type == TKINED_GROUP) && (argc > 3)) {
	    Tcl_SetResult(interp, argv[3], TCL_VOLATILE);
	}
        if ((type == TKINED_NODE || type == TKINED_NETWORK) && (argc > 4)) {
	    Tcl_SetResult(interp, argv[4], TCL_VOLATILE);
	}
	return TCL_OK;

    } else if ((strcmp(cmd, "links") == 0) && (argc > 5)) {
        if ((type == TKINED_NODE) || (type == TKINED_NETWORK))
	    Tcl_SetResult(interp, argv[5], TCL_VOLATILE);
        return TCL_OK;

    } else if ((strcmp(cmd, "member") == 0) && (argc > 4)) {
        if (type == TKINED_GROUP)
	    Tcl_SetResult(interp, argv[4], TCL_VOLATILE);
        return TCL_OK;

    } else if ((strcmp(cmd, "src") == 0) && (argc > 2)) {
        if (type == TKINED_LINK)
	    Tcl_SetResult(interp, argv[2], TCL_VOLATILE);
        return TCL_OK;

    } else if ((strcmp(cmd, "dst") == 0) && (argc > 3)) {
        if (type == TKINED_LINK)
	    Tcl_SetResult(interp, argv[3], TCL_VOLATILE);
        return TCL_OK;

    } else if ((strcmp(cmd, "text") == 0) && (argc > 2)) {
        if (type == TKINED_LINK)
	    Tcl_SetResult(interp, argv[2], TCL_VOLATILE);
        return TCL_OK;

    }

    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_InedCmd --
 *
 *	This procedure is invoked to process the "ined" command.
 *	See the user documentation for details on what it does.
 *
 *	We send the command to the tkined editor and waits for a
 *	response containing the answer or the error description.
 *	Everything received while waiting for the response is
 *	queued for later execution.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

int
Tnm_InedCmd(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    Tcl_Channel channel;
    int i;
    char *p;
    static int initialized = 0;

    if (! initialized) {
	InedInitialize(interp);
	initialized = 1;
    }

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			 " command ?arg ...?\"", (char *) NULL);
	return TCL_ERROR;
    }

    /* 
     * The loop command is used by programs that do not wish
     * to use their own main or event loop mechanism - this is
     * only for historical reasons
     */

    if ((argc == 2) && (strcmp(argv[1], "loop") == 0)) {
	char *line;
	InedFlushQueue(interp);
	while ((line = InedGets(interp)) != (char *) NULL) {
	    InedAppendQueue(interp, line);
	    InedFlushQueue(interp);
	}
	return TCL_OK;
    }

    /* 
     * Check for commands that can be implemented locally (based on 
     * the list representation of tkined objects).
     */

    if (argc == 3) {
        int largc;
	char **largv;
        int rc = Tcl_SplitList(interp, argv[2], &largc, &largv);
	if (rc == TCL_OK && largc > 0) {
	    if (InedCompCmd(argv[1], interp, largc, largv) == TCL_OK) {
		ckfree((char *) largv);
		return TCL_OK;
	    }
 	    ckfree((char *) largv);
	}
    }

    /*
     * Write the command string to the tkined editor.
     */

    channel = Tcl_GetChannel(interp, "stdout", NULL);
    if (channel == NULL) {
	InedFatal(interp);
	return TCL_ERROR;
    }

    for (i = 0; i < argc; i++) {
	if (Tcl_Write(channel, "{", 1) < 0) InedFatal(interp);
        for (p = argv[i]; *p; p++) {
	    if (*p == '\r') {
		continue;
	    } else if (*p == '\n') {
	        if (Tcl_Write(channel, "\\n", 2) < 0) InedFatal(interp);
	    } else {
	        if (Tcl_Write(channel, p, 1) < 0) InedFatal(interp);
	    }
	}
        if (Tcl_Write(channel, "} ", 2) < 0) InedFatal(interp);
    }
    if (Tcl_Write(channel, "\n", 1) < 0) InedFatal(interp);
    Tcl_Flush(channel);
 
    /*
     * Wait for the response.
     */

    channel = Tcl_GetChannel(interp, "stdin", NULL);
    if (channel == NULL) {
	InedFatal(interp);
	return TCL_ERROR;
    }

    while ((p = InedGets(interp)) != (char *) NULL) {
        if (*p == '\0') continue;
	if (strncmp(p, "ined ok", 7) == 0) {
	    char *r = p+7;
	    while (*r && isspace(*r)) r++;
	    Tcl_SetResult(interp, r, TCL_VOLATILE);
	    ckfree(p);
	    return TCL_OK;
	} else if (strncmp(p, "ined error", 10) == 0) {
	    char *r = p+10;
	    while (*r && isspace(*r)) r++;
	    Tcl_SetResult(interp, r, TCL_VOLATILE);
	    ckfree(p);
	    return TCL_ERROR;
	} else {
	    InedAppendQueue(interp, p);
	    Tcl_CreateTimerHandler(0, InedFlushProc, (ClientData) interp);
	}
    }

    InedFatal(interp);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * InedReceiveProc --
 *
 *	This procedure is called from the event handler whenever
 *	a command can be read from the tkined editor (stdin).
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static void
InedReceiveProc (clientData, mask)
    ClientData clientData;
    int mask;
{
    Tcl_Interp *interp = (Tcl_Interp *) clientData;
    char *cmd = InedGets(interp);

    if (cmd != NULL) {
        InedAppendQueue(interp, cmd);
        InedFlushQueue(interp);
    } else {
        Tcl_Channel channel;
	channel = Tcl_GetChannel(interp, "stdin", NULL);
        Tcl_DeleteChannelHandler(channel, 
				 InedReceiveProc, (ClientData) interp);
    }
}
