/*
 * tnmSnmpTcl.c --
 *
 *	Extend a Tcl-Interpreter about the ability to talk SNMP
 *	(Version 1 as well as Version 2).
 *
 * Copyright (c) 1994-1996 Technical University of Braunschweig.
 *
 * See the file "license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 */

#include "tnmSnmp.h"
#include "tnmMib.h"

/*
 * The global variable SNMP_Session list contains all existing
 * session handles.
 */

SNMP_Session *sessionList = NULL;

int hexdump = 0;

/*
 * A hash table to store agent session configurations and
 * a hash table to cache resolved host names.
 */

static Tcl_HashTable aliasTable;

/*
 * Forward declarations for procedures defined later in this file:
 */

static void
EvalCmdProc	_ANSI_ARGS_((SNMP_Session *session, SNMP_PDU *pdu, 
			     ClientData clientData));
static char *
GetOption	_ANSI_ARGS_((Tcl_Interp *interp, ClientData object, 
			     int option));
static int
SetOption	_ANSI_ARGS_((Tcl_Interp *interp, ClientData object, 
			     int option, char *value));
static int
Configured	_ANSI_ARGS_((Tcl_Interp *interp, SNMP_Session *session));

static void
DeleteAgentInterp _ANSI_ARGS_((ClientData clientData, Tcl_Interp *interp));

static int
SnmpCmd		_ANSI_ARGS_((ClientData	clientData, Tcl_Interp *interp,
			     int argc, char **argv));
static int
SessionCmd	_ANSI_ARGS_((ClientData	clientData, Tcl_Interp *interp,
			     int argc, char **argv));
static int
WaitSession	_ANSI_ARGS_((Tcl_Interp *interp, SNMP_Session *session, 
			     char *id));
static void
DestroySession	_ANSI_ARGS_((ClientData clientdata));

static int
Request		_ANSI_ARGS_((Tcl_Interp *interp, SNMP_Session *session,
			     int pdu_type, int argc, char **argv));
static int
Walk		_ANSI_ARGS_((Tcl_Interp *interp, SNMP_Session *session,
			     int argc, char **argv));
static int
ExpandTable	_ANSI_ARGS_((Tcl_Interp *interp, 
			     char *tList, Tcl_DString *dst));
static int
ExpandScalars	_ANSI_ARGS_((Tcl_Interp *interp, 
			     char *sList, Tcl_DString *dst));
static int
Table		_ANSI_ARGS_((Tcl_Interp *interp, SNMP_Session *session,
			     int argc, char **argv));
static int
Scalars		_ANSI_ARGS_((Tcl_Interp *interp, SNMP_Session *session,
			     int argc, char **argv));
static void
ScalarSetVar	_ANSI_ARGS_((Tcl_Interp *interp, char *vbl,
			     char *varName, Tcl_DString *result));

/*
 * The options used to configure snmp session objects.
 */

#define TNM_SNMP_OPT_ADDRESS	     1
#define TNM_SNMP_OPT_TRAPADDRESS     2
#define TNM_SNMP_OPT_PORT	     3
#define TNM_SNMP_OPT_TRAPPORT	     4
#define TNM_SNMP_OPT_VERSION	     5
#define TNM_SNMP_OPT_READCOMMUNITY   6
#define TNM_SNMP_OPT_WRITECOMMUNITY  7
#define TNM_SNMP_OPT_TRAPCOMMUNITY   8
#define TNM_SNMP_OPT_USER	     9
#define TNM_SNMP_OPT_PASSWORD	    10
#define TNM_SNMP_OPT_CONTEXT	    11
#define TNM_SNMP_OPT_AGENT	    12
#define TNM_SNMP_OPT_ALIAS	    13
#define TNM_SNMP_OPT_TIMEOUT	    14
#define TNM_SNMP_OPT_RETRIES	    15
#define TNM_SNMP_OPT_WINDOW	    16
#define TNM_SNMP_OPT_DELAY	    17
#ifdef SNMP_BENCH
#define TNM_SNMP_OPT_RTT	    18
#define TNM_SNMP_OPT_SENDSIZE	    19
#define TNM_SNMP_OPT_RECVSIZE	    20
#endif

static TnmTable optionTable[] = {
    { TNM_SNMP_OPT_ADDRESS,        "-address" },
    { TNM_SNMP_OPT_TRAPADDRESS,    "-trapaddress" },
    { TNM_SNMP_OPT_PORT,           "-port" },
    { TNM_SNMP_OPT_TRAPPORT,       "-trapport" },
    { TNM_SNMP_OPT_VERSION,        "-version" },
    { TNM_SNMP_OPT_READCOMMUNITY,  "-community" },
    { TNM_SNMP_OPT_WRITECOMMUNITY, "-writecommunity" },
    { TNM_SNMP_OPT_TRAPCOMMUNITY,  "-trapcommunity" },
    { TNM_SNMP_OPT_USER,           "-user" },
    { TNM_SNMP_OPT_PASSWORD,       "-password" },
    { TNM_SNMP_OPT_CONTEXT,        "-context" },
    { TNM_SNMP_OPT_AGENT,          "-agent" },
    { TNM_SNMP_OPT_ALIAS,          "-alias" },
    { TNM_SNMP_OPT_TIMEOUT,        "-timeout" },
    { TNM_SNMP_OPT_RETRIES,        "-retries" },
    { TNM_SNMP_OPT_WINDOW,         "-window" },
    { TNM_SNMP_OPT_DELAY,          "-delay" },
#ifdef SNMP_BENCH
    { TNM_SNMP_OPT_RTT,            "-rtt" },
    { TNM_SNMP_OPT_SENDSIZE,       "-sendSize" },
    { TNM_SNMP_OPT_RECVSIZE,       "-recvSize" },
#endif
    { 0, NULL }
};

static TnmConfig config = {
    optionTable,
    SetOption,
    GetOption
};

/*
 * The following structure describes a Tcl command that should be
 * evaluated once we receive a response for a SNMP request.
 */

typedef struct EvalCmd {
    Tcl_Interp *interp;
    char *cmd;
} EvalCmd;


/*
 *----------------------------------------------------------------------
 *
 * EvalCmdProc --
 *
 *	This procedure is called once we have received the response
 *	for an asynchronous SNMP request. It evaluates a Tcl script.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Arbitrary side effects since commands are evaluated.
 *
 *----------------------------------------------------------------------
 */

static void
EvalCmdProc(session, pdu, clientData)
    SNMP_Session *session;
    SNMP_PDU *pdu;
    ClientData clientData;
{
    EvalCmd *ec = (EvalCmd *) clientData;
    Tnm_SnmpEvalCallback(ec->interp, session, pdu, ec->cmd,
			 NULL, NULL, NULL, NULL);
    ckfree((char *) ec);
}

/*
 *----------------------------------------------------------------------
 *
 * GetOption --
 *
 *	This procedure retrieves the value of a session option.
 *
 * Results:
 *	A pointer to the value formatted as a string.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static char *
GetOption(interp, object, option)
    Tcl_Interp *interp;
    ClientData object;
    int option;
{
    SNMP_Session *session = (SNMP_Session *) object;
    static char buffer[256];
    int code;

    switch (option) {
    case TNM_SNMP_OPT_ADDRESS:
	return inet_ntoa(session->maddr.sin_addr);
#if 0
    case TNM_SNMP_OPT_TRAPADDRESS:
	return inet_ntoa(session->taddr.sin_addr);
#endif
    case TNM_SNMP_OPT_PORT:
	sprintf(buffer, "%u", ntohs(session->maddr.sin_port));
	return buffer;
#if 0
    case TNM_SNMP_OPT_TRAPPORT:
	sprintf(buffer, "%u", ntohs(session->taddr.sin_port));
	return buffer;
#endif
    case TNM_SNMP_OPT_VERSION:
	switch(session->version) {
	case TNM_SNMPv1:
	    return "SNMPv1";
#ifdef TNM_SNMPv2C
	case TNM_SNMPv2C:
	    return "SNMPv2c";
#endif
#ifdef TNM_SNMPv2U
	case TNM_SNMPv2U:
	    return "SNMPv2u";
	}
#endif
    case TNM_SNMP_OPT_READCOMMUNITY:
	if (session->version!=TNM_SNMPv1 && session->version!=TNM_SNMPv2C) {
	    return NULL;
	}
	return session->readCommunity ? session->readCommunity : "";
    case TNM_SNMP_OPT_WRITECOMMUNITY:
	if (session->version!=TNM_SNMPv1 && session->version!=TNM_SNMPv2C) {
	    return NULL;
	}
	return session->writeCommunity;
#if 0
    case TNM_SNMP_OPT_TRAPCOMMUNITY:
	if (session->version!=TNM_SNMPv1 && session->version!=TNM_SNMPv2C) {
	    return NULL;
	}
	return session->trapCommunity;
#endif
#ifdef TNM_SNMPv2U
    case TNM_SNMP_OPT_USER:
	if (session->version != TNM_SNMPv2U) {
	    return NULL;
	}
	memset(buffer, 0, USEC_MAX_USER);
	memcpy(buffer, session->userName, session->userNameLen);
	return buffer;
    case TNM_SNMP_OPT_PASSWORD:
	if (session->version != TNM_SNMPv2U) {
	    return NULL;
	}
	return session->password;
    case TNM_SNMP_OPT_CONTEXT:
	if (session->version != TNM_SNMPv2U) {
	    return NULL;
	}
	memset(buffer, 0, USEC_MAX_CONTEXT);
	memcpy(buffer, session->cntxt, session->cntxtLen);
	return buffer;
#endif
    case TNM_SNMP_OPT_AGENT:
	if (! session->agentInterp) {
	    return NULL;
	}
	Tcl_ResetResult(interp);
	code = Tcl_GetInterpPath(interp, session->agentInterp);
	if (code == TCL_OK) {
	    return interp->result;
	}
	return "";
    case TNM_SNMP_OPT_TIMEOUT:
	sprintf(buffer, "%d", session->timeout);
	return buffer;
    case TNM_SNMP_OPT_RETRIES:
	sprintf(buffer, "%d", session->retries);
	return buffer;
    case TNM_SNMP_OPT_WINDOW:
	sprintf(buffer, "%d", session->window);
	return buffer;
    case TNM_SNMP_OPT_DELAY:
	sprintf(buffer, "%d", session->delay);
	return buffer;
#ifdef SNMP_BENCH
    case TNM_SNMP_OPT_RTT:
	sprintf(buffer, "%d",
		(session->stats.recvTime.sec 
		 - session->stats.sendTime.sec) * 1000
		+ (session->stats.recvTime.usec 
		   - session->stats.sendTime.usec) / 1000);
	return buffer;
    case TNM_SNMP_OPT_SENDSIZE:
	sprintf(buffer, "%d", session->stats.sendSize);
	return buffer;
    case TNM_SNMP_OPT_RECVSIZE:
	sprintf(buffer, "%d", session->stats.recvSize);
	return buffer;
#endif
    }
    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * SetOption --
 *
 *	This procedure modifies a single option of a session object.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
SetOption(interp, object, option, value)
    Tcl_Interp *interp;
    ClientData object;
    int option;
    char *value;
{
    SNMP_Session *session = (SNMP_Session *) object;
    int num;

    switch (option) {
    case TNM_SNMP_OPT_ADDRESS:
	return TnmSetIPAddress(interp, value, &session->maddr);
    case TNM_SNMP_OPT_TRAPADDRESS:
	return TnmSetIPAddress(interp, value, &session->taddr);
    case TNM_SNMP_OPT_PORT:
	return TnmSetIPPort(interp, "udp", value, &session->maddr);
    case TNM_SNMP_OPT_TRAPPORT:
	return TnmSetIPPort(interp, "udp", value, &session->taddr);
    case TNM_SNMP_OPT_VERSION:
	if (strcmp(value, "SNMPv1") == 0) {
	    session->version = TNM_SNMPv1;
	    return TCL_OK;
	}
#ifdef TNM_SNMPv2C
	if (strcmp(value, "SNMPv2C") == 0 || strcmp(value, "SNMPv2c") == 0) {
	    session->version = TNM_SNMPv2C;
	    return TCL_OK;
	}
#endif
#ifdef TNM_SNMPv2U
	if (strcmp(value, "SNMPv2U") == 0 || strcmp(value, "SNMPv2u") == 0) {
	    session->version = TNM_SNMPv2U;
	    return TCL_OK;
	}
#endif
	Tcl_AppendResult(interp, "unknown SNMP version \"",
			 value, "\"", (char *) NULL);
	return TCL_ERROR;
    case TNM_SNMP_OPT_READCOMMUNITY:
#ifdef TNM_SNMPv2U
	if (session->version == TNM_SNMPv2U) {
	    session->version = TNM_SNMPv1;
	}
#endif
	if (session->readCommunity) ckfree(session->readCommunity);
	session->readCommunity = ckstrdup(value);
	return TCL_OK;
    case TNM_SNMP_OPT_WRITECOMMUNITY:
#ifdef TNM_SNMPv2U
	if (session->version == TNM_SNMPv2U) {
	    session->version = TNM_SNMPv1;
	}
#endif
	if (session->writeCommunity) ckfree(session->writeCommunity);
	session->writeCommunity = (*value) ? ckstrdup(value) : NULL;
	return TCL_OK;
    case TNM_SNMP_OPT_TRAPCOMMUNITY:
#ifdef TNM_SNMPv2U
	if (session->version == TNM_SNMPv2U) {
	    session->version = TNM_SNMPv1;
	}
#endif
	if (session->trapCommunity) ckfree(session->trapCommunity);
	session->trapCommunity = (*value) ? ckstrdup(value) : NULL;
	return TCL_OK;
#ifdef TNM_SNMPv2U
    case TNM_SNMP_OPT_USER:
	session->version = TNM_SNMPv2U;
	if (strlen(value) > USEC_MAX_USER) {
	    Tcl_SetResult(interp, "user name too long", TCL_STATIC);
	    return TCL_ERROR;
	}
	session->userNameLen = strlen(value);
	memcpy(session->userName, value, session->userNameLen);
	return TCL_OK;
    case TNM_SNMP_OPT_PASSWORD:
	session->version = TNM_SNMPv2U;
	if (session->password) {
	    ckfree(session->password);
	    session->password = NULL;
	}
	if (strlen(value) == 0) {
	    session->qos &= ~ USEC_QOS_AUTH;
	} else {
	    session->password = ckstrdup(value);
	    session->qos |= USEC_QOS_AUTH;
	}
	return TCL_OK;
    case TNM_SNMP_OPT_CONTEXT:
	if (strlen(value) > USEC_MAX_CONTEXT) {
	    Tcl_SetResult(interp, "context name too long", TCL_STATIC);
	    return TCL_ERROR;
	}
	session->cntxtLen = strlen(value);
	memcpy(session->cntxt, value, session->cntxtLen);
	return TCL_OK;
#endif
    case TNM_SNMP_OPT_AGENT:
	session->agentInterp = Tcl_GetSlave(interp, value);
	if (! session->agentInterp) {
	    Tcl_AppendResult(interp, "unknown interp \"", value, "\"",
			     (char *) NULL);
	    return TCL_ERROR;
	}
	Tcl_CallWhenDeleted(session->agentInterp, DeleteAgentInterp, 
			    (ClientData) session);
	if (session->agentInterp != interp) {
	    Tcl_CreateCommand(session->agentInterp, session->name,
			      SessionCmd, (ClientData) session,
			      DestroySession);
	}
	if (! Tcl_IsSafe(session->agentInterp)) {
	    TnmWriteMessage(interp, "Warning: SNMP agent created based on ");
	    TnmWriteMessage(interp, "an unsafe Tcl interpreter!\n");
	}
	return TCL_OK;
    case TNM_SNMP_OPT_ALIAS:
	{
	    Tcl_HashEntry *entryPtr;
	    int i, largc, code;
	    char **largv, **argv;
	    char *alias;
	    entryPtr = Tcl_FindHashEntry(&aliasTable, value);
	    if (! entryPtr) {
		Tcl_AppendResult(interp, "unknown alias \"",
				 value, "\"", (char *) NULL);
		return TCL_ERROR;
	    }
	    alias = (char *) Tcl_GetHashValue(entryPtr);
	    if (! alias) {
		Tcl_SetResult(interp, "alias loop detected", TCL_STATIC);
		return TCL_ERROR;
	    }
	    if (Tcl_SplitList(interp, alias, &largc, &largv) != TCL_OK) {
		return TCL_ERROR;
	    }
	    Tcl_SetHashValue(entryPtr, NULL);
	    argv = (char **) ckalloc((largc + 3) * sizeof(char *));
	    argv[0] = session->name;
	    argv[1] = "configure";
	    for (i = 0; i < largc; i++) {
		argv[i+2] = largv[i];
	    }
	    code = TnmSetConfig(interp, &config, session, largc + 2, argv);
	    Tcl_SetHashValue(entryPtr, alias);
	    ckfree((char *) argv);
	    ckfree((char *) largv);
	    return code;
	}
    case TNM_SNMP_OPT_TIMEOUT:
	if (TnmGetPositive(interp, value, &num) != TCL_OK) {
	    return TCL_ERROR;
	}
	session->timeout = num;
	return TCL_OK;
    case TNM_SNMP_OPT_RETRIES:
	if (TnmGetUnsigned(interp, value, &num) != TCL_OK) {
	    return TCL_ERROR;
	}
	session->retries = num;
	return TCL_OK;
    case TNM_SNMP_OPT_WINDOW:
	if (TnmGetUnsigned(interp, value, &num) != TCL_OK) {
	    return TCL_ERROR;
	}
	session->window = num;
	return TCL_OK;
    case TNM_SNMP_OPT_DELAY:
	if (TnmGetUnsigned(interp, value, &num) != TCL_OK) {
	    return TCL_ERROR;
	}
	session->delay = num;
	return TCL_OK;
    }

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Configured --
 *
 *	This procedure determines if all required parameters for
 *	communication with a remote SNMP entity are set.
 *
 * Results:
 *	A standard TCL result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
Configured(interp, session)
    Tcl_Interp *interp;
    SNMP_Session *session;
{
    char *name = Tcl_GetCommandName(interp, session->token);

    if (! session->version) {
        Tcl_AppendResult(interp, "session \"", name, 
			 "\" not configured", (char *) NULL);
	return TCL_ERROR;
    }

    switch (session->version) {
    case TNM_SNMPv1: 
	if (! session->readCommunity) {
	    Tcl_AppendResult(interp, "session \"", name,
			     "\" has no community string", (char *) NULL);
	    return TCL_ERROR;
	}
	break;
	
#ifdef TNM_SNMPv2C
    case TNM_SNMPv2C: 
	if (! session->readCommunity) {
	    Tcl_AppendResult(interp, "session \"", name,
			     "\" has no community string", (char *) NULL);
	    return TCL_ERROR;
	}
	break;
#endif
	
#ifdef TNM_SNMPv2U
    case TNM_SNMPv2U: 
	if (session->userNameLen == 0) {
	    Tcl_AppendResult(interp, "session \"", name,
			     "\" has no user name", (char *) NULL);
	    return TCL_ERROR;
	}
	break;
#endif
	
    default: 
	Tcl_SetResult(interp, "unknown SNMP version", TCL_STATIC);
	return TCL_ERROR;
    }

    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Tnm_SnmpInit --
 *
 *	This procedure initializes the SNMP extension. Note, it 
 *	does not initialize the `mib' command.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

int
Tnm_SnmpInit(interp)
    Tcl_Interp *interp;
{
    Tnm_SnmpSysUpTime();
    memset((char *) &snmpStats, 0, sizeof(SNMP_Statistics));

    Tcl_CreateCommand(interp, "snmp", SnmpCmd, (ClientData) NULL,
		      (Tcl_CmdDeleteProc *) NULL);

    Tcl_InitHashTable(&aliasTable, TCL_STRING_KEYS);

    srand(time(NULL) * getpid());

    return Tnm_MibInit(interp);
}

/*
 *----------------------------------------------------------------------
 *
 * DeleteAgentInterp --
 *
 *	This procedure is called whenever the interpreter registered
 *	for an agent session is deleted. This makes sure that we
 *	cleanup all state in the parent interpreter.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static void
DeleteAgentInterp(clientData, interp)
    ClientData clientData;
    Tcl_Interp *interp;
{
    SNMP_Session *session = (SNMP_Session *) clientData;
    session->agentInterp = NULL;
    Tcl_DeleteCommand(interp, session->name);
}

/*
 *----------------------------------------------------------------------
 *
 * SnmpCmd --
 *
 *	This procedure is invoked to process the "snmp" command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

static int
SnmpCmd(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    SNMP_Session *session;
    int code;

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			 " option ?arg arg ...?\"", (char *) NULL);
	return TCL_ERROR;
    }

    /*
     * snmp subcommand "alias"
     */
    
    if (strcmp(argv[1], "alias") == 0) {
        Tcl_HashEntry *entryPtr;
        if (argc == 2) {
	    Tcl_HashSearch search;
	    entryPtr = Tcl_FirstHashEntry(&aliasTable, &search);
	    while (entryPtr) {
	        Tcl_AppendElement(interp,
				  Tcl_GetHashKey(&aliasTable, entryPtr));
	        entryPtr = Tcl_NextHashEntry(&search);
	    }
	    return TCL_OK;
	} else if (argc == 3) {
	    entryPtr = Tcl_FindHashEntry(&aliasTable, argv[2]);
	    if (entryPtr) {
	        Tcl_SetResult(interp, (char *) Tcl_GetHashValue(entryPtr),
			      TCL_STATIC);
	    }
	    return TCL_OK;
	} else if (argc == 4) {
	    int isNew;
	    entryPtr = Tcl_CreateHashEntry(&aliasTable, argv[2], &isNew);
	    if (!isNew) {
		ckfree((char *) Tcl_GetHashValue(entryPtr));
	    }
	    if (*argv[3] == '\0') {
		Tcl_DeleteHashEntry(entryPtr);
	    } else {
		Tcl_SetHashValue(entryPtr, ckstrdup(argv[3]));
	    }
	    return TCL_OK;
	} else {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
			     " alias ?agent? ?config?\"", (char *) NULL);
	    return TCL_ERROR;
	}
    }

    if (strcmp(argv[1], "session") == 0) {

	/*
	 * snmp subcommand "session" -- create socket if neccessary 
	 * (this installs the event handler).
	 */
	
	if (Tnm_SnmpManagerOpen(interp) != TCL_OK) {
	    return TCL_ERROR;
	}

	/* 
	 * Call an arbitrary mib command to trigger the autoload
 	 * mechanism -- ugly but correct in most cases.
	 */

	Tcl_Eval(interp, "mib oid 1");
	Tcl_ResetResult(interp);
	
	/*
	 * Allocate and configure a new session handle.
	 */
	
	session = Tnm_SnmpCreateSession();
	session->interp = interp;
	code = TnmSetConfig(interp, &config, session, argc, argv);
	if (code == TCL_OK && session->agentInterp) {
	    code = Tnm_SnmpAgentInit(session->agentInterp, session);
	}
	if (code != TCL_OK) {
	    Tnm_SnmpDeleteSession(session);
	    return TCL_ERROR;
	}
#ifdef TNM_SNMPv2U
	if (session->version == TNM_SNMPv2U && session->qos & USEC_QOS_AUTH) {
	    Tnm_SnmpUsecGetAgentID(session);
	}
#endif

	session->nextPtr = sessionList;
	sessionList = session;

	/*
	 * Finally create a Tcl command for this session.
	 */
	
	session->token = Tcl_CreateCommand(interp, session->name, SessionCmd,
			  (ClientData) session, DestroySession);
	Tcl_SetResult(interp, session->name, TCL_STATIC);
	return TCL_OK;

    } else if (strcmp(argv[1], "info") == 0) {

	/*
	 * snmp subcommand "info" -- simply output the list of
	 * open sessions.
	 */
	
        if (argc != 2) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
			     argv[0], " info\"", (char *) NULL);
	    return TCL_ERROR;
	}
	for (session = sessionList; session; session = session->nextPtr) {
	    if (session->interp == interp) {
		Tcl_AppendElement(interp,
				  Tcl_GetCommandName(interp, session->token));
	    }
	}
	return TCL_OK;
    
    } else if (strcmp(argv[1], "wait") == 0) {

	/*
	 * snmp subcommand "wait" -- we have to start the loop every time
	 * we have done an event because the event may have changed the 
	 * session list.
	 */
	
        if (argc != 2) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"",
			     argv[0], " wait\"", (char *) NULL);
	    return TCL_ERROR;
	}
      repeat:
	for (session = sessionList; session; session = session->nextPtr) {
	    if (Tnm_SnmpQueueRequest(session, NULL)) {
		Tcl_DoOneEvent(0);
		goto repeat;
	    }
	}
	return TCL_OK;
    
    } else if (strcmp(argv[1], "watch") == 0) {

	/*
	 * snmp subcomand "watch" -- simply toggle the hexdump variable.
	 */

	if (argc > 3) {
            Tcl_AppendResult(interp, "wrong # args: should be \"",
			     argv[0], " watch ?bool?\"", (char *) NULL);
            return TCL_ERROR;
        }
	if (argc < 3) {
	    Tcl_SetResult(interp, hexdump ? "1" : "0", TCL_STATIC);
	    return TCL_OK;
	}
	return (Tcl_GetBoolean(interp, argv[2], &hexdump));
    }
    
    /*
     * invalid "snmp" subcommand
     */

    Tcl_AppendResult(interp, "bad option \"", argv[1], 
		     "\": should be alias, info, session, ",
		     "wait, or watch", (char *) NULL);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * SessionCmd --
 *
 *	This procedure is invoked to process a session command.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *----------------------------------------------------------------------
 */

static int
SessionCmd(clientData, interp, argc, argv)
    ClientData clientData;
    Tcl_Interp *interp;
    int argc;
    char **argv;
{
    SNMP_Session *session = (SNMP_Session *) clientData;
    int code;

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
			 argv[0], " option ?arg arg ...?\"", (char *) NULL);
	return TCL_ERROR;
    }

    if (strcmp(argv[1], "configure") == 0) {

	/*
	 * This call to WaitSession() is actually required so that a 
	 * configuration change does not affect queued requests.
	 */

	Tcl_Preserve((ClientData) session);
	WaitSession(interp, session, NULL);
 	code = TnmSetConfig(interp, &config, session, argc, argv);
	if (code == TCL_OK && session->agentInterp) {
	    code = Tnm_SnmpAgentInit(session->agentInterp, session);
	}
	if (code != TCL_OK) {
	    Tcl_Release((ClientData) session);
	    return TCL_ERROR;
	}
#ifdef TNM_SNMPv2U
	if (session->version == TNM_SNMPv2U && session->qos & USEC_QOS_AUTH) {
	    Tnm_SnmpUsecGetAgentID(session);
	}
#endif
	Tcl_Release((ClientData) session);
	return TCL_OK;
    } else if (strcmp(argv[1], "wait") == 0) {
	if (argc == 2) {
	    return WaitSession(interp, session, NULL);
	} else if (argc == 3) {
	    return WaitSession(interp, session, argv[2]);
	}
	Tcl_AppendResult(interp, "wrong # args: should be \"",
			 argv[0], " wait ?request?\"", (char *) NULL);
	return TCL_ERROR;
    } else if (strcmp(argv[1], "destroy") == 0) {
	
	/*
	 * If this is an agent session and the agent interpreter
	 * is the same as this one, make sure that the command
	 * is not removed twice. Therefore, we set the agent
	 * interpreter pointer to NULL.
	 */

	if (session->agentInterp == interp) {
	    session->agentInterp = NULL;
	}
	Tcl_DeleteCommand(interp, argv[0]);
	return TCL_OK;
    }

    if (strcmp(argv[1], "cget") == 0) {
	return TnmGetConfig(interp, &config, session, argc, argv);
    }

    /*
     * All commands below need a well configured session.
     */

    if (Configured(interp, session) != TCL_OK) {
	return TCL_ERROR;
    }

    if (strcmp(argv[1], "get") == 0) {
	return Request(interp, session, TNM_SNMP_GET, argc-1, argv+1);

    } else if (strcmp(argv[1], "getnext") == 0) {
	return Request(interp, session, TNM_SNMP_GETNEXT, argc-1, argv+1);

    } else if (strcmp(argv[1], "getbulk") == 0) {
	return Request(interp, session, TNM_SNMP_GETBULK, argc-1, argv+1);

    } else if (strcmp(argv[1], "set") == 0) {
	return Request(interp, session, TNM_SNMP_SET, argc-1, argv+1);
	
    } else if (strcmp(argv[1], "walk") == 0) {
	return Walk(interp, session, argc-1, argv+1);

    } else if (strcmp(argv[1], "table") == 0) {
	return Table(interp, session, argc-1, argv+1);

    } else if (strcmp(argv[1], "scalars") == 0) {
	return Scalars(interp, session, argc-1, argv+1);

    } else if (strcmp(argv[1], "inform") == 0) {
	if (session->version == TNM_SNMPv1) {
	    char *name = Tcl_GetCommandName(interp, session->token);
	    Tcl_AppendResult(interp, "inform option not allowed on ",
			     "SNMPv1 session \"", name, "\"", (char *) NULL);
	    return TCL_ERROR;
	}  else if (session->version & TNM_SNMPv2) {
	    return Request(interp, session, TNM_SNMP_INFORM, argc-1, argv+1);
	} else {
	    Tcl_SetResult(interp, "unknown SNMP version", TCL_STATIC);
            return TCL_ERROR;
	}

    } else if (strcmp(argv[1], "trap") == 0) {
	if (session->version == TNM_SNMPv1) {
	    return Request(interp, session, TNM_SNMPv1_TRAP, argc-1, argv+1);
	} else if (session->version & TNM_SNMPv2) {
	    return Request(interp, session, TNM_SNMPv2_TRAP, argc-1, argv+1);
	} else {
	    Tcl_SetResult(interp, "unknown SNMP version", TCL_STATIC);
            return TCL_ERROR;
        }

    } else if (strcmp(argv[1], "bind") == 0) {
	int event;
	SNMP_Binding *bindPtr = session->bindPtr;
	if (argc < 4 || argc > 5) {
	    Tcl_AppendResult(interp, "wrong # of args: should be \"",
			     argv[0], " bind label event ?command?\"", 
			     (char *) NULL);
	    return TCL_ERROR;
	}
	event = TnmGetTableKey(tnmSnmpEventTable, argv[3]);
	if (argv[2][0] == '\0') {
	    
	    if (event < 0 || (event & TNM_SNMP_GENERIC_BINDINGS) == 0) {
		Tcl_AppendResult(interp, "unknown event \"", argv[3], 
				 "\": use trap, inform, recv, send, ",
				 "begin, end, or report", (char *) NULL);
		return TCL_ERROR;
	    }
	    if (event & (TNM_SNMP_TRAP_EVENT | TNM_SNMP_INFORM_EVENT)) {
		if (! session->traps) {
		    if (Tnm_SnmpTrapOpen(interp) != TCL_OK) {
			return TCL_ERROR;
		    }
		    session->traps = 1;
		}
	    }
	    while (bindPtr) {
		if (bindPtr->event == event) break;
		bindPtr = bindPtr->nextPtr;
	    }
	    if (argc == 4) {
		if (bindPtr) {
		    Tcl_SetResult(interp, bindPtr->command, TCL_STATIC);
		}
	    } else {
		if (! bindPtr) {
		    bindPtr = (SNMP_Binding *) ckalloc(sizeof(SNMP_Binding));
		    memset ((char *) bindPtr, 0, sizeof(SNMP_Binding));
		    bindPtr->event = event;
		    bindPtr->nextPtr = session->bindPtr;
		    session->bindPtr = bindPtr;
		}
		if (bindPtr->command) {
		    ckfree (bindPtr->command);
		}
		bindPtr->command = ckstrdup(argv[4]);
	    }
	} else {

	    char *oidstr = Tnm_MibGetOid(argv[2], 0);
	    Tnm_Oid *oid;
	    int code, oidlen;
	    
	    if (!oidstr) {
		Tcl_AppendResult(interp, "no object \"", argv[2], "\"",
				 (char *) NULL);
		return TCL_ERROR;
	    }
	    
	    if (event < 0 || (event & TNM_SNMP_INSTANCE_BINDINGS) == 0) {
		Tcl_AppendResult(interp, "unknown event \"", argv[3],
				 "\": use get, set, create, check, ",
				 "commit, rollback, format, or scan", 
				 (char *) NULL);
		return TCL_ERROR;
	    }
	    
	    if (argc == 5) {
		oid = Tnm_StrToOid(oidstr, &oidlen);
		code = Tnm_SnmpSetNodeBinding(session, oid, oidlen,
					      event, argv[4]);
		if (code != TCL_OK) {
		    Tcl_AppendResult(interp, "unknown instance \"",
				     argv[2], "\"", (char *) NULL);
		    return TCL_ERROR;
		}
	    } else {
		char *cmd;
		oid = Tnm_StrToOid(oidstr, &oidlen);
		cmd = Tnm_SnmpGetNodeBinding(session, oid, oidlen, event);
		Tcl_SetResult(interp, cmd ? cmd : "", TCL_STATIC);
	    }
	}
	return TCL_OK;

    } else if (strcmp(argv[1], "instance") == 0) {
	int code;
        if (argc < 4 || argc > 5) {
	    Tcl_AppendResult(interp,  "wrong # args: should be \"",
			     argv[0], " instance oid varName ?defval?\"",
			     (char *) NULL);
	    return TCL_ERROR;
	}
	
        if (! session->agentInterp) {
	    char *name = Tcl_GetCommandName(interp, session->token);
	    Tcl_AppendResult(interp, "invalid agent session \"", 
			     name, "\"", (char *) NULL);
	    return TCL_ERROR;
	}
	code = Tnm_SnmpCreateNode(session->agentInterp, argv[2], argv[3],
				  (argc > 4) ? argv[4] : "");
	if (code != TCL_OK) {
	    if (interp != session->agentInterp) {
		Tcl_SetResult(interp, session->agentInterp->result, 
			      TCL_VOLATILE);
		Tcl_ResetResult(session->agentInterp);
	    }
	    return code;
	}
	return TCL_OK;
    }

    Tcl_AppendResult(interp, "bad option \"", argv[1], "\": should be ",
		     "configure, cget, wait, destroy, ",
		     "get, getnext, getbulk, set, trap, inform, walk, ",
		     "scalars, table instance, or bind", (char *) NULL);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * WaitSession --
 *
 *	This procedure processes events until either the list of
 *	outstanding requests is empty or until the given request
 *	is no longer in the list of outstanding requests.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	Tcl events are processed which can cause arbitrary side effects.
 *
 *----------------------------------------------------------------------
 */

static int
WaitSession(interp, session, request)
    Tcl_Interp *interp;
    SNMP_Session *session;
    char *request;
{
    u_int id = 0;
    char *name = Tcl_GetCommandName(interp, session->token);

    if (! name) {
	return TCL_OK;
    }

    if (request) {
	char *p;
	for (p = request; isdigit(*p); p++) {
	    id = 10 * id + *p - '0';
	}
    }
    
    /*
     * Do not use the session pointer! We have to search for the
     * session name after each single event because the session
     * may be deleted as a side effect of the event.
     */

    name = ckstrdup(name);
  repeat:
    for (session = sessionList; session; session = session->nextPtr) {
	if (strcmp(session->name, name) != 0) continue;
	if (! id) {
	    if (Tnm_SnmpQueueRequest(session, NULL)) {
		Tcl_DoOneEvent(0);
		goto repeat;
	    }
	} else {
	    if (Tnm_SnmpFindRequest(id)) {
		Tcl_DoOneEvent(0);
		goto repeat;
	    }
	}
    }
    
    ckfree(name);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * DestroySession --
 *
 *	This procedure is invoked when a session handle is deleted.
 *	It frees the associated session structure and de-installs all
 *	pending events. If it is the last session, we also close the
 *	socket for manager communication.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static void
DestroySession(clientData)
    ClientData clientData;
{
    SNMP_Session *p, *q, *session = (SNMP_Session *) clientData;

    if (session->agentInterp) {
	Tcl_Interp *agentInterp = session->agentInterp;
	session->agentInterp = NULL;
	Tcl_DontCallWhenDeleted(agentInterp, DeleteAgentInterp,
				(ClientData) session);
	Tcl_DeleteCommand(agentInterp, session->name);
    }
    
    for (p = sessionList, q = NULL; p != NULL; q = p, p = p->nextPtr) {
	if (p == session) break;
    }
    if (!p) return;

    if (q == NULL) {
	sessionList = p->nextPtr;
    } else {
	q->nextPtr = p->nextPtr;
    }

    Tnm_SnmpDeleteSession(session);

    if (sessionList == NULL) {
	Tnm_SnmpManagerClose();
    }
}

/*
 *----------------------------------------------------------------------
 *
 * Request --
 *
 *	This procedure creates a pdu structure and calls Tnm_SnmpEncode
 *	to send the packet to the destination.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
Request(interp, session, pdu_type, argc, argv)
    Tcl_Interp *interp;
    SNMP_Session *session;
    int pdu_type;
    int argc;
    char **argv;
{
    char *cmd = NULL;
    SNMP_PDU _pdu;
    SNMP_PDU *pdu = &_pdu;
    
    /* 
     * Initialize the PDU:
     */

    pdu->addr	      = session->maddr;
    pdu->type         = pdu_type;
    pdu->request_id   = TnmSnmpGetRequestId();
    pdu->error_status = TNM_SNMP_NOERROR;
    pdu->error_index  = 0;    
    pdu->trapOID      = NULL;
    Tcl_DStringInit (&pdu->varbind);

#ifdef SNMP_BENCH
    memset((char *) &session->stats, 0, sizeof(session->stats));
#endif

    /*
     * Check # of arguments:
     */
    
    if ((pdu->type == TNM_SNMP_GETBULK && argc < 4) 
	|| (pdu->type == TNM_SNMPv1_TRAP && argc < 3)
	|| (pdu->type == TNM_SNMPv2_TRAP && argc < 3)
	|| (pdu->type == TNM_SNMP_INFORM && argc < 3)
	|| (argc < 2)) {
	goto usage;
    }
    
    /*
     * Read NonRepeaters and MaxRepetitions for GetBulkRequest:
     */
    
    if (pdu->type == TNM_SNMP_GETBULK) {
	int num;
	if (--argc) {
	    if (Tcl_GetInt(interp, *++argv, &num) != TCL_OK) goto errorExit;
	    pdu->error_status = (num < 0) ? 0 : num;
	}
	if (--argc) {
	    if (Tcl_GetInt(interp, *++argv, &num) != TCL_OK) goto errorExit;
	    pdu->error_index  = (num < 0) ? 0 : num;
	}
    } else if (pdu->type == TNM_SNMPv1_TRAP 
	       || pdu->type == TNM_SNMPv2_TRAP
	       || pdu->type == TNM_SNMP_INFORM) {
	argc--;
	if (Tnm_IsOid (*++argv)) {
	    pdu->trapOID = ckstrdup(*argv);
	} else {
	    char *tmp = Tnm_MibGetOid(*argv, 0);
	    if (! tmp) {
		Tcl_AppendResult(interp,  "no object \"", *argv, "\"",
				 (char *) NULL);
		goto errorExit;	
	    }    
	    pdu->trapOID = ckstrdup(tmp);
	}
    } else {
	pdu->error_status = TNM_SNMP_NOERROR;
	pdu->error_index  = 0;
    }
    
    /*
     * Check for varbind-list and split it into args:
     */
    
    if (!argc) goto usage;    
    Tcl_DStringAppend(&pdu->varbind, *++argv, -1);

    /*
     * Check for the callback function:
     */
    
    if (--argc && *++argv != NULL) {
	cmd = *argv;
    }

    if (cmd) {
	EvalCmd *ec = (EvalCmd *) ckalloc(sizeof(EvalCmd) + strlen(cmd) + 1);
	ec->interp = interp;
	ec->cmd = (char *) ec + sizeof(EvalCmd);
	strcpy(ec->cmd, cmd);
	if (Tnm_SnmpEncode(interp, session, pdu, EvalCmdProc, ec) != TCL_OK) {
	    goto errorExit;
	}
    } else {
	if (Tnm_SnmpEncode(interp, session, pdu, NULL, NULL) != TCL_OK) {
	    goto errorExit;
	}
    }

    if (pdu->trapOID) ckfree(pdu->trapOID);
    Tcl_DStringFree(&pdu->varbind);
    return TCL_OK;
    
  usage:
    if (pdu->type == TNM_SNMP_GETBULK) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", session->name,
			 " getbulk non-repeaters max-repetitions ",
			 "list ?callback?\"", (char *) NULL);
    } else if (pdu->type == TNM_SNMPv1_TRAP 
	       || pdu->type == TNM_SNMPv2_TRAP) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", session->name,
			 " trap snmpTrapOID list\"", (char *) NULL);
    } else if (pdu->type == TNM_SNMP_INFORM) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", session->name,
			 " inform snmpTrapOID list\"", (char *) NULL);
    } else {
	Tcl_AppendResult(interp, "wrong # args: should be \"", session->name,
			 " ", *argv, " list ?callback?\"", (char *) NULL);
    }
    
  errorExit:
    if (pdu->trapOID) ckfree(pdu->trapOID);
    Tcl_DStringFree(&pdu->varbind);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * Walk --
 *
 *	This procedure walks a MIB tree. It evaluates the given Tcl
 *	command foreach varbind retrieved using getbulk requests.
 *	First, all variables contained in the list argument are
 *	converted to their OIDs. Then we loop using gebulk requests
 *	until we get an error or until one returned variable starts
 *	with an OID not being a valid prefix.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
Walk(interp, session, argc, argv)
    Tcl_Interp *interp;
    SNMP_Session *session;
    int argc;
    char **argv;
{
    int i, j, k, result;
    int oidc, respc;
    char **oidv = NULL, **respv = NULL;
    SNMP_PDU _pdu, *pdu = &_pdu;
    int numRepeaters = 0;
    
    if (argc != 4) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
			 session->name, " walk varName list command\"",
			 (char *) NULL);
	return TCL_ERROR;
    }

    /*
     * Initialize the PDU.
     */

    pdu->addr         = session->maddr;
    pdu->type         = TNM_SNMP_GETBULK;
    pdu->request_id   = TnmSnmpGetRequestId();
    pdu->error_status = TNM_SNMP_NOERROR;
    pdu->error_index  = 0;    
    pdu->trapOID      = NULL;
    Tcl_DStringInit(&pdu->varbind);

    /*
     * Save the oid prefix contained in list in oidv and oidc.
     */
    
    result = Tcl_SplitList(interp, argv[2], &oidc, &oidv);
    if (result != TCL_OK) {
	return result;
    }
    if (oidc == 0) {
	result = TCL_OK;
	goto loopDone;
    }
    
    for (i = 0; i < oidc; i++) {
	char *tmp = Tnm_MibGetOid(oidv[i], 0);
	if (!tmp) {
	    Tcl_AppendResult(interp,  "no object \"", oidv[i], "\"",
			     (char *) NULL);
	    ckfree((char *) oidv);
	    Tcl_DStringFree(&pdu->varbind);
            return TCL_ERROR;
	}
	oidv[i] = ckalloc(strlen(tmp) + 2);
	strcpy(oidv[i], tmp);
	strcat(oidv[i], ".");
	Tcl_DStringAppendElement(&pdu->varbind, tmp);
    }

    while (1) {

	pdu->type         = TNM_SNMP_GETBULK;
	pdu->request_id   = TnmSnmpGetRequestId();

	/* 
	 * Set the non-repeaters and the max-repetitions for the getbulk
	 * operation. We use the sequence 8 16 24 32 40 48 to increase
	 * the `warp' factor with every repetition.
	 *
	 * Some measurements show some real bad effects. If you increase
	 * the warp factor too much, you will risk timeouts because the
	 * agent might need a lot of time to build the response. If the
	 * agent does not cache response packets, you will get very bad 
	 * performance. Therefore, I have limited the `warp' factor to
	 * 24 which works well with scotty's default parameters on an
	 * Ethernet. (This number should not be hard coded but perhaps
	 * it is better so because everyone should read this comment.
	 */

	if (numRepeaters < 24 ) {
	    numRepeaters += 8;
	}

	pdu->error_status = 0;
	pdu->error_index  = (numRepeaters / oidc > 0) 
	    ? numRepeaters / oidc : 1;

	result = Tnm_SnmpEncode(interp, session, pdu, NULL, NULL);
	if (result == TCL_ERROR 
	    && (strncmp(interp->result, "noSuchName ", 11) == 0)) {
	    result = TCL_OK;
	    goto loopDone;
	}
	if (result != TCL_OK) {
            break;
        }
	
	if (respv) ckfree((char *) respv);
	result = Tcl_SplitList(interp, interp->result, &respc, &respv);
	if (result != TCL_OK) {
	    goto loopDone;
	}

	if (respc < oidc) {
	    Tcl_SetResult(interp, "response with wrong # of varbinds",
			  TCL_STATIC);
	    result = TCL_ERROR;
	    goto loopDone;
	}

	for (j = 0; j < respc / oidc; j++) {

	    for (i = 0; i < oidc; i++) {
		if (strncmp(oidv[i], respv[j * oidc + i], 
			    strlen(oidv[i])) != 0) {
		    result = TCL_OK;
		    goto loopDone;
		}
	    }

	    Tcl_DStringFree(&pdu->varbind);
	    for (k = j * oidc; k < (j+1) * oidc; k++) {
		int vbc;
		char **vbv;
		result = Tcl_SplitList(interp, respv[k], &vbc, &vbv);
		if (result != TCL_OK) {
		    goto loopDone;
		}
		if (strcmp(vbv[1], "endOfMibView") == 0) {
		    ckfree((char *) vbv);
		    result = TCL_OK;
		    goto loopDone;
		}
		ckfree((char *) vbv);
		Tcl_DStringAppendElement(&pdu->varbind, respv[k]);
	    }

	    if (Tcl_SetVar(interp, argv[1], Tcl_DStringValue(&pdu->varbind),
			   TCL_LEAVE_ERR_MSG) == NULL) {
		result = TCL_ERROR;
		goto loopDone;
	    }

	    result = Tcl_Eval(interp, argv[3]);
	    if (result != TCL_OK) {
		if (result == TCL_CONTINUE) {
		    result = TCL_OK;
		} else if (result == TCL_BREAK) {
		    result = TCL_OK;
		    goto loopDone;
		} else if (result == TCL_ERROR) {
		    char msg[100];
		    sprintf(msg, "\n    (\"%s walk\" body line %d)",
			    session->name, interp->errorLine);
		    Tcl_AddErrorInfo(interp, msg);
		    goto loopDone;
		} else {
		    goto loopDone;
		}
	    }
	}
    }

  loopDone:
    for (i = 0; i < oidc; i++) {
	ckfree(oidv[i]);
    }
    ckfree((char *) oidv);
    if (respv) ckfree((char *) respv);
    Tcl_DStringFree(&pdu->varbind);

    /*
     * noSuchName errors mark the end of a SNMPv1 MIB view and hence 
     * they are no real errors. So we ignore them here.
     */

    if (result == TCL_ERROR 
	&& (strncmp(interp->result, "noSuchName", 10) == 0)) {
	result = TCL_OK;
    }

    if (result == TCL_OK) {
	Tcl_ResetResult(interp);
    }
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * ExpandTable --
 *
 *	This procedure expands the list of table variables or a single
 *	table name into a list of MIB instances.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
ExpandTable(interp, tList, dst)
    Tcl_Interp *interp;
    char *tList;
    Tcl_DString *dst;
{
    int i, argc, code;
    char **argv = NULL;
    Tnm_MibNode *nodePtr, *entryPtr = NULL, *tablePtr = NULL;
    
    code = Tcl_SplitList(interp, tList, &argc, &argv);
    if (code != TCL_OK) {
	return TCL_ERROR;
    }

    for (i = 0; i < argc; i++) {

	/*
	 * Lookup the given object.
	 */

	nodePtr = Tnm_MibFindNode(argv[i], NULL, 0);
	if (! nodePtr) {
	    Tcl_AppendResult(interp, "unknown mib table \"", argv[i], "\"",
			     (char *) NULL);
	    ckfree((char *) argv);
	    return TCL_ERROR;
	}

	/*
	 * Locate the entry (SEQUENCE) that contains this object.
	 */

	switch (nodePtr->syntax) {
	  case ASN1_SEQUENCE:
	    entryPtr = nodePtr;
	    break;
	  case ASN1_SEQUENCE_OF: 
	    if (nodePtr->childPtr) {
		entryPtr = nodePtr->childPtr;
	    }
	    break;
	  default:
	    if (nodePtr->parentPtr && nodePtr->childPtr == NULL
		&& nodePtr->parentPtr->syntax == ASN1_SEQUENCE) {
		entryPtr = nodePtr->parentPtr;
	    } else {
	    unknownTable:
		Tcl_AppendResult(interp, "not a table \"", argv[i], "\"",
				 (char *) NULL);
		ckfree((char *) argv);
		return TCL_ERROR;
	    }
	}

	/*
	 * Check whether all objects belong to the same table.
	 */

	if (entryPtr == NULL || entryPtr->parentPtr == NULL) {
	    goto unknownTable;
	}

	if (tablePtr == NULL) {
	    tablePtr = entryPtr->parentPtr;
	}
	if (tablePtr != entryPtr->parentPtr) {
	    Tcl_AppendResult(interp, "instances not in the same table",
			     (char *) NULL);
	    ckfree((char *) argv);
	    return TCL_ERROR;
	}

	/*
	 * Now add the nodes to the list. Expand SEQUENCE nodes to
	 * include all child nodes. Check the access mode which must
	 * allow at least read access.
	 */

	if (nodePtr == entryPtr || nodePtr == tablePtr) {
	    Tnm_MibNode *nPtr;
	    for (nPtr = entryPtr->childPtr; nPtr; nPtr=nPtr->nextPtr) {
		if (nPtr->access != TNM_MIB_NOACCESS) {
		    Tcl_DStringAppendElement(dst, nPtr->label);
		}
	    }
	} else {
	    if (nodePtr->access != TNM_MIB_NOACCESS) {
		Tcl_DStringAppendElement(dst, nodePtr->label);
	    }
	}
    }

    ckfree((char *) argv);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Table --
 *
 *	This procedure retrieves a conceptual SNMP table and stores
 *	the values in a Tcl array.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	Tcl variables are modified.
 *
 *----------------------------------------------------------------------
 */

static int
Table(interp, session, argc, argv)
    Tcl_Interp *interp;
    SNMP_Session *session;
    int argc;
    char **argv;
{
    int i, largc, code;
    SNMP_PDU _pdu, *pdu = &_pdu;
    Tcl_DString varList;
    char **largv;
    
    if (argc !=  3) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
			 session->name, " table label varName\"",
			 (char *) NULL);
	return TCL_ERROR;
    }

    /*
     * A special hack to make sure that the given array name 
     * is actually known as an array.
     */

    Tcl_SetVar2(interp, argv[2], "foo", "", 0);
    Tcl_UnsetVar(interp, argv[2], 0);

    /*
     * Initialize the PDU.
     */

    pdu->addr         = session->maddr;
    pdu->type         = TNM_SNMP_GETBULK;
    pdu->request_id   = TnmSnmpGetRequestId();
    pdu->error_status = TNM_SNMP_NOERROR;
    pdu->error_index  = 0;    
    pdu->trapOID      = NULL;
    Tcl_DStringInit(&pdu->varbind);
    Tcl_DStringInit(&varList);

    /*
     * Expand the given table list to create the complete getnext varbind.
     */

    code = ExpandTable(interp, argv[1], &varList);
    if (code != TCL_OK) {
        return TCL_ERROR;
    }

    if (Tcl_DStringLength(&varList) == 0) {
	return TCL_OK;
    }

    /*
     *
     */

    code = Tcl_SplitList(interp, Tcl_DStringValue(&varList),
			 &largc, &largv);
    if (code != TCL_OK) {
	Tcl_DStringFree(&varList);
	return TCL_ERROR;
    }

    for (i = 0; i < largc; i++) {
	TnmWriteMessage(interp, largv[i]);
	TnmWriteMessage(interp, "\n");
    }

    ckfree((char *) largv);
    Tcl_DStringFree(&varList);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ExpandScalars --
 *
 *	This procedure expands the list of scalar or group names
 *	into a list of MIB instances.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	None.
 *
 *----------------------------------------------------------------------
 */

static int
ExpandScalars(interp, sList, dst)
    Tcl_Interp *interp;
    char *sList;
    Tcl_DString *dst;
{
    int argc, code, i;
    char **argv = NULL;
    Tnm_MibNode *nodePtr;
    Tnm_Oid oid[TNM_OIDMAXLEN];
    int oidLen;

    code = Tcl_SplitList(interp, sList, &argc, &argv);
    if (code != TCL_OK) {
	return TCL_ERROR;
    }

    for (i = 0; i < argc; i++) {

	nodePtr = Tnm_MibFindNode(argv[i], NULL, 0);
	if (nodePtr == NULL) {
	    Tcl_AppendResult(interp, "unknown mib object \"", argv[i], "\"",
			     (char *) NULL);
	    ckfree((char *) argv);
	    return TCL_ERROR;
	}

	/*
	 * Skip the node if it is a table or an entry node.
	 */

	if (nodePtr->syntax == ASN1_SEQUENCE 
	    || nodePtr->syntax == ASN1_SEQUENCE_OF) {
	    continue;
	}

	/*
	 * Try to expand to child nodes if the node has child nodes, 
	 * Ignore all nodes which itsef have childs and which are
	 * not accessible.
	 */

	if (nodePtr->childPtr) {
	    for (nodePtr = nodePtr->childPtr; 
		 nodePtr; nodePtr=nodePtr->nextPtr) {
		if (nodePtr->access == TNM_MIB_NOACCESS || nodePtr->childPtr) {
		    continue;
		}
		oidLen = Tnm_MibNodeGetOid(nodePtr, oid);
		oid[oidLen++] = 0;
		Tcl_DStringAppendElement(dst, Tnm_OidToStr(oid, oidLen));
	    }

	} else if (nodePtr->access != TNM_MIB_NOACCESS) {
	    oidLen = Tnm_MibNodeGetOid(nodePtr, oid);
	    oid[oidLen++] = 0;
	    Tcl_DStringAppendElement(dst, Tnm_OidToStr(oid, oidLen));

	} else {
	    Tcl_AppendResult(interp, "object \"", argv[0],
			     "\" not accessible", (char *) NULL);
	    ckfree((char *) argv);
	    return TCL_ERROR;
	}
    }

    ckfree((char *) argv);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * Scalars --
 *
 *	This procedure extracts the variables and values contained in
 *	the varbindlist vbl and set corresponding array Tcl variables.
 *	The list of array names modified is appended to result.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	Tcl variables are modified.
 *
 *----------------------------------------------------------------------
 */

static int
Scalars(interp, session, argc, argv)
    Tcl_Interp *interp;
    SNMP_Session *session;
    int argc;
    char **argv;
{
    int i, largc, code;
    SNMP_PDU _pdu, *pdu = &_pdu;
    Tcl_DString varList;
    Tcl_DString result;
    char **largv;
    
    if (argc !=  3) {
	Tcl_AppendResult(interp, "wrong # args: should be \"",
			 session->name, " scalars label varName\"",
			 (char *) NULL);
	return TCL_ERROR;
    }

    /*
     * A special hack to make sure that the given array name 
     * is actually known as an array.
     */

    Tcl_SetVar2(interp, argv[2], "foo", "", 0);
    Tcl_UnsetVar(interp, argv[2], 0);

    /*
     * Initialize the PDU.
     */

    pdu->addr         = session->maddr;
    pdu->type         = TNM_SNMP_GET;
    pdu->request_id   = TnmSnmpGetRequestId();
    pdu->error_status = TNM_SNMP_NOERROR;
    pdu->error_index  = 0;    
    pdu->trapOID      = NULL;
    Tcl_DStringInit(&pdu->varbind);
    Tcl_DStringInit(&varList);
    Tcl_DStringInit(&result);

    /*
     * Expand the given scalar list to create the complete get varbind.
     */

    code = ExpandScalars(interp, argv[1], &varList);
    if (code != TCL_OK) {
        return TCL_ERROR;
    }

    if (Tcl_DStringLength(&varList) == 0) {
	return TCL_OK;
    }

    /*
     * First try to retrieve all variables in one get request. This
     * may fail because the PDU causes a tooBig error or the agent
     * responds to missing variables with a noSuchName error.
     */

    Tcl_DStringAppend(&pdu->varbind, Tcl_DStringValue(&varList),
		      Tcl_DStringLength(&varList));
    code = Tnm_SnmpEncode(interp, session, pdu, NULL, NULL);
    if (code == TCL_OK) {	
	ScalarSetVar(interp, interp->result, argv[2], &result);
	Tcl_DStringFree(&varList);
	Tcl_DStringResult(interp, &result);
	return TCL_OK;
    }

    /*
     * Stop if we got no response since the agent is not
     * talking to us. This saves some time-outs.
     */

    if (strcmp(interp->result, "noResponse") == 0) {
	return TCL_ERROR;
    }

    /*
     * If we had no success, try every single varbind with one
     * request. Ignore errors so we just collect existing variables.
     */

    code = Tcl_SplitList(interp, Tcl_DStringValue(&varList),
			 &largc, &largv);
    if (code != TCL_OK) {
	Tcl_DStringFree(&varList);
	return TCL_ERROR;
    }

    for (i = 0; i < largc; i++) {

	pdu->type         = TNM_SNMP_GET;
	pdu->request_id   = TnmSnmpGetRequestId();
	pdu->error_status = TNM_SNMP_NOERROR;
	pdu->error_index  = 0;    
	Tcl_DStringInit(&pdu->varbind);
	Tcl_DStringAppend(&pdu->varbind, largv[i], -1);

	code = Tnm_SnmpEncode(interp, session, pdu, NULL, NULL);
	if (code != TCL_OK) {
	    continue;
	}

	ScalarSetVar(interp, interp->result, argv[2], &result);
    }
    ckfree((char *) largv);
    Tcl_DStringFree(&varList);
    Tcl_DStringResult(interp, &result);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ScalarSetVar --
 *
 *	This procedure extracts the variables and values contained in
 *	the varbindlist vbl and set corresponding array Tcl variables.
 *	The list of array names modified is appended to result.
 *
 * Results:
 *	None.
 *
 * Side effects:
 *	Tcl variables are modified.
 *
 *----------------------------------------------------------------------
 */

static void
ScalarSetVar(interp, vbl, varName, result)
    Tcl_Interp *interp;
    char *vbl;
    char *varName;
    Tcl_DString *result;
{
    int i, code, varBindSize;
    SNMP_VarBind *varBindPtr;

    code = Tnm_SnmpSplitVBList(interp, vbl, &varBindSize, &varBindPtr);
    if (code != TCL_OK) {
	return;
    }
    
    for (i = 0; i < varBindSize; i++) {
        Tnm_MibNode *nodePtr = Tnm_MibFindNode(varBindPtr[i].soid, NULL, 0);
	char *name = nodePtr ? nodePtr->label : varBindPtr[i].soid;
	
	if ((strcmp(varBindPtr[i].syntax, "noSuchObject") == 0)
	    || (strcmp(varBindPtr[i].syntax, "noSuchInstance") == 0)
	    || (strcmp(varBindPtr[i].syntax, "endOfMibView") == 0)) {
	    continue;
	}
	
	Tcl_SetVar2(interp, varName, name, varBindPtr[i].value, 0);
	Tcl_DStringAppendElement(result, name);
    }
    
    Tnm_SnmpFreeVBList(varBindSize, varBindPtr);
}
