/***************************************
  $Header: /home/amb/wwwoffle/RCS/purge.c 2.7 1997/12/27 16:58:20 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 2.0a.
  Purge old files from the cache.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1996,97 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <time.h>
#include <utime.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <time.h>
#include <dirent.h>
#include <unistd.h>

#include "wwwoffle.h"
#include "misc.h"
#include "proto.h"
#include "config.h"
#include "errors.h"


/* Local functions */
static int PurgeFiles(char *proto,char *host,time_t now,int age,int *bytes_age);

/*+ Set this to 0 for debugging so that nothing is deleted. +*/
#define DO_DELETE 1


/*++++++++++++++++++++++++++++++++++++++
  Purge files from the cache that meet the age criteria.

  int fd the file descriptor of the wwwoffle client.
  ++++++++++++++++++++++++++++++++++++++*/

void PurgeCache(int fd)
{
 time_t now=time(NULL)+60;
 int total_bytes=0,total_dirs=0;
 int i,p,pass;
 int age_for_size=-1;
 int *nbytes_by_age=(int*)malloc((DefaultPurgeAge+1)*sizeof(int));

 for(i=0;i<=DefaultPurgeAge;i++)
    nbytes_by_age[i]=0;

 if(PurgeUseMTime)
    write_string(fd,"(Using modification time.)\n");
 else
    write_string(fd,"(Using last access time.)\n");

 for(pass=1;pass<=2;pass++)
   {
    if(PurgeCacheSize)
       if(pass==1)
          write_string(fd,"\nPass 1: Checking dates and sizes of files.\n");
       else
          write_string(fd,"\nPass 2: Purging files down to specified size.\n");

    write_string(fd,"\n");

    for(p=0;p<NProtocols;p++)
      {
       DIR *dir;
       struct dirent* ent;
       struct stat buf;
       char *proto=Protocols[p].name;

       /* Open the spool directory. */

       if(stat(proto,&buf))
         {PrintMessage(Inform,"Cannot stat directory '%s' [%!s]; not purged",proto);continue;}

       if(chdir(proto))
         {PrintMessage(Warning,"Cannot change to directory '%s' [%!s]; not purged.",proto);continue;}

       dir=opendir(".");
       if(!dir)
         {PrintMessage(Warning,"Cannot open directory '%s' [%!s]; not purged.",proto);chdir("..");continue;}

       ent=readdir(dir);  /* skip .  */
       if(!ent)
         {PrintMessage(Warning,"Cannot read directory '%s' [%!s]; not purged.",proto);closedir(dir);chdir("..");continue;}
       ent=readdir(dir);  /* skip .. */

       /* Search through all of the sub directories. */

       while((ent=readdir(dir)))
         {
          struct stat buf;

          if(lstat(ent->d_name,&buf))
             PrintMessage(Inform,"Cannot stat file '%s/%s' [%!s]; race condition?",proto,ent->d_name);
          else if(S_ISDIR(buf.st_mode))
            {
             int bytes=0;
             int age=WhatPurgeAge(proto,ent->d_name);

             if(age<0)
                bytes=PurgeFiles(proto,ent->d_name,now,-1,nbytes_by_age);
             else
               {
                if(pass==2)
                   age=age_for_size;
                bytes=PurgeFiles(proto,ent->d_name,now,age,nbytes_by_age);

                if(bytes==-1)
                  {
#if DO_DELETE
                   if(rmdir(ent->d_name))
                      PrintMessage(Warning,"Cannot delete what should be an empty directory '%s/%s' [%!s].",proto,ent->d_name);
#else
                   PrintMessage(Debug,"rmdir(%s/%s).",proto,ent->d_name);
#endif
                  }
                else
                  {
                   struct utimbuf utbuf;

                   utbuf.actime=buf.st_atime;
                   utbuf.modtime=buf.st_mtime;
                   utime(ent->d_name,&utbuf);
                  }
               }

             if(bytes>0)
                total_bytes+=bytes;
             if(bytes>=0)
                total_dirs++;

             if(age<0)
                write_formatted(fd,"Not Purged       %8s://%-32s ; %5d kB.\n",proto,ent->d_name,(bytes+512)/1024);
             else if(bytes==-1)
                write_formatted(fd,"Purged (%2d days) %8s://%-32s ; (empty) - deleted.\n",age,proto,ent->d_name);
             else
                write_formatted(fd,"Purged (%2d days) %8s://%-32s ; %5d kB.\n",age,proto,ent->d_name,(bytes+512)/1024);
            }
         }

       closedir(dir);
       chdir("..");
      }

    write_formatted(fd,"Total of %d directories ; %d kB.\n",total_dirs,(total_bytes+512)/1024);

    if(pass==1)
      {
       write_string(fd,"\n");

       for(total_bytes=i=0;i<=DefaultPurgeAge;i++)
         {
          total_bytes+=nbytes_by_age[i];

          if(PurgeCacheSize && age_for_size<0 && ((total_bytes+512)/1024)>(1024*PurgeCacheSize))
            {
             age_for_size=i;
             write_formatted(fd,"Cutoff Age is %2d days for %3d MB.\n",age_for_size,PurgeCacheSize);
            }

          if(i==DefaultPurgeAge)
             write_formatted(fd,"Total all ages     ; %5d kB.\n",(total_bytes+512)/1024);
          else
             write_formatted(fd,"Newer than %2d day%c ; %5d kB.\n",i+1,i?'s':' ',(total_bytes+512)/1024);
         }
       total_bytes=0;
      }

    if(!PurgeCacheSize || age_for_size==-1)
       break;
   }

 write_string(fd,"\n");

 free(nbytes_by_age);
}


/*++++++++++++++++++++++++++++++++++++++
  Delete the file in the current directory that are older than the specified age.

  int PurgeFiles Returns the number of bytes in files that are left.

  char *proto The name of the protocol directory to purge.

  char *host The name of the host directory to purge.

  time_t now The current time (in seconds).

  int age The age that is specified for the directory in days.

  int *bytes_age An array that contains the number of bytes for a given age.
  ++++++++++++++++++++++++++++++++++++++*/

static int PurgeFiles(char *proto,char *host,time_t now,int age,int *bytes_age)
{
 int bytes_left=-1;
 long oldest=0;
 DIR *dir;
 struct dirent* ent;

 if(age>=0)
    oldest=now-age*(24*3600);

 /* Open the spool subdirectory. */

 if(chdir(host))
   {PrintMessage(Warning,"Cannot change to directory '%s/%s' [%!s]; not purged.",proto,host);return(1);}

 dir=opendir(".");
 if(!dir)
   {PrintMessage(Warning,"Cannot open directory '%s/%s' [%!s]; not purged.",proto,host);chdir("..");return(1);}

 ent=readdir(dir);  /* skip .  */
 if(!ent)
   {PrintMessage(Warning,"Cannot read directory '%s/%s' [%!s]; not purged.",proto,host);closedir(dir);chdir("..");return(1);}
 ent=readdir(dir);  /* skip .. */

 /* Check all of the files for age, and delete as needed. */

 while((ent=readdir(dir)))
   {
    struct stat buf;

    if(stat(ent->d_name,&buf))
       PrintMessage(Inform,"Cannot stat file '%s/%s/%s' [%!s]; race condition?",proto,host,ent->d_name);
    else
      {
       time_t t;

       if(PurgeUseMTime)
          t=buf.st_mtime;
       else
          t=buf.st_atime;

       if(t>oldest)
         {
          if(bytes_left==-1)
             bytes_left=0;
          bytes_left+=buf.st_size;
          if(age>0)
            {
             int days=(now-t)/(24*3600);
             if(days>DefaultPurgeAge)
                days=DefaultPurgeAge;
             bytes_age[days]+=buf.st_size;
            }
         }
       else
         {
#if DO_DELETE
          if(unlink(ent->d_name))
             PrintMessage(Warning,"Cannot unlink file '%s/%s/%s' [%!s].",proto,host,ent->d_name);
#else
          PrintMessage(Debug,"unlink(%s/%s/%s).",proto,host,ent->d_name);
#endif

          if(*ent->d_name=='D')
            {
             *ent->d_name='U';

#if DO_DELETE
             if(unlink(ent->d_name))
                PrintMessage(Warning,"Cannot unlink file '%s/%s/%s' [%!s].",proto,host,ent->d_name);
#else
             PrintMessage(Debug,"unlink(%s/%s/%s).",proto,host,ent->d_name);
#endif
            }
         }
      }
   }

 closedir(dir);
 chdir("..");

 return(bytes_left);
}
