/***************************************
  $Header: /home/amb/wwwoffle/RCS/wwwoffle-tools.c 1.4 1998/04/22 16:38:30 amb Exp $

  WWWOFFLE - World Wide Web Offline Explorer - Version 2.1c.
  Tools for use in the cache for version 2.x.
  ******************/ /******************
  Written by Andrew M. Bishop

  This file Copyright 1997,98 Andrew M. Bishop
  It may be distributed under the GNU Public License, version 2, or
  any higher version.  See section COPYING of the GNU Public license
  for conditions under which this file may be redistributed.
  ***************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <time.h>
#include <utime.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <dirent.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>

#include "wwwoffle.h"
#include "misc.h"
#include "errors.h"
#include "config.h"


#define LS         1
#define MV         2
#define RM         3
#define LS_SPECIAL 4

static void wwwoffle_ls(URL *Url);
static void wwwoffle_ls_special(char *name);
static void wwwoffle_mv(URL *Url1,URL *Url2);
static void wwwoffle_rm(URL *Url);

static void ls(char *file);


/*++++++++++++++++++++++++++++++++++++++
  The main program
  ++++++++++++++++++++++++++++++++++++++*/

int main(int argc,char **argv)
{
 struct stat buf;
 URL *Url[2];
 int mode=0;
 int i;
 char *p;

 p=argv[0]+strlen(argv[0])-1;
 while(p>=argv[0] && *p!='/')
    p--;

 p++;

 if(!strcmp(p,"wwwoffle-ls"))
    mode=LS;
 else if(!strcmp(p,"wwwoffle-mv"))
    mode=MV;
 else if(!strcmp(p,"wwwoffle-rm"))
    mode=RM;
 else
   {
    fprintf(stderr,"Program must be started as wwwoffle-ls, wwwoffle-mv or wwwoffle-rm\n");
    exit(1);
   }

 /* Initialise */

 if(mode==LS && argc!=2)
   {fprintf(stderr,"Usage: wwwoffle-ls ( <dir>/<subdir> | <protocol>://<host> | <URL> | \n"
                   "                     outgoing | lasttime )\n");exit(0);}
 else if(mode==MV && argc!=3)
   {fprintf(stderr,"Usage: wwwoffle-mv (<dir1>/<subdir1> | <protocol1>://<host1>)\n"
                   "                   (<dir2>/<subdir2> | <protocol2>://<host2>)\n");exit(0);}
 else if(mode==RM && argc!=2)
   {fprintf(stderr,"Usage: wwwoffle-rm <URL>\n");exit(0);}

 if(stat("outgoing",&buf) || !S_ISDIR(buf.st_mode))
   {fprintf(stderr,"The wwwoffle-tools programs must be started from the spool directory\n"
                   "There is no 'outgoing' directory here so it can't be right.\n");exit(1);}

 InitErrorHandler(p,0,1);

 LogLevel=Inform;

 /* Get the arguments */

 for(i=1;i<argc;i++)
   {
    char *colon=strchr(argv[i],':');
    char *slash=strchr(argv[i],'/');

    if(mode==LS && (!strcmp(argv[i],"outgoing") || !strcmp(argv[i],"lasttime")))
      {
       mode=LS_SPECIAL;
      }
    else if(colon && slash && colon<slash)
      {
       Url[i-1]=SplitURL(argv[i]);
      }
    else
      {
       char *slash,*url;

       slash=strchr(argv[i],'/');
       if(!slash)
         {fprintf(stderr,"Cannot parse the argument '%s'\n",argv[i]);exit(1);}

       *slash=0;
       url=(char*)malloc(strlen(slash+1)+strlen(argv[i])+8);
       sprintf(url,"%s://%s",argv[i],slash+1);

       Url[i-1]=SplitURL(url);
      }
   }

 if(mode==LS)
    wwwoffle_ls(Url[0]);
 else if(mode==MV)
    wwwoffle_mv(Url[0],Url[1]);
 else if(mode==RM)
    wwwoffle_rm(Url[0]);
 else if(mode==LS_SPECIAL)
    wwwoffle_ls_special(argv[1]);

 exit(0);
}


/*++++++++++++++++++++++++++++++++++++++
  List the URLs within a directory of the cache.

  URL *Url The URL to list.
  ++++++++++++++++++++++++++++++++++++++*/

static void wwwoffle_ls(URL *Url)
{
 if(chdir(Url->proto))
   {PrintMessage(Warning,"Cannot change to directory '%s' [%!s].",Url->proto);return;}

 if(chdir(Url->host))
   {PrintMessage(Warning,"Cannot change to directory '%s/%s' [%!s].",Url->proto,Url->host);chdir("..");return;}

 if(strcmp(Url->path,"/"))
   {
    char *name=URLToFileName(Url);

    *name='D';
    ls(name);

    free(name);
   }
 else
   {
    struct dirent* ent;
    DIR *dir=opendir(".");

    if(!dir)
      {PrintMessage(Warning,"Cannot open current directory '%s/%s' [%!s].",Url->proto,Url->host);chdir("../..");return;}

    ent=readdir(dir);  /* skip .  */
    if(!ent)
      {PrintMessage(Warning,"Cannot read current directory '%s/%s' [%!s].",Url->proto,Url->host);closedir(dir);chdir("../..");return;}
    ent=readdir(dir);  /* skip .. */

    while((ent=readdir(dir)))
      {
       if(*ent->d_name=='D')
          ls(ent->d_name);
      }

    closedir(dir);
   }

 chdir("../..");
}


/*++++++++++++++++++++++++++++++++++++++
  List the URLs within the outgoing or lasttime special directory of the cache.

  char *name The name of the directory to list.
  ++++++++++++++++++++++++++++++++++++++*/

static void wwwoffle_ls_special(char *name)
{
 struct dirent* ent;
 DIR *dir;

 if(chdir(name))
   {PrintMessage(Warning,"Cannot change to directory '%s' [%!s].",name);return;}

 dir=opendir(".");

 if(!dir)
   {PrintMessage(Warning,"Cannot open current directory '%s' [%!s].",name);chdir("..");return;}

 ent=readdir(dir);  /* skip .  */
 if(!ent)
   {PrintMessage(Warning,"Cannot read current directory '%s' [%!s].",name);closedir(dir);chdir("..");return;}
 ent=readdir(dir);  /* skip .. */

 while((ent=readdir(dir)))
   {
    if(*ent->d_name=='D' || *ent->d_name=='O')
       ls(ent->d_name);
   }

 closedir(dir);

 chdir("..");
}


/*++++++++++++++++++++++++++++++++++++++
  List one file.

  char *file The name of the file to ls.
  ++++++++++++++++++++++++++++++++++++++*/

static void ls(char *file)
{
 static char *month[12]={"Jan","Feb","Mar","Apr","May","Jun","Jul","Aug","Sep","Oct","Nov","Dec"};
 struct stat buf;
 struct tm *tim;
 time_t now=-1;

 if(now==-1)
    now=time(NULL);

 if(stat(file,&buf))
   {PrintMessage(Warning,"Cannot stat the file '%s' [%!s].",file);return;}
 else
   {
    char *url=FileNameToURL(file);

    tim=localtime(&buf.st_mtime);

    if(buf.st_mtime<(now-(180*24*3600)))
       printf("%s %7ld %3s %2d %d %s\n",file,(long)buf.st_size,month[tim->tm_mon],tim->tm_mday,tim->tm_year+1900,url);
    else
       printf("%s %7ld %3s %2d %2d:%02d %s\n",file,(long)buf.st_size,month[tim->tm_mon],tim->tm_mday,tim->tm_hour,tim->tm_min,url);

    free(url);
   }
}


/*++++++++++++++++++++++++++++++++++++++
  Move one URL or host to another.

  URL *Url1 The source URL.

  URL *Url2 The destination URL.
  ++++++++++++++++++++++++++++++++++++++*/

static void wwwoffle_mv(URL *Url1,URL *Url2)
{
 struct dirent* ent;
 DIR *dir;

 if(chdir(Url1->proto))
   {PrintMessage(Warning,"Cannot change to directory '%s' [%!s].",Url1->proto);return;}

 if(chdir(Url1->host))
   {PrintMessage(Warning,"Cannot change to directory '%s/%s' [%!s].",Url1->proto,Url1->host);chdir("..");return;}

 dir=opendir(".");

 if(!dir)
   {PrintMessage(Warning,"Cannot open current directory '%s/%s' [%!s].",Url1->proto,Url1->host);chdir("../..");return;}

 ent=readdir(dir);  /* skip .  */
 if(!ent)
   {PrintMessage(Warning,"Cannot read current directory '%s/%s' [%!s].",Url1->proto,Url1->host);closedir(dir);chdir("../..");return;}
 ent=readdir(dir);  /* skip .. */

 while((ent=readdir(dir)))
   {
    if(*ent->d_name=='D')
      {
       char *url1=FileNameToURL(ent->d_name),*url2;
       URL *Url;
       char *path2,*name1,*name2;
       int fd2;

       Url=SplitURL(url1);
       url2=(char*)malloc(strlen(Url->pathp)+strlen(Url2->host)+strlen(Url2->proto)+8);
       sprintf(url2,"%s://%s%s",Url2->proto,Url2->host,Url->pathp);
       FreeURL(Url);

       name1=(char*)malloc(strlen(ent->d_name)+1);
       strcpy(name1,ent->d_name);

       Url=SplitURL(url2);
       name2=URLToFileName(Url);

       path2=(char*)malloc(strlen(Url2->proto)+strlen(Url2->host)+strlen(name2)+16);

       sprintf(path2,"../../%s",Url2->proto);
       mkdir(path2,0755);

       sprintf(path2,"../../%s/%s",Url2->proto,Url2->host);
       mkdir(path2,0755);

       *name1=*name2='D';
       sprintf(path2,"../../%s/%s/%s",Url2->proto,Url2->host,name2);
       rename(name1,path2);

       *name1=*name2='U';
       sprintf(path2,"../../%s/%s/%s",Url2->proto,Url2->host,name2);
       fd2=open(path2,O_WRONLY|O_CREAT|O_TRUNC);
       write_string(fd2,Url->name);
       close(fd2);
       unlink(name2);

       free(url1);
       free(url2);
       free(name1);
       free(name2);
       free(path2);
       FreeURL(Url);
      }
   }

 closedir(dir);

 chdir("../..");
}


/*++++++++++++++++++++++++++++++++++++++
  Delete a URL.

  URL *Url The URL to delete.
  ++++++++++++++++++++++++++++++++++++++*/

static void wwwoffle_rm(URL *Url)
{
 DeleteWebpageSpoolFile(Url,0);
}
