/* Copyright (C) 1979-1996 TcX AB & Monty Program KB & Detron HB

   This software is distributed with NO WARRANTY OF ANY KIND.  No author or
   distributor accepts any responsibility for the consequences of using it, or
   for whether it serves any particular purpose or works at all, unless he or
   she says so in writing.  Refer to the Free Public License (the "License")
   for full details.

   Every copy of this file must include a copy of the License, normally in a
   plain ASCII text file named PUBLIC.	The License grants you the right to
   copy, modify and redistribute this file, but only under certain conditions
   described in the License.  Among other things, the License requires that
   the copyright notice and this notice be preserved on all copies. */

/* Function with list databases, tables or fields */

#include "mysql_priv.h"
#include "sql_select.h"				// For select_describe
#include <my_dir.h>

static int mysql_find_files(THD *thd,const char *path,const char *wild,
			    bool dir);

/****************************************************************************
** Send list of databases
** A database is a directory in the mysql_data_home directory
****************************************************************************/


int
mysqld_show_dbs(THD *thd,const char *wild)
{
  Item_string *field=new Item_string("",0);
  List<Item> field_list;
  char *end;
  DBUG_ENTER("mysqld_show_dbs");

  field->name=(char*) sql_alloc(20+ (wild ? strlen(wild)+4: 0));
  field->max_length=NAME_LEN;
  end=strmov(field->name,"Database");
  if (wild && wild[0])
    strxmov(end," (",wild,")",NullS);
  field_list.push_back(field);

  if (send_fields(thd,field_list,1))
    DBUG_RETURN(1);
  DBUG_RETURN(mysql_find_files(thd,mysql_data_home,wild,1));
}

/***************************************************************************
** List all tables in a database
** A table is a .frm file in the current databasedir
***************************************************************************/

int mysqld_show_tables(THD *thd,const char *db,const char *wild)
{
  Item_string *field=new Item_string("",0);
  List<Item> field_list;
  char path[FN_LEN],*end;
  DBUG_ENTER("mysqld_show_tables");

  field->name=(char*) sql_alloc(20+strlen(db)+(wild ? strlen(wild)+4:0));
  end=strxmov(field->name,"Tables in ",db,NullS);
  if (wild && wild[0])
    strxmov(end," (",wild,")",NullS);
  field->max_length=NAME_LEN;
  (void) sprintf(path,"%s/%s",mysql_data_home,db);
  (void) unpack_filename(path,path);
  field_list.push_back(field);
  if (send_fields(thd,field_list,1))
    DBUG_RETURN(1);
  DBUG_RETURN(mysql_find_files(thd,path,wild,0));
}


static int
mysql_find_files(THD *thd,const char *path,const char *wild,bool dir)
{
  uint i;
  char *ext;
  MY_DIR *dirp;
  FILEINFO *file;
  DBUG_ENTER("mysql_find_files");

  if (!(dirp = my_dir(path,MYF(MY_WME | (dir ? MY_WANT_STAT : 0)))))
    DBUG_RETURN(-1);

  for (i=0 ; i < (uint) dirp->number_off_files	; i++)
  {
    file=dirp->dir_entry+i;
    if (dir)
    {						/* Return databases */
      if (file->name[0] == '.' || !MY_S_ISDIR(file->mystat.st_mode) ||
	  (wild && wild[0] && wild_compare(file->name,wild)))
	continue;
    }
    else
    {
	// Return only .frm files which isn't temp files.
      if (strcmp(ext=fn_ext(file->name),reg_ext) ||
	  strchr(file->name,'-'))		// Mysql temp table
	continue;
      *ext=0;
      if (wild && wild[0] && wild_compare(file->name,wild))
	continue;
    }
    thd->packet.length(0);
    net_store_data(&thd->packet,file->name);
    if (my_net_write(&thd->net,(byte*) thd->packet.ptr(),thd->packet.length()))
      break;
  }
  send_eof(&thd->net);
  my_dirend(dirp);
  DBUG_RETURN(0);
}


int
mysqld_show_fields(THD *thd, const char *db, const char *table_name,
		  const char *wild)
{
  TABLE *table;
  char tmp[MAX_FIELD_WIDTH];
  DBUG_ENTER("mysqld_show_fields");
  DBUG_PRINT("enter",("db: %s  table: %s",db,table_name));

  if (!(table = open_table(thd,db,table_name,table_name)))
  {
    send_error(&thd->net);
    DBUG_RETURN(1);
  }
  ha_info(table,2);			// sync record count

  List<Item> field_list;
  field_list.push_back(new Item_empty_string("Field",NAME_LEN));
  field_list.push_back(new Item_empty_string("Type",40));
  field_list.push_back(new Item_empty_string("Null",1));
  field_list.push_back(new Item_empty_string("Key",3));
  field_list.push_back(new Item_empty_string("Default",NAME_LEN));
  field_list.push_back(new Item_empty_string("Extra",20));

	// Send first number of fields and records
  {
    byte *pos;
    pos=net_store_length((byte*) tmp,field_list.elements);
    pos=net_store_length(pos,table->keyfile_info.records);
    (void) my_net_write(&thd->net, (byte*) tmp,(uint) (pos-(byte*) tmp));
  }

  if (send_fields(thd,field_list,0))
    DBUG_RETURN(1);
  restore_record(table,2);	// Get empty record

  Field **ptr,*field;
  for (ptr=table->field; (field= *ptr) ; ptr++)
  {
    if (!wild || !wild[0] || !wild_case_compare(field->field_name,wild))
    {
      byte *pos;
      uint flags=field->flags;
      String *packet= &thd->packet,type(tmp,sizeof(tmp));

      packet->length(0);
      net_store_data(packet,field->field_name);
      field->sql_type(type);
      net_store_data(packet,(byte*) type.ptr(),type.length());

      pos=(byte*) ((flags & NOT_NULL_FLAG) &&
		   field->type() != FIELD_TYPE_TIMESTAMP ?
		   "" : "YES");
      net_store_data(packet,(const char*) pos);
      pos=(byte*) ((field->flags & PRI_KEY_FLAG) ? "PRI" :
		   (field->flags & UNIQUE_KEY_FLAG) ? "UNI" :
		   (field->flags & MULTIPLE_KEY_FLAG) ? "MUL":"");
      net_store_data(packet,(char*) pos);

      if (!(flags & BLOB_FLAG) && field->type() != FIELD_TYPE_TIMESTAMP &&
	  !field->is_null())
      {						// Not null by default
	type.set(tmp,sizeof(tmp));
	field->val_str(&type);
	net_store_data(packet,(byte*) type.ptr(),type.length());
      }
      else if (field->maybe_null() || field->type() == FIELD_TYPE_TIMESTAMP)
	net_store_null(packet);			// Null as default
      else
	net_store_data(packet,(byte*) tmp,0);

      char *end=tmp;
      if (field->unireg_check == Field::NEXT_NUMBER)
	end=strmov(tmp,"auto_increment");
      net_store_data(packet,(byte*) tmp,(uint) (end-tmp));
      if (my_net_write(&thd->net,(byte*) packet->ptr(),packet->length()))
	DBUG_RETURN(1);
    }
  }
  send_eof(&thd->net);
  DBUG_RETURN(0);
}


int
mysqld_show_keys(THD *thd, const char *db, const char *table_name)
{
  TABLE *table;
  DBUG_ENTER("mysqld_show_keys");
  DBUG_PRINT("enter",("db: %s  table: %s",db,table_name));

  if (!(table = open_table(thd,db,table_name,table_name)))
  {
    send_error(&thd->net);
    DBUG_RETURN(1);
  }

  List<Item> field_list;
  Item *item;
  field_list.push_back(new Item_empty_string("Table",NAME_LEN));
  field_list.push_back(new Item_int("Non_unique",0,1));
  field_list.push_back(new Item_empty_string("Key_name",NAME_LEN));
  field_list.push_back(new Item_int("Seq_in_index",0,2));
  field_list.push_back(new Item_empty_string("Column_name",NAME_LEN));
  field_list.push_back(item=new Item_empty_string("Collation",1));
  item->maybe_null=1;
  field_list.push_back(item=new Item_int("Cardinality",0,11));
  item->maybe_null=1;
  field_list.push_back(item=new Item_int("Sub_part",0,3));

  if (send_fields(thd,field_list,1))
    DBUG_RETURN(1);

  TABLE *form= table;
  KEY *key_info=form->key_info;
  ha_info(form,2);
  for (uint i=0 ; i < form->keys ; i++,key_info++)
  {
    KEY_PART_INFO *key_part= key_info->key_part;
    byte  buff[10],*end;
    String *packet= &thd->packet;
    for (uint j=0 ; j < key_info->key_parts ; j++,key_part++)
    {
      packet->length(0);
      net_store_data(packet,table->table_name);
      net_store_data(packet,(byte*) (key_info->dupp_key ? "1" :"0"), 1);
      net_store_data(packet,key_info->name);
      end=(byte*) int2str((long) (j+1),(char*) buff,10);
      net_store_data(packet,buff,(uint) (end-buff));
      net_store_data(packet,key_part->field ? key_part->field->field_name :
		     "?unknown field?");
      if (ha_option_flag[form->db_type] & HA_READ_ORDER)
	net_store_data(packet,(byte*) (key_part->key_part_flag == 0 ? "A" : "D"), 1);
      else
	net_store_null(packet); /* purecov: inspected */
      if (form->db_type == DB_TYPE_ISAM && form->keyfile_info.rec_per_key[i])
      {
	ulong records=(form->keyfile_info.records /
		       form->keyfile_info.rec_per_key[i]);
	end=(byte*) int2str((long) records,(char*) buff,10);
	net_store_data(packet,buff,(uint) (end-buff));
      }
      else
	net_store_null(packet);
      if (!key_part->field ||
	  key_part->length !=
	  table->field[key_part->fieldnr-1]->pack_length())
      {
	end=(byte*) int2str((long) key_part->length,(char*) buff,10); /* purecov: inspected */
	net_store_data(packet,buff,(uint) (end-buff)); /* purecov: inspected */
      }
      else
	net_store_null(packet);
      if (my_net_write(&thd->net,(byte*) packet->ptr(),packet->length()))
	DBUG_RETURN(1); /* purecov: inspected */
    }
  }
  send_eof(&thd->net);
  DBUG_RETURN(0);
}


/****************************************************************************
** Return only fields for API mysql_list_fields
** Use "show table wildcard" in mysql instead of this
****************************************************************************/

void
mysqld_list_fields(THD *thd, const char *table_name, const char *wild)
{
  TABLE *table;
  DBUG_ENTER("mysqld_list_fields");
  DBUG_PRINT("enter",("table: %s",table_name));

  if (!(table = open_table(thd,thd->db,table_name,table_name)))
    send_error(&thd->net);
  else
  {
    List<Item> field_list;

    Field **ptr,*field;
    for (ptr=table->field ; (field= *ptr); ptr++)
    {
      if (!wild || !wild[0] || !wild_case_compare(field->field_name,wild))
	field_list.push_back(new Item_field(field));
    }
    restore_record(table,2);		// Get empty record
    if (send_fields(thd,field_list,2))
      DBUG_VOID_RETURN;
    VOID(net_flush(&thd->net));
#ifdef DELETE_ITEMS
    field_list.delete_elements();
#endif
  }
  DBUG_VOID_RETURN;
}


/****************************************************************************
** Return info about all processes
** returns for each thread: thread id, user, host, db, command, info
****************************************************************************/

class thread_info :public Sql_alloc, public ilink {
public:
  ulong thread_id;
  uint	 command;
  char	*user,*host,*db,*query,*proc_info;
};

#ifdef __GNUC__
template class I_List<thread_info>;
template class I_List_iterator<THD>;
#endif


void mysqld_list_processes(THD *thd)
{
  Item *field;
  List<Item> field_list;
  I_List<thread_info> thread_infos;
  DBUG_ENTER("mysql_list_processes");

  field_list.push_back(new Item_int("Id",0,7));
  field_list.push_back(new Item_empty_string("User",16));
  field_list.push_back(new Item_empty_string("Host",64));
  field_list.push_back(field=new Item_empty_string("db",NAME_LEN));
  field->maybe_null=1;
  field_list.push_back(new Item_empty_string("Command",16));
  field_list.push_back(field=new Item_empty_string("Info",100));
  field->maybe_null=1;
  if (send_fields(thd,field_list,1))
    DBUG_VOID_RETURN;

  VOID(pthread_mutex_lock(&LOCK_thread_count)); // For unlink from list
  if (!thd->killed)
  {
    I_List_iterator<THD> it(threads);
    THD *tmp;
    while ((tmp=it++))
    {
      if (tmp->net.fd >= 0)
      {
	thread_info *thd_info=new thread_info;

	thd_info->thread_id=tmp->thread_id;
	thd_info->user=sql_strdup(tmp->user);
	thd_info->host=sql_strdup(tmp->host ? tmp->host : "unknown host");
	thd_info->db=0;
	if (tmp->db)
	  thd_info->db=sql_strdup(tmp->db);
	thd_info->command=(int) tmp->command;
	thd_info->proc_info=
	  tmp->proc_info ? tmp->proc_info :
	  tmp->killed ? "Killed" :
	  tmp->locked ? "Locked" : NullS;
	if (pthread_kill(tmp->real_id,0))
	  tmp->proc_info="*** DEAD ***";	// This shouldn't happen
	thd_info->query=0;
	if (tmp->query)
	{
	  uint length=strlen(tmp->query);
	  if (length > 100)
	    length=100;
	  thd_info->query=(char*) sql_memdup(tmp->query,length+1);
	  thd_info->query[length]=0;
	}
	thread_infos.append(thd_info);
      }
    }
  }
  VOID(pthread_mutex_unlock(&LOCK_thread_count));

  thread_info *thd_info;
  String *packet= &thd->packet;
  while ((thd_info=thread_infos.get()))
  {
    byte buff[20],*end;
    packet->length(0);
    end=(byte*) int2str((long) thd_info->thread_id,(char*) buff,10);
    net_store_data(packet,(byte*) buff,(uint) (end-buff));
    net_store_data(packet,thd_info->user);
    net_store_data(packet,thd_info->host);
    if (thd_info->db)
      net_store_data(packet,thd_info->db);
    else
      net_store_null(packet);
    if (thd_info->proc_info)
      net_store_data(packet,thd_info->proc_info);
    else
      net_store_data(packet,command_name[thd_info->command]);
    if (thd_info->query)
      net_store_data(packet,thd_info->query);
    else
      net_store_null(packet);
    if (my_net_write(&thd->net,(byte*) packet->ptr(),packet->length()))
      break; /* purecov: inspected */
  }
  send_eof(&thd->net);
  DBUG_VOID_RETURN;
}


/*****************************************************************************
** Status functions
*****************************************************************************/

int mysqld_show_status(THD *thd)
{
  DBUG_ENTER("mysqld_show_status");

  List<Item> field_list;
  field_list.push_back(new Item_int("Uptime",0,10));
  field_list.push_back(new Item_int("Running_threads",0,5));
  field_list.push_back(new Item_int("Questions",0,10));
  field_list.push_back(new Item_int("Opened_tables",0,10));
  field_list.push_back(new Item_int("Reloads",0,10));
  field_list.push_back(new Item_int("Open_tables",0,5));
  if (send_fields(thd,field_list,1))
    DBUG_RETURN(1);

#ifdef SAFEMALLOC
  if (lCurMemory)				// Using SAFEMALLOC
  {
    field_list.push_back(new Item_int("Memory_in_use",0,10));
    field_list.push_back(new Item_int("Max_memory_used",0,10));
  }
#endif
  byte *packet=(byte*) thd->packet.ptr();
  packet=net_store_data(packet,(long) (thd->query_start() - start_time));
  packet=net_store_data(packet,(long) thread_count);
  packet=net_store_data(packet,(long) thd->query_id);
  packet=net_store_data(packet,(long) opened_tables);
  packet=net_store_data(packet,(long) thd->version);
  packet=net_store_data(packet,(long) cached_tables());

#ifdef SAFEMALLOC
  if (lCurMemory)				// Using SAFEMALLOC
  {
    packet=net_store_data(packet,(long) (lCurMemory+1023L)/1024L);
    packet=net_store_data(packet,(long) (lMaxMemory+1023L)/1024L);
  }
#endif
  if (my_net_write(&thd->net,(byte*) thd->packet.ptr(),
		   (uint) (packet - (byte*) thd->packet.ptr())))
    DBUG_RETURN(1);
  send_eof(&thd->net);
  DBUG_RETURN(0);
}


int mysqld_show_variables(THD *thd,const char *wild)
{
  uint i;
  char buff[1024];
  String packet2(buff,sizeof(buff));
  List<Item> field_list;
  DBUG_ENTER("mysqld_show_variables");
  field_list.push_back(new Item_empty_string("Variable_name",30));
  field_list.push_back(new Item_empty_string("Value",256));
  if (send_fields(thd,field_list,1))
    DBUG_RETURN(1);

  for (i=0; variables[i].name; i++)
  {
    if (!(wild && wild[0] && wild_compare(variables[i].name,wild)))
    {
      packet2.length(0);
      net_store_data(&packet2,variables[i].name);
      switch (variables[i].type){
      case SHOW_LONG:
	net_store_data(&packet2,*(ulong*) variables[i].value);
	break;
      case SHOW_INT:
	net_store_data(&packet2,(ulong) *(int*) variables[i].value);
	break;
      case SHOW_CHAR:
	net_store_data(&packet2,variables[i].value);
	break;
      case SHOW_CHAR_PTR:
	{
	  char *value= *(char**) variables[i].value;
	  net_store_data(&packet2,value ? value : "");
	  break;
	}
      }
      if (my_net_write(&thd->net, (byte*) packet2.ptr(),packet2.length()))
	DBUG_RETURN(1);
    }
  }
  send_eof(&thd->net);
  DBUG_RETURN(0);
}
