//============================================================================
//
//    SSSS    tt          lll  lll              
//   SS  SS   tt           ll   ll                
//   SS     tttttt  eeee   ll   ll   aaaa    "An Atari 2600 VCS Emulator"
//    SSSS    tt   ee  ee  ll   ll      aa      
//       SS   tt   eeeeee  ll   ll   aaaaa   Copyright (c) 1995,1996,1997
//   SS  SS   tt   ee      ll   ll  aa  aa         Bradford W. Mott
//    SSSS     ttt  eeeee llll llll  aaaaa    
//
//============================================================================

/**
  This is an abstract base class for classes that emulate the
  6507 microprocessor.  For the most part the 6507 is just like
  the 6502 except it has an 8K address space.

  @author  Bradford W. Mott
  @version $Id: M6507.cxx,v 1.2 1997/05/17 19:00:06 bwmott Exp $
*/

#include <assert.h>
#include <iostream.h>
#include <iomanip.h>
#include <fstream.h>

#include "Error.hxx"
#include "M6507.hxx"

//============================================================================
// Default constructor
//============================================================================
M6507::M6507(System& system)
    : mySystem(system)
{
#ifdef DEBUG
  // Open debug stream
  debugStream.open("M6507.debug", ios::out);
#endif

  // Compute the BCD lookup table
  for(uWord t = 0; t < 256; ++t)
  {
    ourBCDTable[0][t] = ((t >> 4) * 10) + (t & 0x0f);
    ourBCDTable[1][t] = (((t % 100) / 10) << 4) | (t % 10);
  }
}

//============================================================================
// Destructor
//============================================================================
M6507::~M6507()
{
}

//============================================================================
// Answers value of the Processor Status register
//============================================================================
uByte M6507::PS() const
{
  uByte ps = 0x20;

  if(N) ps |= 0x80;
  if(V) ps |= 0x40;
  if(B) ps |= 0x10;
  if(D) ps |= 0x08;
  if(I) ps |= 0x04;
  if(Z) ps |= 0x02;
  if(C) ps |= 0x01;

  return ps;
}

//============================================================================
// Change the processor flags to correspond to the given PS value
//============================================================================
void M6507::PS(uByte ps)
{
  N = ps & 0x80;
  V = ps & 0x40;
  B = ps & 0x10;
  D = ps & 0x08;
  I = ps & 0x04;
  Z = ps & 0x02;
  C = ps & 0x01;
}

//============================================================================
// Preform a system reset
//============================================================================
void M6507::reset()
{
  // Set the halt execution flag
  myHaltExecution = false;

  // Reset cycle count to zero
  myCycles = 0;

  // Set registers to default values
  A = X = Y = 0;
  SP = 0xff;
  PS(0x20);

  // Load PC from the reset vector
  PC = mySystem.peek(0xfffc) | (mySystem.peek(0xfffd) << 8);
}

//============================================================================
// Invalid instruction was fetched
//============================================================================
inline void M6507::invalidInstruction()
{
#ifdef DEBUG
  debugStream << "*INVALID*";
#endif
  Error err;
  err.message() << "An invalid instruction was executed by the 6507!";
  err.description() << "The ROM image is not supported or Cartridge.Type"
      << " needs to be set!";
  Throw(err);
}  

//============================================================================
// Output operator for addressing modes
//============================================================================
ostream& operator<<(ostream& out, const M6507::AddressingMode& mode)
{
  switch(mode)
  {
    case M6507::Implied:
      out << "implied";
      break;
    case M6507::Immediate:
      out << "#$nn   ";
      break;
    case M6507::Absolute:
      out << "$nnnn  ";
      break;
    case M6507::AbsoluteX:
      out << "$nnnn,X";
      break;
    case M6507::AbsoluteY:
      out << "$nnnn,Y";
      break;
    case M6507::Zero:
      out << "$nn    ";
      break;
    case M6507::ZeroX:
      out << "$nn,X  ";
      break;
    case M6507::ZeroY:
      out << "$nn,Y  ";
      break;
    case M6507::Indirect:
      out << "($nnnn)";
      break;
    case M6507::IndirectX:
      out << "($nn,X)";
      break;
    case M6507::IndirectY:
      out << "($nn),Y";
      break;
    case M6507::Invalid:
      out << "invalid";
      break;
  }
  return out;
}

//============================================================================
// Lookup table used for binary-code-decimal math
//============================================================================
uByte M6507::ourBCDTable[2][256];

//============================================================================
// Addressing mode for each of the 256 opcodes
//============================================================================
M6507::AddressingMode M6507::ourAddressingModeTable[256] = {
    Implied,    IndirectX, Invalid,   Invalid,    // 0x0?
    Invalid,    Zero,      Zero,      Invalid,
    Implied,    Immediate, Implied,   Invalid,
    Invalid,    Absolute,  Absolute,  Invalid,

    Implied,    IndirectY, Invalid,   Invalid,    // 0x1?
    Invalid,    ZeroX,     ZeroX,     Invalid,
    Implied,    AbsoluteY, Invalid,   Invalid,
    Invalid,    AbsoluteX, AbsoluteX, Invalid,

    Absolute,   IndirectX, Invalid,   Invalid,    // 0x2?
    Zero,       Zero,      Zero,      Invalid,
    Implied,    Immediate, Implied,   Invalid,
    Absolute,   Absolute,  Absolute,  Invalid,

    Implied,    IndirectY, Invalid,   Invalid,    // 0x3?
    Invalid,    ZeroX,     ZeroX,     Invalid,
    Implied,    AbsoluteY, Invalid,   Invalid,
    Invalid,    AbsoluteX, AbsoluteX, Invalid,

    Implied,    IndirectX, Invalid,   Invalid,    // 0x4?
    Invalid,    Zero,      Zero,      Invalid,
    Implied,    Immediate, Implied,   Invalid,
    Absolute,   Absolute,  Absolute,  Invalid,

    Implied,    IndirectY, Invalid,   Invalid,    // 0x5?
    Invalid,    ZeroX,     ZeroX,     Invalid,
    Implied,    AbsoluteY, Invalid,   Invalid,
    Invalid,    AbsoluteX, AbsoluteX, Invalid,

    Implied,    IndirectX, Invalid,   Invalid,    // 0x6?
    Invalid,    Zero,      Zero,      Invalid,
    Implied,    Immediate, Implied,   Invalid,
    Indirect,   Absolute,  Absolute,  Invalid,

    Implied,    IndirectY, Invalid,   Invalid,    // 0x7?
    Invalid,    ZeroX,     ZeroX,     Invalid,
    Implied,    AbsoluteY, Invalid,   Invalid,
    Invalid,    AbsoluteX, AbsoluteX, Invalid,

    Invalid,    IndirectX, Invalid,   Invalid,    // 0x8?
    Zero,       Zero,      Zero,      Invalid,
    Implied,    Invalid,   Implied,   Invalid,
    Absolute,   Absolute,  Absolute,  Invalid,

    Implied,    IndirectY, Invalid,   Invalid,    // 0x9?
    ZeroX,      ZeroX,     ZeroY,     Invalid,
    Implied,    AbsoluteY, Implied,   Invalid,
    Invalid,    AbsoluteX, Invalid,   Invalid,

    Immediate,  IndirectX, Immediate, Invalid,    // 0xA?
    Zero,       Zero,      Zero,      Invalid,
    Implied,    Immediate, Implied,   Invalid,
    Absolute,   Absolute,  Absolute,  Invalid,

    Implied,    IndirectY, Invalid,   Invalid,    // 0xB?
    ZeroX,      ZeroX,     ZeroY,     Invalid,
    Implied,    AbsoluteY, Implied,   Invalid,
    AbsoluteX,  AbsoluteX, AbsoluteY, Invalid,

    Immediate,  IndirectX, Invalid,   Invalid,    // 0xC?
    Zero,       Zero,      Zero,      Invalid,
    Implied,    Immediate, Implied,   Invalid,
    Absolute,   Absolute,  Absolute,  Invalid,

    Implied,    IndirectY, Invalid,   Invalid,    // 0xD?
    Invalid,    ZeroX,     ZeroX,     Invalid,
    Implied,    AbsoluteY, Invalid,   Invalid,
    Invalid,    AbsoluteX, AbsoluteX, Invalid,

    Immediate,  IndirectX, Invalid,   Invalid,    // 0xE?
    Zero,       Zero,      Zero,      Invalid,
    Implied,    Immediate, Implied,   Invalid,
    Absolute,   Absolute,  Absolute,  Invalid,

    Implied,    IndirectY, Invalid,   Invalid,    // 0xF?
    Invalid,    ZeroX,     ZeroX,     Invalid,
    Implied,    AbsoluteY, Invalid,   Invalid,
    Invalid,    AbsoluteX, AbsoluteX, Invalid
  };

//============================================================================
// CPU cycles for each of the 256 opcodes
//============================================================================
uLong M6507::ourInstructionCycleTable[256] = {
//  0  1  2  3  4  5  6  7  8  9  a  b  c  d  e  f
    7, 6, 2, 2, 2, 3, 5, 2, 3, 2, 2, 2, 2, 4, 6, 2,  // 0
    2, 5, 2, 2, 2, 4, 6, 2, 2, 4, 2, 2, 2, 4, 7, 2,  // 1
    6, 6, 2, 2, 3, 3, 5, 2, 4, 2, 2, 2, 4, 4, 6, 2,  // 2
    2, 5, 2, 2, 2, 4, 6, 2, 2, 4, 2, 2, 2, 4, 7, 2,  // 3
    6, 6, 2, 2, 2, 3, 5, 2, 3, 2, 2, 2, 3, 4, 6, 2,  // 4
    2, 5, 2, 2, 2, 4, 6, 2, 2, 4, 2, 2, 2, 4, 7, 2,  // 5
    6, 6, 2, 2, 2, 3, 5, 2, 4, 2, 2, 2, 5, 4, 6, 2,  // 6
    2, 5, 2, 2, 2, 4, 6, 2, 2, 4, 2, 2, 2, 4, 7, 2,  // 7
    2, 6, 2, 2, 3, 3, 3, 2, 2, 2, 2, 2, 4, 4, 4, 2,  // 8
    2, 6, 2, 2, 4, 4, 4, 2, 2, 5, 2, 2, 2, 5, 2, 2,  // 9
    2, 6, 2, 2, 3, 3, 3, 2, 2, 2, 2, 2, 4, 4, 4, 2,  // a
    2, 5, 2, 2, 4, 4, 4, 2, 2, 4, 2, 2, 4, 4, 4, 2,  // b
    2, 6, 2, 2, 3, 3, 5, 2, 2, 2, 2, 2, 4, 4, 6, 2,  // c
    2, 5, 2, 2, 2, 4, 6, 2, 2, 4, 2, 2, 2, 4, 7, 2,  // d
    2, 6, 2, 2, 3, 3, 5, 2, 2, 2, 2, 2, 4, 4, 6, 2,  // e
    2, 5, 2, 2, 2, 4, 6, 2, 2, 4, 2, 2, 2, 4, 7, 2   // f
  };

//============================================================================
// Instruction mnemonic table
//============================================================================
const char* M6507::ourInstructionMnemonicTable[256] = {
  "BRK",  "ORA",  "n/a",  "n/a",  "n/a",  "ORA",  "ASL",  "n/a",
  "PHP",  "ORA",  "ASLA", "n/a",  "n/a",  "ORA",  "ASL",  "n/a",
  "BPL",  "ORA",  "n/a",  "n/a",  "n/a",  "ORA",  "ASL",  "n/a",
  "CLC",  "ORA",  "n/a",  "n/a",  "n/a",  "ORA",  "ASL",  "n/a",

  "JSR",  "AND",  "n/a",  "n/a",  "BIT",  "AND",  "ROL",  "n/a",
  "PLP",  "AND",  "ROLA", "n/a",  "BIT",  "AND",  "ROL",  "n/a",
  "BMI",  "AND",  "n/a",  "n/a",  "n/a",  "AND",  "ROL",  "n/a",
  "SEC",  "AND",  "n/a",  "n/a",  "n/a",  "AND",  "ROL",  "n/a",
  
  "RTI",  "EOR",  "n/a",  "n/a",  "n/a",  "EOR",  "LSR",  "n/a",
  "PHA",  "EOR",  "LSRA", "n/a",  "JMP",  "EOR",  "LSR",  "n/a",
  "BVC",  "EOR",  "n/a",  "n/a",  "n/a",  "EOR",  "LSR",  "n/a",
  "CLI",  "EOR",  "n/a",  "n/a",  "n/a",  "EOR",  "LSR",  "n/a",

  "RTS",  "ADC",  "n/a",  "n/a",  "n/a",  "ADC",  "ROR",  "n/a",
  "PLA",  "ADC",  "RORA", "n/a",  "JMP",  "ADC",  "ROR",  "n/a",
  "BVS",  "ADC",  "n/a",  "n/a",  "n/a",  "ADC",  "ROR",  "n/a",
  "SEI",  "ADC",  "n/a",  "n/a",  "n/a",  "ADC",  "ROR",  "n/a",

  "n/a",  "STA",  "n/a",  "n/a",  "STY",  "STA",  "STX",  "n/a",
  "DEY",  "n/a",  "TXA",  "n/a",  "STY",  "STA",  "STX",  "n/a",
  "BCC",  "STA",  "n/a",  "n/a",  "STY",  "STA",  "STX",  "n/a",
  "TYA",  "STA",  "TXS",  "n/a",  "n/a",  "STA",  "n/a",  "n/a",

  "LDY",  "LDA",  "LDX",  "n/a",  "LDY",  "LDA",  "LDX",  "n/a",
  "TAY",  "LDA",  "TAX",  "n/a",  "LDY",  "LDA",  "LDX",  "n/a",
  "BCS",  "LDA",  "n/a",  "n/a",  "LDY",  "LDA",  "LDX",  "n/a",
  "CLV",  "LDA",  "TSX",  "n/a",  "LDY",  "LDA",  "LDX",  "n/a",

  "CPY",  "CMP",  "n/a",  "n/a",  "CPY",  "CMP",  "DEC",  "n/a",
  "INY",  "CMP",  "DEX",  "n/a",  "CPY",  "CMP",  "DEC",  "n/a",
  "BNE",  "CMP",  "n/a",  "n/a",  "n/a",  "CMP",  "DEC",  "n/a",
  "CLD",  "CMP",  "n/a",  "n/a",  "n/a",  "CMP",  "DEC",  "n/a",

  "CPX",  "SBC",  "n/a",  "n/a",  "CPX",  "SBC",  "INC",  "n/a",
  "INX",  "SBC",  "NOP",  "n/a",  "CPX",  "SBC",  "INC",  "n/a",
  "BEQ",  "SBC",  "n/a",  "n/a",  "n/a",  "SBC",  "INC",  "n/a",
  "SED",  "SBC",  "n/a",  "n/a",  "n/a",  "SBC",  "INC",  "n/a"
};


