package biss.awt.kernel;

import biss.awt.Awt;
import java.awt.Color;
import java.awt.Font;
import java.awt.FontMetrics;
import java.awt.Image;
import java.awt.Rectangle;
import java.awt.image.ImageObserver;
import java.awt.peer.ComponentPeer;

/**
 * kernel implementation of java.awt.Graphics.
 * Most of the work is done by native methods of class NativeLib. Graphics
 * adds just some functionality for clipping and image rendering.
 * What really is important are the fields:
 *
 * 'pData' holds whatever is appropriate to link a Graphics object to its
 * native counterpart (some kind of a handle to a GC, PS, Pen,..).
 *
 * 'Lib' is the link to the native lib wrapper (which does the real work).
 *
 * 'Wnd' or 'Img' are exclusive pointers to the environment object (the object
 * is used to draw within) which are required for propper initialization
 * (colors, fonts, ..). 
 *
 * The remaining fields save specific states of the Graphics object.
 *
 * Note that the caching of Graphics objects is done on the native side and
 * does not show up in this class (other than providing all the required links
 * and states to set/reset reused native Graphics objects).
 *
 * (C) 1996,97 BISS GmbH Germany, see file 'LICENSE.BISS-AWT' for details
 * @author P.C.Mehlitz
 */
public class Graphics
  extends java.awt.Graphics
{
	NativeLib Lib;
	ComponentPeer Wnd;
	Image Img;
	Rectangle ClipRect;
	Color CurrentColor;
	Font CurrentFont;
	Color XorColor;
	long PData;
	int XOrigin;
	int YOrigin;

Graphics ( NativeLib lib ) {
	Lib = lib;
}

Graphics ( NativeLib lib, ComponentPeer wnd ) {
	Lib = lib;
	Wnd = wnd;

	Lib.initGraphics( this);
}

Graphics ( NativeLib lib, Image img ) {
	Lib = lib;
	Img = img;

	Lib.initGraphics( this);
}

public void clearRect(int x, int y, int width, int height) {
	Lib.clearRect( this, x, y, width, height);
}

public void clipRect(int x, int y, int width, int height) {
	Rectangle r = new Rectangle( x, y, width, height);

	if ( ClipRect == null ) 
		ClipRect = r;
	else
		ClipRect = ClipRect.intersection( r);

	if ( ClipRect.width == 0 || ClipRect.height == 0 )
		ClipRect = null;

	Lib.clipGraphics( this);
}

public void copyArea(int x, int y, int width, int height, int dx, int dy) {
	Lib.copyArea( this, x, y, width, height, dx, dy);
}

public java.awt.Graphics create () {

	if ( Wnd != null )
		return new Graphics( Lib, Wnd);

	if ( Img != null )
		return new Graphics( Lib, Img);

	return null;
}

public void dispose () {
	Lib.releaseGraphics( this);

	Wnd = null;
	Img = null;
	ClipRect = null;
	CurrentColor = null;
	CurrentFont = null;
}

public void draw3DRect(int x, int y, int width, int height, boolean raised) {
	Lib.draw3DRect( this, x, y, width, height, raised,
	                Awt.BorderWidth, Awt.ShadowBrightClr, Awt.ShadowDarkClr);
}

public void drawArc(int x, int y, int width, int height, int startAngle, int arcAngle){
	Lib.drawArc( this, x, y, width, height, startAngle, arcAngle);
}

public void drawChars(char data[], int offset, int length, int x, int y) {
	Lib.drawChars( this, data, offset, length, x, y);
}

public boolean drawImage (Image img, int x, int y, Color bgcolor, ImageObserver observer) {
	return drawImage( img, x, y, 0, 0, bgcolor, observer);
}

public boolean drawImage ( Image img, int x, int y, ImageObserver observer) {
	return drawImage( img, x, y, 0, 0, null, observer);
}

public boolean drawImage ( Image img, int x, int y, int width, int height,
                    Color bgColor, ImageObserver observer ) {

	int state = Awt.DefToolkit.checkImage( img, width, height, observer);

	if ( state == 0 ) {
		Awt.DefToolkit.prepareImage( img, width, height, observer);
		return false;
	}
	else if ( state >= 3 ) {
		Lib.drawImage( this, img, x, y, width, height, bgColor);
		return true;
	}

	return false;
}

public boolean drawImage ( Image img, int x, int y, int width, int height, ImageObserver observer){
	return drawImage( img, x, y, width, height, null, observer);
}

public void drawLine(int x1, int y1, int x2, int y2) {
	Lib.drawLine( this, x1, y1, x2, y2);
}

public void drawOval(int x, int y, int width, int height) {
	Lib.drawOval( this, x, y, width, height);
}

public void drawPolygon ( int xPoints[], int yPoints[], int nPoints ) {
	Lib.drawPolygon( this, xPoints, yPoints, nPoints);
}

public void drawRect ( int x, int y, int width, int height ) {
	Lib.drawRect( this, x, y, width, height);
}

public void drawRoundRect(int x, int y, int width, int height, int arcWidth, int arcHeight){
	Lib.drawRoundRect( this, x, y, width, height, arcWidth, arcHeight);
}

public void drawString ( String str, int x, int y ) {
	Lib.drawString( this, str, x, y);
}

public void fillArc(int x, int y, int width, int height, int startAngle, int arcAngle){
	Lib.fillArc( this, x, y, width, height, startAngle, arcAngle);
}

public void fillOval ( int x, int y, int width, int height ){
	Lib.fillOval( this, x, y, width, height);
}

public void fillPolygon ( int xPoints[], int yPoints[], int nPoints ) {
	Lib.fillPolygon( this, xPoints, yPoints, nPoints);
}

public void fillRect(int x, int y, int width, int height) {
	Lib.fillRect( this, x, y, width, height);
}

public void fillRoundRect(int x, int y, int width, int height, int arcWidth, int arcHeight){
	Lib.fillRoundRect( this, x, y, width, height, arcWidth, arcHeight);
}

public Rectangle getClipRect() {
	return ClipRect;
}

public Color getColor() {
	return CurrentColor;
}

public Font getFont() {
	return CurrentFont;
}

public FontMetrics getFontMetrics ( Font f ) {
	return new biss.awt.kernel.FontMetrics( Lib, f);
}

public void setColor ( Color c ){
	if ( c != CurrentColor ) {
		Lib.setColor( this, c);
		CurrentColor = c;
	}
}

public void setFont ( Font font ) {
	if ( font != CurrentFont ){
		Lib.setFont( this, font);
		CurrentFont = font;
	}
}

public void setPaintMode() {
	Lib.setPaintMode( this);
	XorColor = null;
}

public void setXORMode ( Color xorColor ) {
	XorColor = xorColor;
	Lib.setXORMode( this);
}

public void translate ( int x, int y ) {
	XOrigin += x;
	YOrigin += y;
}
}
