#include "cthugha.h"
#include "interface.h"
#include "keys.h"
#include "sound.h"
#include "imath.h"
#include "CthughaBuffer.h"
#include "CthughaDisplay.h"
#include "AutoChanger.h"

#include <ctype.h>
#include <signal.h>

Interface * Interface::interfaces[32] = {
    NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 
    NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 
    NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 
    NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL, 
};
Interface * Interface::current = NULL;

int Interface::showStatus = 0;





////////////////////////////////////////////////////////////////////////////

Interface::Interface(InterfaceElement ** el, int nEl, int pos) : 
    elements(el), nElements(nEl), sel(0) {

    if(pos >= 0)
	interfaces[pos] = this;
}




void Interface::set(Interface * interface) {

    if(current == interface)
	interface = interfaces[0];

    if(interface == NULL)
	return;

    current = interface;
}



ErrorMessages errors;


//
// standard interface
//

// default key handler
int Interface::do_key(int key) {

    int new_sel = -1;
    int jump = 0, dir = 1;

    switch(key) {
    case CK_HOME:
	new_sel = 0;	dir = 1;
	break;
    case CK_PGUP:
	jump = -10;	dir = -1;
	break;
    case CK_UP:
	jump = -1;	dir = -1;
	break;
	
    case CK_END:
	new_sel = nElements - 1; dir = -1;
	break;
    case CK_PGDN:
	jump = 10;	dir = 1;
	break;
    case CK_DOWN:
	jump = 1;	dir = 1;
	break;

    case 'o':
	set( interfaceCoreOption );
	return 0;
 
    case 'O':
	set( interfaceOptions );
	return 0;

    case '?': case '':		//  is the on the same key as ? on a German keyboard
	key = CK_FKT(1);		

    case CK_FKT(1):	case CK_FKT(2):    case CK_FKT(3):	case CK_FKT(4):
    case CK_FKT(5):	case CK_FKT(6):    case CK_FKT(7):	case CK_FKT(8):
    case CK_FKT(9):	case CK_FKT(10):   case CK_FKT(11):	case CK_FKT(12): 
    case CK_FKT(13):	case CK_FKT(14):   case CK_FKT(15):	case CK_FKT(16): 
    case CK_FKT(17):	case CK_FKT(18):   case CK_FKT(19):	case CK_FKT(20):
	key = key - CK_FKT(1) + 1;		// convert keycode to integer

	set( interfaces[key]);
	return 0;
	
    case 'q': case 'Q': case CK_ESC:
	set( interfaces[0]);
	return 0;

    case '.': case ',':
	showStatus = 1 - showStatus;
	return 0;
    }

    // update selection line
    if(jump && (nElements > 1)) {
	new_sel = mod(sel + jump, nElements);
    }
    if( (new_sel >= 0) && (nElements > 1)) {
	while( !elements[new_sel]->selectable && (sel != new_sel) )
	    new_sel = mod(new_sel + dir, nElements);
	
	if(sel == new_sel)
	    return 0;
	
	elements[sel]->selected = 0;
	sel = new_sel;
	elements[new_sel]->selected = 1;
	return 0;
    }

    return key;
}


// default display handler
void Interface::display() {
    for(int i=0; i < nElements; i++)
	elements[i]->display(i);

    if(showStatus) {
	static char str[512];
	sprintf(str, "%s%s%s", 
		(cthughaDisplay != NULL) ? cthughaDisplay->status() : "", 
		(autoChanger != NULL) ? autoChanger->status() : "",
		(CthughaBuffer::current != NULL) ? CthughaBuffer::current->translate.status():"");

	displayDevice->print(str, text_size.y - 1, 'l', TEXT_COLOR_NORMAL, 1);
    }

}

// default runner
void Interface::run() {

    // handle keys
    int key;
    while( (key = getkey()) != CK_NONE) {
	if( sel < nElements)
	    if( elements[sel]->do_key(key) == 0)
		continue;
	this->do_key(key);
    }
}













void InterfaceElement::display(int y) {
    displayDevice->print(text, y, 'l', selected ? c2 : c1);
}


int InterfaceElement::do_key(int key) {
    return key;
}




void InterfaceElementText::display(int y) {
    displayDevice->print(text, y, 'l', selected ? c2 : c1);
}



void Interface::next() {
    for(int i=0; i < 32; i++) {
	if( current == interfaces[i]) {
	    for(int j=i+1; j < (32+i); j++) {
		if(interfaces[j % 32] != NULL) {
		    set(interfaces[j]);
		    return;
		}
	    }
	}
    }
}
void Interface::prev() {
    for(int i=0; i < 32; i++) {
	if( current == interfaces[i]) {
	    for(int j=i-1; j > (i-32); j--) {
		if(interfaces[(j+32) % 32] != NULL) {
		    set(interfaces[j]);
		    return;
		}
	    }
	}
    }
}


int InterfaceElementTitle::do_key(int key) {
	
    switch(key) {
    case '+': case '>':
    case CK_RIGHT:
	Interface::next();
	return 0;
    case '-': case '<':
    case CK_LEFT:
	Interface::prev();
	return 0;
    }
    return key;
};


void InterfaceElementOption::display(int line) {
    char str[512];
    char fmt[512];
    char in[512];

    sprintf(fmt, "%%c%%-%ds%%c", min(text_size.x - 3, 77) );
    sprintf(in, text, option->text());

    // make format and include the > <
    sprintf(str, fmt, selected ? '>' : ' ', in, selected ? '<' : ' ');

    // and bring it to the screen
    displayDevice->print(str, line, 'l', selected ? c2 : c1);
}

int InterfaceElementOption::do_key(int key) {

    switch(key) {
    case '+':
	option->change(inc1);
	return 0;
    case '-':
	option->change(-inc1);
	return 0;
    case CK_RIGHT:
	option->change(inc2);
	return 0;
    case CK_LEFT:
	option->change(-inc2);
	return 0;
    case '*': 
	option->change(inc3);
	return 0;
    case '/':
	option->change(-inc3);
	return 0;
    case '1': case '2': case '3': case '4': case '5':
    case '6': case '7': case '8': case '9': case '0':
	option->setValue( inc2 * (key - '0') );
	option->change(0);
	return 0;
    }
    return key;
}



void ErrorMessages::addMessage(const char * text) {
    if(nMsgs == 128) {
	printfe("too many errors: %s\n", text);
	return;
    }
    strncpy(msgs[nMsgs], text, 128);
    on_screen[nMsgs] = gettime();

    nMsgs ++;
}
void ErrorMessages::display() {
    
    // bring messages to screen
    for(int i = 0; i < nMsgs; i ++) {
	displayDevice->print( msgs[i], - (nMsgs-i), 'r', TEXT_COLOR_ERROR);
    }

    // remove old messages
    const int errorTime = 300;
    while( (nMsgs > 0) && ((gettime() - on_screen[0]) > errorTime) ) {
	for(int i = 1; i < nMsgs; i ++) {
	    strncpy(msgs[i-1], msgs[i], 128);
	    on_screen[i-1] = on_screen[i];
	}
	nMsgs --;
    }
}

    





