/* transformw.c */
/************************************************************
 *                                                          *
 *  Permission is hereby granted  to  any  individual   or  *
 *  institution   for  use,  copying, or redistribution of  *
 *  the xgobi code and associated documentation,  provided  *
 *  that   such  code  and documentation are not sold  for  *
 *  profit and the  following copyright notice is retained  *
 *  in the code and documentation:                          *
 *        Copyright (c) 1990, ..., 1996 Bellcore            *
 *                                                          *
 *  We welcome your questions and comments, and request     *
 *  that you share any modifications with us.               *
 *                                                          *
 *    Deborah F. Swayne            Dianne Cook              *
 *   dfs@research.att.com       dicook@iastate.edu          *
 *      (973) 360-8423    www.public.iastate.edu/~dicook/   *
 *                                                          *
 *                    Andreas Buja                          *
 *                andreas@research.att.com                  *
 *              www.research.att.com/~andreas/              *
 *                                                          *
 ************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <values.h>
#include <math.h>
#include "xincludes.h"
#include "xgobitypes.h"
#include "xgobivars.h"
#include "xgobiexterns.h"
#include "DrawingA.h"

#define NDOMAINBTNS 3
static int domain_ind;
#define DOMAIN_OK       0
#define RAISE_MIN_TO_0  1
#define RAISE_MIN_TO_1  2
static Widget domain_menu_cmd, domain_menu_btn[NDOMAINBTNS] ;
static char *domain_menu_btn_label[] = {
  "No adjustment",
  "Raise minimum to 0",
  "Raise minimum to 1",
};
static Widget *varlabel;

#define NTFORMS 13
#define RESTORE      0
#define POWER        1
#define ABSVALUE     2
#define NEGATIVE     3
#define INVERSE      4
#define LOG10        5
#define SCALE        6
#define STANDARDIZE  7
#define DISCRETE2    8
#define NEGLN        9
#define PERMUTE     10
#define SORT        11
#define NORMSCORE   12

static char *tform_names[] = {
  "Restore",
  "Power family",
  "Absolute Value",
  "Negative",
  "Inverse",
  "Common Log",
  "Scale to [0,1]",
  "Standardize",
  "Discretize: 2 levels",
  "Abs(Natural Log)",
  "Permute",
  "Sort",
  "Normal Score"
};
/*
  "Natural Log",
  "Natural Log (x+1)",
  "Natural Log (x+min)",
  "Common Log (x+1)",
*/

char message[MSGLENGTH];
#define DOMAIN_ERROR sprintf(message, "Data outside the domain of function.\n")

static Widget *var_cbox, *tf_cbox;
static Widget tpane, tpopup = NULL;
static Position popupx = -1, popupy = -1;
static Widget exponent_lbl;
static float exponent = 1.0;
static float domain_incr;
static int ntform_cols, *tform_cols = NULL;

/*
 * this will have to become more complex, to include the
 * domain fixer and the parameter
*/
typedef struct {
  int tform;
  float domain_incr;
  float param;
} TFormType;
TFormType *tform_tp;  /* the transformation applied to each variable */

void
alloc_transform_tp(xgobidata *xg)
{
  tform_tp = (TFormType *) XtRealloc((char *) tform_tp,
    (Cardinal) xg->ncols * sizeof(TFormType));
}

static void
reset_tform_new(xgobidata *xg) {
  int j;

  for (j=0; j<xg->ncols_used; j++) {
    tform_tp[j].tform = RESTORE;
    tform_tp[j].domain_incr = 0.;
    tform_tp[j].param = 0.;
  }
}

static int 
sort_compare (float *val1, float *val2)
{
  if (*val1 < *val2) 
    return (-1);
  else if (*val1 == *val2)
    return (0);
  else 
    return (1);
}

int
transform_new(xgobidata *xg, int *cols, int ncols, float domain_incr,
int tfnum, double param)
{
  int i, j, n;
  float min, max, diff;
  float mean, stddev;
  float median, mad, ref;
  Boolean allequal;
  int tmpi, numperm, indx_flag;

  switch(tfnum)
  {
    case RESTORE:    /* Restore original values */
      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++)
          xg->tform_data[i][j] = xg->raw_data[i][j] + domain_incr;

        (void) strcpy(xg->collab_tform[j], xg->collab[j]);
        XtVaSetValues(varlabel[j], XtNlabel, xg->collab_tform[j], NULL);
      }
      break;

    case POWER:  /* Power transform family, with natural log at 0 */

      if (fabs(param-0) < .001) {       /* Natural log */
        for (n=0; n<ncols; n++) {
          j = cols[n];
          for (i=0; i<xg->nrows; i++) {
            if (xg->raw_data[i][j] + domain_incr <= 0) {
              DOMAIN_ERROR;
              show_message(message, xg);
              return(0);
            }
          }
        }
        for (n=0; n<ncols; n++) {
          j = cols[n];
          for (i=0; i<xg->nrows; i++) {
            xg->tform_data[i][j] = (float)
              log((double) (xg->raw_data[i][j] + domain_incr));
          }

          (void) sprintf(xg->collab_tform[j], "ln(%s)", xg->collab[j]);
          XtVaSetValues(varlabel[j], XtNlabel, xg->collab_tform[j], NULL);
        }
      }

      else {
        double dtmp;

        for (n=0; n<ncols; n++) {
          j = cols[n];
          for (i=0; i<xg->nrows; i++) {
            dtmp = pow((double) (xg->raw_data[i][j] + domain_incr), param);
            /* If dtmp no good, then use -1 * xg->raw_data[i][j] ? */
            xg->tform_data[i][j] = (float) dtmp;
          }
          (void) sprintf(xg->collab_tform[j], "%s^(%.2f)",
            xg->collab[j], param);
          XtVaSetValues(varlabel[j], XtNlabel, xg->collab_tform[j], NULL);
        }
      }
      break;

    case ABSVALUE:    /* Absolute value */
      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++) {
          /* quicker to fabs everything, or do this test? */
          if ((xg->raw_data[i][j] + domain_incr) < 0)
            xg->tform_data[i][j] = (float)
              fabs((double)(xg->raw_data[i][j] + domain_incr)) ;
          else
            xg->tform_data[i][j] = xg->raw_data[i][j] + domain_incr ;
        }

        (void) sprintf(xg->collab_tform[j], "Abs(%s)", xg->collab[j]);
        XtVaSetValues(varlabel[j], XtNlabel, xg->collab_tform[j], NULL);
      }
      break;

    case NEGATIVE:    /* Multiply original values by -1 */
      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++)
          xg->tform_data[i][j] =
            (float) -1.0 * (xg->raw_data[i][j] + domain_incr) ;

        (void) sprintf(xg->collab_tform[j], "-%s", xg->collab[j]);
        XtVaSetValues(varlabel[j], XtNlabel, xg->collab_tform[j], NULL);
      }
      break;

    case INVERSE:    /* 1/x */
      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++) {
          if (xg->raw_data[i][j] + domain_incr == 0) {
            DOMAIN_ERROR;
            show_message(message, xg);
            return(0);
          }
        }
      }

      for (n=0; n<ncols; n++)
      {
        j = cols[n];
        for (i=0; i<xg->nrows; i++)
        {
          xg->tform_data[i][j] = (float)
            pow((double) (xg->raw_data[i][j] + domain_incr),
              (double) (-1.0));
        }

        (void) sprintf(xg->collab_tform[j], "1/%s", xg->collab[j]);
        XtVaSetValues(varlabel[j], XtNlabel, xg->collab_tform[j], NULL);
      }
      break;

    case NEGLN:    
      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++) {
          if (-1.0 * (xg->raw_data[i][j] + domain_incr) <= 0) {
            DOMAIN_ERROR;
            show_message(message, xg);
            return(0);
          }
        }
      }
      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++) {
          xg->tform_data[i][j] = (float)
            log((double) (-1.0 * (xg->raw_data[i][j] + domain_incr)));
        }

        (void) sprintf(xg->collab_tform[j], "ln(%s)", xg->collab[j]);
        XtVaSetValues(varlabel[j], XtNlabel, xg->collab_tform[j], NULL);
      }
      break;

    case LOG10:    /* Common log */
      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++) {
          if (xg->raw_data[i][j] + domain_incr <= 0) {
            DOMAIN_ERROR;
            show_message(message, xg);
            return(0);
          }
        }
      }
      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++) {
          xg->tform_data[i][j] = (float)
            log10((double) (xg->raw_data[i][j] + domain_incr));
        }

        (void) sprintf(xg->collab_tform[j], "log10(%s)", xg->collab[j]);
        XtVaSetValues(varlabel[j], XtNlabel, xg->collab_tform[j], NULL);
      }
      break;

    case SCALE:    /* Map onto [0,1] */
      /* First find min and max; they get updated after transformations */
/* this is computed without regard to the value of domain_incr */
      min_max(xg, xg->raw_data, cols, ncols, &min, &max);
      adjust_limits(&min, &max);
      diff = max - min;

      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++)
          xg->tform_data[i][j] = (float)
            (xg->raw_data[i][j] + domain_incr - min)/diff;

        (void) sprintf(xg->collab_tform[j], "%s [0,1]", xg->collab[j]);
        XtVaSetValues(varlabel[j], XtNlabel, xg->collab_tform[j], NULL);
      }
      break;

    case STANDARDIZE:    /* (x-mean)/sigma */
      /* First find min and max */
      mean_stddev(xg, xg->raw_data, cols, ncols, &min, &max, &mean, &stddev);

      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++)
          xg->tform_data[i][j] = (float)
            (xg->raw_data[i][j] + domain_incr - mean)/stddev;

        (void) sprintf(xg->collab_tform[j], "(%s-m)/s", xg->collab[j]);
        XtVaSetValues(varlabel[j], XtNlabel, xg->collab_tform[j], NULL);
      }
      break;

    case DISCRETE2:    /* x>median */
      /* refuse to discretize if all values are the same */
      allequal = True;
      ref = xg->raw_data[0][cols[0]] + domain_incr ;
      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++) {
          if (xg->raw_data[i][j] + domain_incr != ref) {
            allequal = False;
            break;
          }
        }
      }
      if (allequal) {
        DOMAIN_ERROR;
        show_message(message, xg);
        return(0);
      }

      /* First find median */
      med_mad(xg, xg->raw_data, cols, ncols, &min, &max, &median, &mad);
      /* Then find the true min and max */
      min_max(xg, xg->raw_data, cols, ncols, &min, &max);
      /* This prevents the collapse of the data in a special case */
      if (max == median)
        median = (min + max)/2.0;

      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++)
          if( xg->raw_data[i][j] + domain_incr > median )
              xg->tform_data[i][j] = (float) 1.0;
            else
              xg->tform_data[i][j] = (float) 0.0;

        (void) sprintf(xg->collab_tform[j], "%s:0,1", xg->collab[j]);
        XtVaSetValues(varlabel[j], XtNlabel, xg->collab_tform[j], NULL);
      }
      break;

    case PERMUTE:    /* (x-mean)/sigma */
    { /* This curly brace allows me to define variables for this case */
      /* First generate a random vector */
      /* Allocate array for permutation indices */
      /* First generate a random vector */
      int *permindx;
      permindx = (int *) XtMalloc((Cardinal) xg->nrows * sizeof(int));

      numperm = 0;
      while (numperm < xg->nrows)
      {
#ifdef USE_DRAND48
        tmpi = (int) (drand48() * (double)xg->nrows);
#else
        tmpi =  (int) ((double) random()/ (double) MAXINT * xg->nrows);
#endif
        indx_flag = 0;
        for (i=0; i<numperm; i++) {
          if (tmpi == permindx[i]) {
            indx_flag = 1;
	      }
	    }
        if (!indx_flag) {
          permindx[numperm] = tmpi;
          numperm++;
        }
      }

      for (n=0; n<ncols; n++) {
        j = cols[n];
        for (i=0; i<xg->nrows; i++)
          xg->tform_data[i][j] = xg->raw_data[permindx[i]][j] + domain_incr;

        (void) sprintf(xg->collab_tform[j], "perm(%s)", xg->collab[j]);
        XtVaSetValues(varlabel[j], XtNlabel, xg->collab_tform[j], NULL);
      }

      XtFree((XtPointer) permindx);
    }
      break;

    case SORT: 
    {
      /*  create sort index  here - mallika */
      float *sort_data; /* mallika */
      /* Allocate array for sorted columns - mallika */
      sort_data = (float *) XtMalloc((Cardinal) xg->nrows * sizeof(float));

      for (n=0; n<ncols; n++)
      {
        j = cols[n];
        for (i=0; i<xg->nrows; i++)
          sort_data[i]=xg->raw_data[i][j] + domain_incr;

        qsort((char *) sort_data, xg->nrows, sizeof(float), sort_compare);
   
        for (i=0; i<xg->nrows; i++)
          xg->tform_data[i][j] = sort_data[i]; 

        (void) sprintf(xg->collab_tform[j], "sort(%s)", xg->collab[j]);
        XtVaSetValues(varlabel[j], XtNlabel, xg->collab_tform[j], NULL);
      }
      XtFree((XtPointer) sort_data);/* mallika */
    }
      break;

    case NORMSCORE:  /* mallika*/
    {
      float *norm_score_data;  /*mallika*/
      float ftmp;

      /* Allocate array for normalized scores  - mallika */
      norm_score_data = (float *)
        XtMalloc((Cardinal) xg->nrows * sizeof(float));

     for (n=0; n<ncols; n++) {
        float normmean=0, normvar=0;
        j = cols[n];
        for (i=0; i<xg->nrows; i++)
        {
          ftmp = xg->raw_data[i][j] + domain_incr;
          norm_score_data[i] = ftmp;
          normmean += ftmp;
          normvar += (ftmp * ftmp);
        }
        normmean/=xg->nrows;
        normvar=(float)sqrt((float)(normvar/xg->nrows-normmean*normmean));
        for (i=0; i<xg->nrows; i++)
          norm_score_data[i]=(norm_score_data[i]-normmean)/normvar;

        for (i=0; i<xg->nrows; i++)
        {
          if (norm_score_data[i]>0)
            norm_score_data[i] = erf(norm_score_data[i]/sqrt(2.))/
              2.8284271+0.5;
          else if (norm_score_data[i]<0)
            norm_score_data[i] = 0.5-erf((float) fabs((double) 
              norm_score_data[i])/sqrt(2.))/2.8284271;
          else 
            norm_score_data[i]=0.5;
	 }
        
        for (i=0; i<xg->nrows; i++)
          xg->tform_data[i][j] = norm_score_data[i]; 

        (void) sprintf(xg->collab_tform[j], "normsc(%s)", xg->collab[j]);
        XtVaSetValues(varlabel[j], XtNlabel, xg->collab_tform[j], NULL);
      }
      XtFree((XtPointer) norm_score_data);/* mallika */
    }
      break;
  }
  return(1);
}

static void
tform_response(xgobidata *xg, int *cols, int ncols, int domain_incr,
int tfno, double param) {
  int j, n;

  if (xg->ncols_used > 2)
    update_sphered(xg, cols, ncols);
  update_lims(xg);
  update_world(xg);

  /* Set tform_tp[] for transformed columns */
  for (n=0; n<ncols; n++) {
    tform_tp[cols[n]].tform = tfno;
    tform_tp[cols[n]].domain_incr = domain_incr;
    tform_tp[cols[n]].param = param;
  }

  world_to_plane(xg);
  plane_to_screen(xg);
  /*
    This bit of init_axes() is needed.
  */
  for (n=0; n<ncols; n++) {
    j = cols[n];
    xg->nicelim[j].min = xg->lim0[j].min;
    xg->nicelim[j].max = xg->lim0[j].max;
    SetNiceRange(j, xg);
    xg->deci[j] = set_deci(xg->tickdelta[j]);
  }

  if (xg->is_xyplotting)
    init_ticks(&xg->xy_vars, xg);
  else if (xg->is_dotplotting)
    init_ticks(&xg->dotplot_vars, xg);

  if (xg->is_brushing) {
    assign_points_to_bins(xg);
    if (xg->brush_mode == transient)
      reinit_transient_brushing(xg);
  }

  plot_once(xg);

  if (xg->is_cprof_plotting)
    update_cprof_plot(xg);
}

static void
reset_exp(xgobidata *xg) {
  char lbl[32];
  Boolean state;
  int j, k, groupno, domain_incr, tfno;

  /* reset the power label */
  sprintf(lbl, "%4.2f", exponent);
  XtVaSetValues(exponent_lbl, XtNlabel, lbl, NULL);

  ntform_cols = 0;
  for (j=0; j<xg->ncols_used; j++) {
    XtVaGetValues(var_cbox[j], XtNstate, &state, NULL);
    if (state) {
      tform_cols[ntform_cols++] = j;
      groupno = xg->vgroup_ids[j];
      for (k=j; k<xg->ncols_used; k++) {
        if (xg->vgroup_ids[k] == groupno)
          tform_cols[ntform_cols++] = k;
      }
    }
  }

  if (ntform_cols > 0) {
    if (transform_new(xg, tform_cols, ntform_cols, domain_incr,
      POWER, (double) exponent))
    {
      tform_response(xg, tform_cols, ntform_cols, domain_incr,
        tfno, (double) exponent);
    }
  }
}

/* ARGSUSED */
static XtCallbackProc
reduce_exp_cback(Widget w, xgobidata *xg, XtPointer slideposp)
{
  exponent -= .01;
  reset_exp(xg);
}
/* ARGSUSED */
static XtCallbackProc
increase_exp_cback(Widget w, xgobidata *xg, XtPointer slideposp)
{
  exponent += .01;
  reset_exp(xg);
}

/* ARGSUSED */
static XtCallbackProc
exp_sbar_cback(Widget w, xgobidata *xg, XtPointer slideposp)
{
  int iexp, third_dec_place;
  float ftmp;
  float fslidepos = * (float *) slideposp;

  /* rescale from [0,1] to [-5, 5] */
  ftmp = (fslidepos - .5) * 10;

/*
 * restrict the exponent to two decimal places
*/

  third_dec_place = ((int) (ftmp * 1000)) % 10;
  iexp = (int) (ftmp * 100.);
  if (third_dec_place < 5)
    exponent = ((float) iexp) / 100.;
  else
    exponent = ((float) (iexp+1)) / 100.;

  reset_exp(xg);
}

/* ARGSUSED */
static XtCallbackProc
set_domain_incr_cback(Widget w, xgobidata *xg, XtPointer callback_data)
{
  int k;

  domain_ind = DOMAIN_OK;
  for (k=0; k<NDOMAINBTNS; k++)
    if (domain_menu_btn[k] == w) {
      domain_ind = k;
      break;
    }

  switch (domain_ind) {
    case DOMAIN_OK:
      domain_incr = 0;
      break;
    case RAISE_MIN_TO_0:
      domain_incr = xg->lim[ tform_cols[0] ].min;
      break;
    case RAISE_MIN_TO_1:
      domain_incr = xg->lim[ tform_cols[0] ].min + 1.0;
      break;
    default:
      domain_incr = 0;
  }

  XtVaSetValues(domain_menu_cmd,
    XtNlabel, domain_menu_btn_label[domain_ind],
    NULL);
}

/* ARGSUSED */
static XtCallbackProc
close_cback(Widget w, xgobidata *xg, XtPointer callback_data)
{
  XtDestroyWidget(tpopup);
  tpopup = NULL;
}

/* ARGSUSED */
XtCallbackProc
open_tform_popup_cback(Widget w, xgobidata *xg, XtPointer callback_data)
{
  Widget close;
  Widget form0;
  Dimension width, height;
  register int j, k;
  Widget box_variables, box_varlabels, box_tforms;
  Widget power_form, power_lbl, reduce_exp_arr, exp_sbar;
  Widget restore_cbox;
  Widget increase_exp_arr;
  char str[32];
  static char *domain_menu_str = "Before transforming:";
  Widget domain_menu_box, domain_menu_lab, domain_menu;

  if (tpopup == NULL) {
    tform_cols = (int *) XtMalloc((Cardinal) xg->ncols * sizeof(int));

    var_cbox = (Widget *) XtMalloc((Cardinal) xg->ncols * sizeof(Widget));
    tf_cbox = (Widget *) XtMalloc((Cardinal) (NTFORMS-2) * sizeof(Widget));

    if (popupx == -1 && popupy == -1) {
      XtVaGetValues(xg->workspace,
        XtNwidth, &width,
        XtNheight, &height, NULL);
      XtTranslateCoords(xg->workspace,
        (Position) width, (Position) (height/2), &popupx, &popupy);
    }

    /*
     * Create the popup to solicit subsetting arguments.
    */
    tpopup = XtVaCreatePopupShell("Variable Transformation",
      topLevelShellWidgetClass, xg->shell,
      XtNx, (Position) popupx,
      XtNy, (Position) popupy,
      XtNinput, (Boolean) True,
      XtNtitle, (String) "Hide or exclude groups",
      XtNiconName, (String) "Tform",
      NULL);
    if (mono) set_mono(tpopup);

    /*
     * Create a paned widget so the 'Click here ...'
     * can be all across the bottom.
    */
    tpane = XtVaCreateManagedWidget("Form",
      panedWidgetClass, tpopup,
      XtNorientation, (XtOrientation) XtorientVertical,
      XtNresizable, False,
      NULL);

    form0 = XtVaCreateManagedWidget("Form",
      formWidgetClass, tpane,
      XtNresizable, False,
      NULL);
    if (mono) set_mono(form0);

    box_tforms = XtVaCreateManagedWidget("Close",
      boxWidgetClass, form0,
      XtNorientation, (XtOrientation) XtorientVertical,
      XtNleft, (XtEdgeType) XtChainLeft,
      XtNright, (XtEdgeType) XtChainLeft,
      XtNtop, (XtEdgeType) XtChainTop,
      XtNbottom, (XtEdgeType) XtChainTop,
      NULL);

    domain_ind = DOMAIN_OK;
    build_labelled_menu(&domain_menu_box, &domain_menu_lab, domain_menu_str,
      &domain_menu_cmd, &domain_menu, domain_menu_btn,
      domain_menu_btn_label, domain_menu_btn_label,  /* no nicknames */
      NDOMAINBTNS, domain_ind, box_tforms, NULL,
      XtorientHorizontal, appdata.font, "Transformations", xg);
    for (k=0; k<NDOMAINBTNS; k++)
      XtAddCallback(domain_menu_btn[k],  XtNcallback,
        (XtCallbackProc) set_domain_incr_cback, (XtPointer) xg);

    /* Restore */
    restore_cbox = CreateCommand(xg, "Restore", True, 
      NULL, NULL, box_tforms, "Transformations");
    XtManageChild(restore_cbox);

    /* Power family */
    power_form = XtVaCreateManagedWidget("Form",
      boxWidgetClass, box_tforms,
      XtNresizable, False,
      XtNorientation, XtorientHorizontal,
      XtNhSpace, 1,
      XtNvSpace, 1,
      NULL);
    power_lbl = XtVaCreateManagedWidget("Label",
      labelWidgetClass, power_form,
      XtNlabel, (String) "Power",
      NULL);

    reduce_exp_arr = XtVaCreateManagedWidget("Icon",
      commandWidgetClass, power_form,
      XtNinternalHeight, (Dimension) 0,
      XtNinternalWidth, (Dimension) 0,
      XtNborderColor, (Pixel) appdata.fg,
      XtNbitmap, (Pixmap) leftarr,
      NULL);
    if (mono) set_mono(reduce_exp_arr);
    XtAddCallback(reduce_exp_arr, XtNcallback,
     (XtCallbackProc) reduce_exp_cback, (XtPointer) xg);

    sprintf(str, "Power Tformations");
    width = XTextWidth(appdata.font, str, strlen(str));

    exp_sbar = XtVaCreateManagedWidget("Scrollbar",
      scrollbarWidgetClass, power_form,
      XtNhorizDistance, (Dimension) 0,
      XtNwidth, (Dimension) width,
      XtNheight, (Dimension) 20,
      XtNorientation, (XtOrientation) XtorientHorizontal,
      NULL);
    if (mono) set_mono(exp_sbar);
    XtAddCallback(exp_sbar, XtNjumpProc,
     (XtCallbackProc) exp_sbar_cback, (XtPointer) xg);

/*
 * The value of std_width ranges between -5 and 5 and the
 * scrollbar is on [0, 1]
*/
    XawScrollbarSetThumb(exp_sbar, 1./10. + .5, -1.);

    increase_exp_arr = XtVaCreateManagedWidget("Icon",
      commandWidgetClass, power_form,
      XtNinternalHeight, (Dimension) 0,
      XtNinternalWidth, (Dimension) 0,
      XtNborderColor, (Pixel) appdata.fg,
      XtNhorizDistance, (Dimension) 0,
      XtNbitmap, (Pixmap) rightarr,
      NULL);
    if (mono) set_mono(increase_exp_arr);
    XtAddCallback(increase_exp_arr, XtNcallback,
     (XtCallbackProc) increase_exp_cback, (XtPointer) xg);

    (void) sprintf(str, "%1.2f", -9.99);
    width = XTextWidth(appdata.font, str,
        strlen(str) + 2*ASCII_TEXT_BORDER_WIDTH);
    (void) sprintf(str, "%4.2f", exponent);
    exponent_lbl = XtVaCreateManagedWidget("StdizeLabel",
      labelWidgetClass,  power_form,
      XtNlabel, (String) str,
      XtNwidth, width,
      NULL);
    if (mono) set_mono(exponent_lbl);

    for (j=0; j<NTFORMS-2; j++)
      tf_cbox[j] = CreateCommand(xg, tform_names[j+2], True, 
        NULL, NULL, box_tforms, "Transformations");
    XtManageChildren(tf_cbox, NTFORMS-2);

    box_variables = XtVaCreateManagedWidget("Box",
      boxWidgetClass, form0,
      XtNorientation, (XtOrientation) XtorientVertical,
      XtNleft, (XtEdgeType) XtChainLeft,
      XtNright, (XtEdgeType) XtChainLeft,
      XtNtop, (XtEdgeType) XtChainTop,
      XtNbottom, (XtEdgeType) XtChainTop,
      XtNfromHoriz, box_tforms,
      NULL);
    for (j=0; j<xg->ncols; j++)
      var_cbox[j] = CreateToggle(xg, xg->collab[j], True, 
        NULL, NULL, NULL, False, ANY_OF_MANY, box_variables, "Transformations");
    XtManageChildren(var_cbox, xg->ncols);

    box_varlabels = XtVaCreateManagedWidget("Form",
      formWidgetClass, form0,
      XtNorientation, (XtOrientation) XtorientVertical,
      XtNleft, (XtEdgeType) XtChainLeft,
      XtNright, (XtEdgeType) XtChainRight,
      XtNtop, (XtEdgeType) XtChainTop,
      XtNbottom, (XtEdgeType) XtChainTop,
      XtNfromHoriz, box_variables,
      NULL);
    varlabel = (Widget *) XtMalloc((Cardinal) xg->ncols * sizeof(Widget));
    for (j=0; j<xg->ncols; j++) {
      varlabel[j] = XtVaCreateWidget("Label",
        labelWidgetClass, box_varlabels,
        XtNleft, (XtEdgeType) XtChainLeft,
        XtNright, (XtEdgeType) XtChainRight,
        XtNlabel, (String) xg->collab_tform[j],
        XtNfromVert, (j>0) ? varlabel[j-1] : NULL,
        NULL);
    }
    XtManageChildren(varlabel, xg->ncols);

    close = XtVaCreateManagedWidget("Close",
      commandWidgetClass, tpane,
      XtNshowGrip, (Boolean) False,
      XtNskipAdjust, (Boolean) True,
      XtNlabel, (String) "Click here to dismiss",
      NULL);
    if (mono) set_mono(close);
    XtAddCallback(close, XtNcallback,
      (XtCallbackProc) close_cback, (XtPointer) xg);
  }

  XtPopup(tpopup, (XtGrabKind) XtGrabNone);
  set_wm_protocols(tpopup);
  XRaiseWindow(display, XtWindow(tpopup));

  XtPopup(tpopup, (XtGrabKind) XtGrabNone);
  set_wm_protocols(tpopup);
  XRaiseWindow(display, XtWindow(tpopup));
}

/*
void
permute_again(xgobidata *xg) {
  if (xg->is_touring && (xg->is_princ_comp || xg->is_pp))
    ;
  else {
    if (ntform_cols > 0 && tform_cols != NULL) {
      if ( transform(xg, tform_cols, ntform_cols, PERMUTE) )
        update_transformed_data(xg, tform_cols, ntform_cols, PERMUTE);
    }
  }
}

void
restore_variables(xgobidata *xg) {
  if (ntform_cols > 0 && tform_cols != NULL) {
    if ( transform(xg, tform_cols, ntform_cols, RESTORE) )
      update_transformed_data(xg, tform_cols, ntform_cols, RESTORE);
  }
}
*/
