/*
 * Copyright (c) 1999 Apple Computer, Inc. All rights reserved.
 *
 * @APPLE_LICENSE_HEADER_START@
 * 
 * Portions Copyright (c) 1999 Apple Computer, Inc.  All Rights
 * Reserved.  This file contains Original Code and/or Modifications of
 * Original Code as defined in and that are subject to the Apple Public
 * Source License Version 1.1 (the "License").  You may not use this file
 * except in compliance with the License.  Please obtain a copy of the
 * License at http://www.apple.com/publicsource and read it before using
 * this file.
 * 
 * The Original Code and all software distributed under the License are
 * distributed on an "AS IS" basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, AND APPLE HEREBY DISCLAIMS ALL SUCH WARRANTIES,
 * INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE OR NON- INFRINGEMENT.  Please see the
 * License for the specific language governing rights and limitations
 * under the License.
 * 
 * @APPLE_LICENSE_HEADER_END@
 */
/*
	File:		StringFormatter.h

	Contains:	Utility class for formatting text to a buffer.
				Construct object with a buffer, then call one
				of many Put methods to write into that buffer.
					
	$Log: StringFormatter.h,v $
	Revision 1.2  1999/02/19 23:13:20  ds
	Created
	

*/

#ifndef __STRINGFORMATTER_H__
#define __STRINGFORMATTER_H__

#include <string.h>
#include "StrPtrLen.h"
#include "MyAssert.h"


class StringFormatter {
public:
	
	//pass in a buffer and length for writing
	StringFormatter(char *buffer, UInt32 length) : 	fCurrentPut(buffer), 
													fStartPut(buffer),
													fEndPut(buffer + length)
													{}

	StringFormatter(StrPtrLen &buffer) : 			fCurrentPut(buffer.Ptr),
													fStartPut(buffer.Ptr),
													fEndPut(buffer.Ptr + buffer.Len)
													{}
	virtual ~StringFormatter() {}
	
	//"erases" all data in the buffer
	void		Reset() { fCurrentPut = fStartPut; }
	//"erases" the specified # of bytes of data
	void		Reset(UInt32 inNumBytesToErase);

	//Object does no bounds checking on the buffer. That is your responsibility!
	void	 	Put(const SInt32 num);
	void		Put(char* buffer, UInt32 bufferSize);
	void		Put(char* str) 		{ Put(str, strlen(str)); }
	void		Put(const StrPtrLen &str) { Put(str.Ptr, str.Len); }
	void		PutSpace() 			{ *(fCurrentPut++) = ' '; }
	void		PutEOL() 			{  Put(sEOL, sEOLLen); }
	void		PutChar(char c)		{ *fCurrentPut = c; fCurrentPut++; }
	void		PutTerminator()		{ PutChar('\0'); }
		
	inline UInt32		GetCurrentOffset();
	inline UInt32		GetSpaceLeft();
	inline UInt32		GetTotalBufferSize();
	char*				GetCurrentPtr()		{ return fCurrentPut; }
	char*				GetBufPtr()			{ return fStartPut; }
	
		
protected:

	//If you fill up the StringFormatter buffer, this function will get called. By
	//default, no action is taken. But derived objects can clear out the data and reset the buffer
	virtual void	BufferIsFull(char* /*inBuffer*/, UInt32 /*inBufferLen*/) { Assert(0); }

	char*		fCurrentPut;
	char*		fStartPut;
	char*		fEndPut;
	
	static char*	sEOL;
	static UInt32	sEOLLen;
		
};

inline UInt32 StringFormatter::GetCurrentOffset()
{
	Assert(fCurrentPut >= fStartPut);
	return (UInt32)(fCurrentPut - fStartPut);
}

inline UInt32 StringFormatter::GetSpaceLeft()
{
	Assert(fEndPut >= fCurrentPut);
	return (UInt32)(fEndPut - fCurrentPut);
}

inline UInt32 StringFormatter::GetTotalBufferSize()
{
	Assert(fEndPut >= fStartPut);
	return (UInt32)(fEndPut - fStartPut);
}
#endif // __STRINGFORMATTER_H__

