/*-
 * Copyright (c) 1998 Joe Greco and sol.net Network Services
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

/* Todo:	option for only filtering headers
 *		option for "early abort" like Cyclone does
 */

#include "defs.h"

Prototype void filter_failed(char *reason);
Prototype void close_filter_program(void);
Prototype void open_filter_program(void);
Prototype int sizeit(int len);
Prototype int diabfilter(char *loc);

int filter_fd_stdin = -1;
int filter_fd_stdout = -1;
int filter_fail_count = 0;
time_t filter_try_again = 0;
time_t filter_last_fail = 0;
pid_t filter_pid = 0;

void 
filter_failed(char *reason)
{
	int delay;
	time_t now = time(NULL);

	/* Yeah, yeah, it's hokey. */
	if (now - filter_last_fail < 300) {
		filter_fail_count++;
	} else {
		filter_fail_count--;
	}
	filter_last_fail = now;
	delay = filter_fail_count * filter_fail_count;
	delay = (delay > 900) ? 900 : delay;
	filter_try_again = time(NULL) + delay;
	syslog(LOG_ERR, "diab-filter: filter failed, %s, sleeping for %d seconds\n", reason, delay);
}

void 
close_filter_program(void)
{
	int status, rval, loop;

	if (! (filter_fd_stdin < 0)) {
		close(filter_fd_stdin);
		filter_fd_stdin = -1;
	}
	if (! (filter_fd_stdout < 0)) {
		close(filter_fd_stdout);
		filter_fd_stdout = -1;
	}
	if (filter_pid) {
		for (loop = 0; loop < 10; loop++) {
			if ((rval = waitpid(filter_pid, &status, WNOHANG)) < 0) {
				syslog(LOG_ERR, "waitpid for %d failed: %m", filter_pid);
				filter_pid = 0;
				return;
			}
			if (WIFEXITED(status)) {
				if (WEXITSTATUS(status)) {
					syslog(LOG_ERR, "filter returned exit %d", WEXITSTATUS(status));
				} else {
					syslog(LOG_NOTICE, "filter exited normally");
				}
				filter_pid = 0;
				return;
			}
			if (WIFSIGNALED(status)) {
				syslog(LOG_ERR, "filter exited on signal %d", WTERMSIG(status));
				filter_pid = 0;
				return;
			}
			sleep(1);
		}
		syslog(LOG_ERR, "filter failed to exit");
		filter_pid = 0;
		return;
	}
}

void
open_filter_program(void)
{
	int stdinfds[2];
	int stdoutfds[2];
	int nfd;
	pid_t newpid;

	if (filter_try_again) {
		if (time(NULL) < filter_try_again) {
			return;
		}
		filter_try_again = 0;
	}

	if (! (PATH_FILTER[0] == '/')) {
		/* Not a path name!  Guess that it is a TCP connection */
		if ((nfd = connect_tcp_socket(PATH_FILTER, 0, 0)) < 0) {
			syslog(LOG_ERR, "couldnt connect to remote filter: %m");
			filter_failed("couldnt connect to remote filter");
			return;
		}
		filter_fd_stdin = nfd;
		filter_fd_stdout = nfd;

		if (fcntl(filter_fd_stdin, F_SETFD, 1) < 0) {
			syslog(LOG_ERR, "fcntl filter stdin: %m");
		}
		if (fcntl(filter_fd_stdout, F_SETFD, 1) < 0) {
			syslog(LOG_ERR, "fcntl filter stdout: %m");
		}

		filter_pid = 0;

		/* "Woohoo!" */
		syslog(LOG_NOTICE, "filter connected to remote filter");
		return;
	}

	if (pipe(stdinfds) < 0) {
		filter_failed("cant create pipe");
		return;
	}

	if (pipe(stdoutfds) < 0) {
		filter_failed("cant create pipe");
		close(stdinfds[0]);
		close(stdinfds[1]);
		return;
	}

	/* We foolishly assume SIGPIPE has been handled elsewhere as SIG_IGN */
	/* Assumption is the mother ... XXX */

	if ((newpid = fork()) < 0) {
		filter_failed("cant create child process");
		close(stdinfds[0]);
		close(stdinfds[1]);
		close(stdoutfds[0]);
		close(stdoutfds[1]);
		return;
	}

	if (! newpid) {
		/* Child processing. */

		if (dup2(stdinfds[0], fileno(stdin)) < 0) {
			filter_failed("cant dup2 stdin");
			close(stdinfds[0]);
			close(stdinfds[1]);
			close(stdoutfds[0]);
			close(stdoutfds[1]);
			exit(1);
		}
		close(stdinfds[0]);
		close(stdinfds[1]);

		if (dup2(stdoutfds[1], fileno(stdout)) < 0) {
			filter_failed("cant dup2 stdout");
			close(fileno(stdin));
			close(stdoutfds[0]);
			close(stdoutfds[1]);
			exit(1);
		}
		close(stdoutfds[0]);
		close(stdoutfds[1]);

		execl(PATH_FILTER, PATH_FILTER, NULL);
		filter_failed("cant exec filter");
		close(fileno(stdin));
		close(fileno(stdout));
		exit(1);
	}
	/* Parent processing. */

	close(stdinfds[0]);
	close(stdoutfds[1]);

	filter_fd_stdin = stdinfds[1];
	filter_fd_stdout = stdoutfds[0];

	if (fcntl(filter_fd_stdin, F_SETFD, 1) < 0) {
		syslog(LOG_ERR, "fcntl filter stdin: %m");
	}
	if (fcntl(filter_fd_stdout, F_SETFD, 1) < 0) {
		syslog(LOG_ERR, "fcntl filter stdout: %m");
	}

	filter_pid = newpid;

	/* "Woohoo!" */
	syslog(LOG_NOTICE, "filter launched");
	return;
}

int 
sizeit(int len)
{
	int c = 0;

	while (len >>= 1) {
		c++;
	}
	len = 2;
	while (c--) {
		len <<= 1;
	}
	return(len);
}

int 
diabfilter(char *loc)
{
	int rval, count, eoln, nbytes, llen;
	static int abufsiz = 0, nbufsiz = 0;
	static char *abuf = NULL, *nbuf = NULL;
	char *aptr, *nptr;

	if (! loc || ! (aptr = strrchr(loc, ','))) {
		return(-1);
	}

	llen = atoi(aptr + 1);	/* get article length */

	if (llen >= abufsiz) {
		count = sizeit(llen);
		if (! (aptr = (char *)realloc(abuf, count))) {
			syslog(LOG_ERR, "realloc %d failed", count);
			return(-1);
		}
		abufsiz = count;
		abuf = aptr;

		count = abufsiz * 2 + 6;
		if (! (nptr = (char *)realloc(nbuf, count))) {
			syslog(LOG_ERR, "realloc %d failed", count);
			return(-1);
		}
		nbufsiz = count;
		nbuf = nptr;
	}

	if (filter_fd_stdin < 0) {
		open_filter_program();
	}
	if (filter_fd_stdin < 0) {
		return(-1);
	}

	if ((rval = diab_read(loc, abuf, abufsiz)) < 0) {
#ifdef DEBUG
		syslog(LOG_ERR, "diab_read failed: %s", loc);
#endif
		return(-1);
	}

	count = rval;
	aptr = abuf;
	nptr = nbuf;
	eoln = 0;
	while (count--) {
		if (*aptr == '\n') {
			*nptr++ = '\r';
			*nptr++ = *aptr++;
			eoln = 1;
		} else {
			if (eoln) {
				if (*aptr == '.') {
					*nptr++ = '.';
				}
				eoln = 0;
			}
			*nptr++ = *aptr++;
		}
	}
	if (! eoln) {
		syslog(LOG_ERR, "article did not end with a return: %s", loc);
		*nptr++ = '\r';
		*nptr++ = '\n';
	}
	*nptr++ = '.';
	*nptr++ = '\r';
	*nptr++ = '\n';

	nbytes = nptr - nbuf;

	/* Send the article to the filter ... */
	if ((rval = write(filter_fd_stdin, nbuf, nbytes)) != nbytes) {
		syslog(LOG_ERR, "filter write failure: wanted to write %d, wrote %d: %m", nbytes, rval);
		filter_failed("write");
		close_filter_program();
		return(-1);
	}

	/* Get the response of the filter ... */
	/* XXX this is Pure Evil(tm) because the response isn't going to 
	 * have to be atomic */
	if ((rval = read(filter_fd_stdout, abuf, abufsiz)) <= 0) {
		syslog(LOG_ERR, "filter read failure: got %d: %m", rval);
		filter_failed("read");
		close_filter_program();
		return(-1);
	}
	abuf[rval] = '\0';

	if (*abuf == '3') {
		return(0);
	}
	if (*abuf == '4') {
		return(1);
	}
	syslog(LOG_ERR, "filter read failure: got unknown response: %s", abuf);
	return(-1);
}

