/********************************************/
/* wrap and mirror UGs by Gabriel Maldonado */
/* and others by same author                */
/* Code adapted by JPff 1998 Sep 19         */
/********************************************/

#include "cs.h"  
#include "uggab.h"
#include <math.h>

void wrap(WRAP *p)
{
    float 	*adest= p->xdest;
    float  	*asig = p->xsig;
    float	xlow, xhigh, xsig; 
    int 	loopcount = ksmps;

    if ((xlow=*p->xlow) >= (xhigh=*p->xhigh)) {
      float 	xaverage;
      xaverage = (xlow + xhigh) * 0.5f;
      do
	*adest++ = xaverage;
      while (--loopcount);
    }
    else 
      do {
	if ((xsig=(float) *asig++) >= xlow )	
	  *adest++ = (float)(xlow + fmod(xsig - xlow, fabs(xlow-xhigh)));
	else	
	  *adest++ = (float)(xhigh- fmod(xhigh- xsig, fabs(xlow-xhigh)));
      } while (--loopcount);
}


void kwrap(WRAP *p)
{
    float xsig, xlow, xhigh;

    if ((xlow=*p->xlow) >= (xhigh=*p->xhigh))
      *p->xdest = (xlow + xhigh)*0.5f;  
    else {
      if ((xsig=*p->xsig) >= xlow )	
	*p->xdest = (float)(xlow + fmod(xsig - xlow, fabs(xlow-xhigh)));
      else							
	*p->xdest = (float)(xhigh- fmod(xhigh- xsig, fabs(xlow-xhigh)));
    }
}


/*---------------------------------------------------------------------*/


void kmirror(WRAP *p)
{
    float  xsig, xlow, xhigh;
    xsig = *p->xsig;
    xhigh= *p->xhigh;
    xlow = *p->xlow;

    if (xlow >= xhigh) *p->xdest = (xlow + xhigh)*0.5f;  
    else {
      while ((xsig > xhigh) || (xsig < xlow)) {
	if (xsig > xhigh)
	  xsig = xhigh + xhigh - xsig;
	else
	  xsig = xlow + xlow - xsig;
      }
      *p->xdest = xsig;
    }
}


void mirror(WRAP *p)
{
    float 	*adest, *asig;
    float	xlow, xhigh, xaverage, xsig; 
    int 	loopcount = ksmps;

    adest = p->xdest;
    asig  = p->xsig;
    xlow = *p->xlow;
    xhigh = *p->xhigh;

    if (xlow >= xhigh)	{
      xaverage = (xlow + xhigh)*0.5f;
      do
	*adest++ = xaverage;
      while (--loopcount);
    }

    do	{
      xsig = *asig++;
      while ((xsig > xhigh) || ( xsig < xlow )) {
	if (xsig > xhigh)
	  xsig = xhigh + xhigh - xsig;
	else
	  xsig = xlow + xlow - xsig;
      }
      *adest++ = xsig;
    } while (--loopcount);
}


void trig_set(TRIG *p)
{
    p->old_sig = 0.0f;
}

void trig(TRIG *p)
{
    switch ((int) (*p->kmode + 0.5f)) {
    case 0:	  /* down-up */
      if (p->old_sig <= *p->kthreshold && *p->ksig > *p->kthreshold) 
	*p->kout = 1.0f;
      else
	*p->kout = 0.0f;
      break;
    case 1:	 /* up-down */
      if (p->old_sig >= *p->kthreshold && *p->ksig < *p->kthreshold) 
	*p->kout = 1.0f;
      else
	*p->kout = 0.0f;
      break;
    case 2:	 /* both */
      if ((p->old_sig <= *p->kthreshold && *p->ksig > *p->kthreshold) ||
	  (p->old_sig >= *p->kthreshold && *p->ksig < *p->kthreshold ) )
	*p->kout = 1.0f;
      else
	*p->kout = 0.0f;
      break;  
    default: 
      perferror(" bad imode value");
    }
    p->old_sig = *p->ksig;
}

/*-------------------------------*/


void interpol(INTERPOL *p)
{
    float point_value = (*p->point - *p->imin ) * (1.0f/(*p->imax - *p->imin));
    *p->r = point_value * (*p->val2 - *p->val1) + *p->val1;
}

void nterpol_init(INTERPOL *p)
{
    p->point_factor = 1.0f/(*p->imax - *p->imin);
}

void knterpol(INTERPOL *p)
{
    float point_value = (*p->point - *p->imin ) * p->point_factor;
    *p->r = point_value * (*p->val2 - *p->val1) + *p->val1;
}

void anterpol(INTERPOL *p)
{
    float point_value = (*p->point - *p->imin ) * p->point_factor;
    float *out = p->r, *val1 = p->val1, *val2 = p->val2;
    int loopcount = ksmps;
    do	{
      *out++ = point_value * (*val2++ - *val1) + *val1++;
    } while (--loopcount);
}

/* Oscilators */

void posc_set( POSC *p)
{
    FUNC *ftp;

    if ((ftp = ftnp2find(p->ift)) == NULL) return;
    p->ftp    = ftp;
    p->tablen = ftp->flen;
    p->phs    = *p->iphs * p->tablen;
}

void posc(POSC *p)
{
    float	*out = p->out, *ft = p->ftp->ftable;
    float	*curr_samp, fract;
    double	phs = p->phs;
    double      si = *p->freq * p->tablen / esr;
    long	n = ksmps;
    float       amp = *p->amp;

    do {
      curr_samp = ft + (long)phs;
      fract     = (float)(phs - (double)((long)phs));
      *out++    = amp * (*curr_samp +(*(curr_samp+1)-*curr_samp)*fract);
      phs      += si;
      while (phs >= p->tablen)
        phs -= p->tablen;
    } while (--n);
    p->phs = phs;
}


void kposc(POSC *p)
{
    double	phs = p->phs;
    double      si = *p->freq * p->tablen / ekr;
    float	*curr_samp = p->ftp->ftable + (long)phs;
    float	fract = (float)(phs - (double)((long)phs));

    *p->out = *p->amp * (*curr_samp +(*(curr_samp+1)-*curr_samp)*fract);
    phs    += si;
    while (phs >= p->tablen)
      phs -= p->tablen;
    p->phs = phs;
}

void posc3(POSC *p)
{
    float	*out = p->out, *ftab = p->ftp->ftable;
    float	fract;
    double	phs  = p->phs;
    double      si   = *p->freq * p->tablen / esr;
    long	n    = ksmps;
    float       amp = *p->amp;
    int         x0;
    float       y0, y1, ym1, y2;

    do {
      x0    = (long)phs;
      fract = (float)(phs - (double)x0);
      x0--;
      if (x0<0) {
        ym1 = ftab[p->tablen-1]; x0 = 0;
      }
      else ym1 = ftab[x0++];
      y0    = ftab[x0++];
      y1    = ftab[x0++];
      if (x0>p->tablen) y2 = ftab[1]; else y2 = ftab[x0];
      {
        float frsq = fract*fract;
        float frcu = frsq*ym1;
        float t1   = y2 + 3.0f*y0;
        *out++     = amp * (y0 + 0.5f*frcu +
                            fract*(y1 - frcu/6.0f - t1/6.0f - ym1/3.0f) +
                            frsq*fract*(t1/6.0f - 0.5f*y1) +
                            frsq*(0.5f* y1 - y0));
      }
      phs += si;
      while (phs >= p->tablen)
        phs -= p->tablen;
    } while (--n);
    p->phs = phs;
}

void kposc3(POSC *p)
{
    double	phs   = p->phs;
    double      si    = *p->freq * p->tablen / ekr;
    float	*ftab = p->ftp->ftable;
    int         x0    = (long)phs;
    float	fract = (float)(phs - (double)x0);
    float       y0, y1, ym1, y2;

    x0--;
    if (x0<0) {
      ym1 = ftab[p->tablen-1]; x0 = 0;
    }
    else ym1 = ftab[x0++];
    y0 = ftab[x0++];
    y1 = ftab[x0++];
    if (x0>p->tablen) y2 = ftab[1]; else y2 = ftab[x0];
    {
      float frsq = fract*fract;
      float frcu = frsq*ym1;
      float t1   = y2 + 3.0f*y0;
      *p->out    = *p->amp * (y0 + 0.5f*frcu +
                              fract*(y1 - frcu/6.0f - t1/6.0f - ym1/3.0f) +
                              frsq*fract*(t1/6.0f - 0.5f*y1) +
                              frsq*(0.5f* y1 - y0));
    }
    phs += si;
    while (phs >= p->tablen)
      phs -= p->tablen;
    p->phs = phs;
}

void lposc_set(LPOSC *p)
{
    FUNC *ftp;
    
    if ((ftp = ftnp2find(p->ift)) == NULL) return;
    if (!(p->fsr=ftp->gen01args.sample_rate)){
      printf("losc: no sample rate stored in function assuming=sr\n");
      p->fsr=esr;
    }
    p->ftp    = ftp;
    p->tablen = ftp->flen;
    p->phs    = *p->iphs * p->tablen;
}


void lposc(LPOSC *p)
{
    float	*out = p->out, *ft = p->ftp->ftable;
    float	*curr_samp, fract;
    double	phs= p->phs, si= *p->freq * (p->fsr/esr);
    long	n = ksmps;
    double	loop, end, looplength = p->looplength;
    float       amp = *p->amp;

    if ((loop = *p->kloop) < 0) loop=0;
    if ((end = *p->kend) > p->tablen || end <=0 ) end = p->tablen;
    looplength = end - loop;

    do {
      curr_samp = ft + (long)phs;
      fract = (float)(phs - (double)((long)phs));
      *out++ = amp * (*curr_samp +(*(curr_samp+1)-*curr_samp)*fract);
      phs += si;
      if (phs >= end) phs -= looplength;
    } while (--n);
    p->phs = phs;
}

void lposc3(LPOSC *p)
{
    float	*out = p->out, *ftab = p->ftp->ftable;
    float	fract;
    double	phs = p->phs, si= *p->freq * (p->fsr/esr);
    long	n = ksmps;
    double	loop, end, looplength = p->looplength;
    float       amp = *p->amp;
    int         x0;
    float       y0, y1, ym1, y2;

    if ((loop = *p->kloop) < 0) loop=0;
    if ((end = *p->kend) > p->tablen || end <=0 ) end = p->tablen;
    looplength = end - loop;

    do {
      x0    = (long)phs;
      fract = (float)(phs - (double)x0);
      x0--;
      if (x0<0) {
        ym1 = ftab[p->tablen-1]; x0 = 0;
      }
      else ym1 = ftab[x0++];
      y0    = ftab[x0++];
      y1    = ftab[x0++];
      if (x0>p->tablen) y2 = ftab[1]; else y2 = ftab[x0];
      {
        float frsq = fract*fract;
        float frcu = frsq*ym1;
        float t1   = y2 + 3.0f*y0;
        *out++     = amp * (y0 + 0.5f*frcu +
                            fract*(y1 - frcu/6.0f - t1/6.0f - ym1/3.0f) +
                            frsq*fract*(t1/6.0f - 0.5f*y1) +
                            frsq*(0.5f* y1 - y0));
      }
      phs += si;
      while (phs >= end) phs -= looplength;
    } while (--n);
    p->phs = phs;
}


