
/* waveguide.c -- primitive functions and operations for waveguides */

/*
 * Code conversion from C++ to C (October 1994)
 * Author: Michael A. Casey MIT Media Labs
 * Language: C
 * Copyright (c) 1994 MIT Media Lab, All Rights Reserved
 */

#include <stdlib.h>
#include <stdio.h>

#include "cs.h"
#include "wavegde.h"

#define EPSILON (0.25f)      /* threshold for small tuning values */
/* prototypes */

void error(const char*, const char*);


/***** circularBuffer class member function definitions *****/

/* ::circularBuffer -- constructor for circular buffer class
 * This routine assumes that the DATA pointer has already been
 * allocated by the calling routine.
 */
void circularBufferCircularBuffer(circularBuffer* cb, len_t N)
{
    if (!cb->data)
      error("Buffer memory not allocated!","<circularBuffer::circularBuffer>");

  /* Initialize pointers and variables */
    cb->size = N;
    cb->_inited=1;
    cb->pointer=cb->data;
    cb->endPoint=cb->data+cb->size-1;
    cb->insertionPoint=cb->data;
    cb->extractionPoint=cb->data;
}



/* ::write -- insert new value in the buffer, update insertion pointer */
void circularBufferWrite(circularBuffer* cb,sampT val)
{
  /* update the extraction point */
  cb->extractionPoint=cb->insertionPoint;
  
  /* place data at the insertion point */
  *cb->insertionPoint--=val;
  
  /* update the insertion point */
  if (cb->insertionPoint<cb->data)
    cb->insertionPoint=cb->endPoint;  
}



/* ::read -- extract the value at the extraction point */
 sampT circularBufferRead(circularBuffer* cb)
{
  sampT val;
  /* Read the value at the extraction point */
  val = *cb->extractionPoint++;

  /* Update the extraction point */
  if (cb->extractionPoint>cb->endPoint)
    cb->extractionPoint=cb->data;

  return val;
}


/* ***** class guideRail -- waveguide rail derived class ***** */
void guideRailGuideRail(guideRail* gr, len_t d){
  circularBufferCircularBuffer(gr,d); /* Guide rail is a circular buffer */
}


/* ::update -- waveguide rail insert and update routine */
 void guideRailUpdate(guideRail *gr,sampT samp){

  *gr->pointer++ = samp;
  if (gr->pointer > gr->endPoint)
    gr->pointer = gr->data;
}

/* ::access -- waveguide rail access routine */
 sampT guideRailAccess(guideRail* gr, len_t pos){
  sampPtr s = gr->pointer - pos;
  while(s < gr->data)
    s+=gr->size;
  while(s > gr->endPoint)
    s -= gr->size;
  return *s;
}

/* ***** class filter -- digital filter routines ****** */
/* ::filter -- constructor, assumes preallocated data and coeffs*/
void filterFilter(filter* filt, len_t n)
{
  if (!filt->coeffs)
    error("Coeffs not allocated!","<filter::filter>");

  /* Initialize circular buffer */
  circularBufferCircularBuffer(&filt->buffer,n);
}

/* ::set -- set the coefficients */
void filterSet(filter* filt, paramPtr c)
{
  int i;
  
  if (!filt->buffer._inited)
    error("Filter not inited, can't set","<filter::set>");
  
  for (i=0;i<filt->buffer.size;i++)
    filt->coeffs[i]=c[i];
}

/* ::reset -- resets the lattice data to zero initial conditions */
void filterReset(filter* filt)
{
  int i;
  for (i=0;i<filt->buffer.size;i++)
    filt->buffer.data[i]=0.0f;
}

/* ::get -- check range of index, return corresponding coefficient */
paramT filterGet(filter* filt, len_t i)
{
  if ((i<0) || (i>(filt->buffer.size-1)))
    error("Index out of range","<filter::get()>");

  return filt->coeffs[i];
}

/* ::IIR -- recursive digital filter routine */
/*          insert new samp, update lattice buffers, return new output */
 sampT filterIIR(filter* filt, sampT s)
{
  paramPtr c=filt->coeffs;
  int i;
  
  /* y[n] = x[n] + c1*y[n-1] + ... + cM*y[n-M+1] */
  for (i=0;i<filt->buffer.size;i++)
    s += *c++ * circularBufferRead(&filt->buffer);
  circularBufferWrite(&filt->buffer,s);
  return s;
}

/* ::FIR -- direct convolution filter routine */
 sampT filterFIR(filter* filt, sampT s)
{
  paramPtr c=filt->coeffs;
  int i;
  
  /* y[n] = c1*x[n] + c2*x[n-1] + ... + cM*x[n-M+1] */
  circularBufferWrite(&filt->buffer,s);
  s=0.0f;
  for (i=0;i<filt->buffer.size;i++)
    s += *c++ * circularBufferRead(&filt->buffer);
  return s;
}

/* ::allpass -- accurate 1st-order allpass filter routine */
/*   c = allpass filter coefficient, input sample */
 sampT filterAllpass(waveguide* wg,sampT s){
  /* p[n] = x[n] + gp[n-1], y[n] = p[n-1] - gp[n] */
  paramT q = s + wg->c*wg->p;
  s= -wg->c*q+wg->p;
  wg->p=q;
  return s;
}

/* ***** Waveguide base-class member definitions ***** */

/* ::waveguide -- constructor
 * sets delay lengths and filter responses for frequency
 * total delay length = (SR/f0)
 * also sets tuning filter for fractional delay for exact tuning
 */
void waveguideWaveguide(waveguide* wg,
			paramT freq,
			sampT* upperData,
			sampT* lowerData)
{
    paramT size,df;

    wg->excited=0;
    wg->p=0.0f; /* tuning filter state variable */
    wg->f0=freq;
    wg->w0 = (TWOPI_F*freq)/(paramT)esr;
  
#ifdef WG_VERBOSE  
    printf("f0=%f, w0=%f\n",wg->f0,wg->w0);
#endif

  /* Calculate the size of the delay lines and set them */
  /* Set pointers to appropriate positions in instrument memory */
    size = (paramT)esr / freq - 1.0f;

  /* construct the fractional part of the delay */
    df = (size - (len_t)size); /* fractional delay amount */
    if (df<EPSILON) {
      df=1.0f+EPSILON;
      size=size-1.0f;
    }
    wg->upperRail.data = upperData;
    wg->lowerRail.data = lowerData;
    guideRailGuideRail(&wg->upperRail,(len_t)(size/2.0));
    guideRailGuideRail(&wg->lowerRail,(len_t)(size/2.0));

#ifdef WG_VERBOSE
    printf("size=%d+1, df=%f\n",(len_t)size,df);
#endif
    waveguideSetTuning(wg,df);
}

/* Set the allpass tuning filter coefficient */
void waveguideSetTuning(waveguide* wg, paramT df)
{
    paramT k=(1/(paramT)esr)*wg->w0;

  /*c = (1.0-df)/(1.0+df);*/ /* Solve for coefficient from df */
    wg->c = -sinf((k-k*df)/2.0)/sinf((k+k*df)/2.0);

#ifdef WG_VERBOSE  
    printf("tuning :c=%f\n",wg->c);
    fflush(stdout);
#endif
}


/* error -- report errors */
void error(const char* a, const char* b)
{
    printf("Error:%s,%b\n",a,b);
    exit(1);
}
