/*	Copyright (C) 1995 Free Software Foundation, Inc.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this software; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * As a special exception, the Free Software Foundation gives permission
 * for additional uses of the text contained in its release of GUILE.
 *
 * The exception is that, if you link the GUILE library with other files
 * to produce an executable, this does not by itself cause the
 * resulting executable to be covered by the GNU General Public License.
 * Your use of that executable is in no way restricted on account of
 * linking the GUILE library code into it.
 *
 * This exception does not however invalidate any other reasons why
 * the executable file might be covered by the GNU General Public License.
 *
 * This exception applies only to the code released by the
 * Free Software Foundation under the name GUILE.  If you copy
 * code from other Free Software Foundation releases into a copy of
 * GUILE, as the General Public License permits, the exception does
 * not apply to the code that you add in this way.  To avoid misleading
 * anyone as to the status of such modified files, you must delete
 * this exception notice from them.
 *
 * If you write modifications of your own for GUILE, it is your choice
 * whether to permit this exception to apply to your modifications.
 * If you do not wish that, delete this exception notice.  
 */


#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include "gscm.h"
#include "_scm.h"


SCM scm_exitval;		/* INUM with return value */

unsigned char scm_upcase[CHAR_CODE_LIMIT];
unsigned char scm_downcase[CHAR_CODE_LIMIT];
unsigned char scm_lowers[] = "abcdefghijklmnopqrstuvwxyz";
unsigned char scm_uppers[] = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";

extern int scm_verbose;
#ifdef __STDC__
void 
scm_tables_prehistory (void)
#else
void 
scm_tables_prehistory ()
#endif
{
  int i;
  for (i = 0; i < CHAR_CODE_LIMIT; i++)
    scm_upcase[i] = scm_downcase[i] = i;
  for (i = 0; i < sizeof scm_lowers / sizeof (char); i++)
    {
      scm_upcase[scm_lowers[i]] = scm_uppers[i];
      scm_downcase[scm_uppers[i]] = scm_lowers[i];
    }
  scm_verbose = 1;		/* Here so that monitor info won't be */
  /* printed while in scm_init_storage. (BOOM) */
}

#ifdef EBCDIC
char *scm_charnames[] =
{
  "nul","soh","stx","etx", "pf", "ht", "lc","del",
   0   , 0   ,"smm", "vt", "ff", "cr", "so", "si",
  "dle","dc1","dc2","dc3","res", "nl", "bs", "il",
  "can", "em", "cc", 0   ,"ifs","igs","irs","ius",
   "ds","sos", "fs", 0   ,"byp", "lf","eob","pre",
   0   , 0   , "sm", 0   , 0   ,"enq","ack","bel",
   0   , 0   ,"syn", 0   , "pn", "rs", "uc","eot",
   0   , 0   , 0   , 0   ,"dc4","nak", 0   ,"sub",
   "space", scm_s_newline, "tab", "backspace", "return", "page", "null"};
char scm_charnums[] =
"\000\001\002\003\004\005\006\007\
\010\011\012\013\014\015\016\017\
\020\021\022\023\024\025\026\027\
\030\031\032\033\034\035\036\037\
\040\041\042\043\044\045\046\047\
\050\051\052\053\054\055\056\057\
\060\061\062\063\064\065\066\067\
\070\071\072\073\074\075\076\077\
 \n\t\b\r\f\0";
#endif /* def EBCDIC */
#ifdef ASCII
char *scm_charnames[] =
{
  "nul","soh","stx","etx","eot","enq","ack","bel",
   "bs", "ht", "nl", "vt", "np", "cr", "so", "si",
  "dle","dc1","dc2","dc3","dc4","nak","syn","etb",
  "can", "em","sub","esc", "fs", "gs", "rs", "us",
  "space", "newline", "tab", "backspace", "return", "page", "null", "del"};
char scm_charnums[] =
"\000\001\002\003\004\005\006\007\
\010\011\012\013\014\015\016\017\
\020\021\022\023\024\025\026\027\
\030\031\032\033\034\035\036\037\
 \n\t\b\r\f\0\177";
#endif /* def ASCII */


/* Local functions needing declarations.
 */

static SCM lreadr P ((SCM tok_buf, SCM port, int case_i));
static SCM lreadparen P ((SCM tok_buf, SCM port, char *name, int case_i));
static SCM lreadrecparen P((SCM tok_buf, SCM port, char *name, int case_i));
static sizet read_token P ((int ic, SCM tok_buf, SCM port, int case_i,
			    int weird));


/* {Names of immediate symbols}
 * 
 * This table must agree with the declarations in scm.h: {Immediate Symbols}.
 */

char *scm_isymnames[] =
{
  /* This table must agree with the declarations */
  "#@and",
  "#@begin",
  "#@case",
  "#@cond",
  "#@do",
  "#@if",
  "#@lambda",
  "#@let",
  "#@let*",
  "#@letrec",
  "#@or",
  "#@quote",
  "#@set!",
  "#@define",
#if 0
  "#@literal-variable-ref",
  "#@literal-variable-set!",
#endif
  "#@apply",
  "#@call-with-current-continuation",

 /* user visible ISYMS */
 /* other keywords */
 /* Flags */

  "#f",
  "#t",
  "#<undefined>",
  "#<eof>",
  "()",
  "#<unspecified>"
};

/* {Printing of Scheme Objects}
 */

/* Print an integer.
 */
#ifdef __STDC__
void 
scm_intprint (long n, int radix, SCM port)
#else
void 
scm_intprint (n, radix, port)
     long n;
     int radix;
     SCM port;
#endif
{
  char num_buf[INTBUFLEN];
  scm_lfwrite (num_buf, (sizet) sizeof (char), scm_iint2str (n, radix, num_buf), port);
}

/* Print an object of unrecognized type.
 */
#ifdef __STDC__
void 
scm_ipruk (char *hdr, SCM ptr, SCM port)
#else
void 
scm_ipruk (hdr, ptr, port)
     char *hdr;
     SCM ptr;
     SCM port;
#endif
{
  scm_puts ("#<unknown-", port);
  scm_puts (hdr, port);
  if (CELLP (ptr))
    {
      scm_puts (" (0x", port);
      scm_intprint (CAR (ptr), 16, port);
      scm_puts (" . 0x", port);
      scm_intprint (CDR (ptr), 16, port);
      scm_puts (") @", port);
    }
  scm_puts (" 0x", port);
  scm_intprint (ptr, 16, port);
  scm_putc ('>', port);
}

/* Print a list.
 */
#ifdef __STDC__
void 
scm_iprlist (char *hdr, SCM exp, char tlr, SCM port, int writing)
#else
void 
scm_iprlist (hdr, exp, tlr, port, writing)
     char *hdr;
     SCM exp;
     char tlr;
     SCM port;
     int writing;
#endif
{
  scm_puts (hdr, port);
  /* CHECK_INTS; */
  scm_iprin1 (CAR (exp), port, writing);
  exp = CDR (exp);
  for (; NIMP (exp); exp = CDR (exp))
    {
      if (NECONSP (exp))
	break;
      scm_putc (' ', port);
      /* CHECK_INTS; */
      scm_iprin1 (CAR (exp), port, writing);
    }
  if (NNULLP (exp))
    {
      scm_puts (" . ", port);
      scm_iprin1 (exp, port, writing);
    }
  scm_putc (tlr, port);
}


/* Print generally.  Handles both write and display according to WRITING.
 */
#ifdef __STDC__
void 
scm_iprin1 (SCM exp, SCM port, int writing)
#else
void 
scm_iprin1 (exp, port, writing)
     SCM exp;
     SCM port;
     int writing;
#endif
{
  register long i;
taloop:
  switch (7 & (int) exp)
    {
    case 2:
    case 6:
      scm_intprint (INUM (exp), 10, port);
      break;
    case 4:
      if (ICHRP (exp))
	{
	  i = ICHR (exp);
	  if (writing)
	    scm_puts ("#\\", port);
	  if (!writing)
	    scm_putc ((int) i, port);
	  else if ((i <= ' ') && scm_charnames[i])
	    scm_puts (scm_charnames[i], port);
#ifndef EBCDIC
	  else if (i == '\177')
	    scm_puts (scm_charnames[(sizeof scm_charnames / sizeof (char *)) - 1], port);
#endif /* ndef EBCDIC */
	  else if (i > '\177')
	    scm_intprint (i, 8, port);
	  else
	    scm_putc ((int) i, port);
	}
      else if (   IFLAGP (exp)
	       && (ISYMNUM (exp) < (sizeof scm_isymnames / sizeof (char *))))
	  scm_puts (ISYMCHARS (exp), port);
      else if (ILOCP (exp))
	{
	  scm_puts ("#@", port);
	  scm_intprint ((long) IFRAME (exp), 10, port);
	  scm_putc (ICDRP (exp) ? '-' : '+', port);
	  scm_intprint ((long) IDIST (exp), 10, port);
	}
      else
	goto idef;
      break;
    case 1:
      /* gloc */
      scm_puts ("#@", port);
      exp = CAR (exp - 1);
      goto taloop;
    default:
    idef:
      scm_ipruk ("immediate", exp, port);
      break;
    case 0:
      switch (TYP7 (exp))
	{
	case tcs_cons_gloc:
	  if (CDR (CAR (exp) - 1L) == 0)
	    {
	      SCM name;
	      scm_lfwrite ("#<latte ",
			   (sizet) sizeof (char),
			   (sizet) 8,
			   port);
	      name = ((SCM *)(STRUCT_TYPE_DATA( exp)))[struct_i_name];
	      scm_lfwrite (CHARS (name),
			   (sizet) sizeof (char),
			   (sizet) LENGTH (name),
			   port);
	      scm_putc (' ', port);
	      scm_intprint(exp, 16, port);
	      scm_putc ('>', port);
	      break;
	    }
	case tcs_cons_imcar:
	case tcs_cons_nimcar:
	  scm_iprlist ("(", exp, ')', port, writing);
	  break;
	case tcs_closures:
#ifdef DEBUG_EXTENSIONS
	  if (PRINT_PROCNAMES)
	    {
	      SCM name;
	      name = gscm_procedure_property (exp, scm_i_name);
	      scm_lputs ("#<procedure", port);
	      if (NFALSEP (name))
		{
		  scm_lputc (' ', port);
		  scm_lputs (CHARS (name), port);
		}
	      scm_lputc ('>', port);
	    }
	  else
#endif
	    {
	      exp = CODE (exp);
	      scm_iprlist ("#<CLOSURE ", exp, '>', port, writing);
	    }
	  break;
	case tc7_string:
	  if (writing)
	    {
	      scm_putc ('\"', port);
	      for (i = 0; i < LENGTH (exp); ++i)
		switch (CHARS (exp)[i])
		  {
		  case '\"':
		  case '\\':
		    scm_putc ('\\', port);
		  default:
		    scm_putc (CHARS (exp)[i], port);
		  }
	      scm_putc ('\"', port);
	      break;
	    }
	  else
	    scm_lfwrite (CHARS (exp),
			 (sizet) sizeof (char),
			 (sizet) LENGTH (exp),
			 port);
	  break;
	case tcs_symbols:
	  {
	    int pos;
	    int end;
	    int len;
	    char * str;
	    int weird;
	    int maybe_weird;
	    int mw_pos;

	    len = LENGTH (exp);
	    str = CHARS (exp);
	    scm_remember (&exp);
	    pos = 0;
	    weird = 0;
	    maybe_weird = 0;

	    for (end = pos; end < len; ++end)
	      switch (str[end])
		{
#ifdef BRACKETS_AS_PARENS
		case '[':
		case ']':
#endif
		case '(':
		case ')':
		case '\"':
		case ';':
		case WHITE_SPACES:
		case LINE_INCREMENTORS:
		weird_handler:
		  if (maybe_weird)
		    {
		      end = mw_pos;
		      maybe_weird = 0;
		    }
		  if (!weird)
		    {
		      scm_lfwrite ("#{", (sizet) sizeof(char), 2, port);
		      weird = 1;
		    }
		  if (pos < end)
		    {
		      scm_lfwrite (str + pos, sizeof (char), end - pos, port);
		    }
		  {
		    char buf[2];
		    buf[0] = '\\';
		    buf[1] = str[end];
		    scm_lfwrite (buf, (sizet) sizeof (char), 2, port);
		  }
		  pos = end + 1;
		  break;
		case '\\':
		  if (weird)
		    goto weird_handler;
		  if (!maybe_weird)
		    {
		      maybe_weird = 1;
		      mw_pos = pos;
		    }
		  break;
		case '}':
		case '#':
		  if (weird)
		    goto weird_handler;
		  break;
		default:
		  break;
		}
	    if (pos < end)
	      scm_lfwrite (str + pos, (sizet) sizeof (char), end - pos, port);
	    if (weird)
	      scm_lfwrite ("}#", (sizet) sizeof (char), 2, port);
	    break;
	  }
	case tc7_wvect:
	  if (SCM_IS_WHVEC (exp))
	    scm_puts ("#wh(", port);
	  else
	    scm_puts ("#w(", port);
	  goto common_vector_printer;

	case tc7_vector:
	  scm_puts ("#(", port);
	common_vector_printer:
	  for (i = 0; i + 1 < LENGTH (exp); ++i)
	    {
	      /* CHECK_INTS; */
	      scm_iprin1 (VELTS (exp)[i], port, writing);
	      scm_putc (' ', port);
	    }
	  if (i < LENGTH (exp))
	    {
	      /* CHECK_INTS; */
	      scm_iprin1 (VELTS (exp)[i], port, writing);
	    }
	  scm_putc (')', port);
	  break;
	case tc7_lvector:
	  {
	    SCM result;
	    SCM hook;
	    hook = scm_get_lvector_hook (exp, LV_PRINT_FN);
	    if (hook == BOOL_F)
	      {
		scm_puts ("#<locked-vector ", port);
		scm_intprint(CDR(exp), 16, port);
		scm_puts (">", port);
	      }
	    else
	      {
		result
		  = scm_apply (hook,
			       scm_listify (exp, port, (writing ? BOOL_T : BOOL_F),
					    SCM_UNDEFINED),
			       EOL);
		if (result == BOOL_F)
		  goto punk;
	      }
	    break;
	  }
	  break;
	case tc7_bvect:
	case tc7_byvect:
	case tc7_svect:
	case tc7_ivect:
	case tc7_uvect:
	case tc7_fvect:
	case tc7_dvect:
	case tc7_cvect:
	case tc7_llvect:
	  scm_raprin1 (exp, port, writing);
	  break;
	case tcs_subrs:
	  scm_puts ("#<primitive-procedure ", port);
	  scm_puts (CHARS (SNAME (exp)), port);
	  scm_putc ('>', port);
	  break;
#ifdef CCLO
	case tc7_cclo:
	  scm_puts ("#<compiled-closure ", port);
	  scm_iprin1 (CCLO_SUBR (exp), port, writing);
	  scm_putc ('>', port);
	  break;
#endif
	case tc7_contin:
	  scm_puts ("#<continuation ", port);
	  scm_intprint (LENGTH (exp), 10, port);
	  scm_puts (" @ ", port);
	  scm_intprint ((long) CHARS (exp), 16, port);
	  scm_putc ('>', port);
	  break;
	case tc7_port:
	  i = PTOBNUM (exp);
	  if (i < scm_numptob && scm_ptobs[i].print && (scm_ptobs[i].print) (exp, port, writing))
	    break;
	  goto punk;
	case tc7_smob:
	  i = SMOBNUM (exp);
	  if (i < scm_numsmob && scm_smobs[i].print
	      && (scm_smobs[i].print) (exp, port, writing))
	    break;
	  goto punk;
	default:
	punk:scm_ipruk ("type", exp, port);
	}
    }
}

/* Various I/O primitives, leading up to READ
 */

#ifdef __IBMC__
# define MSDOS
#endif
#ifdef MSDOS
# ifndef GO32
#  include <io.h>
#  include <conio.h>
#ifdef __STDC__
static int 
input_waiting (FILE *f)
#else
static int 
input_waiting (f)
     FILE *f;
#endif
{
  if (feof (f))
    return 1;
  if (fileno (f) == fileno (stdin) && (isatty (fileno (stdin))))
    return kbhit ();
  return -1;
}
# endif
#else
# ifdef _DCC
#  include <ioctl.h>
# else
#  ifndef AMIGA
#   ifndef vms
#    ifdef MWC
#     include <sys/io.h>
#    else
#     ifndef THINK_C
#      ifndef ARM_ULIB
#       include <sys/ioctl.h>
#      endif
#     endif
#    endif
#   endif
#  endif
# endif


#ifdef __STDC__
static int
input_waiting(FILE *f)
#else
static int
input_waiting(f)
     FILE *f;
#endif
{
# ifdef FIONREAD
  long remir;
  if (feof(f)) return 1;
  ioctl(fileno(f), FIONREAD, &remir);
  return remir;
# else
  return -1;
# endif
}
#endif

/* perhaps should undefine MSDOS from __IBMC__ here */
#ifndef GO32
PROC (s_char_ready_p, "char-ready?", 1, 0, 0, scm_char_ready_p);
#ifdef __STDC__
SCM 
scm_char_ready_p (SCM port)
#else
SCM 
scm_char_ready_p (port)
     SCM port;
#endif
{
  if (UNBNDP (port))
    port = cur_inp;
  else
    ASSERT (NIMP (port) && OPINPORTP (port), port, ARG1, s_char_ready_p);
  if (CRDYP (port) || !FPORTP (port))
    return BOOL_T;
  return input_waiting (STREAM (port)) ? BOOL_T : BOOL_F;
}
#endif

PROC (s_eof_object_p, "eof-object?", 1, 0, 0, scm_eof_object_p);
#ifdef __STDC__
SCM 
scm_eof_object_p (SCM x)
#else
SCM 
scm_eof_object_p (x)
     SCM x;
#endif
{
  return (EOF_VAL == x) ? BOOL_T : BOOL_F;
}

/* internal SCM call */
#ifdef __STDC__
void 
scm_fflush (SCM port)
#else
void 
scm_fflush (port)
     SCM port;
#endif
{
  sizet i = PTOBNUM (port);
  (scm_ptobs[i].fflush) (STREAM (port));
}

PROC (s_force_output, "force-output", 0, 1, 0, scm_force_output);
#ifdef __STDC__
SCM 
scm_force_output (SCM port)
#else
SCM 
scm_force_output (port)
     SCM port;
#endif
{
  if (UNBNDP (port))
 port = cur_outp;
  else
    ASSERT (NIMP (port) && OPOUTPORTP (port), port, ARG1, s_force_output);
  {
    sizet i = PTOBNUM (port);
    SYSCALL ((scm_ptobs[i].fflush) (STREAM (port)));
    return UNSPECIFIED;
  }
}

PROC (s_write, "write", 1, 1, 0, scm_write);
#ifdef __STDC__
SCM 
scm_write (SCM obj, SCM port)
#else
SCM 
scm_write (obj, port)
     SCM obj;
     SCM port;
#endif
{
  if (UNBNDP (port))
    port = cur_outp;
  else
    ASSERT (NIMP (port) && OPOUTPORTP (port), port, ARG2, s_write);
  scm_iprin1 (obj, port, 1);
#ifdef HAVE_PIPE
# ifdef EPIPE
  if (EPIPE == errno)
    scm_close_port (port);
# endif
#endif
  return UNSPECIFIED;
}


PROC (s_display, "display", 1, 1, 0, scm_display);
#ifdef __STDC__
SCM 
scm_display (SCM obj, SCM port)
#else
SCM 
scm_display (obj, port)
     SCM obj;
     SCM port;
#endif
{
  if (UNBNDP (port))
    port = cur_outp;
  else
    ASSERT (NIMP (port) && OPOUTPORTP (port), port, ARG2, s_display);
  scm_iprin1 (obj, port, 0);
#ifdef HAVE_PIPE
# ifdef EPIPE
  if (EPIPE == errno)
    scm_close_port (port);
# endif
#endif
  return UNSPECIFIED;
}

PROC (s_newline, "newline", 0, 1, 0, scm_newline);
#ifdef __STDC__
SCM
scm_newline(SCM port)
#else
SCM 
scm_newline (port)
     SCM port;
#endif
{
  if (UNBNDP (port))
 port = cur_outp;
  else
    ASSERT (NIMP (port) && OPOUTPORTP (port), port, ARG1, s_newline);
  scm_putc ('\n', port);
#ifdef HAVE_PIPE
# ifdef EPIPE
  if (EPIPE == errno)
    scm_close_port (port);
  else
# endif
#endif
  if (port == cur_outp)
    scm_fflush (port);
  return UNSPECIFIED;
}

PROC (s_write_char, "write-char", 1, 1, 0, scm_write_char);
#ifdef __STDC__
SCM 
scm_write_char (SCM chr, SCM port)
#else
SCM 
scm_write_char (chr, port)
     SCM chr;
     SCM port;
#endif
{
  if (UNBNDP (port))
 port = cur_outp;
  else
    ASSERT (NIMP (port) && OPOUTPORTP (port), port, ARG2, s_write_char);
  ASSERT (ICHRP (chr), chr, ARG1, s_write_char);
  scm_putc ((int) ICHR (chr), port);
#ifdef HAVE_PIPE
# ifdef EPIPE
  if (EPIPE == errno)
    scm_close_port (port);
# endif
#endif
  return UNSPECIFIED;
}


#ifdef __STDC__
void 
scm_putc (int c, SCM port)
#else
void 
scm_putc (c, port)
     int c;
     SCM port;
#endif
{
  sizet i = PTOBNUM (port);
  SYSCALL ((scm_ptobs[i].fputc) (c, STREAM (port)));
#ifdef TRANSCRIPT_SUPPORT
  if (scm_trans && (port == def_outp || port == cur_errp))
    SYSCALL (fputc (c, scm_trans));
#endif
}

#ifdef __STDC__
void 
scm_puts (char *s, SCM port)
#else
void 
scm_puts (s, port)
     char *s;
     SCM port;
#endif
{
  sizet i = PTOBNUM (port);
  SYSCALL ((scm_ptobs[i].fputs) (s, STREAM (port)));
#ifdef TRANSCRIPT_SUPPORT
  if (scm_trans && (port == def_outp || port == cur_errp))
    SYSCALL (fputs (s, scm_trans));
#endif
}

#ifdef __STDC__
int 
scm_lfwrite (char *ptr, sizet size, sizet nitems, SCM port)
#else
int 
scm_lfwrite (ptr, size, nitems, port)
     char *ptr;
     sizet size;
     sizet nitems;
     SCM port;
#endif
{
  int ret;
  sizet i = PTOBNUM (port);
  SYSCALL (ret = (scm_ptobs[i].fwrite(ptr, size, nitems, STREAM (port))));
#ifdef TRANSCRIPT_SUPPORT
  if (scm_trans && (port == def_outp || port == cur_errp))
    SYSCALL (fwrite (ptr, size, nitems, scm_trans));
#endif
  return ret;
}

#ifdef __STDC__
int 
scm_lgetc (SCM port)
#else
int 
scm_lgetc (port)
     SCM port;
#endif
{
  FILE *f;
  int c;
  sizet i;
  /* One char may be stored in the high bits of (car port) orre@nada.kth.se. */
  if (CRDYP (port))
    {
      c = CGETUN (port);
      CLRDY (port);		/* Clear ungetted char */
      return c;
    }
  f = STREAM (port);
  i = PTOBNUM (port);
#ifdef linux
  c = (scm_ptobs[i].fgetc) (f);
#else
  SYSCALL (c = (scm_ptobs[i].fgetc) (f));
#endif
#ifdef TRANSCRIPT_SUPPORT
  if (scm_trans && (f == stdin))
    SYSCALL (fputc (c, scm_trans));
#endif
  return c;
}

#ifdef __STDC__
void 
scm_lungetc (int c, SCM port)
#else
void 
scm_lungetc (c, port)
     int c;
     SCM port;
#endif
{
/*	ASSERT(!CRDYP(port), port, ARG2, "too many scm_lungetc");*/
  CUNGET (c, port);
}



PROC (s_read_char, "read-char", 0, 1, 0, scm_read_char);
#ifdef __STDC__
SCM 
scm_read_char (SCM port)
#else
SCM 
scm_read_char (port)
     SCM port;
#endif
{
  int c;
  if (UNBNDP (port))
 port = cur_inp;
  else
    ASSERT (NIMP (port) && OPINPORTP (port), port, ARG1, s_read_char);
  c = scm_lgetc (port);
  if (EOF == c)
    return EOF_VAL;
  return MAKICHR (c);
}


PROC (s_peek_char, "peek-char", 0, 1, 0, scm_peek_char);
#ifdef __STDC__
SCM 
scm_peek_char (SCM port)
#else
SCM 
scm_peek_char (port)
     SCM port;
#endif
{
  int c;
  if (UNBNDP (port))
    port = cur_inp;
  else
    ASSERT (NIMP (port) && OPINPORTP (port), port, ARG1, s_peek_char);
  c = scm_lgetc (port);
  if (EOF == c)
    return EOF_VAL;
  scm_lungetc (c, port);
  return MAKICHR (c);
}


#ifdef __STDC__
char *
scm_grow_tok_buf (SCM tok_buf)
#else
char *
scm_grow_tok_buf (tok_buf)
     SCM tok_buf;
#endif
{
  sizet len = LENGTH (tok_buf);
  len += len / 2;
  scm_resizuve (tok_buf, (SCM) MAKINUM (len));
  return CHARS (tok_buf);
}

static scm_cell scm_tmp_loadpath = {(SCM) BOOL_F, (SCM) EOL};
SCM *scm_loc_loadpath = (SCM *) & scm_tmp_loadpath;
SCM loadport = SCM_UNDEFINED;
SCM scm_filename;
long scm_linum;
long scm_colnum;
long scm_linum = 1;
#ifdef READER_EXTENSIONS
SCM scm_filename = BOOL_F;
long scm_colnum = -1;
#endif


static char s_eofin[] = "end of file in ";
#ifdef __STDC__
static int 
flush_ws (SCM port, char *eoferr)
#else
static int 
flush_ws (port, eoferr)
     SCM port;
     char *eoferr;
#endif
{
  register int c;
  while (1)
    switch (c = GETC (port))
      {
      case EOF:
      goteof:
	if (eoferr)
	  scm_wta (SCM_UNDEFINED, s_eofin, eoferr);
	return c;
      case ';':
      lp:
	switch (c = scm_lgetc (port))
	  {
	  case EOF:
	    goto goteof;
	  default:
	    goto lp;
	  case LINE_INCREMENTORS:
	    break;
	  }
      case LINE_INCREMENTORS:
	if (port==loadport) NEXTL;
      case SINGLE_SPACES:
	break;
      case TAB:
#ifdef READER_EXTENSIONS
	scm_colnum += 8 - (scm_colnum + 1) % 8;
#endif
	break;
      default:
	return c;
      }
}

#ifdef GUILE
static int default_case_i = 0;
#else 
static int default_case_i = 1;
#endif


PROC (s_read, "read", 0, 2, 0, scm_read);
#ifdef __STDC__
SCM 
scm_read (SCM port, SCM casep)
#else
SCM 
scm_read (port, casep)
     SCM port;
     SCM casep;
#endif
{
  int c;
  SCM tok_buf;
  int case_i;

  if (UNBNDP (port))
    port = cur_inp;
  else
    ASSERT (NIMP (port) && OPINPORTP (port), port, ARG1, s_read);

  case_i = (UNBNDP (casep)
	    ? default_case_i
	    : (casep == BOOL_F));

  do
    {
      c = flush_ws (port, (char *) NULL);
      if (EOF == c)
	return EOF_VAL;
      scm_lungetc (c, port);
      tok_buf = scm_makstr (30L, 0);
    }
  while (EOF_VAL == (tok_buf = lreadr (tok_buf, port, case_i)));
  return tok_buf;
}

#ifdef __STDC__
static int
casei_streq (char * s1, char * s2)
#else
static int
casei_streq (s1, s2)
     char * s1;
     char * s2;
#endif
{
  while (*s1 && *s2)
    if (scm_downcase[(int)*s1] != scm_downcase[(int)*s2])
      return 0;
    else
      {
	++s1;
	++s2;
      }
  return !(*s1 || *s2);
}


static char s_unknown_sharp[] = "unknown # object";
#ifdef __STDC__
static SCM 
lreadr (SCM tok_buf, SCM port, int case_i)
#else
static SCM 
lreadr (tok_buf, port, case_i)
     SCM tok_buf;
     SCM port;
     int case_i;
#endif
{
  int c;
  sizet j;
  SCM p;
tryagain:
  c = flush_ws (port, s_read);
  switch (c)
    {
/*	case EOF: return EOF_VAL;*/
#ifdef BRACKETS_AS_PARENS
    case '[':
#endif
    case '(':
#ifdef READER_EXTENSIONS
      return (RECORD_POSITIONS
	      ? lreadrecparen (tok_buf, port, "list", case_i)
	      : lreadparen (tok_buf, port, "list", case_i));
#else
      return lreadparen (tok_buf, port, "list", case_i);
#endif
#ifdef BRACKETS_AS_PARENS
    case ']':
#endif
    case ')':
      scm_warn ("unexpected \")\"", "");
      goto tryagain;
    case '\'':
      return scm_cons2 (scm_i_quote, lreadr (tok_buf, port, case_i), EOL);
    case '`':
      return scm_cons2 (scm_i_quasiquote, lreadr (tok_buf, port, case_i), EOL);
    case ',':
      c = GETC (port);
      if ('@' == c)
	p = scm_i_uq_splicing;
      else
	{
	  UNGETC (c, port);
	  p = scm_i_unquote;
	}
      return scm_cons2 (p, lreadr (tok_buf, port, case_i), EOL);
    case '#':
      c = GETC (port);
      switch (c)
	{
#ifdef BRACKETS_AS_PARENS
	case '[':
#endif
	case '(':
	  p = lreadparen (tok_buf, port, "vector", case_i);
	  return NULLP (p) ? nullvect : scm_vector (p);
	case 't':
	case 'T':
	  return BOOL_T;
	case 'f':
	case 'F':
	  return BOOL_F;
	case 'b':
	case 'B':
	case 'o':
	case 'O':
	case 'd':
	case 'D':
	case 'x':
	case 'X':
	case 'i':
	case 'I':
	case 'e':
	case 'E':
	  UNGETC (c, port);
	  c = '#';
	  goto num;
	case '*':
	  j = read_token (c, tok_buf, port, case_i, 0);
	  p = scm_istr2bve (CHARS (tok_buf) + 1, (long) (j - 1));
	  if (NFALSEP (p))
	    return p;
	  else
	    goto unkshrp;
	case '{':
	  j = read_token (c, tok_buf, port, case_i, 1);
	  p = scm_intern (CHARS (tok_buf), j);
	  return CAR (p);
	case '\\':
	  c = GETC (port);
	  j = read_token (c, tok_buf, port, case_i, 0);
	  if (j == 1)
	    return MAKICHR (c);
	  if (c >= '0' && c < '8')
	    {
	      p = scm_istr2int (CHARS (tok_buf), (long) j, 8);
	      if (NFALSEP (p))
		return MAKICHR (INUM (p));
	    }
	  for (c = 0; c < sizeof scm_charnames / sizeof (char *); c++)
	    if (scm_charnames[c]
		&& (casei_streq (scm_charnames[c], CHARS (tok_buf))))
	      return MAKICHR (scm_charnums[c]);
	  scm_wta (SCM_UNDEFINED, "unknown # object: #\\", CHARS (tok_buf));
	default:
	callshrp:
	  p = CDR (scm_intern ("read:sharp", (sizeof "read:sharp") - 1));
	  if (NIMP (p))
	    {
	      p = scm_apply (p, MAKICHR (c), scm_acons (port, EOL, EOL));
	      if ((UNSPECIFIED == p) || (UNSPECIFIED == p))
		goto unkshrp;
	      return p;
	    }
	unkshrp:scm_wta ((SCM) MAKICHR (c), s_unknown_sharp, "");
	}
    case '"':
      j = 0;
      while ('"' != (c = GETC (port)))
	{
	  ASSERT (EOF != c, SCM_UNDEFINED, s_eofin, "string");
	  if (j + 1 >= LENGTH (tok_buf))
	    scm_grow_tok_buf (tok_buf);
	  if (c == '\\')
	    switch (c = GETC (port))
	      {
	      case '\n':
		continue;
	      case '0':
		c = '\0';
		break;
	      case 'f':
		c = '\f';
		break;
	      case 'n':
		c = '\n';
		break;
	      case 'r':
		c = '\r';
		break;
	      case 't':
		c = '\t';
		break;
	      case 'a':
		c = '\007';
		break;
	      case 'v':
		c = '\v';
		break;
	      }
	  CHARS (tok_buf)[j] = c;
	  ++j;
	}
      if (j == 0)
	return nullstr;
      CHARS (tok_buf)[j] = 0;
      return scm_makfromstr (CHARS (tok_buf), j, 0);
    case DIGITS:
    case '.':
    case '-':
    case '+':
    num:
      j = read_token (c, tok_buf, port, case_i, 0);
      p = scm_istring2number (CHARS (tok_buf), (long) j, 10L);
      if (NFALSEP (p))
	return p;
      if (c == '#')
	{
	  if ((j == 2) && (GETC (port) == '('))
	    {
	      UNGETC ('(', port);
	      c = CHARS (tok_buf)[1];
	      goto callshrp;
	    }
	  scm_wta (SCM_UNDEFINED, s_unknown_sharp, CHARS (tok_buf));
	}
      goto tok;
    case ':':
      j = read_token ('-', tok_buf, port, case_i, 0);
      p = scm_intern (CHARS (tok_buf), j);
      return scm_make_keyword (CAR (p));
    default:
      j = read_token (c, tok_buf, port, case_i, 0);
    tok:
      p = scm_intern (CHARS (tok_buf), j);
      return CAR (p);
    }
}

#ifdef _UNICOS
_Pragma ("noopt");		/* # pragma _CRI noopt */
#endif
#ifdef __STDC__
static sizet 
read_token (int ic, SCM tok_buf, SCM port, int case_i, int weird)
#else
static sizet 
read_token (ic, tok_buf, port, case_i, weird)
     int ic;
     SCM tok_buf;
     SCM port;
     int case_i;
     int weird;
#endif
{
  register sizet j;
  register int c;
  register char *p;

  c = ic;
  p = CHARS (tok_buf);

  if (!weird)
    {
      p[0] = (case_i ? scm_downcase[c] : c);
      j = 1;
    }
  else
    j = 0;

  while (1)
    {
      if (j + 1 >= LENGTH (tok_buf))
	p = scm_grow_tok_buf (tok_buf);
      c = GETC (port);
      switch (c)
	{
#ifdef BRACKETS_AS_PARENS
	case '[':
	case ']':
#endif
	case '(':
	case ')':
	case '"':
	case ';':
	case WHITE_SPACES:
	case LINE_INCREMENTORS:
	  if (weird)
	    goto default_case;

	  UNGETC (c, port);
	case EOF:
	eof_case:
	  p[j] = 0;
	  return j;
	case '\\':
	  if (!weird)
	    goto default_case;
	  else
	    {
	      c = GETC (port);
	      if (c == EOF)
		goto eof_case;
	      else
		goto default_case;
	    }
	case '}':
	  if (!weird)
	    goto default_case;

	  c = GETC (port);
	  if (c == '#')
	    {
	      p[j] = 0;
	      return j;
	    }
	  else
	    {
	      UNGETC (c, port);
	      c = '}';
	      goto default_case;
	    }

	default:
	default_case:
	  p[j++] = (case_i ? scm_downcase[c] : c);
	}
    }
}
#ifdef _UNICOS
_Pragma ("opt");		/* # pragma _CRI opt */
#endif

#ifdef __STDC__
static SCM 
lreadparen (SCM tok_buf, SCM port, char *name, int case_i)
#else
static SCM 
lreadparen (tok_buf, port, name, case_i)
     SCM tok_buf;
     SCM port;
     char *name;
     int case_i;
#endif
{
  SCM tmp, tl, ans;
  int c = flush_ws (port, name);
  if (')' == c
#ifdef BRACKETS_AS_PARENS
      || ']' == c
#endif
    )
    return EOL;
  UNGETC (c, port);
  if (scm_i_dot == (tmp = lreadr (tok_buf, port, case_i)))
    {
      ans = lreadr (tok_buf, port, case_i);
    closeit:
      if (')' != (c = flush_ws (port, name))
#ifdef BRACKETS_AS_PARENS
	  && ']' != c
#endif
	)
	scm_wta (SCM_UNDEFINED, "missing close paren", "");
      return ans;
    }
  ans = tl = scm_cons (tmp, EOL);
  while (')' != (c = flush_ws (port, name))
#ifdef BRACKETS_AS_PARENS
	 && ']' != c
#endif
    )
    {
      UNGETC (c, port);
      if (scm_i_dot == (tmp = lreadr (tok_buf, port, case_i)))
	{
	  CDR (tl) = lreadr (tok_buf, port, case_i);
	  goto closeit;
	}
      tl = (CDR (tl) = scm_cons (tmp, EOL));
    }
  return ans;
}





#ifdef READER_EXTENSIONS
#ifdef __STDC__
static SCM 
lreadrecparen (SCM tok_buf, SCM port, char *name, int case_i)
#else
static SCM 
lreadrecparen (tok_buf, port, name, case_i)
     SCM tok_buf;
     SCM port;
     char *name;
     int case_i;
#endif
{
  SCM tmp, tl, ans, ans2, tl2;
  int lin = scm_linum, col = scm_colnum;
  int c = flush_ws (port, name);
  /* These variables contain the index of the entry of the last read list
     in the scm_list_data table and it's copy respectively. */
  static SCM last_copied_expr;
  if (')' == c
#ifdef BRACKETS_AS_PARENS
      || ']' == c
#endif
    )
    return EOL;
  UNGETC (c, port);
  if (scm_i_dot == (tmp = lreadr (tok_buf, port, case_i)))
    {
      ans = lreadr (tok_buf, port, case_i);
      if (')' != (c = flush_ws (port, name))
#ifdef BRACKETS_AS_PARENS
	  && ']' != c
#endif
	)
	scm_wta (SCM_UNDEFINED, "missing close paren", "");
      return ans;
    }
  /* We now know that we will cons. */
  ans = tl = scm_cons (tmp, EOL);
  if (COPY_SOURCE)
    if (NIMP (tmp) && CONSP (tmp))
      ans2 = tl2 = scm_cons (last_copied_expr, EOL);
    else
      ans2 = tl2 = scm_cons (tmp, EOL);
  while (')' != (c = flush_ws (port, name))
#ifdef BRACKETS_AS_PARENS
	 && ']' != c
#endif
    )
    {
      UNGETC (c, port);
      if (scm_i_dot == (tmp = lreadr (tok_buf, port, case_i)))
	{
	  CDR (tl) = tmp = lreadr (tok_buf, port, case_i);
	  if (COPY_SOURCE)
	    if (NIMP (tmp) && CONSP (tmp))
	      CDR (tl2) = last_copied_expr;
	    else
	      CDR (tl2) = tmp;
	  if (')' != (c = flush_ws (port, name))
#ifdef BRACKETS_AS_PARENS
	      && ']' != c
#endif
	      )
	    scm_wta (SCM_UNDEFINED, "missing close paren", "");
	  goto exit;
	}
      tl = (CDR (tl) = scm_cons (tmp, EOL));
      if (COPY_SOURCE)
	if (NIMP (tmp) && CONSP (tmp))
	  tl2 = (CDR (tl2) = scm_cons (last_copied_expr, EOL));
	else
	  tl2 = (CDR (tl2) = scm_cons (tmp, EOL));
    }
exit:
  WHASHSET (scm_object_whash,
	    scm_weak_hash_create_handle (scm_object_whash, ans),
	    _scm_make_srcprops (lin, col, scm_filename,
				COPY_SOURCE
				? last_copied_expr = ans2
				: SCM_UNDEFINED));
  return ans;
}
#endif /* READER_EXTENSIONS */

/* {Loading from source files.}
 */



static char s_load[]="load";

PROC (s_sys_try_load, "%try-load", 1, 0, 0, scm_sys_try_load);
#ifdef __STDC__
SCM 
scm_sys_try_load (SCM filename)
#else
SCM 
scm_sys_try_load (filename)
     SCM filename;
#endif
{
  ASSERT (NIMP (filename) && ROSTRINGP (filename), filename, ARG1, s_load);
  {
    SCM oloadpath = *scm_loc_loadpath;
    SCM oloadport = loadport;
    long olinum = scm_linum;
#ifdef READER_EXTENSIONS
    long ofilename = scm_filename;
    long ocolnum = scm_colnum;
#endif
    SCM form, port;
    port = scm_open_file (filename,
			  scm_makfromstr ("r", (sizet) sizeof (char), 0));
    if (FALSEP (port))
      return port;
    *scm_loc_loadpath = filename;
    loadport = port;
    scm_linum = 1;
#ifdef READER_EXTENSIONS
    scm_colnum = -1;
    scm_filename = filename;
#endif
    while (1)
      {
	form = scm_read (port, UNSPECIFIED);
	if (EOF_VAL == form)
	  break;
	scm_eval_x (form);
      }
    scm_close_port (port);
    scm_linum = olinum;
#ifdef READER_EXTENSIONS
    scm_colnum = ocolnum;
    scm_filename = ofilename;
#endif
    loadport = oloadport;
    *scm_loc_loadpath = oloadpath;
  }
  return BOOL_T;
}


/* {Way Out}
 */

PROC (s_quit, "quit", 0, 1, 0, scm_quit);
#ifdef __STDC__
SCM 
scm_quit (SCM n)
#else
SCM 
scm_quit (n)
     SCM n;
#endif
{
  if (UNBNDP (n) || BOOL_T == n)
    n = MAKINUM (EXIT_SUCCESS);
  else if (INUMP (n))
    scm_exitval = n;
  else
    scm_exitval = MAKINUM (EXIT_FAILURE);
  if (scm_errjmp_bad)
    exit (INUM (scm_exitval));
  scm_dowinds (EOL, scm_ilength (dynwinds));
#ifdef DEBUG_EXTENSIONS
  last_debug_info_frame = DFRAME (rootcont);
#endif
  longjmp (JMPBUF (rootcont), -1);
}


PROC (s_abort, "abort", 0, 0, 0, scm_abort);
#ifdef __STDC__
SCM 
scm_abort (void)
#else
SCM 
scm_abort ()
#endif
{
  if (scm_errjmp_bad)
    exit (INUM (scm_exitval));
  scm_dowinds (EOL, scm_ilength (dynwinds));
#ifdef DEBUG_EXTENSIONS
  last_debug_info_frame = DFRAME (rootcont);
#endif
  longjmp (JMPBUF (rootcont), -2);
}


/* {call-with-dynamic-root}
 *
 * Suspending the current thread to evaluate a thunk on the
 * same C stack but in a new dynamic context.
 *
 * Calls to call-with-dynamic-root return exactly once (unless
 * the process is somehow exitted).
 */

SCM scm_exitval;		/* INUM with return value */
static int n_dynamic_roots = 0;

#ifdef __STDC__
static SCM 
_cwdr (SCM thunk, SCM a1, SCM args, SCM error_thunk, STACKITEM * stack_start)
#else
static SCM 
_cwdr (thunk, a1, args, error_thunk, stack_start)
     SCM thunk;
     SCM a1;
     SCM args;
     SCM error_thunk;
     STACKITEM * stack_start;
#endif
{
#ifdef _UNICOS
  int i;
#else
  long i;
#endif

  SCM inferior_exitval;		/* INUM with return value */
  SCM old_dynamic_winds;
#ifdef DEBUG_EXTENSIONS
  debug_info *old_dframe;
#endif
  SCM old_rootcont;
  SCM answer;

  /* Exit the caller's dynamic state. 
   */
  old_dynamic_winds = dynwinds;
  scm_dowinds (EOL, scm_ilength (dynwinds));

  /* Create a fresh root continuation.
   * Temporarily substitute it for the native root continuation.
   */
  old_rootcont = rootcont;
  {
    SCM new_root;
    NEWCELL (new_root);
    DEFER_INTS;
    SETJMPBUF (new_root,
	       scm_must_malloc ((long) sizeof (regs),
				"inferior root continuation"));
    CAR (new_root) = tc7_contin;
    DYNENV (new_root) = EOL;
    BASE (new_root) = stack_start;
    SEQ (new_root) = n_dynamic_roots++;
#ifdef DEBUG_EXTENSIONS
    old_dframe = last_debug_info_frame;
    DFRAME (new_root) = last_debug_info_frame = BIGPTR;
#endif
    ALLOW_INTS;
    rootcont = new_root;
  }


  /* Establish a jump-buffer for returns to this dynamic root.
   */
  i = setjmp (JMPBUF (rootcont));

  switch ((int) i)
    {
    default:
      {
	/* An error condition.
	 */
	char *name = scm_errmsgs[i - WNA].s_response;
	if (name)
	  {
	    SCM proc = CDR (scm_intern (name, (sizet) strlen (name)));
	    if (NIMP (proc))
	      scm_apply (proc, EOL, EOL);
	  }
	if ((i = scm_errmsgs[i - WNA].parent_err))
	  goto error_exit;
	def_err_response ();
	scm_errjmp_bad = 0;
	scm_alrm_deferred = 0;
	scm_sig_deferred = 0;
	scm_ints_disabled = 0;
	goto error_exit;
      }

    case 0:
      inferior_exitval = MAKINUM (EXIT_SUCCESS);
      scm_errjmp_bad = 0;
      errno = 0;
      scm_alrm_deferred = 0;
      scm_sig_deferred = 0;
      scm_ints_disabled = 0;
      scm_errjmp_bad = 0;
      scm_alrm_deferred = 0;
      scm_sig_deferred = 0;
      scm_ints_disabled = 0;
      *scm_loc_loadpath = BOOL_F;
      answer = scm_apply (thunk, a1, args);
      goto return_answer;

    case -2:
      /* (...fallthrough)
       *
       * Inferior executed (abort).
       *
       */
      scm_errjmp_bad = 0;
      scm_alrm_deferred = 0;
      scm_sig_deferred = 0;
      scm_ints_disabled = 0;
      /*
       * (...fallthrough)
       */
    case -1:
      /* 
       * Inferior executed (quit).
       *
       * (...fallthrough)
       */
    case -3:
      /* (...fallthrough)
       *
       * Inferior executed (restart).
       *
       * (...fallthrough)
       */
    error_exit:
      /*
       *
       * Inferior caused an error.
       *
       */
      *scm_loc_loadpath = BOOL_F;
      answer = scm_apply (error_thunk, scm_cons (MAKINUM (i), EOL), EOL);
      rootcont = old_rootcont;
#ifdef DEBUG_EXTENSIONS
      last_debug_info_frame = old_dframe;
#endif
      scm_dowinds (old_dynamic_winds,   - scm_ilength (old_dynamic_winds));
      return answer;
    }

 return_answer:
  rootcont = old_rootcont;
#ifdef DEBUG_EXTENSIONS
      last_debug_info_frame = old_dframe;
#endif
  scm_dowinds (old_dynamic_winds,   - scm_ilength (old_dynamic_winds));
  return answer;
}


PROC (s_with_dynamic_root, "with-dynamic-root", 2, 0, 0, scm_with_dynamic_root);
#ifdef __STDC__
SCM
scm_with_dynamic_root (SCM thunk, SCM error_thunk)
#else
SCM
scm_with_dynamic_root (thunk, error_thunk)
     SCM thunk;
     SCM error_thunk;
#endif
{
  STACKITEM stack_place;

  return _cwdr (thunk, EOL, EOL, error_thunk, &stack_place);
}

#ifdef __STDC__
SCM
scm_app_wdr (SCM proc, SCM a1, SCM args, SCM error)
#else
SCM
scm_app_wdr (proc, a1, args, error)
     SCM proc;
     SCM a1;
     SCM args;
     SCM error;
#endif
{
  STACKITEM stack_place;
  return _cwdr (proc, a1, args, error, &stack_place);
}



/* {Read-eval-print Loops}
 */

int scm_verbose = 1;
long scm_cells_allocated = 0;
long scm_lcells_allocated = 0;
long scm_mallocated = 0;
long scm_lmallocated = 0;
long scm_rt = 0;
long scm_gc_rt;
long scm_gc_time_taken;
long scm_gc_cells_collected;
long scm_gc_malloc_collected;
long scm_gc_ports_collected;


#ifdef __STDC__
int
scm_ldfile(char *path)
#else
int
scm_ldfile(path)
     char *path;
#endif
{
  SCM name = scm_makfromstr(path, (sizet)(strlen(path))*sizeof(char), 0);
  *scm_loc_errobj = name;
  return BOOL_F==scm_sys_try_load(name);
}


#ifdef __STDC__
int
scm_ldprog(char *path)
#else
int
scm_ldprog(path)
     char *path;
#endif
{
  SCM name = scm_makfromstr(path, (sizet)(strlen(path))*sizeof(char), 0);
  *scm_loc_errobj = name;
  return
    BOOL_F==scm_evstr("(try-load (in-vicinity (program-vicinity) errobj))");
}


PROC (s_eval_string, "eval-string", 1, 0, 0, scm_eval_string);
#ifdef __STDC__
SCM
scm_eval_string(SCM str)
#else
SCM
scm_eval_string(str)
     SCM str;
#endif
{
  str = scm_mkstrport(INUM0, str, OPN | RDNG, s_eval_string);
  str = scm_read(str, default_case_i);
  return EVAL(str, (SCM)EOL);
}


#ifdef __STDC__
SCM
scm_evstr(char *str)
#else
SCM
scm_evstr(str)
     char *str;
#endif
{
  SCM lsym;
  NEWCELL(lsym);
  SETLENGTH(lsym, strlen(str)+0L, tc7_ssymbol);
  SETCHARS(lsym, str);
  return scm_eval_string(lsym);
}


PROC (s_load_string, "load-string", 1, 0, 0, scm_load_string);
#ifdef __STDC__
SCM
scm_load_string(SCM str)
#else
SCM
scm_load_string(str)
     SCM str;
#endif
{
  ASSERT(NIMP(str) && (STRINGP(str) || SYMBOLP(str)), str, ARG1,
	 s_load_string);
  str = scm_mkstrport(INUM0, str, OPN | RDNG, s_load_string);
#ifdef DEBUG_EXTENSIONS
  if (DEBUGGINGP)
    while(1) {
      SCM form;
      form = scm_read(str, default_case_i);
      if (EOF_VAL==form) break;
      DSIDEVAL(form, EOL);
    }
  else
#endif
    while(1) {
      SCM form = scm_read(str, default_case_i);
      if (EOF_VAL==form) break;
      SIDEVAL(form, EOL);
    }
  return BOOL_T;
}


#ifdef __STDC__
void
scm_ldstr(char *str)
#else
void
scm_ldstr(str)
     char *str;
#endif
{
  SCM lsym;
  NEWCELL(lsym);
  SETLENGTH(lsym, strlen(str)+0L, tc7_ssymbol);
  SETCHARS(lsym, str);
  scm_load_string(lsym);
}



PROC (s_line_number, "line-number", 0, 0, 0, scm_line_number);
#ifdef __STDC__
SCM 
scm_line_number (void)
#else
SCM 
scm_line_number ()
#endif
{
  return MAKINUM (scm_linum);
}

PROC (s_column_num, "column-num", 0, 0, 0, scm_column_num);
#ifdef __STDC__
SCM
scm_column_num (void)
#else
SCM
scm_column_num ()
#endif
{
#ifdef READER_EXTENSIONS
  return MAKINUM (scm_colnum);
#else
  return BOOL_F;
#endif
}


PROC (s_program_arguments, "program-arguments", 0, 0, 0, scm_program_arguments);
#ifdef __STDC__
SCM 
scm_program_arguments (void)
#else
SCM 
scm_program_arguments ()
#endif
{
  return progargs;
}

extern char s_heap[];
extern CELLPTR *scm_hplims;
#ifdef __STDC__
void 
scm_growth_mon (char *obj, long size, char *units)
#else
void 
scm_growth_mon (obj, size, units)
     char *obj;
     long size;
     char *units;
#endif
{
  if (scm_verbose > 2)
    {
      scm_puts ("; grew ", cur_errp);
      scm_puts (obj, cur_errp);
      scm_puts (" to ", cur_errp);
      scm_intprint (size, 10, cur_errp);
      scm_putc (' ', cur_errp);
      scm_puts (units, cur_errp);
      if ((scm_verbose > 4) && !strcmp (obj, "heap"))
	scm_heap_report ();
      scm_puts ("\n", cur_errp);
    }
}

#ifdef __STDC__
void 
scm_gc_start (char *what)
#else
void 
scm_gc_start (what)
     char *what;
#endif
{
  if (scm_verbose > 3 && FPORTP (cur_errp))
    {
      ALLOW_INTS;
      scm_puts (";GC(", cur_errp);
      scm_puts (what, cur_errp);
      scm_puts (")", cur_errp);
      scm_fflush (cur_errp);
      DEFER_INTS;
    }
  scm_gc_rt = INUM (scm_my_time ());
  scm_gc_cells_collected = 0;
  scm_gc_malloc_collected = 0;
  scm_gc_ports_collected = 0;
}

#ifdef __STDC__
void 
scm_gc_end (void)
#else
void 
scm_gc_end ()
#endif
{
  scm_gc_rt = INUM (scm_my_time ()) - scm_gc_rt;
  scm_gc_time_taken = scm_gc_time_taken + scm_gc_rt;
  if (scm_verbose > 3)
    {
      ALLOW_INTS;
      if (!FPORTP (cur_errp))
	scm_puts (";GC ", cur_errp);
      scm_intprint (scm_time_in_msec (scm_gc_rt), 10, cur_errp);
      scm_puts (" cpu mSec, ", cur_errp);
      scm_intprint (scm_gc_cells_collected, 10, cur_errp);
      scm_puts (" cells, ", cur_errp);
      scm_intprint (scm_gc_malloc_collected, 10, cur_errp);
      scm_puts (" malloc, ", cur_errp);
      scm_intprint (scm_gc_ports_collected, 10, cur_errp);
      scm_puts (" ports collected\n", cur_errp);
      scm_fflush (cur_errp);
      DEFER_INTS;
    }
}

PROC (s_repl_report_reset, "repl-report-reset", 0, 0, 0, scm_repl_report_reset);
#ifdef __STDC__
SCM
scm_repl_report_reset (void)
#else
SCM
scm_repl_report_reset ()
#endif
{
  scm_lcells_allocated = scm_cells_allocated;
  scm_lmallocated = scm_mallocated;
  return UNSPECIFIED;
}

PROC (s_repl_report_start_timing, "repl-report-start-timing", 0, 0, 0, scm_repl_report_start_timing);
#ifdef __STDC__
SCM 
scm_repl_report_start_timing (void)
#else
SCM 
scm_repl_report_start_timing ()
#endif
{
  scm_rt = INUM (scm_my_time ());
  scm_gc_time_taken = 0;
  return BOOL_F;
}


PROC (s_repl_report, "repl-report", 0, 0, 0, scm_repl_report);
#ifdef __STDC__
SCM 
scm_repl_report (void)
#else
SCM
scm_repl_report ()
#endif
{
  scm_fflush (cur_outp);
  scm_puts (";Evaluation took ", cur_errp);
  scm_intprint (scm_time_in_msec (INUM (scm_my_time ()) - scm_rt), 10, cur_errp);
  scm_puts (" mSec (", cur_errp);
  scm_intprint (scm_time_in_msec (scm_gc_time_taken), 10, cur_errp);
  scm_puts (" in scm_gc) ", cur_errp);
  scm_intprint (scm_cells_allocated - scm_lcells_allocated, 10, cur_errp);
  scm_puts (" cells work, ", cur_errp);
  scm_intprint (scm_mallocated - scm_lmallocated, 10, cur_errp);
  scm_puts (" bytes other\n", cur_errp);
  scm_fflush (cur_errp);
  return UNSPECIFIED;
}

PROC (s_room, "room", 0, 0, 1, scm_room);
#ifdef __STDC__
SCM 
scm_room (SCM args)
#else
SCM 
scm_room (args)
     SCM args;
#endif
{
  scm_intprint (scm_cells_allocated, 10, cur_errp);
  scm_puts (" out of ", cur_errp);
  scm_intprint (scm_heap_size, 10, cur_errp);
  scm_puts (" cells in use, ", cur_errp);
  scm_intprint (scm_mallocated, 10, cur_errp);
  scm_puts (" bytes allocated (of ", cur_errp);
  scm_intprint (scm_mtrigger, 10, cur_errp);
  scm_puts (")\n", cur_errp);
  if (NIMP (args))
    {
      scm_heap_report ();
      scm_puts ("\n", cur_errp);
      scm_stack_report ();
    }
  return UNSPECIFIED;
}

extern int scm_n_heap_segs;
#ifdef __STDC__
void 
scm_heap_report (void)
#else
void 
scm_heap_report ()
#endif
{
  sizet i = 0;
  scm_puts ("; heap segments:", cur_errp);
  while (i < scm_n_heap_segs)
    {
      scm_puts ("\n; 0x", cur_errp);
      scm_intprint ((long) scm_heap_table[i].bounds[0], 16, cur_errp);
      scm_puts (" - 0x", cur_errp);
      scm_intprint ((long) scm_heap_table[i].bounds[1], 16, cur_errp);
      ++i;
    }
}

#ifdef __STDC__
void 
scm_exit_report (void)
#else
void 
scm_exit_report ()
#endif
{
  if (scm_verbose > 2)
    {
      scm_puts (";Totals: ", cur_errp);
      scm_intprint (scm_time_in_msec (INUM (scm_my_time ())), 10, cur_errp);
      scm_puts (" mSec my time, ", cur_errp);
      scm_intprint (scm_time_in_msec (INUM (scm_your_time ())), 10, cur_errp);
      scm_puts (" mSec your time\n", cur_errp);
    }
}


PROC (s_verbose, "verbose", 0, 1, 0, scm_prolixity);
#ifdef __STDC__
SCM 
scm_prolixity (SCM arg)
#else
SCM 
scm_prolixity (arg)
     SCM arg;
#endif
{
  int old = scm_verbose;
  if (!UNBNDP (arg))
    {
      if (FALSEP (arg))
	scm_verbose = 1;
      else
	scm_verbose = INUM (arg);
    }
  return MAKINUM (old);
}



/* {Standard Ports}
 */
PROC (s_current_input_port, "current-input-port", 0, 0, 0, scm_current_input_port);
#ifdef __STDC__
SCM 
scm_current_input_port (void)
#else
SCM 
scm_current_input_port ()
#endif
{
  return cur_inp;
}

PROC (s_current_output_port, "current-output-port", 0, 0, 0, scm_current_output_port);
#ifdef __STDC__
SCM 
scm_current_output_port (void)
#else
SCM 
scm_current_output_port ()
#endif
{
  return cur_outp;
}

PROC (s_current_error_port, "current-error-port", 0, 0, 0, scm_current_error_port);
#ifdef __STDC__
SCM 
scm_current_error_port (void)
#else
SCM 
scm_current_error_port ()
#endif
{
  return cur_errp;
}

PROC (s_set_current_input_port, "set-current-input-port", 1, 0, 0, scm_set_current_input_port);
#ifdef __STDC__
SCM 
scm_set_current_input_port (SCM port)
#else
SCM 
scm_set_current_input_port (port)
     SCM port;
#endif
{
  SCM oinp = cur_inp;
  ASSERT (NIMP (port) && OPINPORTP (port), port, ARG1, s_current_error_port);
  cur_inp = port;
  return oinp;
}


PROC (s_set_current_output_port, "set-current-output-port", 1, 0, 0, scm_set_current_output_port);
#ifdef __STDC__
SCM 
scm_set_current_output_port (SCM port)
#else
SCM 
scm_set_current_output_port (port)
     SCM port;
#endif
{
  SCM ooutp = cur_outp;
  ASSERT (NIMP (port) && OPOUTPORTP (port), port, ARG1, s_current_error_port);
  cur_outp = port;
  return ooutp;
}


PROC (s_set_current_error_port, "set-current-error-port", 1, 0, 0, scm_set_current_error_port);
#ifdef __STDC__
SCM 
scm_set_current_error_port (SCM port)
#else
SCM 
scm_set_current_error_port (port)
     SCM port;
#endif
{
  SCM oerrp = cur_errp;
  ASSERT (NIMP (port) && OPOUTPORTP (port), port, ARG1, s_current_error_port);
  cur_errp = port;
  return oerrp;
}

/* {Help finding slib}
 */


PROC (s_compiled_library_path, "compiled-library-path", 0, 0, 0, scm_compiled_library_path);
#ifdef __STDC__
SCM
scm_compiled_library_path (void)
#else
SCM
scm_compiled_library_path ()
#endif
{
#ifndef LIBRARY_PATH
  return BOOL_F;
#else
  return makfrom0str (LIBRARY_PATH);
#endif
}



/* {Initializing the Module}
 */


char s_ccl[] = "char-code-limit";

#ifdef __STDC__
void
scm_final_repl (void)
#else
void
scm_final_repl ()
#endif
{
  scm_loc_errobj = (SCM *) & scm_tmp_errobj;
  scm_loc_loadpath = (SCM *) & scm_tmp_loadpath;
  loadport = SCM_UNDEFINED;
#ifdef TRANSCRIPT_SUPPORT
  transcript = BOOL_F;
  scm_trans = 0;
#endif
  scm_linum = 1;
}



#ifdef __STDC__
void
scm_init_repl (int iverbose)
#else
void
scm_init_repl (iverbose)
     int iverbose;
#endif
{
  scm_sysintern (s_ccl, MAKINUM (CHAR_CODE_LIMIT));
  scm_i_name = CAR (scm_sysintern ("name", SCM_UNDEFINED));
  scm_permenant_object (scm_i_name);
  scm_loc_errobj = &CDR (scm_sysintern ("errobj", SCM_UNDEFINED));
  scm_loc_loadpath = &CDR (scm_sysintern ("*load-pathname*", BOOL_F));
#ifdef TRANSCRIPT_SUPPORT
  transcript = BOOL_F;
  scm_trans = 0;
#endif
  scm_linum = 1;
  scm_verbose = iverbose;
#ifndef GO32
  scm_add_feature(s_char_ready_p);
#endif
#ifdef ARM_ULIB
  set_erase ();
#endif
  system_error_sym = CAR (scm_intern0 ("%%system-error"));
  scm_permenant_object (system_error_sym);
#include "repl.x"
}

