/*	Copyright (C) 1995 Free Software Foundation, Inc.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this software; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * As a special exception, the Free Software Foundation gives permission
 * for additional uses of the text contained in its release of GUILE.
 *
 * The exception is that, if you link the GUILE library with other files
 * to produce an executable, this does not by itself cause the
 * resulting executable to be covered by the GNU General Public License.
 * Your use of that executable is in no way restricted on account of
 * linking the GUILE library code into it.
 *
 * This exception does not however invalidate any other reasons why
 * the executable file might be covered by the GNU General Public License.
 *
 * This exception applies only to the code released by the
 * Free Software Foundation under the name GUILE.  If you copy
 * code from other Free Software Foundation releases into a copy of
 * GUILE, as the General Public License permits, the exception does
 * not apply to the code that you add in this way.  To avoid misleading
 * anyone as to the status of such modified files, you must delete
 * this exception notice from them.
 *
 * If you write modifications of your own for GUILE, it is your choice
 * whether to permit this exception to apply to your modifications.
 * If you do not wish that, delete this exception notice.  
 */


#include <stdio.h>
#include "_scm.h"



/* {Strings}
 */

static char s_string[];

#ifdef __STDC__
SCM 
scm_makstr (long len, int slots)
#else
SCM 
scm_makstr (len, slots)
     long len;
     int slots;
#endif
{
  SCM s;
  SCM * mem;
  NEWCELL (s);
  --slots;
  REDEFER_INTS;
  mem = (SCM *)scm_must_malloc (sizeof (SCM) * (slots + 1) + len + 1,
				s_string);
  if (slots >= 0)
    {
      int x;
      mem[slots] = (SCM)mem;
      for (x = 0; x < slots; ++x)
	mem[x] = BOOL_F;
    }
  SETCHARS (s, (char *) (mem + slots + 1));
  SETLENGTH (s, len, tc7_string);
  REALLOW_INTS;
  CHARS (s)[len] = 0;
  return s;
}

/* converts C scm_array of strings to SCM scm_list of strings. */
/* If argc < 0, a null terminated scm_array is assumed. */
#ifdef __STDC__
SCM 
scm_makfromstrs (int argc, char **argv)
#else
SCM 
scm_makfromstrs (argc, argv)
     int argc;
     char **argv;
#endif
{
  int i = argc;
  SCM lst = EOL;
  if (0 > i)
    for (i = 0; argv[i]; i++);
  while (i--)
    lst = scm_cons (scm_makfromstr (argv[i], (sizet) strlen (argv[i]), 0), lst);
  return lst;
}

#ifdef __STDC__
SCM
scm_take0str (char * it)
#else
SCM
scm_take0str (it)
     char * it;
#endif
{
  SCM answer;
  NEWCELL (answer);
  DEFER_INTS;
  SETLENGTH (answer, strlen (it), tc7_string);
  SETCHARS (answer, it);
  ALLOW_INTS;
  return answer;
}

#ifdef __STDC__
SCM 
scm_makfromstr (char *src, sizet len, int slots)
#else
SCM 
scm_makfromstr (src, len, slots)
     char *src;
     sizet len;
     int slots;
#endif
{
  SCM s;
  register char *dst;
  s = scm_makstr ((long) len, slots);
  dst = CHARS (s);
  while (len--)
    *dst++ = *src++;
  return s;
}


#ifdef __STDC__
SCM 
makfrom0str (char *src)
#else
SCM 
makfrom0str (src)
     char *src;
#endif
{
  if (!src) return BOOL_F;
  return scm_makfromstr (src, (sizet) strlen (src), 0);
}

#ifdef __STDC__
SCM 
makfrom0str_opt (char *src)
#else
SCM 
makfrom0str_opt (src)
     char *src;
#endif
{
  return makfrom0str (src);
}


PROC (s_string_p, "string?", 1, 0, 0, scm_string_p);
#ifdef __STDC__
SCM
scm_string_p (SCM x)
#else
SCM
scm_string_p (x)
     SCM x;
#endif
{
  if (IMP (x))
    return BOOL_F;
  return STRINGP (x) ? BOOL_T : BOOL_F;
}

PROC (s_list_to_string, "list->string", 1, 0, 0, scm_string);
PROC (s_string, "string", 0, 0, 1, scm_string);
#ifdef __STDC__
SCM
scm_string (SCM chrs)
#else
SCM
scm_string (chrs)
     SCM chrs;
#endif
{
  SCM res;
  register char *data;
  long i = scm_ilength (chrs);
  ASSERT (i >= 0, chrs, ARG1, s_string);
  res = scm_makstr (i, 0);
  data = CHARS (res);
  for (;NNULLP (chrs);chrs = CDR (chrs)) {
    ASSERT (ICHRP (CAR (chrs)), chrs, ARG1, s_string);
    *data++ = ICHR (CAR (chrs));
  }
  return res;
}

PROC (s_make_string, "make-string", 1, 1, 0, scm_make_string);
#ifdef __STDC__
SCM
scm_make_string (SCM k, SCM chr)
#else
SCM
scm_make_string (k, chr)
     SCM k;
     SCM chr;
#endif
{
  SCM res;
  register char *dst;
  register long i;
  ASSERT (INUMP (k) && (k >= 0), k, ARG1, s_make_string);
  i = INUM (k);
  res = scm_makstr (i, 0);
  dst = CHARS (res);
  if ICHRP (chr) for (i--;i >= 0;i--) dst[i] = ICHR (chr);
  return res;
}

PROC (s_string_length, "string-length", 1, 0, 0, scm_string_length);
#ifdef __STDC__
SCM
scm_string_length (SCM str)
#else
SCM
scm_string_length (str)
     SCM str;
#endif
{
  ASSERT (NIMP (str) && ROSTRINGP (str), str, ARG1, s_string_length);
  return MAKINUM (LENGTH (str));
}

PROC (s_string_ref, "string-ref", 2, 0, 0, scm_string_ref);
#ifdef __STDC__
SCM
scm_string_ref (SCM str, SCM k)
#else
SCM
scm_string_ref (str, k)
     SCM str;
     SCM k;
#endif
{
  ASSERT (NIMP (str) && ROSTRINGP (str), str, ARG1, s_string_ref);
  ASSERT (INUMP (k), k, ARG2, s_string_ref);
  ASSERT (INUM (k) < LENGTH (str) && INUM (k) >= 0, k, OUTOFRANGE, s_string_ref);
  return MAKICHR (CHARS (str)[INUM (k)]);
}

PROC (s_string_set_x, "string-set!", 3, 0, 0, scm_string_set_x);
#ifdef __STDC__
SCM
scm_string_set_x (SCM str, SCM k, SCM chr)
#else
SCM
scm_string_set_x (str, k, chr)
     SCM str;
     SCM k;
     SCM chr;
#endif
{
  ASSERT (NIMP (str) && STRINGP (str), str, ARG1, s_string_set_x);
  ASSERT (INUMP (k), k, ARG2, s_string_set_x);
  ASSERT (ICHRP (chr), chr, ARG3, s_string_set_x);
  ASSERT (INUM (k) < LENGTH (str) && INUM (k) >= 0, k, OUTOFRANGE, s_string_set_x);
  CHARS (str)[INUM (k)] = ICHR (chr);
  return UNSPECIFIED;
}


PROC1 (s_string_equal_p, "string=?", tc7_rpsubr, scm_string_equal_p);
#ifdef __STDC__
SCM
scm_string_equal_p (SCM s1, SCM s2)
#else
SCM
scm_string_equal_p (s1, s2)
     SCM s1;
     SCM s2;
#endif
{
  register sizet i;
  register char *c1, *c2;
  ASSERT (NIMP (s1) && ROSTRINGP (s1), s1, ARG1, s_string_equal_p);
  ASSERT (NIMP (s2) && ROSTRINGP (s2), s2, ARG2, s_string_equal_p);
  i = LENGTH (s2);
  if (LENGTH (s1) != i) return BOOL_F;
  c1 = CHARS (s1);
  c2 = CHARS (s2);
  while (0 != i--) if (*c1++ != *c2++) return BOOL_F;
  return BOOL_T;
}

PROC1 (s_string_ci_equal_p, "string-ci=?", tc7_rpsubr, scm_string_ci_equal_p);
#ifdef __STDC__
SCM
scm_string_ci_equal_p (SCM s1, SCM s2)
#else
SCM
scm_string_ci_equal_p (s1, s2)
     SCM s1;
     SCM s2;
#endif
{
  register sizet i;
  register unsigned char *c1, *c2;
  ASSERT (NIMP (s1) && ROSTRINGP (s1), s1, ARG1, s_string_ci_equal_p);
  ASSERT (NIMP (s2) && ROSTRINGP (s2), s2, ARG2, s_string_ci_equal_p);
  i = LENGTH (s2);
  if (LENGTH (s1) != i) return BOOL_F;
  c1 = UCHARS (s1);
  c2 = UCHARS (s2);
  while (0 != i--) if (scm_upcase[*c1++] != scm_upcase[*c2++]) return BOOL_F;
  return BOOL_T;
}

PROC1 (s_string_less_p, "string<?", tc7_rpsubr, scm_string_less_p);
#ifdef __STDC__
SCM
scm_string_less_p (SCM s1, SCM s2)
#else
SCM
scm_string_less_p (s1, s2)
     SCM s1;
     SCM s2;
#endif
{
  register sizet i, len;
  register unsigned char *c1, *c2;
  register int c;
  ASSERT (NIMP (s1) && ROSTRINGP (s1), s1, ARG1, s_string_less_p);
  ASSERT (NIMP (s2) && ROSTRINGP (s2), s2, ARG2, s_string_less_p);
  len = LENGTH (s1);
  i = LENGTH (s2);
  if (len>i) i = len;
  c1 = UCHARS (s1);
  c2 = UCHARS (s2);
  for (i = 0;i<len;i++) {
    c = (*c1++ - *c2++);
    if (c>0) return BOOL_F;
    if (c<0) return BOOL_T;
  }
  return (LENGTH (s2) != len) ? BOOL_T : BOOL_F;
}

PROC1 (s_string_leq_p, "string<=?", tc7_rpsubr, scm_string_leq_p);
#ifdef __STDC__
SCM
scm_string_leq_p (SCM s1, SCM s2)
#else
SCM
scm_string_leq_p (s1, s2)
     SCM s1;
     SCM s2;
#endif
{
  return BOOL_NOT (scm_string_less_p (s2, s1));
}

PROC1 (s_string_gr_p, "string>?", tc7_rpsubr, scm_string_gr_p);
#ifdef __STDC__
SCM
scm_string_gr_p (SCM s1, SCM s2)
#else
SCM
scm_string_gr_p (s1, s2)
     SCM s1;
     SCM s2;
#endif
{
  return scm_string_less_p (s2, s1);
}

PROC1 (s_string_geq_p, "string>=?", tc7_rpsubr, scm_string_geq_p);
#ifdef __STDC__
SCM
scm_string_geq_p (SCM s1, SCM s2)
#else
SCM
scm_string_geq_p (s1, s2)
     SCM s1;
     SCM s2;
#endif
{
  return BOOL_NOT (scm_string_less_p (s1, s2));
}

PROC1 (s_string_ci_less_p, "string-ci<?", tc7_rpsubr, scm_string_ci_less_p);
#ifdef __STDC__
SCM
scm_string_ci_less_p (SCM s1, SCM s2)
#else
SCM
scm_string_ci_less_p (s1, s2)
     SCM s1;
     SCM s2;
#endif
{
  register sizet i, len;
  register unsigned char *c1, *c2;
  register int c;
  ASSERT (NIMP (s1) && ROSTRINGP (s1), s1, ARG1, s_string_ci_less_p);
  ASSERT (NIMP (s2) && ROSTRINGP (s2), s2, ARG2, s_string_ci_less_p);
  len = LENGTH (s1);
  i = LENGTH (s2);
  if (len>i) i=len;
  c1 = UCHARS (s1);
  c2 = UCHARS (s2);
  for (i = 0;i<len;i++) {
    c = (scm_upcase[*c1++] - scm_upcase[*c2++]);
    if (c>0) return BOOL_F;
    if (c<0) return BOOL_T;
  }
  return (LENGTH (s2) != len) ? BOOL_T : BOOL_F;
}

PROC1 (s_string_ci_leq_p, "string-ci<=?", tc7_rpsubr, scm_string_ci_leq_p);
#ifdef __STDC__
SCM
scm_string_ci_leq_p (SCM s1, SCM s2)
#else
SCM
scm_string_ci_leq_p (s1, s2)
     SCM s1;
     SCM s2;
#endif
{
  return BOOL_NOT (scm_string_ci_less_p (s2, s1));
}

PROC1 (s_string_ci_gr_p, "string-ci>?", tc7_rpsubr, scm_string_ci_gr_p);
#ifdef __STDC__
SCM
scm_string_ci_gr_p (SCM s1, SCM s2)
#else
SCM
scm_string_ci_gr_p (s1, s2)
     SCM s1;
     SCM s2;
#endif
{
  return scm_string_ci_less_p (s2, s1);
}

PROC1 (s_string_ci_geq_p, "string-ci>=?", tc7_rpsubr, scm_string_ci_geq_p);
#ifdef __STDC__
SCM
scm_string_ci_geq_p (SCM s1, SCM s2)
#else
SCM
scm_string_ci_geq_p (s1, s2)
     SCM s1;
     SCM s2;
#endif
{
  return BOOL_NOT (scm_string_ci_less_p (s1, s2));
}

PROC (s_substring, "substring", 3, 0, 0, scm_substring);
#ifdef __STDC__
SCM
scm_substring (SCM str, SCM start, SCM end)
#else
SCM
scm_substring (str, start, end)
     SCM str;
     SCM start;
     SCM end;
#endif
{
  long l;
  ASSERT (NIMP (str) && ROSTRINGP (str),
	 str, ARG1, s_substring);
  ASSERT (INUMP (start), start, ARG2, s_substring);
  ASSERT (INUMP (end), end, ARG3, s_substring);
  ASSERT (INUM (start) <= LENGTH (str), start, OUTOFRANGE, s_substring);
  ASSERT (INUM (end) <= LENGTH (str), end, OUTOFRANGE, s_substring);
  l = INUM (end)-INUM (start);
  ASSERT (l >= 0, MAKINUM (l), OUTOFRANGE, s_substring);
  return scm_makfromstr (&CHARS (str)[INUM (start)], (sizet)l, 0);
}

PROC (s_string_append, "string-append", 0, 0, 1, scm_string_append);
#ifdef __STDC__
SCM
scm_string_append (SCM args)
#else
SCM
scm_string_append (args)
     SCM args;
#endif
{
  SCM res;
  register long i = 0;
  register SCM l, s;
  register char *data;
  for (l = args;NIMP (l);) {
    ASSERT (CONSP (l), l, ARGn, s_string_append);
    s = CAR (l);
    ASSERT (NIMP (s) && ROSTRINGP (s),
	   s, ARGn, s_string_append);
    i += LENGTH (s);
    l = CDR (l);
  }
  ASSERT (NULLP (l), args, ARGn, s_string_append);
  res = scm_makstr (i, 0);
  data = CHARS (res);
  for (l = args;NIMP (l);l = CDR (l)) {
    s = CAR (l);
    for (i = 0;i<LENGTH (s);i++) *data++ = CHARS (s)[i];
  }
  return res;
}


#ifdef __STDC__
void
scm_init_strings (void)
#else
void
scm_init_strings ()
#endif
{
#include "strings.x"
}

