/*
 ** Copyright (c) 1991-1994 Xerox Corporation.  All Rights Reserved.
 **
 ** Unlimited use, reproduction, and distribution of this software is
 ** permitted.  Any copy of this software must include both the above
 ** copyright notice of Xerox Corporation and this paragraph.  Any
 ** distribution of this software must comply with all applicable United
 ** States export control laws.  This software is made available AS IS,
 ** and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
 ** INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
 ** AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
 ** PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
 ** THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
 ** CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
 ** XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
 */
/* $Id: server.c,v 1.66 1996/03/12 04:03:33 janssen Exp $ */
/* Last edited by Mike Spreitzer February 13, 1996 12:02 pm PST */

#include "cstubber.h"

static boolean  FirstClass = TRUE;

static void declareReturnValue (Type type, Context context)
{
  if (type != NULL AND
      type_basic_type(type) != void_Type)
    {
      fprintf (context->file, "  %s _retvalue;\n", c_return_type(type));
    }
}

static void listArgument (refany elt, refany rock)
{
  Argument a = (Argument) elt;
  Context context = (Context) rock;
  enum PrimitiveTypes t = type_basic_type (ur_type(a->type));

  fprintf (context->file, ", %s%s",
	   ((a->direction == In AND
	     (t == union_Type OR t == record_Type OR
	      (t == sequence_Type AND NOT TypeIsString(ur_type(a->type)) AND NOT TypeIsWString(ur_type(a->type))))) OR
	    ((a->direction == Out OR a->direction == InOut) AND
	     t != array_Type)) ? "&" : "",
	   c_argument_name(a));
}

void FreeValue (Type type, char *name, Context context, int indent)
{
  TypeKind        t = type_ur_kind(type);
  Type            ut = ur_type(type);

  if (HasFreeRoutine(ut))
    fprintf(context->file, "%*.*s%s__Free (%s%s);\n", indent, indent, "",
	    c_type_name(ut),
	    ((t == array_Type) ? "" : "&"),
	    name);
}

static void freeArgument (refany elt, refany rock)
{
  Argument arg = (Argument) elt;
  Context context = (Context) rock;
  FreeValue (arg->type, c_argument_name(arg), context, 2);
}     

static void freeReturnValue (Type type, Context context)
{
  FreeValue (type, "_retvalue", context, 2);
}

static void sizeArgument (refany elt, refany rock)
{
  Argument a = (Argument) elt;
  Context context = (Context) rock;
  char b[ 1000 ];
  Type ut = ur_type(a->type);
  enum PrimitiveTypes t = type_basic_type (ur_type(a->type));

  if  (a->direction == In)
    return;
  if (t == union_Type OR
      t == record_Type OR
      (t == sequence_Type AND NOT TypeIsString(ut) AND NOT TypeIsWString(ut)))
    {
      b[0] = '&';
      strcpy (b+1, c_argument_name(a));
    }
  else
    strcpy(b, c_argument_name(a));
  fprintf (context->file, "      _size += ");
  SizeValue (context, ut, b);
  fprintf (context->file, ";\n");
  fprintf (context->file, "      if (ILU_ERRNOK(*_err)) goto errexit;\n");
}

static void outputArgument (refany elt, refany rock)
{
  Argument a = (Argument) elt;
  Context context = (Context) rock;
  char b[ 1000 ];
  Type ut = ur_type(a->type);
  enum PrimitiveTypes t = type_basic_type (ur_type(a->type));

  if  (a->direction == In)
    return;
  if (t == union_Type OR
      t == record_Type OR
      (t == sequence_Type AND NOT TypeIsString(ut) AND NOT TypeIsWString(ut)))
    {
      b[0] = '&';
      strcpy (b+1, c_argument_name(a));
    }
  else
    strcpy(b, c_argument_name(a));
  MarshallValue (context, ut, b, 6);
  fprintf (context->file, "      if (ILU_ERRNOK(*_err)) goto errexit;\n");
}

static void inputArgument (refany elt, refany rock)
{
  Argument arg = (Argument) elt;
  Context context = (Context) rock;
  char buf[1000];

  if (arg->direction == Out)
    return;
  sprintf (buf, "%s%s", TypeIsPointer(ur_type(arg->type)) ? "" : "&",
	   c_argument_name(arg));
  UnmarshallValue (context, ur_type(arg->type), arg->def, buf, 2, TRUE, FALSE);
  fprintf (context->file, "  if (ILU_ERRNOK(*_err)) goto errexit;\n");
}

static void declareArgument (refany elt, refany rock)
{
  Argument arg = (Argument) elt;
  TypeKind t = type_ur_kind(arg->type);
  Context context = (Context) rock;

  fprintf(context->file, "  %s %s;\n", c_type_name(arg->type), c_argument_name(arg));
}

static void initType (char *name, Type type, Context context, ArgDirection dir)
{
  TypeKind t = type_ur_kind(type);

  if (t == object_Type || TypeIsString(type) ||
      t == shortinteger_Type || t == integer_Type ||
      t == shortcardinal_Type || t == cardinal_Type ||
      t == shortreal_Type || t == real_Type || t == boolean_Type ||
      t == optional_Type || t == enumeration_Type || t == byte_Type)
    fprintf (context->file, "  %s = (%s) 0;\n", name, c_type_name(type));
  else if ((dir == In) && (t == union_Type || t == record_Type || (t == sequence_Type && !TypeIsString(type))))
    fprintf (context->file, "  memset (&%s, 0, sizeof(%s));\n", name, c_type_name(type));
  else if ((dir == In) && t == array_Type)
    fprintf (context->file, "  memset (%s, 0, sizeof(%s));\n", name, c_type_name(type));

}

static void initArgument (refany elt, refany rock)
{
  Argument arg = (Argument) elt;

  initType (c_argument_name(arg), arg->type, (Context) rock, arg->direction == Out);
}

static void initReturnValue (Type t, Context context)
{
  initType ("_retvalue", t, context, TRUE);
}

static void addExceptionOutputFn (refany elt, refany rock)
{
  Exception       e = (Exception) elt;
  Context         context = (Context) rock;
  Exception       ure = e->import ? e->import : e;
  Type            ut = ur_type(ure->type);
  TypeKind        tk = type_kind(ut);
  char            a;
  char           *p = &a;

  codeType (ut, &p, In);
  fprintf (context->file, "%s\n    { %d, ", context->counter == 0 ? "" : ",", (unsigned int) a);
  context->counter += 1;

  if (tk == object_Type)
    fprintf (context->file, "_%s__ILUType, ", c_type_name(ut));
  else
    fprintf (context->file, "(ilu_Class) 0, ");

  switch (tk)
    {
    case integer_Type:
      fprintf (context->file, "(void (*) (void)) ilu_SizeOfInteger, (void (*) (void)) ilu_OutputInteger");
      break;
      
    case cardinal_Type:
      fprintf (context->file, "(void (*) (void)) ilu_SizeOfCardinal, (void (*) (void)) ilu_OutputCardinal");
      break;
      
    case shortinteger_Type:
      fprintf (context->file, "(void (*) (void)) ilu_SizeOfShortInteger, (void (*) (void)) ilu_OutputShortInteger");
      break;
      
    case shortcardinal_Type:
      fprintf (context->file, "(void (*) (void)) ilu_SizeOfShortCardinal, (void (*) (void)) ilu_OutputShortCardinal");
      break;
      
    case longinteger_Type:
      fprintf (context->file, "(void (*) (void)) ilu_SizeOfLongInteger, (void (*) (void)) ilu_OutputLongInteger");
      break;
      
    case longcardinal_Type:
      fprintf (context->file, "(void (*) (void)) ilu_SizeOfLongCardinal, (void (*) (void)) ilu_OutputLongCardinal");
      break;

    case character_Type:
      fprintf (context->file, "(void (*) (void)) ilu_SizeOfCharacter, (void (*) (void)) ilu_OutputCharacter");
      break;

    case shortcharacter_Type:
      fprintf (context->file, "(void (*) (void)) ilu_SizeOfShortCharacter, (void (*) (void)) ilu_OutputShortCharacter");
      break;

    case real_Type:
      fprintf (context->file, "(void (*) (void)) ilu_SizeOfReal, (void (*) (void)) ilu_OutputReal");
      break;

    case shortreal_Type:
      fprintf (context->file, "(void (*) (void)) ilu_SizeOfShortReal, (void (*) (void)) ilu_OutputShortReal");
      break;

    case longreal_Type:
      fprintf (context->file, "(void (*) (void)) ilu_SizeOfLongReal, (void (*) (void)) ilu_OutputLongReal");
      break;

    case byte_Type:
      fprintf (context->file, "(void (*) (void)) ilu_SizeOfByte, (void (*) (void)) ilu_OutputByte");
      break;

    case boolean_Type:
      fprintf (context->file, "(void (*) (void)) ILU_C_SizeOfBoolean, (void (*) (void)) ILU_C_OutputBoolean");
      break;

    case enumeration_Type:
      fprintf (context->file, "(void (*) (void)) ilu_SizeOfEnum, (void (*) (void)) ilu_OutputEnum");
      break;

    case object_Type:
      fprintf(context->file, "(void (*) (void)) 0, (void (*) (void)) 0",
	      c_type_name(ut));
      break;

    case array_Type:
    case record_Type:
    case union_Type:
    case sequence_Type:
    case optional_Type:
      fprintf(context->file,
	      "(void (*) (void)) _%s__SizeOf, (void (*) (void)) _%s__Output",
	      c_type_name(ut), c_type_name(ut));
      break;

    case void_Type:
      fprintf (context->file, "(void (*) (void)) 0, (void (*) (void)) 0");
      break;

  default:
    fatal ("Error: Bad parameter type %s.\n", c_type_name (ure->type));
  }
  fprintf (context->file, " }");
}

static void declareExceptionValOutputFns (list exceptions, Context context)
{
  if (exceptions == NULL || list_size(exceptions) < 1)
    return;

  fprintf (context->file, "  static struct _ILU_C_ExceptionDescription_s _evec[] = {");
  context->counter = 0;
  list_enumerate (exceptions, addExceptionOutputFn, context);
  fprintf (context->file, "};\n\n");
}

static void serverMethods (Procedure  m, Context  context)
{
  fprintf(context->file, "    server_%s_%s,\n", c_type_name(context->class), c_simple_name(m->name));
}

static void serverProto (Procedure  m, Context  context)
{
  fprintf(context->file, "extern void server_%s_%s ();\n", c_type_name(context->class), c_simple_name(m->name));
}

static void generateCalleeStub (Procedure  m, Context  context)
{
  enum PrimitiveTypes t = type_basic_type(ur_type(m->returnType));

  if (methodInList(c_simple_name(m->name))) {
    MethodRecordID++;
    return;
  }
  addMethodToList(c_simple_name(m->name));

  fprintf(context->file, "static void _%s_%s__truestub(ilu_Call _call, ilu_Error *_err)\n{\n",
	  c_type_name(context->class), c_simple_name(m->name));
  fprintf(context->file, "  %s _h;\n", c_type_name(context->class));
  fprintf(context->file, "  ILU_C_ENVIRONMENT _status;\n");
  list_enumerate(m->arguments, declareArgument, context);
  declareReturnValue(m->returnType, context);
  declareExceptionValOutputFns (m->exceptions, context);
  fprintf(context->file, "\n");

  list_enumerate(m->arguments, initArgument, context);
  initReturnValue(m->returnType, context);
  fprintf(context->file, "  ILU_CLER(*_err);\n");
  fprintf(context->file, "  _h = (%s) ",
	  c_type_name(context->class));
  if (SINGLETON(context->class))
    fprintf(context->file, "_ILU_C_GetServerSingleton (_call, _err);\n");
  else
    fprintf(context->file,
	    "_ILU_C_InputObject (_call, _%s__ILUType, ilu_TRUE, _err);\n",
	    c_type_name(context->class));
  fprintf (context->file, "  if (ILU_ERRNOK(*_err)) goto errexit;\n");
  list_enumerate(m->arguments, inputArgument, context);
  fprintf(context->file, "\n");
  fprintf(context->file,
	  "  if (_ILU_C_FinishParameters (_call, _h, _err)) {\n");
  fprintf(context->file, "    _status.returnCode = ILU_NIL;\n");
  fprintf(context->file, "    _status.callerPassport = ilu_CallerPassportOfCall(_call);\n    ");
  if (t != void_Type)
    fprintf(context->file, "_retvalue = ");
  fprintf(context->file, "%s_%s (_h",
	  c_type_name(context->class), c_simple_name(m->name));
  list_enumerate(m->arguments, listArgument, context);
  fprintf(context->file, ", &_status);\n\n");
  if (m->asynch) {
    fprintf(context->file, "    /* asynchronous method -- no reply */\n");
    fprintf(context->file, "  }\n");
    fprintf(context->file, "  if (!_ILU_C_NoReply(_call, _err)) goto errexit;\n");
  } else {
    char            buf[1000];
    sprintf(buf, "%s_retvalue",
	    (t == record_Type
	     || NonStringSequence(ur_type(m->returnType))
	     || t == union_Type)
	    ? "&" : "");
    fprintf(context->file, "    /* check for errors */\n");
    fprintf(context->file, "    if (_status.returnCode == ILU_NIL) {\n");
    fprintf(context->file, "      ilu_cardinal _size = ");
    fprintf(context->file, "ilu_BeginSizingReply(_call, %s, _err);\n",
	(list_size(m->exceptions) > 0) ? "ilu_TRUE" : "ilu_FALSE");
    fprintf(context->file, "      if (ILU_ERRNOK(*_err))\n");
    fprintf(context->file, "        goto errexit;\n");
    if (t != void_Type) {
      fprintf(context->file, "      _size += ");
      SizeValue(context, ur_type(m->returnType), buf);
      fprintf(context->file, ";\n");
      fprintf(context->file, "      if (ILU_ERRNOK(*_err))\n");
      fprintf(context->file, "        goto errexit;\n");
    }
    list_enumerate(m->arguments, sizeArgument, context);
    fprintf(context->file,
	    "      if (!_ILU_C_BeginReply (_call, %s, _size, _err)) goto errexit;\n",
	(list_size(m->exceptions) > 0) ? "ilu_TRUE" : "ilu_FALSE");
    if (type_basic_type(ur_type(m->returnType)) != void_Type)
      {
	MarshallValue(context, m->returnType, buf, 6);
	fprintf (context->file, "      if (ILU_ERRNOK(*_err)) goto errexit;\n");
      }
    list_enumerate(m->arguments, outputArgument, context);
    fprintf(context->file,
	    "      if (! _ILU_C_FinishReply (_call, _err)) goto errexit;\n");
    fprintf(context->file, "    }\n");
    fprintf(context->file, "    else {\n");
    if (list_size(m->exceptions) > 0) {
      fprintf(context->file,
	      "      _ILU_C_SendException (_call, _evec, &_status, _err);\n",
	      c_interface_name(m->interface));
      fprintf (context->file, "      if (ILU_ERRNOK(*_err)) goto errexit;\n");
    } else {
      fprintf(context->file,
	      "      (void) ILU_ERR_CONS1(bad_param, _err, minor, ilu_bpm_some_raise, 6);\n");
    }
    fprintf(context->file, "    }\n");
    fprintf(context->file, "  }\n\n");
  }
  fprintf (context->file, " errexit:\n");
  fprintf(context->file, "  _ILU_C_FinishServingCall(_call, _err);\n");
  if (t != void_Type)
    freeReturnValue(m->returnType, context);
  list_enumerate(m->arguments, freeArgument, context);
  fprintf(context->file, "  return;\n}\n\n");
}

static void classTable (Type class, Context context)
{
  Class od;

  if  (class == NULL || 
       type_basic_type (class) != object_Type || 
       (od = class_object (class)) == NULL)
    return;
  if  (list_size (od->methods) > 0)
    list_enumerate (od->methods, (void (*)(refany, refany)) serverMethods, context);
}

static void generateProto (Type  class, Context  context)
{
  Class  od;

  if  (class == NULL || 
       type_basic_type (class) != object_Type || 
       (od = class_object (class)) == NULL)
    return;
  if  (list_size (od->methods) > 0)
    list_enumerate (od->methods, (void (*)(refany, refany)) serverProto, context);
}


static void generateServerMiMethodPtrs(refany elt, refany rock)
{
  Type            class = (Type) elt;
  Context         context = (Context) rock;
  Type            ut = ur_type(class);

  if (class_object(ut)->superclasses != NULL)
    list_enumerate(class_object(ut)->superclasses,
		   generateServerMiMethodPtrs, context);
  if (methodInList(c_type_name(ut)))
    return;
  addMethodToList(c_type_name(ut));
  generateProto(ut, context);
  fprintf(context->file,
  "static struct _%s__MethodBlock_s _%s_%s__TrueMethodBlock = {\n",
	  c_type_name(ut), c_type_name(context->class),
	  c_type_name(ut));
  fprintf(context->file, "  NULL /* will be set to _%s__ILUType */",
	  c_type_name(ut));
  if (list_size(class_object(ut)->methods) > 0) {
    fprintf(context->file, ",\n  {\n");
    classTable(ut, context);
    fprintf(context->file, "  }\n");
  }
  fprintf(context->file, "};\n");
  return;
}

static void generateServerClsPtrs (refany elt, refany rock)
{
  Type            t = (Type) elt;
  Context         c = (Context) rock;
  if (class_object(t)->superclasses != NULL)
    list_enumerate(class_object(t)->superclasses,
		   generateServerClsPtrs, c);
  fprintf(c->file,
	  "  (_ILU_C_MethodBlock *) &_%s_%s__TrueMethodBlock,\n",
	  c_type_name(c->class), c_type_name(ur_type(t)));
}

static void generateServerClassTable (refany elt, refany rock)
{
  Type            t = (Type) elt;
  Context         c = (Context) rock;
  if (type_basic_type(t) != object_Type)
    return;

  c->class = t;
  clearMethodList();
  generateServerMiMethodPtrs(t, c);
  fprintf(c->file,
	  "static _ILU_C_MethodBlock *_%s__TrueTypeVector[] = {\n",
	  c_type_name(t));
  fprintf(c->file,
	  "  (_ILU_C_MethodBlock *) &_%s_%s__TrueMethodBlock,\n",
	  c_type_name(c->class), c_type_name(t));
  if (class_object(t)->superclasses != NULL)
    list_enumerate(class_object(t)->superclasses, generateServerClsPtrs,
		   c);
  fprintf(c->file, "  ILU_NIL\n};\n\n");
}

static void trueMethBlkInitsPerIluType(refany elt, refany rock)
{
  Type            class = (Type) elt;
  Context         context = (Context) rock;
  Type            ut = ur_type(class);

  if (class_object(ut)->superclasses != NULL)
    list_enumerate(class_object(ut)->superclasses,
		   trueMethBlkInitsPerIluType, context);
  if (methodInList(c_type_name(ut)))
    return;
  addMethodToList(c_type_name(ut));
  fprintf(context->file, "  _%s_%s__TrueMethodBlock.c = _%s__ILUType;\n",
	  c_type_name(context->class), c_type_name(ut),
	  c_type_name(ut));
}

static void trueMethBlkInitsPerCType (refany elt, refany rock)
{
  Type            t = (Type) elt;
  Context         c = (Context) rock;
  if (type_basic_type(t) != object_Type)
    return;
  c->class = t;
  clearMethodList();
  trueMethBlkInitsPerIluType(t, c);
}

static void classSetupStubs (Type  class, Context  context)
{
  Class  od;

  if  (class == NULL || 
       type_basic_type (class) != object_Type || 
       (od = class_object (class)) == NULL)
    return;
  if  (list_size (od->methods) > 0)
    list_enumerate (od->methods, (void (*)(refany, refany)) generateCalleeStub, context);
}


static void generateServerCodeForClass (Type class, Context context)
{
  if (type_basic_type(class) != object_Type)
    return;

  context->class = class;
  clearMethodList ();
  classSetupStubs (class, context);
}

static void initializeStubPointer (Procedure  method, Context  context)
{
  fprintf (context->file, "  _%s__ILUType->cl_methods[%ld].me_stubproc = (void (*)(ilu_Call)) _%s_%s__truestub;\n",
	   c_type_name (context->class), MethodRecordID++,
	   c_type_name (context->class /*method->object*/),
	   c_simple_name (method->name));
}

static void setupServerStubsInMethodTable (Type  type, Context  context)
{
  Class od;
  
  if  (type == NULL || 
       type_basic_type (type) != object_Type || 
       (od = class_object (type)) == NULL)
    return;
  fprintf (context->file, "\n");
  list_enumerate (od->methods, (void (*)(refany, refany)) initializeStubPointer, context);
}

static void setupServerStubsTable (Type  type, Context  context)
{
  MethodRecordID = 0;
  FirstClass = True;
  context->class = type;
  setupServerStubsInMethodTable (type, context);
}

static void generateHandles (Type  type, Context  context)
{
  fprintf (context->file, "ILU_C_OBJECT %s__CreateTrue  (ilu_string instance_handle, ilu_Server server, void * data)\n{\n", c_type_name(type));
  fprintf (context->file, "  return (_ILU_C_CreateTrueObject (_%s__TrueTypeVector, instance_handle, server, data, ilu_FALSE));\n}\n\n", c_type_name (type));
  fprintf (context->file, "ILU_C_OBJECT %s__OTCreateTrue  (ilu_string instance_handle, ilu_Server server, void * data)\n{\n", c_type_name(type));
  fprintf (context->file, "  if (server == ILU_NIL || instance_handle == ILU_NIL) return ILU_NIL;\n");
  fprintf (context->file, "  return (_ILU_C_CreateTrueObject (_%s__TrueTypeVector, instance_handle, server, data, ilu_TRUE));\n}\n\n", c_type_name (type));
}

static void 
generateServerRegistrationCode(Interface interface, Context context)
{
  FILE           *f = context->file;
  char           *interface_name = (char *) c_interface_name(interface);

  list_enumerate(interface->classes,
	     (void (*) (refany, refany)) generateHandles, context);
  fprintf(f, "void %s__InitializeServer(void)\n{\n",
	  interface_name);
  fprintf(f, "  extern void _%s__GeneralInitialization(void);\n",
	  c_interface_name(interface));
  fprintf(f, "  static ilu_boolean initialized = ilu_FALSE;\n");
  fprintf(f, "  if (initialized) return;\n  initialized = ilu_TRUE;\n\n");
  fprintf(f, "  _%s__GeneralInitialization ();\n", interface_name);
  list_enumerate(interface->classes, trueMethBlkInitsPerCType, context);
  list_enumerate(interface->classes,
       (void (*) (refany, refany)) setupServerStubsTable, context);
  fprintf(f, "}\n\n");
}

void generateServerCode (Interface parse, FILE *file)
{
  struct context_s context;
  char           *pc_interfacename;

  context.file = file;
  context.interface = parse;

  /*
   * get any translation of what the header file for the interface
   * is
   */
  pc_interfacename = interface_header_name(c_interface_name(parse));

  fprintf(file, "#include <stdio.h>\n\n");
  fprintf(file, "#define _%s_TRUE_STUB_CODE_\n\n",
	  c_interface_name(parse));
  fprintf(file, "#include \"%s.h\"\n\n", pc_interfacename);

  list_enumerate(parse->classes, generateServerClassTable, &context);

  list_enumerate(parse->classes,
	    (void (*) (refany, refany)) generateServerCodeForClass,
		 &context);

  generateServerRegistrationCode(parse, &context);
}
