/* Copyright (C) 1994 
            Olav Woelfelschneider (wosch@rbg.informatik.th-darmstadt.de)

 This library is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

 This library is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <X11/Xlib.h>
#include <stdio.h>
#include <string.h>
#include <math.h>

#include "McAlloc.h"
#include "McApp.h"
#include "McGadget.h"
#include "McKnob.h"

#define KNOB ((McKnob *)(gadget->specialInfo))

/****************************************************************************/

static void KnobUpdate(McGadget *gadget, int busy, int all);
static int KnobEvent(McGadget *gadget, XEvent *event);
static void KnobCleanup(McGadget *gadget);

/****************************************************************************/

McSpecialInfo *McCreateKnob(unsigned int flags, int max) {
  McKnob *knob = (McKnob *) McAlloc(sizeof(McKnob));
  knob->flags = flags;
  knob->max=max;
  knob->value=knob->lastvalue=max>>1;
  
  knob->specialInfo.updateProc  = KnobUpdate;
  knob->specialInfo.eventProc   = KnobEvent;
  knob->specialInfo.cleanupProc = KnobCleanup;

  return (McSpecialInfo *)knob;
}

/****************************************************************************/

static void KnobCleanup(McGadget *gadget) {
  McFree(gadget->specialInfo);
}

/****************************************************************************/

/*
 * Whats better? Wasting memory for a table of sines,
 * or wasting cpu cycles by doing it with floats?
 * Dunno, but coding was easier this way (:
 */
static void KnobUpdate(McGadget *gadget, int busy,int all) {
  McWindow *mcw=gadget->mcw;
  McApp *app=mcw->app;
  Display *display=app->display;
  McKnob *knob=KNOB;
  double foo;
  int x, y, w_2, h_2;

  if (knob->value!=knob->lastvalue) {
    h_2=gadget->height>>1; w_2=gadget->width>>1;
    foo=(knob->max-knob->lastvalue) * (3*M_PI_2) / knob->max - M_PI_4;
    y=sin(foo)*h_2; x=cos(foo)*w_2;
    XDrawLine(display, mcw->window, app->gc[GC_CLEAR],
	      gadget->x+w_2, gadget->y+h_2, gadget->x+w_2+x, gadget->y+h_2-y);
  }

  h_2=gadget->height>>1; w_2=gadget->width>>1;
  foo=(knob->max-knob->value) * (3*M_PI_2) / knob->max - M_PI_4;
  y=sin(foo)*h_2; x=cos(foo)*w_2;
  XDrawLine(display, mcw->window, app->gc[GC_NORMAL],
	    gadget->x+w_2, gadget->y+h_2, gadget->x+w_2+x, gadget->y+h_2-y);

  knob->lastvalue=knob->value;
  XDrawArc(display, mcw->window, app->gc[GC_DARK], gadget->x, gadget->y,
	   gadget->width, gadget->height,  225*64, 180*64);
  XDrawArc(display, mcw->window, app->gc[GC_NORMAL], gadget->x, gadget->y,
	   gadget->width, gadget->height,  270*64, 90*64);
  XDrawArc(display, mcw->window, app->gc[GC_BRIGHT], gadget->x, gadget->y,
	   gadget->width, gadget->height, 45*64,  180*64);
}

/****************************************************************************/

static void CalcValue(McGadget *gadget, McKnob *knob, int x, int y, int set) {

  double d;

  x=x-(gadget->x+(gadget->width>>1));
  y=gadget->y+(gadget->height>>1)-y;

  if (y>0) {
    d=acos(y/sqrt(x*x + y*y));
    if (x>0) d=M_PI_2+M_PI_4+d; else d=M_PI_2+M_PI_4-d;
  } else if (y<0) {
    d=acos(-y/sqrt(x*x + y*y));
    if (x>0) {
      d=M_PI+M_PI_2+M_PI_4-d;
      if (d>M_PI+M_PI_2) d=M_PI+M_PI_2;
    } else {
      d=d-M_PI_4;
      if (d<0) d=0;
    }
  } else {
    if (x>0) d=M_PI+M_PI_4; else d=M_PI_4;
  }

  if (set) {
    knob->value = knob->max * d / (M_PI+M_PI_2);
  } else {
    int now = knob->max * d / (M_PI+M_PI_2);
    if (now>knob->value)
      knob->value++;
    else
      if (now<knob->value)
	knob->value--;
  }
}

/****************************************************************************/

static int KnobEvent(McGadget *gadget, XEvent *event) {
  McWindow *mcw=gadget->mcw;
  McApp *app=mcw->app;
  McKnob *knob=KNOB;

  if (event->type == Expose) return 0;

  if (gadget->flags&GAD_PRESSED) {
    switch (event->type) {
    case ButtonRelease:
      gadget->flags&=~GAD_PRESSED;
      if (gadget->callbackUp) {
	(*gadget->callbackUp)(gadget);
      }
      return 1;

    case MotionNotify:
      CalcValue(gadget, knob, event->xmotion.x, event->xmotion.y, 1);
      KnobUpdate(gadget, 0, 0);
      if (gadget->callbackDown) {
	XFlush(mcw->app->display);
	(*gadget->callbackDown)(gadget);
      }
      return 1;
    case ButtonPress:
      return 1;
    default:
      return 0;
    }
  } else { /* not currently pressed */
    switch (event->type) {
    case ButtonPress:
      if (McInRectangle(event->xbutton.x, event->xbutton.y,
			gadget->x, gadget->y, gadget->width, gadget->height)) {
	switch (event->xbutton.button) {
	case 1:
	  CalcValue(gadget, knob, event->xmotion.x, event->xmotion.y, 0);
	  KnobUpdate(gadget, 0, 0);
	  if (gadget->callbackDown) {
	    XFlush(mcw->app->display);
	    (*gadget->callbackDown)(gadget);
	  }
	  if (gadget->callbackUp) {
	    XFlush(mcw->app->display);
	    (*gadget->callbackUp)(gadget);
	  }
	  gadget->flags&=~GAD_PRESSED;
	  McMoveGadgetToStart(gadget);
	  return 1;
	case 2:
	  CalcValue(gadget, knob, event->xmotion.x, event->xmotion.y, 1);
	  KnobUpdate(gadget, 0, 0);
	  if (gadget->callbackDown) {
	    XFlush(app->display);
	    (*gadget->callbackDown)(gadget);
	  }
	  gadget->flags|=GAD_PRESSED;
	  McMoveGadgetToStart(gadget);
	  return 1;
	}
      } else {
	return 0;
      }
    default:
      return 0;
    }
  }
  return 0;
}

/****************************************************************************/

McGadget *MakeKnob(McWindow *mcw, int x, int y, int d, int max) {
  McGadget *gad = McCreateGadget(mcw, GAD_ACTIVE, KNOBGADGET, x, y, d, d);
  gad->specialInfo=McCreateKnob(0, max);
  return gad;
}
