/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.component;

import org.apache.avalon.framework.activity.Disposable;
import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.component.ComponentManager;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.logger.AbstractLoggable;
import org.apache.avalon.framework.thread.SingleThreaded;
import org.apache.avalon.framework.thread.ThreadSafe;
import org.apache.avalon.excalibur.pool.Poolable;
import org.apache.avalon.excalibur.logger.LogKitManager;

/**
 * This class acts like a Factory to instantiate the correct version
 * of the ComponentHandler that you need.
 *
 * @author <a href="mailto:bloritsch@apache.org">Berin Loritsch</a>
 * @version CVS $Revision: 1.4 $ $Date: 2001/12/11 09:53:27 $
 * @since 4.0
 */
public abstract class ComponentHandler extends AbstractLoggable
                                       implements Initializable, Disposable {

    public static ComponentHandler getComponentHandler(
                             final Class componentClass,
                             final Configuration config,
                             final ComponentManager manager,
                             final Context context,
                             final RoleManager roles,
                             final LogKitManager logkit )
    throws Exception
    {
        int numInterfaces = 0;

        if (SingleThreaded.class.isAssignableFrom(componentClass))
        {
            numInterfaces++;
        }

        if (ThreadSafe.class.isAssignableFrom(componentClass))
        {
            numInterfaces++;
        }

        if (Poolable.class.isAssignableFrom(componentClass))
        {
            numInterfaces++;
        }

        if (numInterfaces > 1)
        {
            throw new Exception("[CONFLICT] lifestyle interfaces: " + componentClass.getName());
        }

        if (Poolable.class.isAssignableFrom(componentClass))
        {
            return new PoolableComponentHandler(componentClass,
                                                config,
                                                manager,
                                                context,
                                                roles,
                                                logkit);
        }
        else if (ThreadSafe.class.isAssignableFrom(componentClass))
        {
            return new ThreadSafeComponentHandler(componentClass,
                                                  config,
                                                  manager,
                                                  context,
                                                  roles,
                                                  logkit);
        }
        else // This is a SingleThreaded component
        {
            return new DefaultComponentHandler(componentClass,
                                               config,
                                               manager,
                                               context,
                                               roles,
                                               logkit);
        }
    }

    public static ComponentHandler getComponentHandler(
                             final Component componentInstance )
    throws Exception
    {
        int numInterfaces = 0;

        if (SingleThreaded.class.isAssignableFrom(componentInstance.getClass()))
        {
            numInterfaces++;
        }

        if (ThreadSafe.class.isAssignableFrom(componentInstance.getClass()))
        {
            numInterfaces++;
        }

        if (Poolable.class.isAssignableFrom(componentInstance.getClass()))
        {
            numInterfaces++;
        }

        if (numInterfaces > 1)
        {
            throw new Exception("[CONFLICT] lifestyle interfaces: " + componentInstance.getClass().getName());
        }

        return new ThreadSafeComponentHandler(componentInstance);
    }

    public abstract Component get() throws Exception;

    public abstract void put(Component component) throws Exception;
}
