/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.util;

import org.apache.avalon.framework.activity.Disposable;
import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.activity.Startable;
import org.apache.avalon.framework.activity.Suspendable;
import org.apache.avalon.framework.component.Composable;
import org.apache.avalon.framework.configuration.Configurable;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.logger.LogEnabled;
import org.apache.avalon.framework.logger.Loggable;
import org.apache.avalon.framework.parameters.Parameterizable;

/**
 * This class provides basic facilities for enforcing Avalon's contracts
 * within your own code.
 *
 * @author <a href="mailto:bloritsch@apache.org">Berin Loritsch</a>
 * @author <a href="mailto:mike@hihat.net">Michael McKibben</a>
 * @version CVS $Revision: 1.7 $ $Date: 2002/01/17 15:02:01 $
 */
public final class ComponentStateValidator
{
    private static final String OOO_FAIL       = "Initialization out of order";
    private static final String LOG_FAIL       = OOO_FAIL + ": LogEnabled";
    private static final String CONTEXT_FAIL   = OOO_FAIL + ": Contextualizable";
    private static final String PARAMETER_FAIL = OOO_FAIL + ": Parameterizable";
    private static final String CONFIGURE_FAIL = OOO_FAIL + ": Configurable";
    private static final String COMPOSE_FAIL   = OOO_FAIL + ": Composable";
    private static final String INIT_FAIL      = OOO_FAIL + ": Initializable";
    private static final String START_FAIL     = OOO_FAIL + ": Startable.start()";
    private static final String STOP_FAIL      = OOO_FAIL + ": Startable.stop()";
    private static final String SUSPEND_FAIL   = OOO_FAIL + ": Suspendable.suspend()";
    private static final String RESUME_FAIL    = OOO_FAIL + ": Suspendable.resume()";
    private static final String DISPOSE_FAIL   = OOO_FAIL + ": Disposable";
    private static final String ACTIVE_FAIL    = "Component is not Active";
    private static final String WRITE_FAIL      = "Value is already bound";

    private static final long LOG_ENABLED    = 0x00000001;
    private static final long CONTEXTUALIZED = 0x00000002;
    private static final long COMPOSED       = 0x00000004;
    private static final long CONFIGURED     = 0x00000008;
    private static final long PARAMETERIZED  = 0x00000010;
    private static final long INITIALIZED    = 0x00000020;
    private static final long STARTED        = 0x00000040;
    private static final long ACTIVE         = 0x10000000;
    private static final long SUSPENDED      = 0x01000000;
    private static final long STOPPED        = 0x00000080;
    private static final long DISPOSED       = 0x00000100;

    private static final long LOG_MASK       = LOG_ENABLED;
    private static final long CONTEXT_MASK   = LOG_MASK | CONTEXTUALIZED;
    private static final long COMPOSE_MASK   = CONTEXT_MASK | COMPOSED;
    private static final long CONFIGURE_MASK = COMPOSE_MASK | CONFIGURED;
    private static final long PARAMETER_MASK = CONFIGURE_MASK | PARAMETERIZED;
    private static final long INIT_MASK      = PARAMETER_MASK | INITIALIZED;
    private static final long START_MASK     = INIT_MASK | STARTED;
    private static final long STOP_MASK      = START_MASK | STOPPED;
    private static final long DISPOSE_MASK   = STOP_MASK | DISPOSED;

    private final long m_mask;
    private long m_state;

    /**
     * Create state validator from object (this can be used for more than just
     * components).
     */
    public ComponentStateValidator( final Object object )
    {
        int mask = 0;

        if ( object instanceof LogEnabled ||
             object instanceof Loggable )
        {
            mask |= LOG_ENABLED;
        }

        if ( object instanceof Contextualizable )
        {
            mask |= CONTEXTUALIZED;
        }

        if ( object instanceof Parameterizable )
        {
            mask |= PARAMETERIZED;
        }

        if ( object instanceof Configurable )
        {
            mask |= CONFIGURED;
        }

        if ( object instanceof Composable )
        {
            mask |= COMPOSED;
        }

        if ( object instanceof Initializable )
        {
            mask |= INITIALIZED;
        }

        if ( object instanceof Disposable )
        {
            mask |= DISPOSED;
        }

        if ( object instanceof Startable )
        {
            mask |= STARTED | STOPPED;
        }

        if ( object instanceof Suspendable )
        {
            mask |= SUSPENDED;
        }

        m_mask = mask & ~ACTIVE;
    }

    /**
     * Throw an exception if the initialization is out of order.  It tests to see
     * if the LOG_ENABLED state has already been set, if the component implements
     * LogEnabled or Logger, and if the state has progressed beyond the Logger
     * stage.
     *
     * @throws IllegalStateException if the state is manage out of order
     */
    public void checkLogEnabled()
    {
        checkLogEnabled( LOG_FAIL );
    }

    /**
     * Throw an exception if the initialization is out of order.  It tests to see
     * if the LOG_ENABLED state has already been set, if the component implements
     * LogEnabled or Logger, and if the state has progressed beyond the Logger
     * stage.
     *
     * @param message the message to include in the thrown exception
     * @throws IllegalStateException if the state is manage out of order
     */
    public void checkLogEnabled( final String message )
    {
        boolean isValid = false;
        isValid = (m_state & m_mask & LOG_ENABLED) == 0;
        isValid = isValid && ((m_mask & LOG_MASK) | LOG_ENABLED) ==
                             ((m_state & DISPOSE_MASK) | LOG_ENABLED);

        if ( ! isValid )
        {
            throw new IllegalStateException( message );
        }

        m_state |= LOG_ENABLED;
        if ( (m_state & START_MASK) == (m_mask & START_MASK) )
        {
            m_state |= ACTIVE;
        }
    }

    /**
     * Throw an exception if the initialization is out of order.  It tests to see
     * if the CONTEXTUALIZED state has already been set, if the component implements
     * Contextualizable, and if the state has progressed beyond the Context stage.
     *
     * @throws IllegalStateException if the state is manage out of order
     */
    public void checkContextualized()
    {
        checkContextualized(CONTEXT_FAIL);
    }

    /**
     * Throw an exception if the initialization is out of order.  It tests to see
     * if the CONTEXTUALIZED state has already been set, if the component implements
     * Contextualizable, and if the state has progressed beyond the Context stage.
     *
     * @param message the message to include in the thrown exception
     * @throws IllegalStateException if the state is manage out of order
     */
    public void checkContextualized( final String message )
    {
        boolean isValid = false;
        isValid = (m_state & m_mask & CONTEXTUALIZED) == 0;
        isValid = isValid && ((m_mask & CONTEXT_MASK) | CONTEXTUALIZED) ==
                             ((m_state & DISPOSE_MASK) | CONTEXTUALIZED);

        if ( ! isValid )
        {
            throw new IllegalStateException( message );
        }

        m_state |= CONTEXTUALIZED;
        if ( (m_state & START_MASK) == (m_mask & START_MASK) )
        {
            m_state |= ACTIVE;
        }
    }

    /**
     * Throw an exception if the initialization is out of order.  It tests to see
     * if the PARAMETERIZED state has already been set, if the component implements
     * Parameterizable, and if the state has progressed beyond the Parameters stage.
     *
     * @throws IllegalStateException if the state is manage out of order
     */
    public void checkParameterized()
    {
        checkParameterized( PARAMETER_FAIL );
    }

    /**
     * Throw an exception if the initialization is out of order.  It tests to see
     * if the PARAMETERIZED state has already been set, if the component implements
     * Parameterizable, and if the state has progressed beyond the Parameters stage.
     *
     * @param message the message to include in the thrown exception
     * @throws IllegalStateException if the state is manage out of order
     */
    public void checkParameterized( final String message )
    {
        boolean isValid = false;
        isValid = (m_state & m_mask & PARAMETERIZED) == 0;
        isValid = isValid && ((m_mask & PARAMETER_MASK) | PARAMETERIZED) ==
                             ((m_state & DISPOSE_MASK) | PARAMETERIZED);

        if ( ! isValid )
        {
            throw new IllegalStateException( message );
        }

        m_state |= PARAMETERIZED;
        if ( (m_state & START_MASK) == (m_mask & START_MASK) )
        {
            m_state |= ACTIVE;
        }
    }

    /**
     * Throw an exception if the initialization is out of order.  It tests to see
     * if the CONFIGURED state has already been set, if the component implements
     * Configurable, and if the state has progressed beyond the Configuration stage.
     *
     * @throws IllegalStateException if the state is manage out of order
     */
    public void checkConfigured()
    {
        checkConfigured( CONFIGURE_FAIL );
    }

    /**
     * Throw an exception if the initialization is out of order.  It tests to see
     * if the CONFIGURED state has already been set, if the component implements
     * Configurable, and if the state has progressed beyond the Configuration stage.
     *
     * @param message the message to include in the thrown exception
     * @throws IllegalStateException if the state is manage out of order
     */
    public void checkConfigured( final String message )
    {
        boolean isValid = false;
        isValid = (m_state & m_mask & CONFIGURED) == 0;
        isValid = isValid && ((m_mask & CONFIGURE_MASK) | CONFIGURED) ==
                             ((m_state & DISPOSE_MASK) | CONFIGURED);

        if ( ! isValid )
        {
            throw new IllegalStateException( message );
        }

        m_state |= CONFIGURED;
        if ( (m_state & START_MASK) == (m_mask & START_MASK) )
        {
            m_state |= ACTIVE;
        }
    }

    /**
     * Throw an exception if the initialization is out of order.  It tests to see
     * if the COMPOSED state has already been set, if the component implements
     * Composable, and if the state has progressed beyond the Configuration stage.
     *
     * @throws IllegalStateException if the state is manage out of order
     */
    public void checkComposed()
    {
        checkComposed( COMPOSE_FAIL );
    }

    /**
     * Throw an exception if the initialization is out of order.  It tests to see
     * if the COMPOSED state has already been set, if the component implements
     * Composable, and if the state has progressed beyond the Configuration stage.
     *
     * @param message the message to include in the thrown exception
     * @throws IllegalStateException if the state is manage out of order
     */
    public void checkComposed( final String message )
    {
        boolean isValid = false;
        isValid = (m_state & m_mask & COMPOSED) == 0;
        isValid = isValid && ((m_mask & COMPOSE_MASK) | COMPOSED) ==
                             ((m_state & DISPOSE_MASK) | COMPOSED);

        if ( ! isValid )
        {
            throw new IllegalStateException( message );
        }

        m_state |= COMPOSED;
        if ( (m_state & START_MASK) == (m_mask & START_MASK) )
        {
            m_state |= ACTIVE;
        }
    }

    /**
     * Throw an exception if the initialization is out of order.  It tests to see
     * if the INITIALIZED state has already been set, if the component implements
     * Initializable, and if the state has progressed beyond the <code>initialize</code> stage.
     *
     * @throws IllegalStateException if the state is manage out of order
     */
    public void checkInitialized()
    {
        checkInitialized( INIT_FAIL );
    }

    /**
     * Throw an exception if the initialization is out of order.  It tests to see
     * if the INITIALIZED state has already been set, if the component implements
     * Initializable, and if the state has progressed beyond the <code>initialize</code> stage.
     *
     * @param message the message to include in the thrown exception
     * @throws IllegalStateException if the state is manage out of order
     */
    public void checkInitialized( final String message )
    {
        boolean isValid = false;
        isValid = (m_state & m_mask & INITIALIZED) == 0;
        isValid = isValid && ((m_mask & INIT_MASK) | INITIALIZED) ==
                             ((m_state & DISPOSE_MASK) | INITIALIZED);

        if ( ! isValid )
        {
            throw new IllegalStateException( message );
        }

        m_state |= INITIALIZED;
        if ( (m_state & START_MASK) == (m_mask & START_MASK) )
        {
            m_state |= ACTIVE;
        }
    }

    /**
     * Throw an exception if the initialization is out of order.  It tests to see
     * if the STARTED state has already been set, if the component implements
     * Startable, and if the state has progressed beyond the <code>start</code> stage.
     *
     * @throws IllegalStateException if the state is manage out of order
     */
    public void checkStarted()
    {
        checkStarted( START_FAIL );
    }

    /**
     * Throw an exception if the initialization is out of order.  It tests to see
     * if the STARTED state has already been set, if the component implements
     * Startable, and if the state has progressed beyond the <code>start</code> stage.
     *
     * @param message the message to include in the thrown exception
     * @throws IllegalStateException if the state is manage out of order
     */
    public void checkStarted( final String message )
    {
        boolean isValid = false;
        isValid = (m_state & m_mask & STARTED) == 0;
        isValid = isValid && ((m_mask & START_MASK) | STARTED) ==
                             ((m_state & DISPOSE_MASK) | STARTED);

        if ( ! isValid )
        {
            throw new IllegalStateException( message );
        }

        m_state |= STARTED;
        if ( (m_state & START_MASK) == (m_mask & START_MASK) )
        {
            m_state |= ACTIVE;
        }
    }

    /**
     * Throw an exception if the initialization is out of order.  It tests to see
     * if the SUSPENDED state has already been set, if the component implements
     * Suspendable, and if the Component is active.
     *
     * @throws IllegalStateException if the state is manage out of order
     */
    public void checkSuspended()
    {
        checkSuspended( SUSPEND_FAIL );
    }

    /**
     * Throw an exception if the initialization is out of order.  It tests to see
     * if the SUSPENDED state has already been set, if the component implements
     * Suspendable, and if the Component is active.
     *
     * @param message the message to include in the thrown exception
     * @throws IllegalStateException if the state is manage out of order
     */
    public void checkSuspended( final String message )
    {
        checkActive( message );
        boolean isValid = false;
        isValid = (m_state & m_mask & SUSPENDED) == 0;
        isValid = isValid && ((m_mask & START_MASK) & ~SUSPENDED) ==
                             ((m_state & DISPOSE_MASK) & ~SUSPENDED);

        if ( ! isValid )
        {
            throw new IllegalStateException( message );
        }

        m_state |= SUSPENDED;
    }

    /**
     * Throw an exception if the initialization is out of order.  It tests to see
     * if the SUSPENDED state has not been set, if the component implements
     * Suspendable, and if the Component is active.
     *
     * @throws IllegalStateException if the state is manage out of order
     */
    public void checkResumed()
    {
        checkResumed( RESUME_FAIL );
    }

    /**
     * Throw an exception if the initialization is out of order.  It tests to see
     * if the SUSPENDED state has not been set, if the component implements
     * Suspendable, and if the Component is active.
     *
     * @param message the message to include in the thrown exception
     * @throws IllegalStateException if the state is manage out of order
     */
    public void checkResumed( final String message )
    {
        checkActive( message );
        boolean isValid = false;
        isValid = (m_state & m_mask & SUSPENDED) > 0;
        isValid = isValid && ((m_mask & START_MASK) | SUSPENDED) ==
                             ((m_state & DISPOSE_MASK) | SUSPENDED);

        if ( ! isValid )
        {
            throw new IllegalStateException( message );
        }

        m_state &= ~SUSPENDED;
    }

    /**
     * Throw an exception if the initialization is out of order.  It tests to see
     * if the STOPPED state has not been set, if the component implements
     * Startable, and if the Component is active.
     *
     * @throws IllegalStateException if the state is manage out of order
     */
    public void checkStopped()
    {
        checkStopped( STOP_FAIL );
    }

    /**
     * Throw an exception if the initialization is out of order.  It tests to see
     * if the STOPPED state has not been set, if the component implements
     * Startable, and if the Component is active.
     *
     * @param message the message to include in the thrown exception
     * @throws IllegalStateException if the state is manage out of order
     */
    public void checkStopped( final String message )
    {
        boolean isValid = false;
        isValid = (m_state & m_mask & STOPPED) == 0;
        isValid = isValid && ((m_mask & STOP_MASK) | STOPPED) ==
                             ((m_state & DISPOSE_MASK) | STOPPED);

        if ( ! isValid )
        {
            throw new IllegalStateException( message );
        }

        m_state &= ~ACTIVE;
        m_state |= STOPPED;
    }

    /**
     * Throw an exception if the initialization is out of order.  It tests to see
     * if the DISPOSED state has not been set, if the component implements
     * Disposable.
     *
     * @throws IllegalStateException if the state is manage out of order
     */
    public void checkDisposed()
    {
        checkDisposed( DISPOSE_FAIL );
    }

    /**
     * Throw an exception if the initialization is out of order.  It tests to see
     * if the DISPOSED state has not been set, if the component implements
     * Disposable.
     *
     * @param message the message to include in the thrown exception
     * @throws IllegalStateException if the state is manage out of order
     */
    public void checkDisposed( final String message )
    {
        boolean isValid = false;
        isValid = (m_state & m_mask & DISPOSED) == 0;
        isValid = isValid && ((m_mask & DISPOSE_MASK) | DISPOSED) ==
                             ((m_state & DISPOSE_MASK) | DISPOSED);

        if ( ! isValid )
        {
            throw new IllegalStateException( message );
        }

        m_state &= ~ACTIVE;
        m_state |= DISPOSED;
    }

    /**
     * Checks to see if the state is active.
     *
     * @throws IllegalStateException if the component is not active
     */
    public void checkActive()
    {
        checkActive( ACTIVE_FAIL );
    }

    /**
     * Checks to see if the state is active.
     *
     * @param message the message to include in the thrown exception
     * @throws IllegalStateException if the component is not active
     */
    public void checkActive( final String message )
    {
        if ( isActive() )
        {
            return;
        }

        throw new IllegalStateException( message );
    }

    /**
     * Checks to see if the state is active, and returns true or false.
     *
     * @returns <code>true</code> if active, <code>false</code> if not
     */
    public boolean isActive()
    {
        if ( (ACTIVE & m_state) > 0 )
        {
            return true;
        }

        return false;
    }

    /**
     * Make sure object has not been assigned yet.
     *
     * @param object to test
     * @throws IllegalStateException if the state is manage out of order
     */
    public void checkNotAssigned( final Object object )
    {
        checkNotAssigned( object, WRITE_FAIL );
    }

    /**
     * Make sure object has not been assigned yet.
     *
     * @param object to test
     * @param message the message to include in the thrown exception
     * @throws IllegalStateException if the state is manage out of order
     */
    public void checkNotAssigned( final Object object, final String message )
    {
        if ( null != object )
        {
            throw new IllegalStateException( message );
        }
    }
}

