/**************************************************************
*   
*   Creation Date: <1998-12-02 03:23:31 samuel>
*   Time-stamp: <2001/04/18 23:35:04 samuel>
*   
*	<mmu_io.c>
*	
*	This file is responsible for mac_phys -> ????
*	translation
*   
*   Copyright (C) 1998, 1999, 2000, 2001 Samuel Rydh
*  
*   This program is free software; you can redistribute it and/or
*   modify it under the terms of the GNU General Public License
*   as published by the Free Software Foundation;
*   
**************************************************************/

#include "compat.h"
#include <linux/sys.h>
#include <linux/config.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/mm.h>
#include <linux/vmalloc.h>
#include <linux/slab.h>
#include <asm/io.h>
#include <asm/irq.h>
#include <asm/uaccess.h>
#include <asm/unistd.h>

#include "kernel_vars.h"
#include "mmu.h"
#include "misc.h"

#define PERFORMANCE_INFO
#include "performance.h"


/* allocation constants */
#define MAX_BLOCK_TRANS		10

/* Block translations are used for ROM,RAM and other big
 * buffers as VRAM etc (everything must of course be page-aligned).
 */

/* IO-translations are a different type of mappings.
 * Whenever an IO-area is accessed, a page fault occurs.
 * If there is a page present (user r/w prohibited),
 * then the low-level exception handler examines if
 * the page has a magic signature in the first 8 bytes.
 * If there is a match, then the page is of the
 * type io_page_t and contains the information necessary
 * to emulate the IO. If no page is present, then
 * the corresponding IO-page is looked up and hashed.
 */ 

typedef struct block_trans {
	ulong 	mbase;
	char	*lvbase;
	size_t	size;
	int	flags;

	int	id;
} block_trans_t;

typedef struct io_data {
	block_trans_t	btable[MAX_BLOCK_TRANS];
	int		num_btrans;
	int		next_free_id;
	io_page_t 	*io_page_head;
} io_data_t;

/* shared scratch page for "uninteresting" mappings (all sessions use the same page) */
static char *scratch_page=0;

static int bat_align( int flags, ulong ea, ulong lphys, ulong size, ulong bat[2] );

#define MMU 		(kv->mmu)
#define DECLARE_IOD	io_data_t *iod = kv->mmu.io_data


int
init_mmu_io( kernel_vars_t *kv )
{
	if( !(MMU.io_data = kmalloc( sizeof(io_data_t), GFP_KERNEL ) ))
		return 1;
	memset( MMU.io_data, 0, sizeof(io_data_t) );
	return 0;
}

void 
cleanup_mmu_io( kernel_vars_t *kv )
{
	DECLARE_IOD;
	io_page_t *next2, *p2;

	if( !iod )
		return;

	for( p2=iod->io_page_head; p2; p2=next2 ){
		next2 = p2->next;
		free_page( (ulong)p2 );
	}
	iod->io_page_head = 0;

	/* release the scratch page (not always allocated) */
	if( !g_num_sessions && scratch_page ) {
		free_page( (int)scratch_page );
		scratch_page = 0;
	}

	kfree( iod );
	MMU.io_data = NULL;
}


/*
 * Handle block translations (translations of mac-physical
 * blocks to linux virtual physical addresses)
 */
int
add_block_trans( kernel_vars_t *kv, ulong mbase, char *lvbase, ulong size, int flags )
{
	DECLARE_IOD;
	block_trans_t *bt;
	
	/* we keep an unsorted list - RAM should be added first,
	 * then ROM, then VRAM etc */

	if( iod->num_btrans >= MAX_BLOCK_TRANS ){
		printk("Maximal number of block translations exceeded!\n");
		return -1;
	}

	/* TRANSL_IO flag must not be set for block translations */
	flags &= ~TRANSL_IO;

	/* scratch pages are always physical - lvbase isn't used */
	if( flags & TRANSL_SCRATCH ) {
		lvbase = NULL;
		flags |= TRANSL_PHYSICAL;
		flags &= ~TRANSL_DBAT;
	}

	/* IMPORTANT: DBATs can _only_ be used when we KNOW that ea == mphys. */
	if( flags & TRANSL_DBAT ){
		/* dbat mappings must always be physical. */
		if( !(flags & TRANSL_PHYSICAL) ) {
			printk("Error: TRANSL_DBAT set but not TRANSL_PHYSICAL\n");
			flags &= ~TRANSL_DBAT;
		}
		if( flags & TRANSL_DBAT ){
			ulong bat[2];
			if( !bat_align( flags, mbase, (ulong)lvbase, size, bat ) ) {
				/* printk("BATS: %08lX %08lX\n", bat[0], bat[1] ); */
				MMU.transl_dbat0.word[0] = bat[0];
				MMU.transl_dbat0.word[1] = bat[1];
			}
		}
	}

	bt = &iod->btable[iod->num_btrans++];
	bt->mbase = mbase;
	bt->lvbase = lvbase;
	bt->size = size;
	bt->flags = flags | TRANSL_VALID;
	bt->id = iod->next_free_id++;

	return bt->id;
}


void 
remove_block_trans( kernel_vars_t *kv, int id )
{
	DECLARE_IOD;
	block_trans_t *p;
	int i;

	/* Remove all mappings in the TLB table... 
	 * (too difficult to find the entries we need to flush)
	 */
	BUMP(remove_block_trans);
	clear_tlb_table( kv );

	for(p=iod->btable, i=0; i<iod->num_btrans; i++,p++ ){
		if( id == p->id ) {
			if( p->flags & TRANSL_DBAT ) {
				MMU.transl_dbat0.word[0] = 0;
				MMU.transl_dbat0.word[1] = 0;
			}
			memmove( p,p+1, (iod->num_btrans-1-i)*sizeof(block_trans_t)  );
			iod->num_btrans--;
			return;
		}
	}
	printk("Trying to remove nonexistent block mapping!\n");
}

/* This is primarily intended for framebuffers */
static int
bat_align( int flags, ulong ea, ulong lphys, ulong size, ulong bat[2] )
{
	ulong s;
	ulong offs1, offs2;

	s=0x20000;	/* 128K */
	if( s> size )
		return 1;
	/* Limit to 128MB in order not to cross segments (256MB is bat-max) */
	if( size > 0x10000000 )
		size = 0x10000000;
	for( ; s<size ; s = (s<<1) )
		;
	offs1 = ea & (s-1);
	offs2 = lphys & (s-1);
	if( offs1 != offs2 ) {
		printk("Can't use DBAT since offsets differ (%ld != %ld)\n", offs1, offs2 );
		return 1;
	}
	/* BEPI | BL | VS | VP */
	bat[0] = (ea & ~(s-1)) | (((s-1)>>17) << 2) | 3;
	bat[1] = (lphys & ~(s-1)) | 2;	/* pp=10, R/W */

	bat[1] |= BIT(27);		/* [M] (memory coherence) */
	if( !(flags & TRANSL_FORCE_CACHE) ){
		bat[1] |= BIT(26);	/* [I] (inhibit cache) */
	} else {
		bat[1] |= BIT(25);	/* [W] (write through) */
	}
	return 0;
}



/* Adds an I/O-translation. It is legal to add the same
 * range multiple times (for instance, to alter usr_data)
 */
int 
add_io_trans( kernel_vars_t *kv, ulong mbase, int size, void *usr_data )
{
	DECLARE_IOD;
	io_page_t 	*ip, **pre_next;
	ulong		mb;
	int 		i, num;
	
	/* align mbase and size to double word boundarys */
	size += mbase & 7;
	mbase -= mbase & 7;
	size = (size+7) & ~7;

	while( size>0 ){
		mb = mbase & 0xfffff000;

		pre_next = &iod->io_page_head;
		for( ip=iod->io_page_head; ip && ip->mphys < mb; ip=ip->next )
			pre_next = &ip->next;

		if( !ip || ip->mphys != mb ) {
			/* create new page */
			ip = (io_page_t*)get_free_page( GFP_KERNEL );
			if( !ip ) {
				printk("Failed allocating IO-page\n");
				return 1;
			}
			ip->next = *pre_next;
			*pre_next = ip;

			/* setup block */
			ip->magic = IO_PAGE_MAGIC_1;
			ip->magic2 = IO_PAGE_MAGIC_2;
			ip->me_phys = virt_to_phys(ip);
			ip->mphys = mb;
		}
		/* Fill in IO */
		num = size>>3;
		i = (mbase & 0xfff) >> 3;
		if( i+num > 512 )
			num = 512-i;
		mbase += num<<3;
		size -= num<<3;
		while( num-- )
			ip->usr_data[i++] = usr_data;
	}
	return 0;
}

int 
remove_io_trans( kernel_vars_t *kv, ulong mbase, int size )
{
	DECLARE_IOD;
	io_page_t 	*ip, **pre_next;
	ulong		mb;
	int 		i, num;

	/* To remove an unused IO-page, we must make sure there are no
	 * dangling references to it. Hence we must search the PTE hash 
	 * table and remove all references. We must also issue a 
	 * tlbia to make sure it is not in the on-chip DTLB/ITLB cashe.
	 *
	 * XXX: Instead of seraching the hash, we simply make sure the magic 
	 * constants are invalid. This is perfectly safe since the exception 
	 * handler doesn't write to the page in question - and the physical 
	 * page always exists even if it is allocated by somebody else. 
	 * It is better to make sure there are no references of it though.
	 */

	/* align mbase and size to double word boundarys */
	size += mbase & 7;
	mbase -= mbase & 7;
	size = (size+7) & ~7;

	while( size>0 ){
		mb = mbase & 0xfffff000;

		pre_next = &iod->io_page_head;
		for( ip=iod->io_page_head; ip && ip->mphys < mb; ip=ip->next )
			pre_next = &ip->next;

		if( !ip || ip->mphys != mb ) {
			/* no page... */
			size -= 0x1000 - (mbase & 0xfff);
			mbase += 0x1000 - (mbase & 0xfff);
			continue;
		}
		/* Clear IO */
		num = size>>3;
		i = (mbase & 0xfff) >> 3;
		if( i+num > 512 )
			num = 512-i;
		mbase += num<<3;
		size -= num<<3;
		while( num-- )
			ip->usr_data[i++] = 0;

		/* May we free the page? */
		for(i=0; i<512 && !ip->usr_data[i]; i++ )
			;
		if( i==512 ){
			/* Free page */
			/* XXX: Remove page fram hash, see above */
			*pre_next = ip->next;
			ip->magic2 = ip->magic = 0;	/* IMPORTANT */
			free_page( (ulong)ip );
		}
	}
	return 0;
	
}


/* Translate a mac-physical address (32 bit, not page-index) 
 * and fill in rpn (and _possibly_ other fields) of the pte.
 * The WIMG bits are not modified after this call. 
 * The calling function is not supposed to alter the pte after
 * this function call.
 *
 *	Retuns:
 *	  0 		no translation found
 *	  block_flags	translation found
 */

int 
mphys_to_pte( kernel_vars_t *kv, ulong mphys, PTE *pte, int is_write )
{
	DECLARE_IOD;
	block_trans_t 	*p;
	io_page_t	*p2;
	int 		i, num_btrans;
	
	num_btrans = iod->num_btrans;
	mphys &= 0xfffff000;

	/* Check for a block mapping. */
	for(p=iod->btable, i=0; i<num_btrans; i++,p++ ){
		if( mphys >= p->mbase && mphys <= p->mbase + p->size - 1 ) {
			if( (p->flags & TRANSL_SCRATCH) ) {
				if( !scratch_page ) {
					scratch_page = (char*)get_free_page( GFP_KERNEL );
					if( !scratch_page )
						continue;
				}
				pte->rpn = virt_to_phys(scratch_page) >> 12;
			} else
				pte->rpn = (mphys - p->mbase + (ulong)p->lvbase)>>12;
			if( p->flags & TRANSL_FORCE_CACHE ){
				pte->w = 1;	/* Use write through for now */
				pte->i = 0;
			} else if( !(p->flags & TRANSL_MACOS_CONTROLS_CACHE) ){
				pte->w = 0;
				pte->i = 0;
			}
			return p->flags;
		}
	}

	/* Check for an I/O mapping. */
	for( p2=iod->io_page_head; p2 && p2->mphys<=mphys; p2=p2->next ) {
		if( p2->mphys != mphys )
			continue;
		pte->rpn = p2->me_phys >> 12;
		pte->pp = 0;	/* supervisor R/W */
		pte->w = 0;
		pte->i = 0;
		return TRANSL_VALID | TRANSL_IO | TRANSL_PHYSICAL;
	}
	return 0;
}
