/* 
 *   Creation Date: <2001/05/06 22:47:23 samuel>
 *   Time-stamp: <2001/05/24 03:03:12 samuel>
 *   
 *	<hfs_fs.c>
 *	
 *	
 *   
 *   Copyright (C) 2001 Samuel Rydh (samuel@ibrium.se)
 *   
 *   This program is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU General Public License
 *   as published by the Free Software Foundation
 *   
 */

#include "mol_config.h"
#include "fs.h"

#include "libhfs.h"

#define MAC_OS_ROM_CREATOR	0x63687270	/* 'chrp' */
#define MAC_OS_ROM_TYPE		0x74627869	/* 'tbxi' */
#define MAC_OS_ROM_NAME		"Mac OS ROM"

#define FINDER_TYPE		0x464E4452	/* 'FNDR' */
#define FINDER_CREATOR		0x4D414353	/* 'MACS' */
#define SYSTEM_TYPE		0x7A737973	/* 'zsys' */
#define SYSTEM_CREATOR		0x4D414353	/* 'MACS' */


/************************************************************************/
/*	File System Interface						*/
/************************************************************************/

static void		close_fs( fs_ops_t *fs );
static file_desc_t 	open_path( fs_ops_t *fs, char *path );
static file_desc_t	search_rom( fs_ops_t *fs );
static file_desc_t 	search_file( fs_ops_t *fs, char *name );

static void		file_close( file_desc_t fd);
static int		file_read( file_desc_t fd, void *buf, size_t count );
static int		file_lseek( file_desc_t fd, off_t offset, int whence );
static char		*get_path( file_desc_t fd, char *buf, int length );

static fs_ops_t hfs_ops = {
	close_fs:	close_fs,
	open_path:	open_path,
	search_rom:	search_rom,
	search_file:	search_file,

	get_path:	get_path,
	close:		file_close,
	read:		file_read,
	lseek:		file_lseek
};


/************************************************************************/
/*	File System Operations						*/
/************************************************************************/

fs_ops_t *
fs_hfs_open( unsigned long long offs, char *path )
{
	hfsvol *vol = hfs_mount( path, 0, offs );
	fs_ops_t *fs;

	if( !vol )
		return NULL;

	fs = malloc( sizeof(fs_ops_t) );
	*fs = hfs_ops;
	fs->fs_data = vol;
	fs->dev_name = strdup( path );
	
	return fs;
}

static void
close_fs( fs_ops_t *fs )
{
	hfsvol *vol = (hfsvol*)fs->fs_data;

	hfs_umount( vol );
	free( fs->dev_name );
	free( fs );
}

static file_desc_t
open_path( fs_ops_t *fs, char *path )
{
	char *s, buf[256];
	hfsvol *vol = (hfsvol*)fs->fs_data;

	buf[0] = 0;
	if( path[0] != ':' && path[0] != '/' )
		strcpy( buf, ":" );
	strncat( buf, path, sizeof(buf) );
	buf[sizeof(buf)-1] = 0;

	s=buf;
	while( (s=strchr(s,'/')) )
		*s++ = ':';

	return hfs_open( vol, buf );
}


/************************************************************************/
/*	Search Functions						*/
/************************************************************************/

static int
_find_file( hfsvol *vol, char *path, ulong type, ulong creator )
{
	hfsdir 		*dir;
	hfsdirent	ent;
	int		ret=1;
	
	if( !(dir=hfs_opendir( vol, path )) )
		return 1;

	/* We make the rather safe assumption that directories will not match */
	while( ret && !hfs_readdir( dir, &ent) )
		ret = !(*(ulong*)ent.u.file.type == type && *(ulong*)ent.u.file.creator == creator );

	hfs_closedir( dir );
	return ret;
}


/* ret: 0=success, 1=not_found, 2=not_a_dir */
static int
_search( hfsvol *vol, char *path, char *sname, file_desc_t *ret_fd )
{
	hfsdir 		*dir;
	hfsdirent	ent;
	int		topdir=0, status = 1;
	char		*p, buf[256];

	strncpy( buf, path, sizeof(buf) );
	if( buf[strlen(buf)-1] != ':' )
		strncat( buf, ":", sizeof(buf) );
	buf[sizeof(buf)-1] = 0;
	p = buf + strlen( buf );

	if( !(dir=hfs_opendir( vol, path )) )
		return 2;

	/* printf("DIRECTORY: %s\n", path ); */

	while( status && !hfs_readdir( dir, &ent) ){
		ulong type, creator;
		
		*p = 0;
		topdir = 0;
		
		strncat( buf, ent.name, sizeof(buf) );
		if( (status=_search( vol, buf, sname, ret_fd)) != 2 )
			continue;
		topdir = 1;
		
		/* Name search? */
		if( sname ){
			status = strcasecmp( ent.name, sname );
			continue;
		}
		
		type = *(ulong*)ent.u.file.type;
		creator = *(ulong*)ent.u.file.creator;
		
		/* Look for Mac OS ROM, System and Finder in the same directory */
		if( type == MAC_OS_ROM_TYPE && creator == MAC_OS_ROM_CREATOR ) {
			if( strcasecmp( ent.name, MAC_OS_ROM_NAME ) )
				continue;

			status = _find_file( vol, path, FINDER_TYPE, FINDER_CREATOR )
				|| _find_file( vol, path, SYSTEM_TYPE, SYSTEM_CREATOR );
		}
	}
	if( !status && topdir && ret_fd && !(*ret_fd = hfs_open(vol, buf) )) {
		printf("Unexpected error: failed to open matched ROM\n");
		status = 1;
	}

	hfs_closedir( dir );
	return status;
}

static file_desc_t
_do_search( fs_ops_t *fs, char *sname ) 
{
	hfsvol *vol = hfs_getvol( NULL );
	file_desc_t ret_fd = NULL;
	
	(void)_search( vol, ":", sname, &ret_fd );
	return ret_fd;
}

static file_desc_t
search_rom( fs_ops_t *fs )
{
	return _do_search( fs, NULL );
}

static file_desc_t
search_file( fs_ops_t *fs, char *sname )
{
	return _do_search( fs, sname );
}


/************************************************************************/
/*	File Operations							*/
/************************************************************************/

static void
file_close( file_desc_t fd )
{
	hfsfile *file = (hfsfile*)fd;
	hfs_close( file );
}

static int
file_lseek( file_desc_t fd, off_t offs, int whence )
{
	hfsfile *file = (hfsfile*)fd;

	switch( whence ){
	case SEEK_CUR:
		whence = HFS_SEEK_CUR;
		break;
	case SEEK_END:
		whence = HFS_SEEK_END;
		break;
	default:
	case SEEK_SET:
		whence = HFS_SEEK_SET;
		break;
	}

	return hfs_seek( file, offs, whence );
}

static int
file_read( file_desc_t fd, void *buf, size_t count )
{
	hfsfile *file = (hfsfile*)fd;
	return hfs_read( file, buf, count );
}


static void
_compose( hfsvol *vol, ulong id, char *buf, int len )
{
	char buf2[256];
	buf[0] = 0;
	
	if( hfs_dirinfo( vol, &id, buf2 ) )
		return;
	_compose( vol, id, buf, len );

	strncat( buf, ":", len );
	strncat( buf, buf2, len );
}


static char *
get_path( file_desc_t fd, char *retbuf, int len )
{
	char buf[256], *s;
	hfsvol *vol = hfs_getvol( NULL );
	hfsfile *file = (hfsfile*)fd;
	hfsdirent ent;
	
	hfs_fstat( file, &ent );
	buf[0] = 0;

	strncpy( buf, ent.name, sizeof(buf) );
	_compose( vol, ent.parid, buf, sizeof(buf) );
	strncat( buf, ":", sizeof( buf ) );
	strncat( buf, ent.name, sizeof( buf ) );

	buf[sizeof(buf)-1] = 0;
	
	if( (s=strchr(buf, ':')) && (s=strchr(s+1,':')) ) {
		s++;
	}
	memmove( buf, s, strlen(s) );
	buf[ strlen(s) ] = 0;

	strncpy( retbuf, buf, len );
	retbuf[len-1] = 0;
	
	return retbuf;
}
